// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeAuth);
} else {
    initializeAuth();
}

function initializeAuth() {
    // Get form elements
    const loginForm = document.getElementById('vigore-login-form');
    const signupForm = document.getElementById('vigore-signup-form');
    const logoutButton = document.getElementById('vigore-logout');

    // Handle login form submission
    if (loginForm) {
        loginForm.addEventListener('submit', function (e) {
            e.preventDefault();

            const email = document.getElementById('vigore-email').value;
            const password = document.getElementById('vigore-password').value;

            authenticateUser(email, password, 'login');
        });
    }

    // Handle signup form submission
    if (signupForm) {
        signupForm.addEventListener('submit', function (e) {
            e.preventDefault();

            const email = document.getElementById('vigore-signup-email').value;
            const password = document.getElementById('vigore-signup-password').value;

            authenticateUser(email, password, 'signup');
        });
    }

    // Handle form switching
    const switchToSignup = document.querySelector('.vigore-switch-to-signup');
    const switchToLogin = document.querySelector('.vigore-switch-to-login');

    if (switchToSignup) {
        switchToSignup.addEventListener('click', function (e) {
            e.preventDefault();
            document.querySelector('.vigore-auth-box:first-child').style.display = 'none';
            document.querySelector('.vigore-auth-box:last-child').style.display = 'block';
        });
    }

    if (switchToLogin) {
        switchToLogin.addEventListener('click', function (e) {
            e.preventDefault();
            document.querySelector('.vigore-auth-box:first-child').style.display = 'block';
            document.querySelector('.vigore-auth-box:last-child').style.display = 'none';
        });
    }

    // Handle logout
    if (logoutButton) {
        logoutButton.addEventListener('click', handleLogout);
    }

    // Handle Google login
    const googleLoginBtn = document.getElementById('google-login-btn');
    if (googleLoginBtn) {
        googleLoginBtn.addEventListener('click', handleGoogleLogin);
    }
}

// Function to handle authentication
function authenticateUser(email, password, authType) {
    // Show loading state
    const submitButton = document.querySelector(`#vigore-${authType}-form button[type="submit"]`);
    if (!submitButton) {
        return;
    }

    const originalText = submitButton.textContent;
    submitButton.disabled = true;
    // submitButton.textContent = 'Authenticating...';
    submitButton.innerHTML = `<svg class="_loader_1mcx7_1" data-testid="loader" width="25" height="25" viewBox="0 0 25 25" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M25 12.5C25 14.3525 24.5882 16.1819 23.7945 17.8558C23.0007 19.5297 21.8448 21.0062 20.4104 22.1786C18.976 23.3509 17.299 24.1898 15.5006 24.6345C13.7022 25.0792 11.8275 25.1186 10.012 24.7499C8.19653 24.3812 6.48571 23.6135 5.00331 22.5025C3.5209 21.3914 2.30398 19.9648 1.44059 18.3258C0.577189 16.6867 0.0889003 14.8762 0.0110439 13.0253C-0.0668125 11.1744 0.26771 9.32935 0.990413 7.62359L3.55802 8.71144C2.99654 10.0367 2.73665 11.4701 2.79714 12.9081C2.85762 14.3461 3.23698 15.7527 3.90777 17.0261C4.57856 18.2995 5.524 19.4079 6.6757 20.2711C7.8274 21.1343 9.15657 21.7307 10.567 22.0171C11.9775 22.3036 13.434 22.273 14.8312 21.9275C16.2284 21.582 17.5313 20.9303 18.6457 20.0194C19.7602 19.1086 20.6582 17.9615 21.2749 16.661C21.8915 15.3605 22.2114 13.9393 22.2114 12.5H25Z" fill="white" fill-opacity="0.5"></path><path d="M25 12.5C25 9.6167 24.0032 6.82206 22.1786 4.58956C20.3539 2.35706 17.8136 0.823994 14.988 0.250105C12.1624 -0.323783 9.22526 0.0967964 6.67425 1.44059C4.12324 2.78438 2.11523 4.96874 0.990413 7.62359L3.58992 8.72495C4.46069 6.66972 6.01518 4.9787 7.99003 3.93842C9.96488 2.89813 12.2386 2.57254 14.4261 3.01681C16.6135 3.46109 18.5801 4.6479 19.9926 6.37618C21.4052 8.10445 22.1768 10.2679 22.1768 12.5H25Z" fill="white"></path></svg>`;


    // Make API request
    const formData = new FormData();
    formData.append('action', 'vigore_api_login');
    formData.append('nonce', vigoreAuth.nonce);
    formData.append('email', email);
    formData.append('password', password);

    fetch(vigoreAuth.ajaxurl, {
        method: 'POST',
        body: formData
    })
        .then(response => {
            return response.json();
        })
        .then(response => {
            if (response.success && response.data && response.data.access_token) {
                // Store the token and user info
                const storeData = new FormData();
                storeData.append('action', 'vigore_store_auth');
                storeData.append('nonce', vigoreAuth.nonce);
                storeData.append('access_token', response.data.access_token);
                storeData.append('email', email);
                storeData.append('organization_id', response.data.organization_id || '');
                // Extract firstName from the original API response if available
                if (response.data.first_name) {
                    storeData.append('first_name', response.data.first_name);
                } else if (response.data.firstName) {
                    storeData.append('first_name', response.data.firstName);
                }
                return fetch(vigoreAuth.ajaxurl, {
                    method: 'POST',
                    body: storeData
                });
            } else {
                throw new Error(response.data || 'Invalid credentials');
            }
        })
        .then(response => response.json())
        .then(() => {
            window.location.href = vigoreAuth.settingsUrl;
        })
        .catch(error => {
            // Show error in a more user-friendly way
            const container = document.querySelector('#notice-error-place');
            if (container) {
                const errorDiv = document.createElement('div');
                errorDiv.className = 'error-message-response';
                errorDiv.innerHTML = `<p>${error.message}</p>`;
                container.insertBefore(errorDiv, container.firstChild);
            } else {
                alert(error.message || 'Authentication failed');
            }
        })
        .finally(() => {
            // Reset button state
            if (submitButton) {
                submitButton.disabled = false;
                submitButton.textContent = originalText;
            }
        });
}

// Handle logout
function handleLogout(e) {
    e.preventDefault();

    const formData = new FormData();
    formData.append('action', 'vigore_logout');
    formData.append('nonce', vigoreAuth.nonce);

    fetch(vigoreAuth.ajaxurl, {
        method: 'POST',
        body: formData
    })
        .then(response => response.json())
        .then(() => {
            window.location.reload();
        })
        .catch(error => {
            alert('Failed to logout. Please try again.');
        });
}

// Handle Google login
function handleGoogleLogin() {
    // Generate a nonce for state parameter
    const nonce = Math.random().toString(36).substring(2, 15);

    // Store nonce in sessionStorage for verification
    sessionStorage.setItem('vigore_oauth_nonce', nonce);

    // Get current origin for return_origin parameter
    const returnOrigin = window.location.origin;

    // Construct the OAuth URL
    const authUrl = `https://vigore.app/api/auth/google/login?state=${nonce}&return_origin=${encodeURIComponent(returnOrigin)}&fromWidget=1`;

    // Open popup window in full screen
    const popup = window.open(
        authUrl,
        'Vigore Google Login',
        `width=${window.screen.availWidth},height=${window.screen.availHeight},left=0,top=0`
    );

    if (!popup) {
        return;
    }

    // Listen for postMessage from popup
    const messageHandler = function (event) {
        // Verify origin
        if (event.origin !== 'https://vigore.app') {
            return;
        }

        // Verify nonce
        const storedNonce = sessionStorage.getItem('vigore_oauth_nonce');
        if (!storedNonce) {
            return;
        }

        // Handle Google login failure
        if (event.data && event.data.isLoginSuccess === false) {
            const container = document.querySelector('#notice-error-place');
            if (container) {
                container.innerHTML = '';
                const errorDiv = document.createElement('div');
                errorDiv.className = 'error-message-response';
                errorDiv.innerHTML = `<p>Login failed. Please try again.</p>`;
                container.insertBefore(errorDiv, container.firstChild);
            }
            // Clean up
            sessionStorage.removeItem('vigore_oauth_nonce');
            if (popup) popup.close();
            window.removeEventListener('message', messageHandler);
            return;
        }

        // Clear stored nonce
        sessionStorage.removeItem('vigore_oauth_nonce');

        // Close popup
        if (popup) {
            popup.close();
        }

        // Remove the message listener
        window.removeEventListener('message', messageHandler);

        // Handle the auth data
        if (event.data && event.data.accessToken && event.data.orgId) {
            // Store the auth data
            const storeData = new FormData();
            storeData.append('action', 'vigore_store_auth');
            storeData.append('nonce', vigoreAuth.nonce);
            storeData.append('access_token', event.data.accessToken);
            storeData.append('organization_id', event.data.orgId);
            storeData.append('first_name', event.data.firstName);

            fetch(vigoreAuth.ajaxurl, {
                method: 'POST',
                body: storeData
            })
                .then(response => response.json())
                .then(() => {
                    window.location.href = vigoreAuth.settingsUrl;
                })
                .catch(error => {
                    const container = document.querySelector('#notice-error-place');
                    if (container) {
                        const errorDiv = document.createElement('div');
                        errorDiv.className = 'error-message-response';
                        errorDiv.innerHTML = `<p>Failed to store authentication data. Please try again.</p>`;
                        container.insertBefore(errorDiv, container.firstChild);
                    }
                });
        }
    };

    window.addEventListener('message', messageHandler);

    // Check if popup is closed manually
    const checkClosed = setInterval(() => {
        if (popup.closed) {
            clearInterval(checkClosed);
            window.removeEventListener('message', messageHandler);
            sessionStorage.removeItem('vigore_oauth_nonce');
        }
    }, 1000);
} 