<?php
/**
 * Validation Handler Class
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Validation handler for booking slots
 */
class BSFEF_Validation {

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 * @var \BSFEF_Validation The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 * @return \BSFEF_Validation An instance of the class.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function init_hooks() {
		// Add form validation hooks
		add_action( 'elementor_pro/forms/validation', array( $this, 'validate_booking_slots' ), 10, 2 );

		// Add custom validation messages
		add_filter( 'elementor_pro/forms/validation/error_messages', array( $this, 'add_custom_error_messages' ) );

		// Pre-process form submission
		add_action( 'elementor_pro/forms/process', array( $this, 'pre_process_booking_slots' ), 5, 2 );
	}

	/**
	 * Validate booking slots in form submission
	 *
	 * @since 1.0.0
	 * @access public
	 * @param array                                            $record Form record
	 * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler Ajax handler
	 */
	public function validate_booking_slots( $record, $ajax_handler ) {
		$form_data = $record->get( 'fields' );

		foreach ( $form_data as $field_id => $field ) {
			if ( $field['type'] !== 'booking_slots' ) {
				continue;
			}

			$this->validate_booking_slot_field( $field, $field_id, $ajax_handler, $record );
		}
	}

	/**
	 * Validate individual booking slot field
	 *
	 * @since 1.0.0
	 * @access private
	 * @param array                                            $field Field data
	 * @param string                                           $field_id Field ID
	 * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler Ajax handler
	 * @param array                                            $record Form record
	 */
	private function validate_booking_slot_field( $field, $field_id, $ajax_handler, $record ) {
		$value       = $field['value'];
		$is_required = ! empty( $field['required'] );

		// Check if field is required and empty
		if ( $is_required && empty( $value ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'required' ) );
			return;
		}

		// If field is empty and not required, skip validation
		if ( empty( $value ) ) {
			return;
		}

		// Validate format (should be "date|time")
		if ( ! $this->validate_slot_format( $value ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'invalid_format' ) );
			return;
		}

		// Parse date and time
		list($date, $time) = explode( '|', $value );

		// Validate date
		if ( ! $this->validate_date( $date ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'invalid_date' ) );
			return;
		}

		// Validate time
		if ( ! $this->validate_time( $time ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'invalid_time' ) );
			return;
		}

		// Check if date is not in the past
		if ( ! $this->validate_future_date( $date ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'past_date' ) );
			return;
		}

		// Check business rules (weekends, holidays, etc.)
		if ( ! $this->validate_business_rules( $date, $time ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'business_rules' ) );
			return;
		}

		// Check if slot is available
		if ( ! $this->validate_slot_availability( $date, $time ) ) {
			$ajax_handler->add_error( $field_id, $this->get_error_message( 'slot_unavailable' ) );
			return;
		}

		// Additional custom validations
		$this->run_custom_validations( $date, $time, $field, $field_id, $ajax_handler, $record );
	}

	/**
	 * Pre-process booking slots before form submission
	 *
	 * @since 1.0.0
	 * @access public
	 * @param array                                            $record Form record
	 * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler Ajax handler
	 */
	public function pre_process_booking_slots( $record, $ajax_handler ) {
		$form_data = $record->get( 'fields' );

		foreach ( $form_data as $field_id => $field ) {
			if ( $field['type'] !== 'booking_slots' || empty( $field['value'] ) ) {
				continue;
			}

			// Reserve the slot temporarily
			list($date, $time) = explode( '|', $field['value'] );

			// Double-check availability just before booking
			if ( ! BSFEF_Database::instance()->is_slot_available( $date, $time ) ) {
				$ajax_handler->add_error( $field_id, $this->get_error_message( 'slot_unavailable' ) );
				return;
			}

			// Book the slot
			$booking_data = array(
				'form_id'         => $record->get_form_settings( 'id' ),
				'user_ip'         => $this->get_user_ip(),
				'user_agent'      => $this->get_user_agent(),
				'submission_data' => $this->get_sanitized_submission_data( $record ),
			);

			$booking_id = BSFEF_Database::instance()->book_slot( $date, $time, $booking_data );

			if ( ! $booking_id ) {
				$ajax_handler->add_error( $field_id, $this->get_error_message( 'booking_failed' ) );
				return;
			}

			// Store booking ID in record for later use
			$record->set_field_value( $field_id . '_booking_id', $booking_id );
		}
	}

	/**
	 * Validate slot format
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $value Slot value
	 * @return bool
	 */
	private function validate_slot_format( $value ) {
		if ( ! is_string( $value ) ) {
			return false;
		}

		$parts = explode( '|', $value );
		return count( $parts ) === 2 && ! empty( $parts[0] ) && ! empty( $parts[1] );
	}

	/**
	 * Validate date format and value
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $date Date string
	 * @return bool
	 */
	private function validate_date( $date ) {
		if ( ! is_string( $date ) ) {
			return false;
		}

		// Check format (Y-m-d)
		if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
			return false;
		}

		// Check if it's a valid date
		$date_parts = explode( '-', $date );
		return checkdate( $date_parts[1], $date_parts[2], $date_parts[0] );
	}

	/**
	 * Validate time format and value
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $time Time string
	 * @return bool
	 */
	private function validate_time( $time ) {
		if ( ! is_string( $time ) ) {
			return false;
		}

		// Check format (H:i)
		return preg_match( '/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $time );
	}

	/**
	 * Validate that date is not in the past
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $date Date string
	 * @return bool
	 */
	private function validate_future_date( $date ) {
		$today = current_time( 'Y-m-d' );
		return strtotime( $date ) >= strtotime( $today );
	}

	/**
	 * Validate business rules (weekends, holidays, etc.)
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $date Date string
	 * @param string $time Time string
	 * @return bool
	 */
	private function validate_business_rules( $date, $time ) {
		// Get day of week (0 = Sunday, 6 = Saturday)
		$day_of_week = gmdate( 'w', strtotime( $date ) );

		// Check if weekends are disabled (can be made configurable)
		$disable_weekends = apply_filters( 'bsfef_disable_weekends', true );
		if ( $disable_weekends && ( $day_of_week == 0 || $day_of_week == 6 ) ) {
			return false;
		}

		// Check business hours (can be made configurable)
		$business_start = apply_filters( 'bsfef_business_start', '09:00' );
		$business_end   = apply_filters( 'bsfef_business_end', '17:00' );

		if ( strtotime( $time ) < strtotime( $business_start ) || strtotime( $time ) > strtotime( $business_end ) ) {
			return false;
		}

		// Check holidays (can be extended)
		$holidays = apply_filters( 'bsfef_holidays', array() );
		if ( in_array( $date, $holidays ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Validate slot availability
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $date Date string
	 * @param string $time Time string
	 * @return bool
	 */
	private function validate_slot_availability( $date, $time ) {
		return BSFEF_Database::instance()->is_slot_available( $date, $time );
	}

	/**
	 * Run custom validations
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string                                           $date Date string
	 * @param string                                           $time Time string
	 * @param array                                            $field Field data
	 * @param string                                           $field_id Field ID
	 * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler Ajax handler
	 * @param array                                            $record Form record
	 */
	private function run_custom_validations( $date, $time, $field, $field_id, $ajax_handler, $record ) {
		// Allow custom validation hooks
		$is_valid = apply_filters( 'bsfef_custom_validation', true, $date, $time, $field, $record );

		if ( ! $is_valid ) {
			$custom_message = apply_filters(
				'bsfef_custom_validation_message',
				$this->get_error_message( 'custom_validation' ),
				$date,
				$time,
				$field,
				$record
			);
			$ajax_handler->add_error( $field_id, $custom_message );
		}
	}

	/**
	 * Add custom error messages
	 *
	 * @since 1.0.0
	 * @access public
	 * @param array $messages Existing error messages
	 * @return array
	 */
	public function add_custom_error_messages( $messages ) {
		$custom_messages = array(
			'bsfef_required'          => __( 'Please select a booking slot.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_invalid_format'    => __( 'Invalid booking slot format.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_invalid_date'      => __( 'Invalid date selected.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_invalid_time'      => __( 'Invalid time selected.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_past_date'         => __( 'Cannot book slots in the past.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_business_rules'    => __( 'Selected date/time is not available for booking.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_slot_unavailable'  => __( 'This time slot is no longer available. Please select another time.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_booking_failed'    => __( 'Failed to complete booking. Please try again.', 'booking-slots-for-elementor-forms-lite' ),
			'bsfef_custom_validation' => __( 'This booking slot is not valid.', 'booking-slots-for-elementor-forms-lite' ),
		);

		return array_merge( $messages, $custom_messages );
	}

	/**
	 * Get error message
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $key Error key
	 * @return string
	 */
	private function get_error_message( $key ) {
		$messages = array(
			'required'          => __( 'Please select a booking slot.', 'booking-slots-for-elementor-forms-lite' ),
			'invalid_format'    => __( 'Invalid booking slot format.', 'booking-slots-for-elementor-forms-lite' ),
			'invalid_date'      => __( 'Invalid date selected.', 'booking-slots-for-elementor-forms-lite' ),
			'invalid_time'      => __( 'Invalid time selected.', 'booking-slots-for-elementor-forms-lite' ),
			'past_date'         => __( 'Cannot book slots in the past.', 'booking-slots-for-elementor-forms-lite' ),
			'business_rules'    => __( 'Selected date/time is not available for booking.', 'booking-slots-for-elementor-forms-lite' ),
			'slot_unavailable'  => __( 'This time slot is no longer available. Please select another time.', 'booking-slots-for-elementor-forms-lite' ),
			'booking_failed'    => __( 'Failed to complete booking. Please try again.', 'booking-slots-for-elementor-forms-lite' ),
			'custom_validation' => __( 'This booking slot is not valid.', 'booking-slots-for-elementor-forms-lite' ),
		);

		return $messages[ $key ] ?? $messages['custom_validation'];
	}

	/**
	 * Get user IP address
	 *
	 * @since 1.0.0
	 * @access private
	 * @return string
	 */
	private function get_user_ip() {
		if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
		} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
		} else {
			return isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		}
	}

	/**
	 * Get user agent
	 *
	 * @since 1.0.0
	 * @access private
	 * @return string
	 */
	private function get_user_agent() {
		return isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
	}

	/**
	 * Get sanitized submission data
	 *
	 * @since 1.0.0
	 * @access private
	 * @param array $record Form record
	 * @return array
	 */
	private function get_sanitized_submission_data( $record ) {
		$form_data      = $record->get( 'fields' );
		$sanitized_data = array();

		foreach ( $form_data as $field_id => $field ) {
			// Exclude booking slots fields and sensitive data
			if ( $field['type'] === 'booking_slots' ) {
				continue;
			}

			$sanitized_data[ $field_id ] = array(
				'type'  => $field['type'],
				'value' => sanitize_text_field( $field['value'] ),
				'title' => sanitize_text_field( $field['title'] ?? '' ),
			);
		}

		return $sanitized_data;
	}
}
