<?php
/**
 * Database Handler Class
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Database management for booking slots
 */
class BSFEF_Database {

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 * @var \BSFEF_Database The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Table name
	 *
	 * @since 1.0.0
	 * @access private
	 * @var string
	 */
	private $table_name;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 * @return \BSFEF_Database An instance of the class.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'bsfef_booking_slots';
	}

	/**
	 * Get effective form ID (for multi-form isolation feature)
	 * Free version: Always returns 0 (shared slots across all forms)
	 * PRO version: Returns actual form ID (isolated slots per form)
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string|int $form_id The original form ID
	 * @return string The effective form ID to use in database
	 */
	private function get_effective_form_id( $form_id ) {
		if ( class_exists( 'BSFEF_Pro_Features' ) ) {
			return (string) BSFEF_Pro_Features::get_effective_form_id( $form_id );
		}
		return '0'; // Free version: shared slots
	}

	/**
	 * Create database tables
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function create_tables() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE {$this->table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            booking_date date NOT NULL,
            booking_time time NOT NULL,
            form_id varchar(50) NOT NULL,
            user_ip varchar(45) DEFAULT NULL,
            user_agent text DEFAULT NULL,
            status varchar(20) DEFAULT 'booked',
            created_at timestamp DEFAULT CURRENT_TIMESTAMP,
            updated_at timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_slot_per_form (form_id, booking_date, booking_time),
            KEY idx_date_time (booking_date, booking_time),
            KEY idx_form_id (form_id),
            KEY idx_status (status),
            KEY idx_created_at (created_at)
        ) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );

		// Update version option
		update_option( 'bsfef_db_version', '1.0.0' );
	}

	/**
	 * Check if a slot is available
	 *
	 * @since 1.0.0
	 * @access public
	 * @param string $date Date in Y-m-d format
	 * @param string $time Time in H:i format
	 * @param string $form_id Form ID (in free version, '0' means check ALL forms for shared slots)
	 * @return bool
	 */
	public function is_slot_available( $date, $time, $form_id = null, $min_notice_val = null, $slot_interval = null ) {
		global $wpdb;

		// Determine timezone (prefer WP timezone API)
		try {
			if ( function_exists( 'wp_timezone' ) ) {
				$tz_obj = wp_timezone();
			} else {
				$tz_string = get_option( 'timezone_string' );
				$tz_obj    = $tz_string ? new DateTimeZone( $tz_string ) : new DateTimeZone( 'UTC' );
			}
		} catch ( Exception $e ) {
			$tz_obj = new DateTimeZone( 'UTC' );
		}

		// Current time in site's timezone
		$now     = new DateTime( 'now', $tz_obj );
		$slot_dt = DateTime::createFromFormat( 'Y-m-d H:i', $date . ' ' . $time, $tz_obj );

		// Lite version: minimum booking notice is not supported and must be ignored
		// Always treat min notice as 0 regardless of saved options or passed values
		$min_notice_val = 0;

		// Determine slot interval: prefer passed value, otherwise use filter/default
		if ( $slot_interval === null ) {
			$slot_interval = apply_filters( 'bsfef_slot_interval', 30, $form_id, $date, $time );
		} else {
			$slot_interval = intval( $slot_interval );
		}

		// If slot datetime couldn't be parsed, deny availability
		if ( ! $slot_dt ) {
			return false;
		}

		// Calculate cutoff time for booking
		if ( $min_notice_val > 0 ) {
			$cutoff = clone $now;
			$cutoff->modify( '+' . $min_notice_val . ' minutes' );
			// Only allow slots where start time is at least notice minutes after now
			if ( $slot_dt < $cutoff ) {
				return false;
			}
		} else {
			// No notice: allow booking up to slot start time
			if ( $slot_dt < $now ) {
				return false;
			}
		}

		// Build query
		$query = "SELECT COUNT(*) FROM {$this->table_name} 
                  WHERE booking_date = %s 
                  AND booking_time = %s 
                  AND status = 'booked'";

		$params = array( $date, $time );

		// In free version, form_id='0' means check across ALL forms (shared slots)
		// In PRO version, check specific form_id only
		if ( $form_id !== null && $form_id !== '0' ) {
			$query   .= ' AND form_id = %s';
			$params[] = $form_id;
		}
		// If form_id is '0' or null, don't add form_id filter (check all forms)

		$params_count = count( $params );
		if ( $params_count > 0 ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query is dynamically built but all user inputs are properly prepared via $wpdb->prepare()
			$result = $wpdb->get_var( $wpdb->prepare( $query, $params ) );
		} else {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Direct query necessary for custom table operations, no user input
			$result = $wpdb->get_var( $query );
		}

		return $result == 0;
	}

	/**
	 * Book a time slot
	 *
	 * @since 1.0.0
	 * @access public
	 * @param string $date Date in Y-m-d format
	 * @param string $time Time in H:i format
	 * @param array  $data Additional booking data
	 * @return bool|int
	 */
	public function book_slot( $date, $time, $data = array() ) {
		global $wpdb;

		$original_form_id = $data['form_id'] ?? '';

		// Ensure the slot is not in the past relative to the site's timezone
		try {
			if ( function_exists( 'wp_timezone' ) ) {
				$tz_obj = wp_timezone();
			} else {
				$tz_string = get_option( 'timezone_string' );
				$tz_obj    = $tz_string ? new DateTimeZone( $tz_string ) : new DateTimeZone( 'UTC' );
			}
		} catch ( Exception $e ) {
			$tz_obj = new DateTimeZone( 'UTC' );
		}

		$now     = new DateTime( 'now', $tz_obj );
		$slot_dt = DateTime::createFromFormat( 'Y-m-d H:i', $date . ' ' . $time, $tz_obj );
		if ( ! $slot_dt ) {
			return false;
		}
		if ( $slot_dt < $now ) {
			// Slot is already in the past for the site timezone — deny booking
			return false;
		}
		// For availability checking: use effective form ID (0 for free version = shared slots)
		$effective_form_id = $this->get_effective_form_id( $original_form_id );

		// Check if slot is still available (using effective form ID for shared slot logic)
		if ( ! $this->is_slot_available( $date, $time, $effective_form_id ) ) {
			return false;
		}

		$insert_data = array(
			'booking_date' => $date,
			'booking_time' => $time,
			'form_id'      => $original_form_id, // Store ORIGINAL form ID for future PRO upgrade
			'user_ip'      => $data['user_ip'] ?? '',
			'user_agent'   => $data['user_agent'] ?? '',
			'status'       => 'booked',
		);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct insert
		$result = $wpdb->insert(
			$this->table_name,
			$insert_data,
			array(
				'%s', // booking_date
				'%s', // booking_time
				'%s', // form_id
				'%s', // user_ip
				'%s', // user_agent
				'%s', // status
			)
		);

		if ( $result === false ) {
			return false;
		}

		return $wpdb->insert_id;
	}

	/**
	 * Get available slots for a specific date
	 *
	 * @since 1.0.0
	 * @access public
	 * @param string $date Date in Y-m-d format
	 * @param array  $all_slots All possible slots for the day
	 * @return array
	 */
	public function get_available_slots_for_date( $date, $all_slots ) {
		global $wpdb;

		// Check if date is in the past
		if ( strtotime( $date ) < strtotime( 'today' ) ) {
			return array();
		}

		// Get booked times for this date
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from class property is trusted, user input handled via placeholder
		$booked_times = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT booking_time FROM {$this->table_name} WHERE booking_date = %s AND status = 'booked'", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$date
			)
		);

		// Convert to simple array of times for comparison
		$booked_times_simple = array_map(
			function ( $time ) {
				return gmdate( 'H:i', strtotime( $time ) );
			},
			$booked_times
		);

		// Filter out booked slots
		$available_slots = array_filter(
			$all_slots,
			function ( $slot ) use ( $booked_times_simple ) {
				return ! in_array( $slot['time'], $booked_times_simple );
			}
		);

		return array_values( $available_slots ); // Re-index array
	}

	/**
	 * Get all bookings for a specific date range
	 *
	 * @since 1.0.0
	 * @access public
	 * @param string $start_date Start date in Y-m-d format
	 * @param string $end_date End date in Y-m-d format
	 * @return array
	 */
	public function get_bookings( $start_date = null, $end_date = null ) {
		global $wpdb;

		$sql    = "SELECT * FROM {$this->table_name} WHERE status = 'booked'";
		$params = array();

		if ( $start_date ) {
			$sql     .= ' AND booking_date >= %s';
			$params[] = $start_date;
		}

		if ( $end_date ) {
			$sql     .= ' AND booking_date <= %s';
			$params[] = $end_date;
		}

		$sql .= ' ORDER BY booking_date ASC, booking_time ASC';

		if ( ! empty( $params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Query dynamically built but all user inputs properly prepared with placeholders
			return $wpdb->get_results( $wpdb->prepare( $sql, $params ) );
		} else {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name trusted from $wpdb->prefix
			return $wpdb->get_results( $sql );
		}
	}

	/**
	 * Cancel a booking
	 *
	 * @since 1.0.0
	 * @access public
	 * @param int $booking_id Booking ID
	 * @return bool
	 */
	public function cancel_booking( $booking_id ) {
		global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update operation
		$result = $wpdb->update(
			$this->table_name,
			array( 'status' => 'cancelled' ),
			array( 'id' => $booking_id ),
			array( '%s' ),
			array( '%d' )
		);

		return $result !== false;
	}

	/**
	 * Delete old bookings (cleanup)
	 *
	 * @since 1.0.0
	 * @access public
	 * @param int $days_old Number of days old to delete
	 * @return int Number of deleted rows
	 */
	public function cleanup_old_bookings( $days_old = 30 ) {
		global $wpdb;

		$cutoff_date = gmdate( 'Y-m-d', strtotime( "-{$days_old} days" ) );

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from class property is trusted, date parameter properly prepared
		$result = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$this->table_name} WHERE booking_date < %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$cutoff_date
			)
		);

		return $result;
	}

	/**
	 * Get booking statistics
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array
	 */
	public function get_stats() {
		global $wpdb;

		$stats = array();

		// Total bookings
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from class property is trusted, no user input
		$stats['total_bookings'] = $wpdb->get_var( "SELECT COUNT(*) FROM {$this->table_name} WHERE status = 'booked'" );

		// Today's bookings
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from class property is trusted, date parameter properly prepared
		$stats['today_bookings'] = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$this->table_name} WHERE status = 'booked' AND booking_date = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				current_time( 'Y-m-d' )
			)
		);

		// This week's bookings
		$start_of_week = gmdate( 'Y-m-d', strtotime( 'monday this week' ) );
		$end_of_week   = gmdate( 'Y-m-d', strtotime( 'sunday this week' ) );

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from class property is trusted, date parameters properly prepared
		$stats['week_bookings'] = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$this->table_name} WHERE status = 'booked' AND booking_date BETWEEN %s AND %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$start_of_week,
				$end_of_week
			)
		);

		// This month's bookings
		$start_of_month = gmdate( 'Y-m-01' );
		$end_of_month   = gmdate( 'Y-m-t' );

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from class property is trusted, date parameters properly prepared
		$stats['month_bookings'] = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$this->table_name} WHERE status = 'booked' AND booking_date BETWEEN %s AND %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$start_of_month,
				$end_of_month
			)
		);

		return $stats;
	}

	/**
	 * Get table name
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string
	 */
	public function get_table_name() {
		return $this->table_name;
	}
}
