<?php
/**
 * Booking Slots Field Class
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Only define the class if Elementor Pro's Field_Base exists
if ( class_exists( '\ElementorPro\Modules\Forms\Fields\Field_Base' ) ) :

	/**
	 * Booking Slots Field for Elementor Forms
	 */
	class BSFEF_Booking_Slots_Field extends \ElementorPro\Modules\Forms\Fields\Field_Base {

		/**
		 * Field type name
		 *
		 * @var string
		 */
		public $depended_scripts = array( 'bsfef-frontend' );

		/**
		 * Field type name
		 *
		 * @var string
		 */
		public $depended_styles = array( 'bsfef-frontend' );

		/**
		 * Constructor
		 */
		public function __construct() {
			parent::__construct();
		}

		/**
		 * Get field type
		 *
		 * @since 1.0.0
		 * @access public
		 * @return string
		 */
		public function get_type() {
			return 'booking_slots';
		}

		/**
		 * Get field name
		 *
		 * @since 1.0.0
		 * @access public
		 * @return string
		 */
		public function get_name() {
			return esc_html__( 'Booking Slots', 'booking-slots-for-elementor-forms-lite' );
		}

		/**
		 * Render field output on the frontend
		 *
		 * @since 1.0.0
		 * @access public
		 * @param mixed $item
		 * @param mixed $item_index
		 * @param mixed $form
		 * @return void
		 */
		public function render( $item, $item_index, $form ) {
			$form_id    = $form->get_id();
			$field_id   = $item['custom_id'] ?: $form_id . $item_index;
			$field_name = 'form_fields[' . $item['custom_id'] . ']';

			// Parse availability ranges
			$availability_ranges = array();
			$use_perpetual_mode  = false;
			$available_weekdays  = array();

			// Get interval FIRST before parsing ranges (needed for backwards compatibility)
			$interval = ! empty( $item['time_interval'] ) ? (int) $item['time_interval'] : 30;

			// Trim and check if availability_ranges has actual content
			$ranges_text = isset( $item['availability_ranges'] ) ? trim( $item['availability_ranges'] ) : '';

			// Only allow availability ranges if PRO license is active
			if ( ! empty( $ranges_text ) && BSFEF_Pro_Features::can_use_availability_ranges() ) {
				// Parse the textarea input
				$lines = explode( "\n", $ranges_text );
				foreach ( $lines as $line ) {
					$line = trim( $line );
					if ( empty( $line ) ) {
						continue;
					}

					// New format with interval: DD/MM/YYYY to DD/MM/YYYY | HH:MM to HH:MM | INTERVAL
					if ( preg_match( '/^(\d{2})\/(\d{2})\/(\d{4})\s+to\s+(\d{2})\/(\d{2})\/(\d{4})\s*\|\s*(\d{2}:\d{2})\s+to\s+(\d{2}:\d{2})\s*\|\s*(\d+)$/i', $line, $matches ) ) {
						// Convert DD/MM/YYYY to YYYY-MM-DD format for internal use
						$date_start = $matches[3] . '-' . $matches[2] . '-' . $matches[1]; // YYYY-MM-DD
						$date_end   = $matches[6] . '-' . $matches[5] . '-' . $matches[4];   // YYYY-MM-DD

						$availability_ranges[] = array(
							'date_start' => $date_start,
							'date_end'   => $date_end,
							'time_start' => $matches[7],
							'time_end'   => $matches[8],
							'interval'   => (int) $matches[9], // Interval per range
						);
					}
					// Old format without interval (backwards compatibility): DD/MM/YYYY to DD/MM/YYYY | HH:MM to HH:MM
					elseif ( preg_match( '/^(\d{2})\/(\d{2})\/(\d{4})\s+to\s+(\d{2})\/(\d{2})\/(\d{4})\s*\|\s*(\d{2}:\d{2})\s+to\s+(\d{2}:\d{2})$/i', $line, $matches ) ) {
						// Convert DD/MM/YYYY to YYYY-MM-DD format for internal use
						$date_start = $matches[3] . '-' . $matches[2] . '-' . $matches[1]; // YYYY-MM-DD
						$date_end   = $matches[6] . '-' . $matches[5] . '-' . $matches[4];   // YYYY-MM-DD

						$availability_ranges[] = array(
							'date_start' => $date_start,
							'date_end'   => $date_end,
							'time_start' => $matches[7],
							'time_end'   => $matches[8],
							'interval'   => $interval, // Use default interval from field settings
						);
					}
				}
			}

			// Fall back to perpetual weekday mode if no ranges defined OR not PRO
			if ( empty( $availability_ranges ) ) {
				$use_perpetual_mode = true;

				// Get selected weekdays (legacy mode - works every week indefinitely)
				$available_weekdays = ! empty( $item['available_weekdays'] ) ? $item['available_weekdays'] : array( '1', '2', '3', '4', '5' );

				// Get time range for perpetual mode
				$time_start = ! empty( $item['perpetual_time_start'] ) ? $item['perpetual_time_start'] :
						( ! empty( $item['time_range_start'] ) ? $item['time_range_start'] : '09:00' );
				$time_end   = ! empty( $item['perpetual_time_end'] ) ? $item['perpetual_time_end'] :
						( ! empty( $item['time_range_end'] ) ? $item['time_range_end'] : '17:00' );

				// For perpetual mode, set date range based on license
				$date_start = current_time( 'Y-m-d' );

				// FREE: Limited to 1 year, PRO: 10 years
				if ( BSFEF_Pro_Features::is_pro() ) {
					$date_end = gmdate( 'Y-m-d', strtotime( '+10 years' ) ); // PRO: Far future
				} else {
					$date_end = gmdate( 'Y-m-d', strtotime( '+1 year' ) ); // FREE: 1 year limit
				}

				// Create a single "range" entry for perpetual mode
				$availability_ranges[] = array(
					'date_start' => $date_start,
					'date_end'   => $date_end,
					'time_start' => $time_start,
					'time_end'   => $time_end,
					'perpetual'  => true,
					'weekdays'   => $available_weekdays,
				);
			}

			// Generate unique IDs
			$date_field_id   = 'bsfef-date-' . $field_id;
			$time_field_id   = 'bsfef-time-' . $field_id;
			$hidden_field_id = 'bsfef-combined-' . $field_id;

			// Prepare data for JavaScript
			$field_data = array(
				'availabilityRanges' => $availability_ranges,
				'interval'           => $interval,
				'fieldId'            => $field_id,
				'formId'             => $form_id,
				'ajaxUrl'            => admin_url( 'admin-ajax.php' ),
				'nonce'              => wp_create_nonce( 'bsfef_get_slots' ),
			);

			// Add server time (site timezone aware) and min booking notice to frontend config
			try {
				if ( function_exists( 'wp_timezone' ) ) {
					$tz  = wp_timezone();
					$now = new DateTime( 'now', $tz );
				} elseif ( $tz_string = get_option( 'timezone_string' ) ) {
					$tz  = new DateTimeZone( $tz_string );
					$now = new DateTime( 'now', $tz );
				} else {
					$offset = (float) get_option( 'gmt_offset' );
					$now    = new DateTime( 'now', new DateTimeZone( 'UTC' ) );
					if ( $offset ) {
						$hours   = floor( $offset );
						$minutes = abs( ( $offset - $hours ) * 60 );
						$now->modify( (int) $hours . ' hours' );
						if ( $minutes ) {
							$now->modify( (int) $minutes . ' minutes' );
						}
					}
				}
				$field_data['serverTime'] = $now->format( 'Y-m-d H:i:s' );
			} catch ( Exception $e ) {
				$field_data['serverTime'] = gmdate( 'Y-m-d H:i:s' );
			}

			// Minimum booking notice (minutes) - legacy field support
			$field_data['minBookingNotice'] = ! empty( $item['min_booking_notice'] ) ? (int) $item['min_booking_notice'] : 0;

			$form->add_render_attribute(
				array(
					'booking_date_input' . $item_index => array(
						'class'                => array(
							'elementor-field-textual',
							'elementor-field',
							'elementor-size-' . $item['input_size'],
							'bsfef-date-picker',
						),
						'id'                   => $date_field_id,
						'type'                 => 'text',
						'readonly'             => 'readonly',
						'placeholder'          => ! empty( $item['placeholder_date'] ) ? $item['placeholder_date'] : esc_attr__( 'Select Date', 'booking-slots-for-elementor-forms-lite' ),
						'data-field-config'    => esc_attr( json_encode( $field_data ) ),
						'data-target-time'     => $time_field_id,
						'data-target-combined' => $hidden_field_id,
					),
				)
			);

			$form->add_render_attribute(
				array(
					'booking_time_input' . $item_index => array(
						'class'                => array(
							'elementor-field-textual',
							'elementor-field',
							'elementor-size-' . $item['input_size'],
							'bsfef-time-picker',
						),
						'id'                   => $time_field_id,
						'type'                 => 'select',
						'data-target-combined' => $hidden_field_id,
						'disabled'             => 'disabled',
					),
				)
			);

			$form->add_render_attribute(
				array(
					'booking_combined_input' . $item_index => array(
						'class' => array(
							'elementor-field',
							'bsfef-combined-field',
						),
						'id'    => $hidden_field_id,
						'name'  => $field_name,
						'type'  => 'hidden',
					),
				)
			);

			if ( ! empty( $item['required'] ) ) {
				$form->add_render_attribute( 'booking_combined_input' . $item_index, 'required', 'required' );
				$form->add_render_attribute( 'booking_combined_input' . $item_index, 'aria-required', 'true' );
			}

			// Render the booking slots field
			?>
			<?php
			// Add custom CSS for PRO style customization
			if ( BSFEF_Pro_Features::can_customize_styles() ) {
				$available_color = ! empty( $item['available_date_bg_color'] ) ? $item['available_date_bg_color'] : '#e8f5e9';
				$booked_color    = ! empty( $item['booked_date_bg_color'] ) ? $item['booked_date_bg_color'] : '#fff3e0';
				$selected_color  = ! empty( $item['selected_date_bg_color'] ) ? $item['selected_date_bg_color'] : '#c8e6c9';

				// Add inline CSS using WordPress function
				$custom_css = sprintf(
					'.ui-datepicker .ui-datepicker-calendar .bsfef-available a { background-color: %s !important; } ' .
					'.ui-datepicker .ui-datepicker-calendar .bsfef-fully-booked a, ' .
					'.ui-datepicker .ui-datepicker-calendar .ui-state-disabled a { background-color: %s !important; } ' .
					'.ui-datepicker .ui-datepicker-calendar td.ui-datepicker-current-day a, ' .
					'.ui-datepicker .ui-datepicker-calendar td.ui-state-active a { background-color: %s !important; }',
					esc_attr( $available_color ),
					esc_attr( $booked_color ),
					esc_attr( $selected_color )
				);
				wp_add_inline_style( 'bsfef-frontend', $custom_css );
			}
			?>
		<div class="bsfef-booking-wrapper" data-field-id="<?php echo esc_attr( $field_id ); ?>">
			
			<!-- Date Picker -->
			<div class="bsfef-date-wrapper">
				<label for="<?php echo esc_attr( $date_field_id ); ?>" class="bsfef-label">
					<?php echo esc_html( ! empty( $item['date_label'] ) ? $item['date_label'] : __( 'Select Date:', 'booking-slots-for-elementor-forms-lite' ) ); ?>
					<?php if ( ! empty( $item['required'] ) ) : ?>
						<span class="bsfef-required">*</span>
					<?php endif; ?>
				</label>
				<input <?php $form->print_render_attribute_string( 'booking_date_input' . $item_index ); ?>>
			</div>

			<!-- Time Picker -->
			<div class="bsfef-time-wrapper">
				<label for="<?php echo esc_attr( $time_field_id ); ?>" class="bsfef-label">
					<?php echo esc_html( ! empty( $item['time_label'] ) ? $item['time_label'] : __( 'Select Time:', 'booking-slots-for-elementor-forms-lite' ) ); ?>
					<?php if ( ! empty( $item['required'] ) ) : ?>
						<span class="bsfef-required">*</span>
					<?php endif; ?>
				</label>
				<select <?php $form->print_render_attribute_string( 'booking_time_input' . $item_index ); ?>>
					<option value=""><?php echo esc_html( ! empty( $item['placeholder_time'] ) ? $item['placeholder_time'] : __( 'First select a date', 'booking-slots-for-elementor-forms-lite' ) ); ?></option>
				</select>
			</div>

			<!-- Loading indicator -->
			<div class="bsfef-loading" style="display: none;">
				<span><?php esc_html_e( 'Loading available times...', 'booking-slots-for-elementor-forms-lite' ); ?></span>
			</div>

			<!-- Combined hidden field for form submission -->
			<input <?php $form->print_render_attribute_string( 'booking_combined_input' . $item_index ); ?>>
		</div>
			<?php
		}

		/**
		 * Field validation
		 *
		 * @since 1.0.0
		 * @access public
		 * @param mixed $field
		 * @param mixed $record
		 * @param mixed $ajax_handler
		 * @return void
		 */
		public function validation( $field, $record, $ajax_handler ) {
			// Wrap in try-catch to prevent breaking Elementor
			try {
				// Skip validation if field is empty and not required
				if ( empty( $field['value'] ) ) {
					if ( ! empty( $field['required'] ) ) {
						$ajax_handler->add_error( $field['id'], esc_html__( 'Please select a booking slot.', 'booking-slots-for-elementor-forms-lite' ) );
					}
					return;
				}

				// Parse the combined value (date|time)
				$parts = explode( '|', $field['value'] );
				if ( count( $parts ) !== 2 ) {
					$ajax_handler->add_error( $field['id'], esc_html__( 'Invalid booking slot format.', 'booking-slots-for-elementor-forms-lite' ) );
					return;
				}

				$date = sanitize_text_field( $parts[0] );
				$time = sanitize_text_field( $parts[1] );

				// Validate date format
				if ( ! $this->validate_date( $date ) ) {
					$ajax_handler->add_error( $field['id'], esc_html__( 'Invalid date format.', 'booking-slots-for-elementor-forms-lite' ) );
					return;
				}

				// Validate time format
				if ( ! $this->validate_time( $time ) ) {
					$ajax_handler->add_error( $field['id'], esc_html__( 'Invalid time format.', 'booking-slots-for-elementor-forms-lite' ) );
					return;
				}

				// Check if database class exists
				if ( ! class_exists( 'BSFEF_Database' ) ) {
					$ajax_handler->add_error( $field['id'], esc_html__( 'System error. Please try again later.', 'booking-slots-for-elementor-forms-lite' ) );
					return;
				}

				// Get form ID for slot checking
				$form_id = $record->get_form_settings( 'id' );

				// Check if slot is still available for THIS specific form
				$db = BSFEF_Database::instance();
				if ( ! $db->is_slot_available( $date, $time, $form_id ) ) {
					$ajax_handler->add_error( $field['id'], esc_html__( 'This time slot is no longer available. Please select another time.', 'booking-slots-for-elementor-forms-lite' ) );
					return;
				}
			} catch ( Exception $e ) {
				// Log error but don't break the validation
				// Add generic error
				$ajax_handler->add_error( $field['id'], esc_html__( 'An error occurred during validation. Please try again.', 'booking-slots-for-elementor-forms-lite' ) );
			}
		}

		/**
		 * Process field - format the value for submission
		 *
		 * @since 1.0.0
		 * @access public
		 * @param mixed                                            $field
		 * @param \ElementorPro\Modules\Forms\Classes\Form_Record  $record
		 * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler
		 * @return void
		 */
		public function process_field( $field, $record, $ajax_handler ) {
			// Wrap in try-catch to prevent breaking Elementor
			try {
				// Get the raw value (date|time format)
				$raw_value = isset( $field['raw_value'] ) ? $field['raw_value'] : '';

				if ( empty( $raw_value ) ) {
					return;
				}

				// Parse and format for display
				$parts = explode( '|', $raw_value );
				if ( count( $parts ) === 2 ) {
					$date = sanitize_text_field( $parts[0] );
					$time = sanitize_text_field( $parts[1] );

					// Format as readable text for emails and display
					$formatted_date  = date_i18n( get_option( 'date_format' ), strtotime( $date ) );
					$formatted_value = sprintf( '%s at %s', $formatted_date, $time );

					// Update the field value with formatted text
					$record->update_field( $field['id'], 'value', $formatted_value );

					// Also update the title to use the field label instead of the ID
					if ( ! empty( $field['title'] ) ) {
						$record->update_field( $field['id'], 'title', $field['title'] );
					}
				}
			} catch ( Exception $e ) {
				// Log error but don't break the submission
			}
		}

		/**
		 * Sanitize field value
		 *
		 * @since 1.0.0
		 * @access public
		 * @param mixed $value
		 * @param mixed $field
		 * @return string
		 */
		public function sanitize_field( $value, $field ) {
			// Simple sanitization - just clean the text
			return sanitize_text_field( $value );
		}

		/**
		 * Update form widget controls
		 *
		 * @since 1.0.0
		 * @access public
		 * @param mixed $widget
		 * @return void
		 */
		public function update_controls( $widget ) {
			$elementor = \ElementorPro\Plugin::elementor();

			$control_data = $elementor->controls_manager->get_control_from_stack( $widget->get_unique_name(), 'form_fields' );

			if ( is_wp_error( $control_data ) ) {
				return;
			}

			// Add booking slots specific controls
			$field_controls = array(
				'editor_notice' => array(
					'name'            => 'editor_notice',
					'type'            => \Elementor\Controls_Manager::RAW_HTML,
					'raw'             => '<div style="padding: 10px; background-color: #fff3cd; border-left: 4px solid #ffc107; color: #856404; font-size: 12px; line-height: 1.5;">' .
							'<strong>⚠️ Editor Preview Note:</strong><br>' .
							'The booking field may disappear in the editor preview when changing settings. ' .
							'This is normal - the field will work perfectly on the actual frontend of your site.' .
							'</div>',
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
					'condition'       => array(
						'field_type' => $this->get_type(),
					),
					'tab'             => 'content',
					'inner_tab'       => 'form_fields_content_tab',
					'tabs_wrapper'    => 'form_fields_tabs',
				),
			);

			// Add availability ranges control (PRO only)
			if ( BSFEF_Pro_Features::can_use_availability_ranges() ) {
				$field_controls['availability_ranges'] = array(
					'name'         => 'availability_ranges',
					'label'        => esc_html__( 'Availability Ranges', 'booking-slots-for-elementor-forms-lite' ) . ' 👑',
					'type'         => \Elementor\Controls_Manager::TEXTAREA,
					'default'      => '',
					'placeholder'  => '16/11/2025 to 17/11/2025 | 09:00 to 17:00 | 30' . "\n" . '23/11/2025 to 24/11/2025 | 10:00 to 18:00 | 15',
					'description'  => esc_html__( 'Define multiple date/time ranges with custom intervals (one per line). Perfect for different time slots across multiple weekends, events, or date periods. Format: DD/MM/YYYY to DD/MM/YYYY | HH:MM to HH:MM | INTERVAL. Example: "20/12/2025 to 22/12/2025 | 09:00 to 17:00 | 30" for one weekend with 30min slots, then "27/12/2025 to 29/12/2025 | 10:00 to 16:00 | 15" for another weekend with 15min slots. Leave empty to use weekday mode.', 'booking-slots-for-elementor-forms-lite' ),
					'rows'         => 5,
					'condition'    => array(
						'field_type' => $this->get_type(),
					),
					'tab'          => 'content',
					'inner_tab'    => 'form_fields_content_tab',
					'tabs_wrapper' => 'form_fields_tabs',
				);
			} else {
				// Show PRO upgrade notice
				$field_controls['availability_ranges_pro'] = array(
					'type'            => \Elementor\Controls_Manager::RAW_HTML,
					'raw'             => BSFEF_Pro_Features::upgrade_notice( 'Availability Ranges' ),
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
					'condition'       => array(
						'field_type' => $this->get_type(),
					),
					'tab'             => 'content',
					'inner_tab'       => 'form_fields_content_tab',
					'tabs_wrapper'    => 'form_fields_tabs',
				);
			}

			$field_controls['available_weekdays'] = array(
				'name'         => 'available_weekdays',
				'label'        => esc_html__( 'Available Days of Week', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::SELECT2,
				'multiple'     => true,
				'options'      => array(
					'0' => esc_html__( 'Sunday', 'booking-slots-for-elementor-forms-lite' ),
					'1' => esc_html__( 'Monday', 'booking-slots-for-elementor-forms-lite' ),
					'2' => esc_html__( 'Tuesday', 'booking-slots-for-elementor-forms-lite' ),
					'3' => esc_html__( 'Wednesday', 'booking-slots-for-elementor-forms-lite' ),
					'4' => esc_html__( 'Thursday', 'booking-slots-for-elementor-forms-lite' ),
					'5' => esc_html__( 'Friday', 'booking-slots-for-elementor-forms-lite' ),
					'6' => esc_html__( 'Saturday', 'booking-slots-for-elementor-forms-lite' ),
				),
				'default'      => array( '1', '2', '3', '4', '5' ),
				'description'  => esc_html__( 'Select which days of the week are available. Used only if Availability Ranges is empty.', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['perpetual_time_start'] = array(
				'name'         => 'perpetual_time_start',
				'label'        => esc_html__( 'Start Time (Legacy)', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'default'      => '09:00',
				'placeholder'  => '09:00',
				'description'  => esc_html__( 'Used only if Availability Ranges is empty', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['perpetual_time_end'] = array(
				'name'         => 'perpetual_time_end',
				'label'        => esc_html__( 'End Time (Legacy)', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'default'      => '17:00',
				'placeholder'  => '17:00',
				'description'  => esc_html__( 'Used only if Availability Ranges is empty', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['time_range_start'] = array(
				'name'         => 'time_range_start',
				'label'        => esc_html__( 'Start Time (Deprecated)', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::HIDDEN,
				'default'      => '09:00',
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['time_range_end'] = array(
				'name'         => 'time_range_end',
				'label'        => esc_html__( 'End Time (Deprecated)', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::HIDDEN,
				'default'      => '17:00',
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['time_interval'] = array(
				'name'         => 'time_interval',
				'label'        => esc_html__( 'Time Interval (minutes)', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::SELECT,
				'default'      => '30',
				'options'      => array(
					'5'   => esc_html__( '5 minutes', 'booking-slots-for-elementor-forms-lite' ),
					'10'  => esc_html__( '10 minutes', 'booking-slots-for-elementor-forms-lite' ),
					'15'  => esc_html__( '15 minutes', 'booking-slots-for-elementor-forms-lite' ),
					'20'  => esc_html__( '20 minutes', 'booking-slots-for-elementor-forms-lite' ),
					'30'  => esc_html__( '30 minutes', 'booking-slots-for-elementor-forms-lite' ),
					'45'  => esc_html__( '45 minutes', 'booking-slots-for-elementor-forms-lite' ),
					'60'  => esc_html__( '1 hour', 'booking-slots-for-elementor-forms-lite' ),
					'90'  => esc_html__( '1.5 hours', 'booking-slots-for-elementor-forms-lite' ),
					'120' => esc_html__( '2 hours', 'booking-slots-for-elementor-forms-lite' ),
				),
				'description'  => BSFEF_Pro_Features::can_use_availability_ranges()
					? esc_html__( 'Used only if Availability Ranges is empty. PRO users can set custom intervals per range.', 'booking-slots-for-elementor-forms-lite' )
					: esc_html__( 'Time between each slot. Upgrade to PRO to set different intervals per date range.', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['date_label'] = array(
				'name'         => 'date_label',
				'label'        => esc_html__( 'Date Label', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'default'      => esc_html__( 'Select Date:', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['time_label'] = array(
				'name'         => 'time_label',
				'label'        => esc_html__( 'Time Label', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'default'      => esc_html__( 'Select Time:', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['placeholder_date'] = array(
				'name'         => 'placeholder_date',
				'label'        => esc_html__( 'Date Placeholder', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'default'      => esc_html__( 'Select Date', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			$field_controls['placeholder_time'] = array(
				'name'         => 'placeholder_time',
				'label'        => esc_html__( 'Time Placeholder', 'booking-slots-for-elementor-forms-lite' ),
				'type'         => \Elementor\Controls_Manager::TEXT,
				'default'      => esc_html__( 'First select a date', 'booking-slots-for-elementor-forms-lite' ),
				'condition'    => array(
					'field_type' => $this->get_type(),
				),
				'tab'          => 'content',
				'inner_tab'    => 'form_fields_content_tab',
				'tabs_wrapper' => 'form_fields_tabs',
			);

			// Add style customization controls (PRO only)
			if ( BSFEF_Pro_Features::can_customize_styles() ) {
				$field_controls['style_heading'] = array(
					'type'         => \Elementor\Controls_Manager::HEADING,
					'label'        => esc_html__( 'Style Customization', 'booking-slots-for-elementor-forms-lite' ) . ' 🎨 PRO',
					'separator'    => 'before',
					'condition'    => array(
						'field_type' => $this->get_type(),
					),
					'tab'          => 'content',
					'inner_tab'    => 'form_fields_content_tab',
					'tabs_wrapper' => 'form_fields_tabs',
				);

				$field_controls['available_date_bg_color'] = array(
					'name'         => 'available_date_bg_color',
					'label'        => esc_html__( 'Available Dates Background', 'booking-slots-for-elementor-forms-lite' ),
					'type'         => \Elementor\Controls_Manager::COLOR,
					'default'      => '#e8f5e9',
					'condition'    => array(
						'field_type' => $this->get_type(),
					),
					'tab'          => 'content',
					'inner_tab'    => 'form_fields_content_tab',
					'tabs_wrapper' => 'form_fields_tabs',
				);

				$field_controls['booked_date_bg_color'] = array(
					'name'         => 'booked_date_bg_color',
					'label'        => esc_html__( 'Fully Booked Dates Background', 'booking-slots-for-elementor-forms-lite' ),
					'type'         => \Elementor\Controls_Manager::COLOR,
					'default'      => '#fff3e0',
					'condition'    => array(
						'field_type' => $this->get_type(),
					),
					'tab'          => 'content',
					'inner_tab'    => 'form_fields_content_tab',
					'tabs_wrapper' => 'form_fields_tabs',
				);

				$field_controls['selected_date_bg_color'] = array(
					'name'         => 'selected_date_bg_color',
					'label'        => esc_html__( 'Selected Date Background', 'booking-slots-for-elementor-forms-lite' ),
					'type'         => \Elementor\Controls_Manager::COLOR,
					'default'      => '#c8e6c9',
					'condition'    => array(
						'field_type' => $this->get_type(),
					),
					'tab'          => 'content',
					'inner_tab'    => 'form_fields_content_tab',
					'tabs_wrapper' => 'form_fields_tabs',
				);
			} else {
				// Show PRO upgrade notice for style customization
				$field_controls['style_customization_pro'] = array(
					'type'            => \Elementor\Controls_Manager::RAW_HTML,
					'raw'             => BSFEF_Pro_Features::upgrade_notice( 'Style Customization' ),
					'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
					'condition'       => array(
						'field_type' => $this->get_type(),
					),
					'tab'             => 'content',
					'inner_tab'       => 'form_fields_content_tab',
					'tabs_wrapper'    => 'form_fields_tabs',
				);
			}

			$control_data['fields'] = $this->inject_field_controls( $control_data['fields'], $field_controls );
			$widget->update_control( 'form_fields', $control_data );
		}

		/**
		 * Generate time slots based on range and interval
		 *
		 * @since 1.0.0
		 * @access public
		 * @param string $start_time Start time (HH:MM)
		 * @param string $end_time End time (HH:MM)
		 * @param int    $interval Interval in minutes
		 * @return array Array of time slots
		 */
		public function generate_time_slots( $start_time, $end_time, $interval ) {
			$slots = array();

			try {
				$start        = new DateTime( $start_time );
				$end          = new DateTime( $end_time );
				$interval_obj = new DateInterval( 'PT' . $interval . 'M' );

				$current = clone $start;
				while ( $current <= $end ) {
					$slots[] = $current->format( 'H:i' );
					$current->add( $interval_obj );
				}
			} catch ( Exception $e ) {
				// Fallback to default slots if there's an error
			}

			return $slots;
		}

		/**
		 * Validate date format
		 *
		 * @since 1.0.0
		 * @access private
		 * @param string $date
		 * @return bool
		 */
		private function validate_date( $date ) {
			$d = DateTime::createFromFormat( 'Y-m-d', $date );
			return $d && $d->format( 'Y-m-d' ) === $date;
		}

		/**
		 * Validate time format
		 *
		 * @since 1.0.0
		 * @access private
		 * @param string $time
		 * @return bool
		 */
		private function validate_time( $time ) {
			return preg_match( '/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $time );
		}

		/**
		 * Get user IP address
		 *
		 * @since 1.0.0
		 * @access private
		 * @return string
		 */
		private function get_user_ip() {
			if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
				return sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
			} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
				return sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
			} else {
				return sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ?? '' ) );
			}
		}
	}

endif; // End check for ElementorPro Field_Base class