<?php
/**
 * Admin Page Class
 *
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Admin page for managing bookings
 */
class BSFEF_Admin_Page {

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_post_bsfef_delete_booking', array( $this, 'delete_booking' ) );
		add_action( 'admin_post_bsfef_delete_all_bookings', array( $this, 'delete_all_bookings' ) );
		add_action( 'admin_post_bsfef_delete_past_bookings', array( $this, 'delete_past_bookings' ) );
		add_action( 'admin_post_bsfef_bulk_delete_bookings', array( $this, 'bulk_delete_bookings' ) );
		add_action( 'admin_post_bsfef_save_settings', array( $this, 'save_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'add_admin_scripts' ) );
	}

	/**
	 * Add admin menu
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_admin_menu() {
		$result = add_menu_page(
			__( 'Booking Slots', 'booking-slots-for-elementor-forms-lite' ),
			__( 'Booking Slots', 'booking-slots-for-elementor-forms-lite' ),
			'manage_options',
			'bsfef-bookings',
			array( $this, 'render_admin_page' ),
			'dashicons-calendar-alt',
			30
		);

		// Add Settings submenu
		add_submenu_page(
			'bsfef-bookings',
			__( 'Settings', 'booking-slots-for-elementor-forms-lite' ),
			__( 'Settings', 'booking-slots-for-elementor-forms-lite' ),
			'manage_options',
			'bsfef-settings',
			array( $this, 'render_settings_page' )
		);

		// Add Upgrade submenu (external link, opens in new tab)
		global $submenu;
		$submenu['bsfef-bookings'][] = array(
			'<span style="color:#d63638;font-weight:bold;">' . esc_html__( 'Upgrade', 'booking-slots-for-elementor-forms-lite' ) . '</span>',
			'manage_options',
			'https://products.dopestudio.co.uk/BSEF/',
		);
	}

	/**
	 * Render admin page
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function render_admin_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'booking-slots-for-elementor-forms-lite' ) );
		}

		global $wpdb;
		$bsfef_table_name = $wpdb->prefix . 'bsfef_booking_slots';

		// Get filter parameters (PRO only)
		$filter_form = '';
		$filter_date = '';

		if ( BSFEF_Pro_Features::can_use_filters() ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameters for filtering display, sanitized and used in prepared statement
			$filter_form = isset( $_GET['filter_form'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_form'] ) ) : '';
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameters for filtering display, sanitized and used in prepared statement
			$filter_date = isset( $_GET['filter_date'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_date'] ) ) : '';
		}

		// Pagination parameters
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter for pagination
		$paged      = isset( $_GET['paged'] ) ? max( 1, (int) $_GET['paged'] ) : 1;
		$per_page   = 20; // Number of bookings per page
		$offset     = ( $paged - 1 ) * $per_page;

		// Build base query for counting total results
		$count_sql    = "SELECT COUNT(*) FROM $bsfef_table_name WHERE status = 'booked'";
		$count_params = array();

		if ( $filter_form ) {
			$count_sql     .= ' AND form_id = %s';
			$count_params[] = $filter_form;
		}

		if ( $filter_date ) {
			$count_sql     .= ' AND booking_date = %s';
			$count_params[] = $filter_date;
		}

		// Get total count for pagination
		if ( ! empty( $count_params ) ) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SQL dynamically built but all user inputs properly prepared with placeholders, table name from $wpdb->prefix is trusted
			$total_items = $wpdb->get_var( $wpdb->prepare( $count_sql, $count_params ) );
		} else {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- No user input, table name trusted
			$total_items = $wpdb->get_var( $count_sql );
		}

		$total_pages = ceil( $total_items / $per_page );

		// Build query dynamically based on filters with pagination
		$sql    = "SELECT * FROM $bsfef_table_name WHERE status = 'booked'";
		$params = array();

		if ( $filter_form ) {
			$sql     .= ' AND form_id = %s';
			$params[] = $filter_form;
		}

		if ( $filter_date ) {
			$sql     .= ' AND booking_date = %s';
			$params[] = $filter_date;
		}

		$sql .= ' ORDER BY booking_date DESC, booking_time DESC LIMIT %d OFFSET %d';
		$params[] = $per_page;
		$params[] = $offset;

		// Get bookings with pagination
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- SQL dynamically built but all user inputs properly prepared with placeholders, table name from $wpdb->prefix is trusted
		$bookings = $wpdb->get_results( $wpdb->prepare( $sql, $params ) );

		// Get unique form IDs for filter
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix is trusted
		$form_ids = $wpdb->get_col( "SELECT DISTINCT form_id FROM {$bsfef_table_name} ORDER BY form_id" );

		// Get stats
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix is trusted
		$total_bookings = $wpdb->get_var( "SELECT COUNT(*) FROM {$bsfef_table_name} WHERE status = 'booked'" );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix is trusted, date parameter properly prepared
		$past_bookings = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM $bsfef_table_name WHERE status = 'booked' AND booking_date < %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				current_time( 'Y-m-d' )
			)
		);

		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Booking Slots Management', 'booking-slots-for-elementor-forms-lite' ); ?></h1>

			<!-- Stats -->
			<div class="bsfef-stats" style="margin: 20px 0; display: flex; gap: 20px;">
				<div style="background: #fff; padding: 15px 20px; border-left: 4px solid #2271b1; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
					<h3 style="margin: 0 0 5px 0;"><?php echo esc_html( $total_bookings ); ?></h3>
					<p style="margin: 0; color: #666;"><?php esc_html_e( 'Total Bookings', 'booking-slots-for-elementor-forms-lite' ); ?></p>
				</div>
				<div style="background: #fff; padding: 15px 20px; border-left: 4px solid #d63638; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
					<h3 style="margin: 0 0 5px 0;"><?php echo esc_html( $past_bookings ); ?></h3>
					<p style="margin: 0; color: #666;"><?php esc_html_e( 'Past Bookings', 'booking-slots-for-elementor-forms-lite' ); ?></p>
				</div>
			</div>

			<?php if ( ! BSFEF_Pro_Features::is_pro() ) : ?>
			<!-- Simple Upgrade Notice (WordPress.org compliant) -->
			<div class="notice notice-info" style="margin: 20px 0; padding: 15px; position: relative;">
				<h3 style="margin-top: 0;"><?php esc_html_e( 'Looking for more features?', 'booking-slots-for-elementor-forms-lite' ); ?></h3>
				<p><?php esc_html_e( 'The PRO version includes: Specific Date Ranges, Per-Range Intervals, Extended Perpetual Mode (10 years), Multi-Form Isolation, Custom Calendar Colors, Filter & Search Bookings, Bulk Delete, and Priority Support.', 'booking-slots-for-elementor-forms-lite' ); ?></p>
				<p><a href="https://products.dopestudio.co.uk/BSEF/purchase.php" target="_blank" class="button button-primary"><?php esc_html_e( 'Learn More', 'booking-slots-for-elementor-forms-lite' ); ?></a></p>
			</div>
			<?php endif; ?>

			<!-- Filters and Actions -->
			<div class="tablenav top">
				<?php if ( BSFEF_Pro_Features::can_use_filters() ) : ?>
					<!-- PRO: Filtering enabled -->
					<form method="get" style="display: inline-block;">
						<input type="hidden" name="page" value="bsfef-bookings">
						<select name="filter_form">
							<option value=""><?php esc_html_e( 'All Forms', 'booking-slots-for-elementor-forms-lite' ); ?></option>
							<?php foreach ( $form_ids as $form_id ) : ?>
								<option value="<?php echo esc_attr( $form_id ); ?>" <?php selected( $filter_form, $form_id ); ?>>
									<?php
									/* translators: %s: Form ID number */
									echo esc_html( sprintf( __( 'Form ID: %s', 'booking-slots-for-elementor-forms-lite' ), $form_id ) );
									?>
								</option>
							<?php endforeach; ?>
						</select>
						<input type="date" name="filter_date" value="<?php echo esc_attr( $filter_date ); ?>" placeholder="<?php esc_attr_e( 'Filter by date', 'booking-slots-for-elementor-forms-lite' ); ?>">
						<button type="submit" class="button"><?php esc_html_e( 'Filter', 'booking-slots-for-elementor-forms-lite' ); ?></button>
						<a href="?page=bsfef-bookings" class="button"><?php esc_html_e( 'Clear', 'booking-slots-for-elementor-forms-lite' ); ?></a>
					</form>
				<?php else : ?>
					<!-- FREE: Show simple notice for filters -->
					<div style="display: inline-block; padding: 5px 10px; background: #f0f0f1; border-left: 3px solid #72aee6; margin-right: 10px;">
						<span style="color: #2c3338; font-size: 13px;"><?php esc_html_e( 'Advanced filtering available in PRO version', 'booking-slots-for-elementor-forms-lite' ); ?></span>
					</div>
				<?php endif; ?>

				<?php if ( BSFEF_Pro_Features::can_use_bulk_delete() ) : ?>
				<div class="alignright" style="margin: 0 0 0 8px;">
					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display: inline-block;" onsubmit="return confirm('<?php esc_attr_e( 'Are you sure you want to delete all past bookings? This cannot be undone.', 'booking-slots-for-elementor-forms-lite' ); ?>');">
						<input type="hidden" name="action" value="bsfef_delete_past_bookings">
						<?php wp_nonce_field( 'bsfef_delete_past_bookings', 'bsfef_nonce' ); ?>
						<button type="submit" class="button"><?php esc_html_e( 'Delete Past Bookings', 'booking-slots-for-elementor-forms-lite' ); ?></button>
					</form>
					
					<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display: inline-block;" onsubmit="return confirm('<?php esc_attr_e( 'Are you sure you want to delete ALL bookings? This cannot be undone.', 'booking-slots-for-elementor-forms-lite' ); ?>');">
						<input type="hidden" name="action" value="bsfef_delete_all_bookings">
						<?php wp_nonce_field( 'bsfef_delete_all_bookings', 'bsfef_nonce' ); ?>
						<button type="submit" class="button button-secondary"><?php esc_html_e( 'Delete All Bookings', 'booking-slots-for-elementor-forms-lite' ); ?></button>
					</form>
				</div>
				<?php endif; ?>
			</div>

			<!-- Bulk Actions Form -->
			<?php if ( BSFEF_Pro_Features::can_use_bulk_delete() ) : ?>
				<!-- PRO: Bulk delete enabled -->
				<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" id="bsfef-bulk-form">
					<input type="hidden" name="action" value="bsfef_bulk_delete_bookings">
					<?php wp_nonce_field( 'bsfef_bulk_delete_bookings', 'bsfef_nonce' ); ?>
					
					<div class="tablenav top">
						<div class="alignleft actions bulkactions">
							<button type="button" id="bsfef-select-all" class="button"><?php esc_html_e( 'Select All', 'booking-slots-for-elementor-forms-lite' ); ?></button>
							<button type="button" id="bsfef-deselect-all" class="button"><?php esc_html_e( 'Deselect All', 'booking-slots-for-elementor-forms-lite' ); ?></button>
							<button type="submit" class="button button-primary" id="bsfef-bulk-delete" disabled onclick="return confirm('<?php esc_attr_e( 'Are you sure you want to delete the selected bookings? This cannot be undone.', 'booking-slots-for-elementor-forms-lite' ); ?>');">
								<?php esc_html_e( 'Delete Selected', 'booking-slots-for-elementor-forms-lite' ); ?>
							</button>
							<span id="bsfef-selected-count" style="margin-left: 10px; color: #666;"></span>
						</div>
					</div>
			<?php else : ?>
				<!-- FREE: No bulk delete functionality -->
				<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" id="bsfef-bulk-form">
			<?php endif; ?>

			<!-- Bookings Table -->
			<table class="wp-list-table widefat fixed striped">
				<thead>
					<tr>
						<?php if ( BSFEF_Pro_Features::can_use_bulk_delete() ) : ?>
						<th style="width: 40px;">
							<input type="checkbox" id="bsfef-select-all-checkbox" />
						</th>
						<?php endif; ?>
						<th><?php esc_html_e( 'ID', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'Date', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'Time', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'Form ID', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'User IP', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'Booked On', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'Status', 'booking-slots-for-elementor-forms-lite' ); ?></th>
						<th><?php esc_html_e( 'Actions', 'booking-slots-for-elementor-forms-lite' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php if ( empty( $bookings ) ) : ?>
						<tr>
							<td colspan="<?php echo BSFEF_Pro_Features::can_use_bulk_delete() ? '9' : '8'; ?>" style="text-align: center; padding: 40px 20px;">
								<?php esc_html_e( 'No bookings found.', 'booking-slots-for-elementor-forms-lite' ); ?>
							</td>
						</tr>
					<?php else : ?>
						<?php
						foreach ( $bookings as $booking ) :
							$is_past = strtotime( $booking->booking_date ) < strtotime( 'today' );
							?>
							<tr <?php echo $is_past ? 'style="background: #fff5f5;"' : ''; ?>>
								<?php if ( BSFEF_Pro_Features::can_use_bulk_delete() ) : ?>
								<td>
									<input type="checkbox" name="booking_ids[]" value="<?php echo esc_attr( $booking->id ); ?>" class="bsfef-booking-checkbox" />
								</td>
								<?php endif; ?>
								<td><?php echo esc_html( $booking->id ); ?></td>
								<td><?php echo esc_html( date_i18n( get_option( 'date_format' ), strtotime( $booking->booking_date ) ) ); ?></td>
								<td><?php echo esc_html( date_i18n( 'g:i A', strtotime( $booking->booking_time ) ) ); ?></td>
								<td><?php echo esc_html( $booking->form_id ); ?></td>
								<td><?php echo esc_html( $booking->user_ip ); ?></td>
								<td><?php echo esc_html( date_i18n( 'Y-m-d H:i', strtotime( $booking->created_at ) ) ); ?></td>
								<td>
									<?php if ( $is_past ) : ?>
										<span style="color: #d63638;"><?php esc_html_e( 'Past', 'booking-slots-for-elementor-forms-lite' ); ?></span>
									<?php else : ?>
										<span style="color: #00a32a;"><?php echo esc_html( ucfirst( $booking->status ) ); ?></span>
									<?php endif; ?>
								</td>
								<td>
									<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display: inline;">
										<input type="hidden" name="action" value="bsfef_delete_booking">
										<input type="hidden" name="booking_id" value="<?php echo esc_attr( $booking->id ); ?>">
										<?php wp_nonce_field( 'bsfef_delete_booking', 'bsfef_nonce' ); ?>
										<button type="submit" class="button button-small button-link-delete" onclick="return confirm('<?php esc_attr_e( 'Are you sure you want to delete this booking?', 'booking-slots-for-elementor-forms-lite' ); ?>');">
											<?php esc_html_e( 'Delete', 'booking-slots-for-elementor-forms-lite' ); ?>
										</button>
									</form>
								</td>
							</tr>
						<?php endforeach; ?>
					<?php endif; ?>
				</tbody>
			</table>
			</form>

			<?php
			// Pagination
			if ( $total_pages > 1 ) :
				$page_links = paginate_links(
					array(
						'base'      => add_query_arg( 'paged', '%#%' ),
						'format'    => '',
						'prev_text' => __( '&laquo; Previous', 'booking-slots-for-elementor-forms-lite' ),
						'next_text' => __( 'Next &raquo;', 'booking-slots-for-elementor-forms-lite' ),
						'total'     => $total_pages,
						'current'   => $paged,
						'type'      => 'plain',
					)
				);

				if ( $page_links ) :
					?>
					<div class="tablenav bottom">
						<div class="tablenav-pages">
							<span class="displaying-num">
								<?php
								/* translators: %s: Number of items */
								printf( esc_html( _n( '%s item', '%s items', $total_items, 'booking-slots-for-elementor-forms-lite' ) ), esc_html( number_format_i18n( $total_items ) ) );
								?>
							</span>
							<?php echo wp_kses_post( $page_links ); ?>
						</div>
					</div>
					<?php
				endif;
			endif;
			?>
		</div>
		<?php
	}

	/**
	 * Delete a single booking
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function delete_booking() {
		check_admin_referer( 'bsfef_delete_booking', 'bsfef_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'booking-slots-for-elementor-forms-lite' ) );
		}

		$booking_id = isset( $_POST['booking_id'] ) ? (int) $_POST['booking_id'] : 0;

		if ( $booking_id ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'bsfef_booking_slots';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table deletion uses $wpdb->delete with parameterized where clause
			$wpdb->delete( $table_name, array( 'id' => $booking_id ), array( '%d' ) );
		}

		wp_safe_redirect(
			add_query_arg(
				array(
					'page'    => 'bsfef-bookings',
					'deleted' => '1',
				),
				admin_url( 'admin.php' )
			)
		);
		exit;
	}

	/**
	 * Delete all bookings
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function delete_all_bookings() {
		check_admin_referer( 'bsfef_delete_all_bookings', 'bsfef_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'booking-slots-for-elementor-forms-lite' ) );
		}

		global $wpdb;
		$bsfef_table_name = $wpdb->prefix . 'bsfef_booking_slots';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix is trusted, TRUNCATE is intentional admin action
		$wpdb->query( "TRUNCATE TABLE {$bsfef_table_name}" );

		wp_safe_redirect(
			add_query_arg(
				array(
					'page'    => 'bsfef-bookings',
					'deleted' => 'all',
				),
				admin_url( 'admin.php' )
			)
		);
		exit;
	}

	/**
	 * Delete past bookings
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function delete_past_bookings() {
		check_admin_referer( 'bsfef_delete_past_bookings', 'bsfef_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'booking-slots-for-elementor-forms-lite' ) );
		}

		global $wpdb;
		$bsfef_table_name = $wpdb->prefix . 'bsfef_booking_slots';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix is trusted, date parameter properly prepared
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM $bsfef_table_name WHERE booking_date < %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				current_time( 'Y-m-d' )
			)
		);

		wp_safe_redirect(
			add_query_arg(
				array(
					'page'    => 'bsfef-bookings',
					'deleted' => 'past',
				),
				admin_url( 'admin.php' )
			)
		);
		exit;
	}

	/**
	 * Bulk delete bookings
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function bulk_delete_bookings() {
		check_admin_referer( 'bsfef_bulk_delete_bookings', 'bsfef_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'booking-slots-for-elementor-forms-lite' ) );
		}

		$booking_ids = isset( $_POST['booking_ids'] ) ? array_map( 'intval', $_POST['booking_ids'] ) : array();

		if ( ! empty( $booking_ids ) ) {
			global $wpdb;
			$bsfef_table_name = $wpdb->prefix . 'bsfef_booking_slots';
			$placeholders     = implode( ',', array_fill( 0, count( $booking_ids ), '%d' ) );
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name from $wpdb->prefix is trusted, booking IDs sanitized with intval
			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM $bsfef_table_name WHERE id IN ($placeholders)", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
					$booking_ids
				)
			);
		}

		$count = count( $booking_ids );
		wp_safe_redirect(
			add_query_arg(
				array(
					'page'    => 'bsfef-bookings',
					'deleted' => $count,
				),
				admin_url( 'admin.php' )
			)
		);
		exit;
	}

	/**
	 * Add admin scripts for checkbox handling and upgrade link
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function add_admin_scripts() {
		// Bail if not in admin or user not logged in
		if ( ! is_admin() || ! is_user_logged_in() ) {
			return;
		}

		// Enqueue jQuery for upgrade link script (used globally for all admin pages)
		wp_enqueue_script( 'jquery' );

		// Add upgrade link script globally for all admin pages
		$upgrade_link_script = "
		jQuery(document).ready(function($) {
			// Fix upgrade link to open in new tab and prevent WordPress from mangling the URL
			$('#adminmenu a[href*=\"dopestudio.co.uk\"]').each(function() {
				var \$link = $(this);
				var originalHref = 'https://products.dopestudio.co.uk/BSEF/';
				
				// Set the correct href and attributes
				\$link.attr({
					'href': originalHref,
					'target': '_blank',
					'rel': 'noopener noreferrer'
				});
				
				// Override click to ensure it opens correctly
				\$link.on('click', function(e) {
					e.preventDefault();
					e.stopPropagation();
					window.open(originalHref, '_blank', 'noopener,noreferrer');
					return false;
				});
			});
		});
		";
		wp_add_inline_script( 'jquery', $upgrade_link_script );

		// Booking page specific scripts
		$screen = get_current_screen();
		if ( $screen && $screen->id === 'toplevel_page_bsfef-bookings' ) {
			// Enqueue jQuery for booking page scripts
			wp_enqueue_script( 'jquery' );

			// Add booking page checkbox handling script
			$booking_script = "
			jQuery(document).ready(function($) {
				// Update selected count and delete button state
				function updateSelection() {
					var checkedCount = $('.bsfef-booking-checkbox:checked').length;
					$('#bsfef-selected-count').text(checkedCount > 0 ? checkedCount + ' selected' : '');
					$('#bsfef-bulk-delete').prop('disabled', checkedCount === 0);
				}

				// Select all button
				$('#bsfef-select-all, #bsfef-select-all-checkbox').on('click', function() {
					$('.bsfef-booking-checkbox').prop('checked', true);
					$('#bsfef-select-all-checkbox').prop('checked', true);
					updateSelection();
				});

				// Deselect all button
				$('#bsfef-deselect-all').on('click', function() {
					$('.bsfef-booking-checkbox').prop('checked', false);
					$('#bsfef-select-all-checkbox').prop('checked', false);
					updateSelection();
				});

				// Individual checkbox change
				$('.bsfef-booking-checkbox').on('change', function() {
					var allChecked = $('.bsfef-booking-checkbox').length === $('.bsfef-booking-checkbox:checked').length;
					$('#bsfef-select-all-checkbox').prop('checked', allChecked);
					updateSelection();
				});

				// Initial state
				updateSelection();
			});
			";
			wp_add_inline_script( 'jquery', $booking_script );

			// Add booking page styles
			$booking_styles = "
				#bsfef-bulk-delete:disabled {
					opacity: 0.5;
					cursor: not-allowed;
				}
				#bsfef-selected-count {
					font-weight: bold;
				}
			";
			wp_add_inline_style( 'common', $booking_styles );
		}
	}

	/**
	 * Render settings page
	 *
	 * @since 1.0.1
	 * @access public
	 */
	public function render_settings_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'booking-slots-for-elementor-forms-lite' ) );
		}

		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Plugin Settings', 'booking-slots-for-elementor-forms-lite' ); ?></h1>
			
			<?php
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only GET parameter for displaying success message
			if ( isset( $_GET['settings_updated'] ) && $_GET['settings_updated'] === '1' ) :
				?>
				<div class="notice notice-success is-dismissible">
					<p><?php esc_html_e( 'Settings saved successfully.', 'booking-slots-for-elementor-forms-lite' ); ?></p>
				</div>
			<?php endif; ?>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<input type="hidden" name="action" value="bsfef_save_settings">
				<?php wp_nonce_field( 'bsfef_save_settings', 'bsfef_settings_nonce' ); ?>

				<table class="form-table" role="presentation">
					<tbody>
						<tr>
							<th scope="row">
								<?php esc_html_e( 'Data Retention', 'booking-slots-for-elementor-forms-lite' ); ?>
							</th>
							<td>
								<label>
									<input type="checkbox" name="bsfef_keep_data_on_uninstall" value="1" <?php checked( get_option( 'bsfef_keep_data_on_uninstall', 'yes' ), 'yes' ); ?>>
									<?php esc_html_e( 'Keep booking data when uninstalling plugin', 'booking-slots-for-elementor-forms-lite' ); ?>
								</label>
								<p class="description">
									<?php esc_html_e( 'Recommended: Keep this checked to preserve your booking data when upgrading to PRO version or temporarily removing the plugin. Uncheck only if you want to completely remove all booking data from the database.', 'booking-slots-for-elementor-forms-lite' ); ?>
								</p>
								<div style="margin-top: 10px; padding: 12px; background: #d1ecf1; border-left: 4px solid #0c5460;">
									<strong><?php esc_html_e( '💡 Upgrading to PRO?', 'booking-slots-for-elementor-forms-lite' ); ?></strong>
									<?php esc_html_e( 'Keep this setting enabled to preserve all your bookings when switching to the PRO version. Your data will be seamlessly transferred!', 'booking-slots-for-elementor-forms-lite' ); ?>
								</div>
							</td>
						</tr>
					</tbody>
				</table>

				<?php submit_button( __( 'Save Settings', 'booking-slots-for-elementor-forms-lite' ) ); ?>
			</form>
		</div>
		<?php
	}

	/**
	 * Save plugin settings
	 *
	 * @since 1.0.1
	 * @access public
	 */
	public function save_settings() {
		// Verify nonce
		if ( ! isset( $_POST['bsfef_settings_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['bsfef_settings_nonce'] ) ), 'bsfef_save_settings' ) ) {
			wp_die( esc_html__( 'Security check failed', 'booking-slots-for-elementor-forms-lite' ) );
		}

		// Check permissions
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized', 'booking-slots-for-elementor-forms-lite' ) );
		}

		// Save data retention setting (store as string to avoid WordPress autoload issues with false)
		$keep_data = isset( $_POST['bsfef_keep_data_on_uninstall'] ) ? 'yes' : 'no';
		update_option( 'bsfef_keep_data_on_uninstall', $keep_data );

		// Redirect back with success message
		wp_safe_redirect(
			add_query_arg(
				array(
					'page'             => 'bsfef-settings',
					'settings_updated' => '1',
				),
				admin_url( 'admin.php' )
			)
		);
		exit;
	}
}