<?php
/**
 * Plugin Name: Booking Slots for Elementor Forms Lite
 * Plugin URI: https://products.dopestudio.co.uk/
 * Description: Add booking time slots to Elementor Forms. Allows visitors to select available date and time slots when submitting forms.
 * Version: 1.0.0
 * Author: Dope Studio
 * Author URI: https://dopestudio.co.uk/
 * Text Domain: booking-slots-for-elementor-forms-lite
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Requires Plugins: elementor
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */


// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Check if PRO version is active and prevent loading
if ( defined( 'BSFEF_PRO' ) ) {
	// PRO version is already loaded, self-deactivate and show notice
	add_action(
		'admin_init',
		function () {
			$plugin = plugin_basename( __FILE__ );
			if ( is_plugin_active( $plugin ) ) {
				deactivate_plugins( $plugin );
				add_action(
					'admin_notices',
					function () {
						/* translators: %1$s: Plugin name */
						$message = sprintf(
						/* translators: %1$s: Plugin name */
							esc_html__( 'The Lite version of %1$s has been deactivated because the PRO version is active. Please reload this page.', 'booking-slots-for-elementor-forms-lite' ),
							'<strong>Booking Slots for Elementor Forms</strong>'
						);
						printf( '<div class="notice notice-info"><p>%1$s</p></div>', wp_kses_post( $message ) );
					}
				);
			}
		}
	);
	return; // Stop loading Lite version this time
}

// Define plugin constants
define( 'BSFEF_VERSION', '1.0.0' );
define( 'BSFEF_PLUGIN_FILE', __FILE__ );
define( 'BSFEF_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'BSFEF_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Main Plugin Class
 */
class BSFEF_Elementor_Form_Booking_Slots {

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 * @access private
	 * @static
	 * @var \BSFEF_Elementor_Form_Booking_Slots The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Tracks whether we already hooked the field registration
	 *
	 * @var bool
	 */
	private $field_registration_hooked = false;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 * @return \BSFEF_Elementor_Form_Booking_Slots An instance of the class.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', array( $this, 'init' ) );
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
	}

	/**
	 * Initialize the plugin
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function init() {
		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
			return;
		}

		// Check if Elementor Pro installed and activated
		if ( ! function_exists( 'elementor_pro_load_plugin' ) && ! class_exists( 'ElementorPro\Plugin' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_missing_elementor_pro' ) );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, '3.0.0', '>=' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_elementor_version' ) );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
			return;
		}

		// Load plugin files
		$this->includes();

		// Initialize database
		add_action( 'init', array( $this, 'init_database' ) );

		// Initialize validation
		add_action( 'init', array( $this, 'init_validation' ) );

		// Register booking field once Elementor Pro is ready
		add_action( 'elementor_pro/init', array( $this, 'on_elementor_pro_init' ) );

		// If Elementor Pro already initialized (possible during late plugin load), run immediately
		if ( did_action( 'elementor_pro/init' ) ) {
			$this->on_elementor_pro_init();
		}

		// Enqueue scripts and styles
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

		// Add AJAX handlers
		add_action( 'wp_ajax_bsfef_check_slot_availability', array( $this, 'ajax_check_slot_availability' ) );
		add_action( 'wp_ajax_nopriv_bsfef_check_slot_availability', array( $this, 'ajax_check_slot_availability' ) );

		add_action( 'wp_ajax_bsfef_get_available_slots', array( $this, 'ajax_get_available_slots' ) );
		add_action( 'wp_ajax_nopriv_bsfef_get_available_slots', array( $this, 'ajax_get_available_slots' ) );

		add_action( 'wp_ajax_bsfef_get_fully_booked_dates', array( $this, 'ajax_get_fully_booked_dates' ) );
		add_action( 'wp_ajax_nopriv_bsfef_get_fully_booked_dates', array( $this, 'ajax_get_fully_booked_dates' ) );

		// Load textdomain
		add_action( 'init', array( $this, 'load_textdomain' ) );
	}

	/**
	 * Include required files
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function includes() {
		// Database class doesn't depend on Elementor
		if ( file_exists( BSFEF_PLUGIN_DIR . 'includes/class-database.php' ) ) {
			require_once BSFEF_PLUGIN_DIR . 'includes/class-database.php';
		}

		// Pro Features class
		if ( file_exists( BSFEF_PLUGIN_DIR . 'includes/class-pro-features.php' ) ) {
			require_once BSFEF_PLUGIN_DIR . 'includes/class-pro-features.php';
		}

		// Admin page
		if ( is_admin() && file_exists( BSFEF_PLUGIN_DIR . 'includes/class-admin-page.php' ) ) {
			require_once BSFEF_PLUGIN_DIR . 'includes/class-admin-page.php';
			BSFEF_Admin_Page::instance();
		}

		// Validation class - DISABLED (handled by field class now)
		// if (file_exists(BSFEF_PLUGIN_DIR . 'includes/class-validation.php')) {
		// require_once BSFEF_PLUGIN_DIR . 'includes/class-validation.php';
		// }
	}

	/**
	 * Load field class when Elementor Pro is ready
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function load_field_class() {
		if ( file_exists( BSFEF_PLUGIN_DIR . 'includes/class-booking-slots-field.php' ) ) {
			require_once BSFEF_PLUGIN_DIR . 'includes/class-booking-slots-field.php';
		}
	}

	/**
	 * Initialize database
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function init_database() {
		if ( class_exists( 'BSFEF_Database' ) ) {
			$db = BSFEF_Database::instance();

			// Check if table exists, create if missing
			global $wpdb;
			$table_name = $wpdb->prefix . 'bsfef_booking_slots';

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Checking table existence is necessary for database initialization
			$table_exists = $wpdb->get_var(
				$wpdb->prepare(
					'SHOW TABLES LIKE %s',
					$table_name
				)
			);

			if ( $table_exists !== $table_name ) {
				// Table doesn't exist, create it
				$db->create_tables();
			}
		}
	}

	/**
	 * Elementor Pro initialization callback
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function on_elementor_pro_init() {
		// Prevent duplicate hook registration
		if ( $this->field_registration_hooked ) {
			return;
		}

		// Ensure Elementor Pro is really loaded before proceeding
		if ( ! did_action( 'elementor_pro/init' ) ) {
			return;
		}

		$this->load_field_class();

		if ( ! class_exists( 'BSFEF_Booking_Slots_Field' ) ) {
			return;
		}

		// Register directly with the forms module
		if ( class_exists( '\ElementorPro\Plugin' ) ) {
			$elementor_pro = \ElementorPro\Plugin::instance();

			if ( property_exists( $elementor_pro, 'modules_manager' ) ) {
				$modules_manager = $elementor_pro->modules_manager;

				if ( method_exists( $modules_manager, 'get_modules' ) ) {
					try {
						$forms_module = $modules_manager->get_modules( 'forms' );
						if ( $forms_module && method_exists( $forms_module, 'add_form_field_type' ) ) {
							// Create and register the field
							$field = new \BSFEF_Booking_Slots_Field();
							$forms_module->add_form_field_type( $field->get_type(), $field );

							// Only hook into new_record for booking slots
							add_action( 'elementor_pro/forms/new_record', array( $this, 'book_slot_on_submission' ), 10, 2 );
						}
					} catch ( \Throwable $e ) {
						// Silent fail
					}
				}
			}
		}
	}

	/**
	 * Actually register the field with Elementor Pro
	 *
	 * @since 1.0.0
	 * @access public
	 * @param object $fields_manager
	 */
	public function register_booking_field( $fields_manager ) {
		if ( ! class_exists( 'BSFEF_Booking_Slots_Field' ) ) {
			return;
		}

		if ( ! $fields_manager ) {
			return;
		}

		if ( ! method_exists( $fields_manager, 'register' ) ) {
			return;
		}

		try {
			$field_instance = new \BSFEF_Booking_Slots_Field();
			$fields_manager->register( $field_instance );
		} catch ( \Throwable $e ) {
			// Silent fail
		}
	}

	/**
	 * Initialize validation
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function init_validation() {
		if ( class_exists( 'BSFEF_Validation' ) ) {
			BSFEF_Validation::instance();
		}
	}

	/**
	 * Enqueue frontend scripts and styles
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( 'jquery-ui-datepicker' );

		// Note: jQuery UI CSS styling is handled in frontend.css
		// WordPress.org prohibits external CDN resources, so we use bundled styles

		wp_enqueue_script(
			'bsfef-frontend',
			BSFEF_PLUGIN_URL . 'assets/js/frontend.js',
			array( 'jquery', 'jquery-ui-datepicker' ),
			BSFEF_VERSION,
			true
		);

		wp_enqueue_style(
			'bsfef-frontend',
			BSFEF_PLUGIN_URL . 'assets/css/frontend.css',
			array(),
			BSFEF_VERSION
		);

		// Localize script (maintain legacy efbs_ajax global for backwards compatibility)
		$ajax_params = array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'bsfef_nonce' ),
			'messages' => array(
				'slot_unavailable' => __( 'This slot is no longer available. Please select another time.', 'booking-slots-for-elementor-forms-lite' ),
				'loading'          => __( 'Loading available slots...', 'booking-slots-for-elementor-forms-lite' ),
				'error'            => __( 'An error occurred. Please try again.', 'booking-slots-for-elementor-forms-lite' ),
			),
		);

		wp_localize_script( 'bsfef-frontend', 'bsfef_ajax', $ajax_params );
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_enqueue_scripts( $hook ) {
		// Only load on our plugin's admin pages and Elementor editor
		$allowed_hooks = array( 'toplevel_page_bsfef-bookings' );

		// Check if we're in Elementor editor
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for Elementor editor context
		$is_elementor_editor = ( isset( $_GET['action'] ) && $_GET['action'] === 'elementor' );

		if ( in_array( $hook, $allowed_hooks ) || $is_elementor_editor ) {
			wp_enqueue_style(
				'bsfef-admin',
				BSFEF_PLUGIN_URL . 'assets/css/admin.css',
				array(),
				BSFEF_VERSION
			);
		}
	}

	/**
	 * AJAX handler for checking slot availability
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function ajax_check_slot_availability() {
		check_ajax_referer( 'bsfef_nonce', 'nonce' );

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$date = isset( $_POST['date'] ) ? sanitize_text_field( wp_unslash( $_POST['date'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$time = isset( $_POST['time'] ) ? sanitize_text_field( wp_unslash( $_POST['time'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$form_id = isset( $_POST['form_id'] ) ? sanitize_text_field( wp_unslash( $_POST['form_id'] ) ) : null;

		// Get effective form ID (PRO: actual ID, FREE: 0 for shared slots)
		$effective_form_id = class_exists( 'BSFEF_Pro_Features' ) ? BSFEF_Pro_Features::get_effective_form_id( $form_id ) : '0';

		if ( empty( $date ) || empty( $time ) ) {
			wp_send_json_error( 'Missing parameters' );
			return;
		}

		if ( ! class_exists( 'BSFEF_Database' ) ) {
			wp_send_json_error( 'Database class not loaded' );
			return;
		}

		$is_available = BSFEF_Database::instance()->is_slot_available( $date, $time, $effective_form_id );

		wp_send_json(
			array(
				'success'   => true,
				'available' => $is_available,
			)
		);
	}

	/**
	 * AJAX handler for getting available slots
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function ajax_get_available_slots() {
		check_ajax_referer( 'bsfef_nonce', 'nonce' );

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$date = isset( $_POST['date'] ) ? sanitize_text_field( wp_unslash( $_POST['date'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$time_start = isset( $_POST['time_start'] ) ? sanitize_text_field( wp_unslash( $_POST['time_start'] ) ) : '09:00';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$time_end = isset( $_POST['time_end'] ) ? sanitize_text_field( wp_unslash( $_POST['time_end'] ) ) : '17:00';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$interval = isset( $_POST['interval'] ) ? (int) $_POST['interval'] : 30;
		$notice   = isset( $_POST['notice'] ) ? intval( $_POST['notice'] ) : null;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$form_id = isset( $_POST['form_id'] ) ? sanitize_text_field( wp_unslash( $_POST['form_id'] ) ) : null;

		// Get effective form ID (PRO: actual ID, FREE: 0 for shared slots)
		$effective_form_id = class_exists( 'BSFEF_Pro_Features' ) ? BSFEF_Pro_Features::get_effective_form_id( $form_id ) : '0';

		if ( empty( $date ) ) {
			wp_send_json_error( 'Missing date parameter' );
			return;
		}

		if ( ! class_exists( 'BSFEF_Database' ) ) {
			wp_send_json_error( 'Database class not loaded' );
			return;
		}

		// Generate time slots based on interval
		$all_slots = $this->generate_time_slots_for_ajax( $time_start, $time_end, $interval );

		// Filter out booked slots for this specific form (or shared if free)
		$available_slots = array();
		$db              = BSFEF_Database::instance();

		// Determine timezone and server time for cutoff (respecting WP timezone)
		try {
			if ( function_exists( 'wp_timezone' ) ) {
				$tz_obj = wp_timezone();
			} else {
				$tz_string = get_option( 'timezone_string' );
				$tz_obj    = $tz_string ? new DateTimeZone( $tz_string ) : new DateTimeZone( 'UTC' );
			}
		} catch ( Exception $e ) {
			$tz_obj = new DateTimeZone( 'UTC' );
		}

		$server_now = new DateTime( 'now', $tz_obj );

		// Lite version does not support minimum booking notice - always use 0
		$min_notice_val = 0;
		$notice         = 0;
		$cutoff_dt      = clone $server_now; // no additional minutes

		foreach ( $all_slots as $time ) {
			if ( $db->is_slot_available( $date, $time, $effective_form_id, $notice, $interval ) ) {
				$available_slots[] = array(
					'time'  => $time,
					'label' => $time,
				);
			}
		}

		wp_send_json(
			array(
				'success'    => true,
				'slots'      => $available_slots,
				'serverTime' => $server_now->format( 'Y-m-d H:i:s' ),
				'notice'     => $min_notice_val,
				'cutoff'     => $cutoff_dt->format( 'Y-m-d H:i:s' ),
			)
		);
	}

	/**
	 * AJAX handler for getting fully booked dates
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function ajax_get_fully_booked_dates() {
		check_ajax_referer( 'bsfef_nonce', 'nonce' );

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verified above, sanitized below
		$ranges = isset( $_POST['ranges'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['ranges'] ) ) : array();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$interval = isset( $_POST['interval'] ) ? (int) $_POST['interval'] : 30;
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
		$form_id = isset( $_POST['form_id'] ) ? sanitize_text_field( wp_unslash( $_POST['form_id'] ) ) : null;

		// Get effective form ID (PRO: actual ID, FREE: 0 for shared slots)
		$effective_form_id = class_exists( 'BSFEF_Pro_Features' ) ? BSFEF_Pro_Features::get_effective_form_id( $form_id ) : '0';

		if ( empty( $ranges ) ) {
			wp_send_json_error( 'Missing ranges parameter' );
			return;
		}

		if ( ! class_exists( 'BSFEF_Database' ) ) {
			wp_send_json_error( 'Database class not loaded' );
			return;
		}

		$db                 = BSFEF_Database::instance();
		$fully_booked_dates = array();

		// Check each range
		foreach ( $ranges as $range ) {
			if ( ! isset( $range['date_start'], $range['date_end'], $range['time_start'], $range['time_end'] ) ) {
				continue;
			}

			$date_start = sanitize_text_field( $range['date_start'] );
			$date_end   = sanitize_text_field( $range['date_end'] );
			$time_start = sanitize_text_field( $range['time_start'] );
			$time_end   = sanitize_text_field( $range['time_end'] );

			// Generate all dates in this range
			$start = new DateTime( $date_start );
			$end   = new DateTime( $date_end );

			while ( $start <= $end ) {
				$current_date = $start->format( 'Y-m-d' );

				// Generate all possible time slots for this date
				$all_slots = $this->generate_time_slots_for_ajax( $time_start, $time_end, $interval );

				// Count available slots (using effective_form_id)
				$available_count = 0;
				foreach ( $all_slots as $time ) {
					if ( $db->is_slot_available( $current_date, $time, $effective_form_id ) ) {
						++$available_count;
					}
				}

				// If no slots are available, this date is fully booked
				if ( $available_count === 0 && count( $all_slots ) > 0 ) {
					$fully_booked_dates[] = $current_date;
				}

				$start->modify( '+1 day' );
			}
		}

		wp_send_json(
			array(
				'success'            => true,
				'fully_booked_dates' => array_unique( $fully_booked_dates ),
			)
		);
	}

	/**
	 * Generate time slots for AJAX response
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $start_time
	 * @param string $end_time
	 * @param int    $interval
	 * @return array
	 */
	private function generate_time_slots_for_ajax( $start_time, $end_time, $interval ) {
		$slots = array();

		try {
			$start        = new DateTime( $start_time );
			$end          = new DateTime( $end_time );
			$interval_obj = new DateInterval( 'PT' . $interval . 'M' );

			$current = clone $start;
			while ( $current <= $end ) {
				$slots[] = $current->format( 'H:i' );
				$current->add( $interval_obj );
			}
		} catch ( Exception $e ) {
			// Silent fail - return empty array
		}

		return $slots;
	}

	/**
	 * Book slot when form is successfully submitted
	 *
	 * @since 1.0.0
	 * @access public
	 * @param \ElementorPro\Modules\Forms\Classes\Form_Record  $record
	 * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler
	 */
	public function book_slot_on_submission( $record, $ajax_handler ) {
		// Wrap everything in try-catch to prevent breaking Elementor
		try {
			$form_id = $record->get_form_settings( 'id' );
			$fields  = $record->get( 'fields' );

			// Quick check - does this form even have a booking field?
			$has_booking_field = false;
			foreach ( $fields as $field ) {
				if ( isset( $field['type'] ) && $field['type'] === 'booking_slots' ) {
					$has_booking_field = true;
					break;
				}
			}

			// Don't process forms without booking fields - bail immediately
			if ( ! $has_booking_field ) {
				return;
			}

			// Process the booking
			foreach ( $fields as $field_id => $field ) {
				if ( isset( $field['type'] ) && $field['type'] === 'booking_slots' ) {
					// Use raw_value which contains the original date|time format
					$raw_value = isset( $field['raw_value'] ) ? $field['raw_value'] : ( isset( $field['value'] ) ? $field['value'] : '' );

					if ( ! empty( $raw_value ) ) {
						// Parse the combined value (date|time)
						$parts = explode( '|', $raw_value );
						if ( count( $parts ) === 2 ) {
							$date = sanitize_text_field( $parts[0] );
							$time = sanitize_text_field( $parts[1] );

							// Book the slot
							if ( class_exists( 'BSFEF_Database' ) ) {
								$db = BSFEF_Database::instance();

								$db->book_slot(
									$date,
									$time,
									array(
										'form_id'    => $form_id,
										'user_ip'    => $this->get_user_ip(),
										'user_agent' => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
									)
								);
							}
						}
					}
				}
			}
		} catch ( Exception $e ) {
			// Silently log error - don't break Elementor's flow
		}
	}


	/**
	 * Get user IP address
	 *
	 * @since 1.0.0
	 * @access private
	 * @return string
	 */
	private function get_user_ip() {
		if ( ! empty( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
		} elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
		} else {
			return isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '';
		}
	}

	/**
	 * Load plugin textdomain
	 * Load plugin textdomain
	 *
	 * Note: WordPress.org automatically loads translations for plugins.
	 * This function is kept for backward compatibility but does nothing.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function load_textdomain() {
		// WordPress.org automatically loads translations since WP 4.6
		// No manual loading needed
	}

	/**
	 * Plugin activation
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function activate() {
		// Load required files for activation
		require_once BSFEF_PLUGIN_DIR . 'includes/class-database.php';

		// Create database tables
		BSFEF_Database::instance()->create_tables();
	}

	/**
	 * Plugin deactivation
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function deactivate() {
		// Cleanup tasks on deactivation if needed
	}

	/**
	 * Admin notice
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin notice display only, not processing form data
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}
		/* translators: 1: Plugin name, 2: Required plugin name */
		$message = sprintf(
			/* translators: 1: Plugin name, 2: Required plugin name (Elementor) */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'booking-slots-for-elementor-forms-lite' ),
			'<strong>' . esc_html__( 'Booking Slots for Elementor Forms', 'booking-slots-for-elementor-forms-lite' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'booking-slots-for-elementor-forms-lite' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post( $message ) );
	}

	/**
	 * Admin notice
	 * Warning when the site doesn't have Elementor Pro installed or activated.
	 *
	 * @since 1.0.1
	 * @access public
	 */
	public function admin_notice_missing_elementor_pro() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin notice display only, not processing form data
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}
		/* translators: 1: Plugin name, 2: Required plugin name (Elementor Pro) */
		$message = sprintf(
			/* translators: 1: Plugin name, 2: Required plugin name (Elementor Pro) */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated. This plugin extends Elementor Forms, which is a PRO feature.', 'booking-slots-for-elementor-forms-lite' ),
			'<strong>' . esc_html__( 'Booking Slots for Elementor Forms', 'booking-slots-for-elementor-forms-lite' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor Pro', 'booking-slots-for-elementor-forms-lite' ) . '</strong>'
		);

		$button = sprintf(
			'<a href="https://elementor.com/pro/" target="_blank" class="button button-primary" style="margin-left: 10px;">%s</a>',
			esc_html__( 'Get Elementor Pro', 'booking-slots-for-elementor-forms-lite' )
		);

		printf( '<div class="notice notice-error is-dismissible"><p>%1$s %2$s</p></div>', wp_kses_post( $message ), wp_kses_post( $button ) );
	}

	/**
	 * Admin notice
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin notice display only, not processing form data
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}
		/* translators: 1: Plugin name, 2: Required plugin name, 3: Minimum version number */
		$message = sprintf(
			/* translators: 1: Plugin name, 2: Required plugin name (Elementor), 3: Minimum version number */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'booking-slots-for-elementor-forms-lite' ),
			'<strong>' . esc_html__( 'Booking Slots for Elementor Forms', 'booking-slots-for-elementor-forms-lite' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'booking-slots-for-elementor-forms-lite' ) . '</strong>',
			'3.0.0'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post( $message ) );
	}

	/**
	 * Admin notice
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Admin notice display only, not processing form data
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}
		/* translators: 1: Plugin name, 2: PHP, 3: Minimum PHP version number */
		$message = sprintf(
			/* translators: 1: Plugin name, 2: PHP, 3: Minimum PHP version number */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'booking-slots-for-elementor-forms-lite' ),
			'<strong>' . esc_html__( 'Booking Slots for Elementor Forms', 'booking-slots-for-elementor-forms-lite' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'booking-slots-for-elementor-forms-lite' ) . '</strong>',
			'7.4'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post( $message ) );
	}
}

// Initialize the plugin
BSFEF_Elementor_Form_Booking_Slots::instance();
