/**
 * Elementor Forms Booking Slots - Frontend JavaScript
 *
 * @since 1.0.0
 */

(function($) {
    'use strict';

    /**
     * Booking Slots Handler
     */
    class BookingSlotsHandler {
        constructor() {
            this.init();
        }

        /**
         * Initialize the handler
         */
        init() {
            this.bindEvents();
            this.initializeDatePickers();
        }

        /**
         * Bind events
         */
        bindEvents() {
            // Date picker change event
            $(document).on('change', '.bsfef-date-picker', (e) => {
                this.handleDateChange(e);
            });

            // Time picker change event
            $(document).on('change', '.bsfef-time-picker', (e) => {
                this.handleTimeChange(e);
            });

            // Form submission validation
            $(document).on('submit', '.elementor-form', (e) => {
                this.validateBookingSlots(e);
            });

            // Real-time slot availability check
            $(document).on('focus', '.bsfef-time-picker', (e) => {
                this.checkSlotAvailability(e);
            });
        }

        /**
         * Initialize date pickers
         */
        initializeDatePickers() {
            $('.bsfef-date-picker').each((index, element) => {
                const $element = $(element);
                
                // Check if already initialized and not destroyed
                if ($element.hasClass('hasDatepicker') && $element.data('datepicker-initialized')) {
                    return;
                }
                
                const config = $element.data('field-config');
                
                if (!config) {
                    return;
                }
                
                // Mark as being initialized
                $element.data('datepicker-initialized', true);
                
                // Get all available dates from ranges
                const availableDates = this.getAvailableDatesFromRanges(config.availabilityRanges);
                
                // Check if we're in perpetual mode
                const isPerpetualMode = config.availabilityRanges.length > 0 && config.availabilityRanges[0].perpetual;
                
                // Initialize datepicker with month-by-month loading
                this.initializeDatepickerWithMonthLoading($element, config, availableDates, isPerpetualMode);
            });
        }
        
        /**
         * Initialize datepicker with month-by-month fully booked date loading
         */
        initializeDatepickerWithMonthLoading($element, config, availableDates, isPerpetualMode) {
            const self = this;
            let currentMonthBookedDates = [];
            
            // Find min and max dates from all ranges
            let minDate = null;
            let maxDate = null;
            config.availabilityRanges.forEach(range => {
                const startParts = range.date_start.split('-');
                const endParts = range.date_end.split('-');
                
                const startDate = new Date(parseInt(startParts[0]), parseInt(startParts[1]) - 1, parseInt(startParts[2]));
                const endDate = new Date(parseInt(endParts[0]), parseInt(endParts[1]) - 1, parseInt(endParts[2]));
                
                if (!minDate || startDate < minDate) minDate = startDate;
                if (!maxDate || endDate > maxDate) maxDate = endDate;
            });
            
            // Initialize jQuery UI datepicker
            $element.datepicker({
                dateFormat: 'yy-mm-dd',
                minDate: minDate,
                maxDate: maxDate,
                onChangeMonthYear: function(year, month, inst) {
                    // Load fully booked dates for the new month
                    self.loadFullyBookedDatesForMonth($element, config, year, month, isPerpetualMode)
                        .then((bookedDates) => {
                            currentMonthBookedDates = bookedDates;
                            // Refresh the datepicker to apply new styling
                            $element.datepicker('refresh');
                        });
                },
                beforeShowDay: (date) => {
                    const year = date.getFullYear();
                    const month = String(date.getMonth() + 1).padStart(2, '0');
                    const day = String(date.getDate()).padStart(2, '0');
                    const dateStr = year + '-' + month + '-' + day;
                    
                    let isAvailable = false;
                    
                    // Check if we're in perpetual mode
                    if (isPerpetualMode) {
                        const range = config.availabilityRanges[0];
                        
                        // Parse the date range boundaries
                        const rangeStart = new Date(range.date_start);
                        const rangeEnd = new Date(range.date_end);
                        
                        // Check if date is within the overall date range
                        if (date >= rangeStart && date <= rangeEnd) {
                            // Check if this weekday is allowed
                            const dayOfWeek = String(date.getDay());
                            if (range.weekdays && range.weekdays.includes(dayOfWeek)) {
                                isAvailable = true;
                            }
                        }
                    } else {
                        // Regular mode - check pre-generated date list
                        isAvailable = availableDates.includes(dateStr);
                    }
                    
                    // Check if date is in the past (before today)
                    const today = new Date();
                    today.setHours(0, 0, 0, 0);
                    const isPast = date < today;
                    
                    // Check if date is fully booked for current month
                    const isFullyBooked = currentMonthBookedDates.includes(dateStr);
                    
                    // Return appropriate class based on availability
                    if (!isAvailable) {
                        return [false, ''];
                    } else if (isPast || isFullyBooked) {
                        return [true, 'bsfef-fully-booked'];
                    } else {
                        return [true, 'bsfef-available'];
                    }
                },
                onSelect: (dateText, inst) => {
                    $element.trigger('change');
                }
            });

            // Hide the datepicker immediately after initialization
            $('#ui-datepicker-div').hide();
            
            // Load fully booked dates for initial month
            const today = new Date();
            this.loadFullyBookedDatesForMonth($element, config, today.getFullYear(), today.getMonth() + 1, isPerpetualMode)
                .then((bookedDates) => {
                    currentMonthBookedDates = bookedDates;
                });
        }
        
        /**
         * Load fully booked dates for a specific month
         */
        loadFullyBookedDatesForMonth($element, config, year, month, isPerpetualMode) {
            return new Promise((resolve, reject) => {
                // Calculate first and last day of the month
                const firstDay = year + '-' + String(month).padStart(2, '0') + '-01';
                const lastDay = new Date(year, month, 0).getDate();
                const lastDayStr = year + '-' + String(month).padStart(2, '0') + '-' + String(lastDay).padStart(2, '0');
                
                // Create a month-specific range
                let monthRanges = [];
                
                if (isPerpetualMode) {
                    const range = config.availabilityRanges[0];
                    monthRanges = [{
                        date_start: firstDay,
                        date_end: lastDayStr,
                        time_start: range.time_start,
                        time_end: range.time_end,
                        perpetual: true,
                        weekdays: range.weekdays
                    }];
                } else {
                    // For specific ranges, only check ranges that overlap with this month
                    config.availabilityRanges.forEach(range => {
                        const rangeStart = new Date(range.date_start);
                        const rangeEnd = new Date(range.date_end);
                        const monthStart = new Date(firstDay);
                        const monthEnd = new Date(lastDayStr);
                        
                        // Check if range overlaps with this month
                        if (rangeStart <= monthEnd && rangeEnd >= monthStart) {
                            monthRanges.push(range);
                        }
                    });
                }
                
                if (monthRanges.length === 0) {
                    resolve([]);
                    return;
                }
                
                // Fetch fully booked dates for this month only
                this.getFullyBookedDates(monthRanges, config.interval, config.formId)
                    .then((fullyBookedDates) => {
                        resolve(fullyBookedDates);
                    })
                    .catch((error) => {
                        resolve([]);
                    });
            });
        }
        
        /**
         * Get all available dates from ranges
         */
        getAvailableDatesFromRanges(ranges) {
            const dates = [];
            
            ranges.forEach(range => {
                // Skip date generation for perpetual mode - we'll check weekdays on the fly
                if (range.perpetual) {
                    return;
                }
                
                // Parse dates without timezone issues
                const startParts = range.date_start.split('-');
                const endParts = range.date_end.split('-');
                
                const startDate = new Date(parseInt(startParts[0]), parseInt(startParts[1]) - 1, parseInt(startParts[2]));
                const endDate = new Date(parseInt(endParts[0]), parseInt(endParts[1]) - 1, parseInt(endParts[2]));
                
                // Generate all dates in the range (inclusive)
                const currentDate = new Date(startDate);
                while (currentDate <= endDate) {
                    const dateStr = currentDate.getFullYear() + '-' + 
                                  String(currentDate.getMonth() + 1).padStart(2, '0') + '-' + 
                                  String(currentDate.getDate()).padStart(2, '0');
                    dates.push(dateStr);
                    currentDate.setDate(currentDate.getDate() + 1);
                }
            });
            
            return dates;
        }

        /**
         * Get fully booked dates from server
         */
        getFullyBookedDates(ranges, interval, formId) {
            return new Promise((resolve, reject) => {
                $.ajax({
                    url: bsfef_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'bsfef_get_fully_booked_dates',
                        nonce: bsfef_ajax.nonce,
                        ranges: ranges,
                        interval: interval,
                        form_id: formId
                    },
                    success: (response) => {
                        if (response.success && response.fully_booked_dates) {
                            resolve(response.fully_booked_dates);
                        } else {
                            resolve([]);
                        }
                    },
                    error: (xhr, status, error) => {
                        reject(error);
                    }
                });
            });
        }

        /**
         * Handle date picker change
         */
        handleDateChange(e) {
            const $dateField = $(e.target);
            const $timeField = $('#' + $dateField.data('target-time'));
            const $combinedField = $('#' + $dateField.data('target-combined'));
            const selectedDate = $dateField.val();
            const config = $dateField.data('field-config');

            if (!config) {
                return;
            }

            // Clear time selection
            $timeField.val('').prop('disabled', true);
            $combinedField.val('');

            if (!selectedDate) {
                return;
            }

            // Find which range this date belongs to
            const range = this.getRangeForDate(selectedDate, config.availabilityRanges);
            if (!range) {
                return;
            }

            // Show loading
            this.showLoading($dateField);

            // Get available slots for the selected date using the range's time settings
            this.getAvailableSlots(selectedDate, range, config)
                .then((availableSlots) => {
                    this.populateTimeOptions($timeField, availableSlots);
                    $timeField.prop('disabled', false);
                })
                .catch((error) => {
                    this.showError($dateField, bsfef_ajax.messages.error);
                })
                .finally(() => {
                    this.hideLoading($dateField);
                });
        }
        
        /**
         * Find which range a date belongs to
         */
        getRangeForDate(date, ranges) {
            const selectedDate = new Date(date);
            
            for (const range of ranges) {
                const startDate = new Date(range.date_start);
                const endDate = new Date(range.date_end);
                
                if (selectedDate >= startDate && selectedDate <= endDate) {
                    return range;
                }
            }
            
            return null;
        }

        /**
         * Handle time picker change
         */
        handleTimeChange(e) {
            const $timeField = $(e.target);
            const $dateField = $timeField.closest('.bsfef-booking-wrapper').find('.bsfef-date-picker');
            const $combinedField = $('#' + $timeField.data('target-combined'));
            
            const selectedDate = $dateField.val();
            const selectedTime = $timeField.val();

            // Update combined field value
            if (selectedDate && selectedTime) {
                $combinedField.val(selectedDate + '|' + selectedTime);
                
                // Double-check availability
                this.checkSlotAvailability(e, true);
            } else {
                $combinedField.val('');
            }
        }

        /**
         * Get available slots for a date
         */
        getAvailableSlots(date, range, config) {
            return new Promise((resolve, reject) => {
                // Lite version: minimum notice is not supported. Force notice to 0.
                const notice = 0;

                $.ajax({
                    url: bsfef_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'bsfef_get_available_slots',
                        nonce: bsfef_ajax.nonce,
                        date: date,
                        time_start: range.time_start,
                        time_end: range.time_end,
                        interval: range.interval || config.interval, // Use range-specific interval or fallback to global
                        form_id: config.formId,
                        notice: notice
                    },
                    success: (response) => {
                        if (response.success) {
                            let slots = response.slots || [];

                            // Use serverTime returned by the AJAX response when available so filtering is based on current server time
                            const serverTimeStr = response.serverTime || config.serverTime;
                            if (serverTimeStr) {
                                try {
                                    const server = new Date(serverTimeStr.replace(' ', 'T'));
                                    const serverDateStr = server.getFullYear() + '-' + String(server.getMonth() + 1).padStart(2, '0') + '-' + String(server.getDate()).padStart(2, '0');
                                    if (serverDateStr === date) {
                                        const cutoff = new Date(server.getTime() + notice * 60000);
                                        slots = slots.filter(slot => {
                                            const parts = slot.time.split(':');
                                            const h = parseInt(parts[0], 10);
                                            const m = parseInt(parts[1], 10);
                                            const slotDt = new Date(server.getFullYear(), server.getMonth(), server.getDate(), h, m);
                                            return slotDt.getTime() > cutoff.getTime();
                                        });
                                    }
                                } catch (e) {
                                    // If parsing fails, fall back to returning unfiltered slots
                                }
                            }

                            resolve(slots);
                        } else {
                            reject(new Error('Failed to load slots'));
                        }
                    },
                    error: (xhr, status, error) => {
                        reject(new Error(error));
                    }
                });
            });
        }

        /**
         * Check if a specific slot is still available
         */
        checkSlotAvailability(e, showWarning = false) {
            const $timeField = $(e.target);
            const $dateField = $timeField.closest('.bsfef-booking-wrapper').find('.bsfef-date-picker');
            
            const selectedDate = $dateField.val();
            const selectedTime = $timeField.val();

            if (!selectedDate || !selectedTime) {
                return;
            }

            $.ajax({
                url: bsfef_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'bsfef_check_slot_availability',
                    nonce: bsfef_ajax.nonce,
                    date: selectedDate,
                    time: selectedTime
                },
                success: (response) => {
                    if (response.success && !response.available) {
                        if (showWarning) {
                            this.showSlotUnavailableWarning($timeField);
                        }
                        // Remove the unavailable option
                        $timeField.find(`option[value="${selectedTime}"]`).remove();
                        $timeField.val('');
                        
                        const $combinedField = $('#' + $timeField.data('target-combined'));
                        $combinedField.val('');
                    }
                }
            });
        }

        /**
         * Populate time options
         */
        populateTimeOptions($timeField, availableSlots) {
            const placeholder = $timeField.find('option[value=""]').first().text();
            
            $timeField.empty();
            $timeField.append(`<option value="">${placeholder}</option>`);

            if (availableSlots.length === 0) {
                $timeField.append(`<option value="" disabled>${bsfef_ajax.messages.no_slots || 'No available slots for this date'}</option>`);
                return;
            }

            availableSlots.forEach((slot) => {
                const displayText = slot.label || slot.time;
                $timeField.append(`<option value="${slot.time}">${displayText}</option>`);
            });
        }

        /**
         * Validate booking slots before form submission
         */
        validateBookingSlots(e) {
            const $form = $(e.target);
            let isValid = true;

            $form.find('.bsfef-combined-field').each((index, element) => {
                const $field = $(element);
                const value = $field.val();
                
                if ($field.prop('required') && !value) {
                    isValid = false;
                    this.showFieldError($field, 'Please select a booking slot.');
                    return;
                }

                if (value) {
                    const parts = value.split('|');
                    if (parts.length !== 2) {
                        isValid = false;
                        this.showFieldError($field, 'Invalid booking slot format.');
                        return;
                    }

                    // Additional real-time validation could be added here
                }
            });

            if (!isValid) {
                e.preventDefault();
                return false;
            }

            return true;
        }

        /**
         * Show loading indicator
         */
        showLoading($field) {
            const $wrapper = $field.closest('.bsfef-booking-wrapper');
            $wrapper.find('.bsfef-loading').show();
        }

        /**
         * Hide loading indicator
         */
        hideLoading($field) {
            const $wrapper = $field.closest('.bsfef-booking-wrapper');
            $wrapper.find('.bsfef-loading').hide();
        }

        /**
         * Show error message
         */
        showError($field, message) {
            const $wrapper = $field.closest('.bsfef-booking-wrapper');
            
            // Remove existing error messages
            $wrapper.find('.bsfef-error-message').remove();
            
            // Add new error message
            $wrapper.append(`<div class="bsfef-error-message">${message}</div>`);
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                $wrapper.find('.bsfef-error-message').fadeOut(() => {
                    $(this).remove();
                });
            }, 5000);
        }

        /**
         * Show slot unavailable warning
         */
        showSlotUnavailableWarning($timeField) {
            const $wrapper = $timeField.closest('.bsfef-booking-wrapper');
            
            // Remove existing warnings
            $wrapper.find('.bsfef-warning-message').remove();
            
            // Add warning message
            $wrapper.append(`<div class="bsfef-warning-message">${bsfef_ajax.messages.slot_unavailable}</div>`);
            
            // Auto-hide after 7 seconds
            setTimeout(() => {
                $wrapper.find('.bsfef-warning-message').fadeOut(() => {
                    $(this).remove();
                });
            }, 7000);
        }

        /**
         * Show field error
         */
        showFieldError($field, message) {
            const $wrapper = $field.closest('.bsfef-booking-wrapper');
            
            // Remove existing error messages
            $wrapper.find('.bsfef-field-error').remove();
            
            // Add field error
            $wrapper.append(`<div class="bsfef-field-error">${message}</div>`);
            
            // Add error class
            $wrapper.addClass('bsfef-has-error');
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                $wrapper.find('.bsfef-field-error').fadeOut(() => {
                    $(this).remove();
                });
                $wrapper.removeClass('bsfef-has-error');
            }, 5000);
        }
    }

    /**
     * Initialize when document is ready
     */
    $(document).ready(() => {
        window.BookingSlotsHandler = BookingSlotsHandler;
        new BookingSlotsHandler();
    });

})(jQuery);