<?php
/**
 * BlogBotz - Main Plugin Class
 *
 * Handles plugin initialization and orchestration
 *
 * @package BlogBotz
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin class.
 */
class BlogBotz {

	/**
	 * Plugin instance.
	 *
	 * @var BlogBotz
	 */
	private static $instance = null;

	/**
	 * Admin instance.
	 *
	 * @var BlogBotz_Admin
	 */
	private $admin = null;

	/**
	 * API instance.
	 *
	 * @var BlogBotz_API
	 */
	private $api = null;

	/**
	 * Connection instance.
	 *
	 * @var BlogBotz_Connection
	 */
	private $connection = null;

	/**
	 * Get singleton instance.
	 *
	 * @return BlogBotz Plugin instance.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init();
	}

	/**
	 * Load required class files.
	 */
	private function load_dependencies() {
		require_once BLOGBOTZ_PLUGIN_DIR . 'includes/class-blogbotz-connection.php';
		require_once BLOGBOTZ_PLUGIN_DIR . 'includes/class-blogbotz-admin.php';
		require_once BLOGBOTZ_PLUGIN_DIR . 'includes/class-blogbotz-api.php';
	}

	/**
	 * Initialize plugin components.
	 */
	private function init() {
		// Initialize connection handler.
		$this->connection = new BlogBotz_Connection();

		// Initialize admin interface.
		$this->admin = new BlogBotz_Admin( $this->connection );

		// Initialize REST API.
		$this->api = new BlogBotz_API( $this->connection );

		// Add settings link to plugins page.
		add_filter( 'plugin_action_links_' . BLOGBOTZ_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );
	}

	/**
	 * Plugin activation.
	 */
	public static function activate() {
		// Load connection class.
		require_once BLOGBOTZ_PLUGIN_DIR . 'includes/class-blogbotz-connection.php';

		// Generate unique site key if not exists.
		if ( ! get_option( 'blogbotz_site_key' ) ) {
			update_option( 'blogbotz_site_key', wp_generate_uuid4() );
		}

		// Set default options.
		add_option( 'blogbotz_api_key', '' );
		add_option( 'blogbotz_site_id', '' );
		add_option( 'blogbotz_connected_at', '' );
		add_option( 'blogbotz_user_email', '' );
		add_option( 'blogbotz_user_id', '' );
		add_option( 'blogbotz_app_password', '' );

		// Create API user with Application Password.
		$connection = new BlogBotz_Connection();
		// Get author name from option if set, otherwise use default.
		$author_name = get_option( 'blogbotz_author_name', 'BlogBotz' );
		$user_result = $connection->create_api_user( $author_name );

		if ( is_wp_error( $user_result ) ) {
			// Log error but do not prevent activation.
			blogbotz_log( 'BlogBotz: Failed to create API user on activation: ' . $user_result->get_error_message() );
		}

		// Flush rewrite rules for REST API.
		flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation.
	 */
	public static function deactivate() {
		// Clear connection data (BlogBotz server side).
		// Keep local WordPress user & app password so reactivation is easy.
		delete_option( 'blogbotz_api_key' );
		delete_option( 'blogbotz_site_id' );
		delete_option( 'blogbotz_connected_at' );
		delete_option( 'blogbotz_user_email' );
		delete_option( 'blogbotz_dashboard_url' );

		// Delete transients.
		delete_transient( 'blogbotz_user_creation_error' );
		delete_transient( 'blogbotz_connection_error' );

		// Note: We keep the following for easy reactivation:
		// - blogbotz_site_key (unique identifier),
		// - blogbotz_user_id (WordPress API user),
		// - blogbotz_app_password (Application Password),
		// - blogbotz_author_name (author preference).

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Add settings link to plugins page.
	 *
	 * @param array $links Plugin action links.
	 * @return array Modified links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = '<a href="' . admin_url( 'admin.php?page=blogbotz-ai-blog-automation' ) . '">' . __( 'Settings', 'blogbotz-ai-blog-automation' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Get admin instance.
	 *
	 * @return BlogBotz_Admin Admin instance.
	 */
	public function get_admin() {
		return $this->admin;
	}

	/**
	 * Get API instance.
	 *
	 * @return BlogBotz_API API instance.
	 */
	public function get_api() {
		return $this->api;
	}

	/**
	 * Get connection instance.
	 *
	 * @return BlogBotz_Connection Connection instance.
	 */
	public function get_connection() {
		return $this->connection;
	}
}
