<?php
/**
 * BlogBotz - Connection Handler Class
 *
 * Handles connection state and communication with BlogBotz app
 *
 * @package BlogBotz
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Connection handler class.
 */
class BlogBotz_Connection {

	/**
	 * Check if connected to BlogBotz.
	 *
	 * @return bool True if connected.
	 */
	public function is_connected() {
		$api_key = get_option( 'blogbotz_api_key' );
		return ! empty( $api_key );
	}

	/**
	 * Get connection data.
	 *
	 * @return array|null Connection data or null if not connected.
	 */
	public function get_connection_data() {
		if ( ! $this->is_connected() ) {
			return null;
		}

		return array(
			'connected_at' => get_option( 'blogbotz_connected_at' ),
			'user_email'   => get_option( 'blogbotz_user_email' ),
		);
	}

	/**
	 * Get site key.
	 *
	 * @return string Site key.
	 */
	public function get_site_key() {
		return get_option( 'blogbotz_site_key', '' );
	}

	/**
	 * Get API key.
	 *
	 * @return string API key.
	 */
	public function get_api_key() {
		return get_option( 'blogbotz_api_key', '' );
	}

	/**
	 * Disconnect from BlogBotz.
	 *
	 * @return bool True on success.
	 */
	public function disconnect() {
		delete_option( 'blogbotz_api_key' );
		delete_option( 'blogbotz_connected_at' );
		delete_option( 'blogbotz_user_email' );
		delete_option( 'blogbotz_dashboard_url' );

		return true;
	}

	/**
	 * Store connection credentials.
	 *
	 * @param string $api_key    API key.
	 * @param string $user_email User email.
	 * @return bool True on success.
	 */
	public function store_credentials( $api_key, $user_email = '' ) {
		update_option( 'blogbotz_api_key', sanitize_text_field( $api_key ) );
		update_option( 'blogbotz_connected_at', current_time( 'mysql' ) );

		if ( $user_email ) {
			update_option( 'blogbotz_user_email', sanitize_email( $user_email ) );
		}

		return true;
	}

	/**
	 * Test connection to BlogBotz app.
	 *
	 * @return array Connection status array.
	 */
	public function test_connection() {
		if ( ! $this->is_connected() ) {
			return array(
				'connected' => false,
				'message'   => __( 'Not connected to BlogBotz', 'blogbotz-ai-blog-automation' ),
			);
		}

		// Make a test request to BlogBotz API.
		$response = wp_remote_get(
			BLOGBOTZ_APP_URL . '/api/platforms/wordpress/verify',
			array(
				'timeout' => 10,
				'headers' => array(
					'X-BlogBotz-Key' => $this->get_api_key(),
					'X-Site-Key'     => $this->get_site_key(),
					'Content-Type'   => 'application/json',
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return array(
				'connected' => false,
				'message'   => $response->get_error_message(),
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );

		if ( 200 === $response_code ) {
			return array(
				'connected' => true,
				'message'   => __( 'Successfully connected to BlogBotz', 'blogbotz-ai-blog-automation' ),
			);
		}

		return array(
			'connected' => false,
			'message'   => sprintf(
				/* translators: %d: HTTP response code */
				__( 'Connection failed with HTTP %d', 'blogbotz-ai-blog-automation' ),
				$response_code
			),
		);
	}

	/**
	 * Get site information for connection.
	 *
	 * @return array Site information.
	 */
	public function get_site_info() {
		return array(
			'site_url'          => get_site_url(),
			'site_name'         => get_bloginfo( 'name' ),
			'site_description'  => get_bloginfo( 'description' ),
			'wordpress_version' => get_bloginfo( 'version' ),
			'php_version'       => PHP_VERSION,
			'is_woocommerce'    => class_exists( 'WooCommerce' ),
			'language'          => get_locale(),
			'timezone'          => wp_timezone_string(),
			'plugin_version'    => BLOGBOTZ_VERSION,
		);
	}

	/**
	 * Check system requirements.
	 *
	 * @return array Requirements check results.
	 */
	public function check_requirements() {
		$wp_version    = get_bloginfo( 'version' );
		$wp_version_ok = version_compare( $wp_version, '5.6', '>=' );

		// Check if Application Passwords are available (available in WordPress 5.6+).
		$app_passwords_available = class_exists( 'WP_Application_Passwords' );

		$site_key_exists     = ! empty( get_option( 'blogbotz_site_key' ) );
		$user_exists         = ! empty( get_option( 'blogbotz_user_id' ) );
		$app_password_exists = ! empty( get_option( 'blogbotz_app_password' ) );

		return array(
			'wp_version_ok'           => $wp_version_ok,
			'wp_version'              => $wp_version,
			'app_passwords_available' => $app_passwords_available,
			'site_key_exists'         => $site_key_exists,
			'user_exists'             => $user_exists,
			'app_password_exists'     => $app_password_exists,
			'all_ok'                  => $wp_version_ok && $app_passwords_available && $site_key_exists,
		);
	}

	/**
	 * Create or get BlogBotz API user.
	 *
	 * @param string $author_name Optional author name for display name and nickname.
	 * @return array|WP_Error User data with Application Password or error.
	 */
	public function create_api_user( $author_name = '' ) {
		// Check WordPress version first.
		$wp_version = get_bloginfo( 'version' );
		if ( version_compare( $wp_version, '5.6', '<' ) ) {
			return new WP_Error(
				'wp_version_insufficient',
				/* translators: %s: Current WordPress version number. */
				sprintf( __( 'WordPress 5.6 or higher is required for Application Passwords. Your WordPress version: %s', 'blogbotz-ai-blog-automation' ), $wp_version )
			);
		}

		// Check if Application Passwords class exists (available in WordPress 5.6+).
		if ( ! class_exists( 'WP_Application_Passwords' ) ) {
			return new WP_Error(
				'app_passwords_unavailable',
				sprintf(
					/* translators: %s: Current WordPress version number. */
					__( 'Application Passwords are not available in WordPress %s. The class file (wp-includes/class-wp-application-passwords.php) was not found. This may indicate a corrupted WordPress installation or the feature being disabled.', 'blogbotz-ai-blog-automation' ),
					$wp_version
				)
			);
		}

		$username = 'blogbotz-ai-blog-automation';
		$email    = 'blogbotz@' . wp_parse_url( get_site_url(), PHP_URL_HOST );

		// Use provided author name or get from stored option, or default to 'BlogBotz'.
		if ( empty( $author_name ) ) {
			$author_name = get_option( 'blogbotz_author_name', 'BlogBotz' );
		}

		// Check if user already exists.
		$user = get_user_by( 'login', $username );

		if ( ! $user ) {
			// Create new user.
			$user_id = wp_create_user(
				$username,
				wp_generate_password( 32, true, true ), // Strong random password.
				$email
			);

			if ( is_wp_error( $user_id ) ) {
				return $user_id;
			}

			$user = get_user_by( 'id', $user_id );

			// Set role to Editor (can create/edit/publish posts, upload media).
			$user->set_role( 'editor' );

			// Set display name and nickname to author name.
			wp_update_user(
				array(
					'ID'           => $user_id,
					'display_name' => $author_name,
					'nickname'     => $author_name,
				)
			);

			// Ensure user has all required capabilities.
			$this->ensure_user_permissions( $user_id );

			// Store user ID and author name.
			update_option( 'blogbotz_user_id', $user_id );
			update_option( 'blogbotz_author_name', $author_name );
		} else {
			// Update existing user's display name if author name is provided.
			if ( $author_name ) {
				wp_update_user(
					array(
						'ID'           => $user->ID,
						'display_name' => $author_name,
						'nickname'     => $author_name,
					)
				);
				update_option( 'blogbotz_author_name', $author_name );
			}

			// Ensure user has Editor role and required capabilities.
			$this->ensure_user_permissions( $user->ID );
		}

		// Check if Application Password already exists.
		$app_passwords         = WP_Application_Passwords::get_user_application_passwords( $user->ID );
		$existing_app_password = null;

		foreach ( $app_passwords as $app_password ) {
			if ( 'BlogBotz API Access' === $app_password['name'] ) {
				// Application Password exists but we can't retrieve it (WordPress doesn't store it).
				// We need to create a new one or use the stored one.
				$stored_app_password = get_option( 'blogbotz_app_password' );
				if ( $stored_app_password ) {
					return array(
						'user_id'      => $user->ID,
						'username'     => $user->user_login,
						'app_password' => $stored_app_password,
					);
				}
				// If stored password doesn't exist, we'll create a new one below.
				break;
			}
		}

		// Create new Application Password.
		// Note: create_new_application_password returns array($new_password, $new_item).
		$app_password_result = WP_Application_Passwords::create_new_application_password(
			$user->ID,
			array(
				'name' => 'BlogBotz API Access',
			)
		);

		if ( is_wp_error( $app_password_result ) ) {
			return $app_password_result;
		}

		// The method returns array($new_password, $new_item).
		// $new_password is the plain text password (index 0).
		// $new_item is the password details array (index 1).
		$new_password = $app_password_result[0];
		$new_item     = $app_password_result[1];

		// Store the Application Password (it's only shown once).
		update_option( 'blogbotz_app_password', $new_password );
		update_option( 'blogbotz_user_id', $user->ID );

		return array(
			'user_id'      => $user->ID,
			'username'     => $user->user_login,
			'app_password' => $new_password,
		);
	}

	/**
	 * Ensure BlogBotz API user has correct role and capabilities.
	 *
	 * @param int $user_id User ID.
	 * @return bool True if user has correct permissions, false otherwise.
	 */
	public function ensure_user_permissions( $user_id ) {
		$user = get_user_by( 'id', $user_id );
		if ( ! $user ) {
			return false;
		}

		// Check if user has editor or administrator role.
		$has_editor_role = in_array( 'editor', $user->roles, true );
		$has_admin_role  = in_array( 'administrator', $user->roles, true );

		if ( ! $has_editor_role && ! $has_admin_role ) {
			// Set to editor role.
			$user->set_role( 'editor' );
			blogbotz_log( 'BlogBotz: Updated user role to Editor for user ID ' . $user_id );
		}

		// Verify user has required capabilities.
		$required_caps = array( 'edit_posts', 'publish_posts', 'upload_files' );
		$missing_caps  = array();

		foreach ( $required_caps as $cap ) {
			if ( ! $user->has_cap( $cap ) ) {
				$missing_caps[] = $cap;
			}
		}

		if ( ! empty( $missing_caps ) ) {
			// Add missing capabilities.
			foreach ( $missing_caps as $cap ) {
				$user->add_cap( $cap );
			}
			blogbotz_log( 'BlogBotz: Added missing capabilities to user ID ' . $user_id . ': ' . implode( ', ', $missing_caps ) );
		}

		return true;
	}

	/**
	 * Get BlogBotz API user credentials.
	 *
	 * @return array|null User credentials or null if not available.
	 */
	public function get_api_user_credentials() {
		$user_id      = get_option( 'blogbotz_user_id' );
		$app_password = get_option( 'blogbotz_app_password' );

		if ( ! $user_id || ! $app_password ) {
			return null;
		}

		$user = get_user_by( 'id', $user_id );
		if ( ! $user ) {
			return null;
		}

		// Ensure user has correct permissions.
		$this->ensure_user_permissions( $user_id );

		return array(
			'user_id'      => $user_id,
			'username'     => $user->user_login,
			'app_password' => $app_password,
		);
	}

	/**
	 * Delete BlogBotz API user.
	 *
	 * @return bool True on success, false on failure.
	 */
	public function delete_api_user() {
		$user_id = get_option( 'blogbotz_user_id' );

		if ( ! $user_id ) {
			return true; // Already deleted or never created.
		}

		$user = get_user_by( 'id', $user_id );
		if ( ! $user ) {
			// User doesn't exist, just clean up options.
			delete_option( 'blogbotz_user_id' );
			delete_option( 'blogbotz_app_password' );
			return true;
		}

		// Revoke all Application Passwords for this user (class available in WordPress 5.6+).
		if ( class_exists( 'WP_Application_Passwords' ) ) {
			$app_passwords = WP_Application_Passwords::get_user_application_passwords( $user_id );
			foreach ( $app_passwords as $app_password ) {
				$app_name = is_array( $app_password ) ? ( $app_password['name'] ?? '' ) : '';
				if ( 'BlogBotz API Access' === $app_name ) {
					$uuid = is_array( $app_password ) ? ( $app_password['uuid'] ?? '' ) : '';
					if ( $uuid ) {
						WP_Application_Passwords::delete_application_password( $user_id, $uuid );
					}
				}
			}
		}

		// Delete the user using wp_delete_user().
		// This function is only available in admin context, so we load the required file
		// and immediately use the function (per WordPress coding standards exception).
		if ( ! function_exists( 'wp_delete_user' ) ) {
			require_once ABSPATH . 'wp-admin/includes/user.php';
		}
		wp_delete_user( $user_id );

		// Clean up options.
		delete_option( 'blogbotz_user_id' );
		delete_option( 'blogbotz_app_password' );

		return true;
	}
}
