<?php
/**
 * BlogBotz - REST API Class
 *
 * Handles REST API endpoints for BlogBotz integration
 *
 * @package BlogBotz
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REST API class.
 */
class BlogBotz_API {

	/**
	 * Connection instance.
	 *
	 * @var BlogBotz_Connection
	 */
	private $connection;

	/**
	 * REST namespace.
	 *
	 * @var string
	 */
	const REST_NAMESPACE = 'blogbotz/v1';

	/**
	 * Constructor.
	 *
	 * @param BlogBotz_Connection $connection Connection instance.
	 */
	public function __construct( $connection ) {
		$this->connection = $connection;
		$this->init();
	}

	/**
	 * Initialize API hooks.
	 */
	private function init() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 */
	public function register_routes() {
		// Verify connection endpoint (public).
		register_rest_route(
			self::REST_NAMESPACE,
			'/verify',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'verify_connection' ),
				'permission_callback' => '__return_true',
			)
		);

		// Site info endpoint — returns rich data for AI blog generation.
		// Authenticated via site_key to allow BlogBotz server to fetch on demand.
		register_rest_route(
			self::REST_NAMESPACE,
			'/site-info',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_site_info' ),
				'permission_callback' => '__return_true',
			)
		);

		// Disconnect endpoint (requires admin).
		register_rest_route(
			self::REST_NAMESPACE,
			'/disconnect',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'disconnect' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);
	}

	/**
	 * Check API key permission.
	 * Only used for disconnect endpoint now (other endpoints removed).
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return bool True if authorized.
	 */
	public function check_api_permission( $request ) {
		$api_key    = $request->get_header( 'X-BlogBotz-Key' );
		$stored_key = get_option( 'blogbotz_api_key' );

		if ( ! $api_key || ! $stored_key ) {
			return false;
		}

		return hash_equals( $stored_key, $api_key );
	}

	/**
	 * Check admin permission.
	 *
	 * @return bool True if admin.
	 */
	public function check_admin_permission() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Verify connection endpoint.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response.
	 */
	public function verify_connection( $request ) {
		$site_key = $request->get_param( 'site_key' );

		// Validate parameter type and presence.
		if ( ! is_string( $site_key ) || empty( $site_key ) ) {
			return new WP_Error(
				'invalid_param',
				__( 'Invalid or missing site_key parameter', 'blogbotz-ai-blog-automation' ),
				array( 'status' => 400 )
			);
		}

		$stored_key = get_option( 'blogbotz_site_key' );

		if ( $site_key !== $stored_key ) {
			return new WP_Error( 'invalid_key', __( 'Invalid site key', 'blogbotz-ai-blog-automation' ), array( 'status' => 401 ) );
		}

		return rest_ensure_response(
			array(
				'success'           => true,
				'site_url'          => get_site_url(),
				'site_name'         => get_bloginfo( 'name' ),
				'wordpress_version' => get_bloginfo( 'version' ),
				'is_woocommerce'    => class_exists( 'WooCommerce' ),
				'plugin_version'    => BLOGBOTZ_VERSION,
			)
		);
	}

	/**
	 * Get comprehensive site information for AI blog generation.
	 * Authenticated via site_key parameter.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response.
	 */
	public function get_site_info( $request ) {
		// Authenticate via site_key.
		$site_key = $request->get_param( 'site_key' );
		if ( ! is_string( $site_key ) || empty( $site_key ) ) {
			return new WP_Error( 'invalid_param', __( 'Missing site_key parameter', 'blogbotz-ai-blog-automation' ), array( 'status' => 400 ) );
		}

		$stored_key = get_option( 'blogbotz_site_key' );
		if ( ! hash_equals( $stored_key, $site_key ) ) {
			return new WP_Error( 'invalid_key', __( 'Invalid site key', 'blogbotz-ai-blog-automation' ), array( 'status' => 401 ) );
		}

		$data = array(
			'site'       => $this->collect_site_data(),
			'categories' => $this->collect_categories(),
			'tags'       => $this->collect_tags(),
			'posts'      => $this->collect_recent_posts(),
			'plugins'    => $this->collect_active_plugins(),
			'theme'      => $this->collect_theme_info(),
			'settings'   => $this->collect_site_settings(),
		);

		// Add WooCommerce data if available.
		if ( class_exists( 'WooCommerce' ) ) {
			$data['products'] = $this->collect_products();
		}

		return rest_ensure_response( array( 'success' => true, 'data' => $data ) );
	}

	/**
	 * Collect basic site metadata.
	 *
	 * @return array Site data.
	 */
	private function collect_site_data() {
		return array(
			'url'               => get_site_url(),
			'home_url'          => home_url(),
			'name'              => get_bloginfo( 'name' ),
			'description'       => get_bloginfo( 'description' ),
			'language'          => get_locale(),
			'timezone'          => wp_timezone_string(),
			'date_format'       => get_option( 'date_format' ),
			'permalink_structure' => get_option( 'permalink_structure' ),
			'wordpress_version' => get_bloginfo( 'version' ),
			'php_version'       => PHP_VERSION,
			'is_multisite'      => is_multisite(),
			'is_woocommerce'    => class_exists( 'WooCommerce' ),
			'plugin_version'    => BLOGBOTZ_VERSION,
			'posts_per_page'    => (int) get_option( 'posts_per_page' ),
			'blog_public'       => (bool) get_option( 'blog_public' ),
		);
	}

	/**
	 * Collect all categories with post counts.
	 *
	 * @return array Categories data.
	 */
	private function collect_categories() {
		$categories = get_categories( array(
			'hide_empty' => false,
			'number'     => 100,
		) );

		$result = array();
		foreach ( $categories as $cat ) {
			$result[] = array(
				'id'          => $cat->term_id,
				'name'        => $cat->name,
				'slug'        => $cat->slug,
				'description' => $cat->description,
				'post_count'  => $cat->count,
				'parent'      => $cat->parent,
			);
		}
		return $result;
	}

	/**
	 * Collect popular tags.
	 *
	 * @return array Tags data.
	 */
	private function collect_tags() {
		$tags = get_tags( array(
			'hide_empty' => false,
			'number'     => 100,
			'orderby'    => 'count',
			'order'      => 'DESC',
		) );

		if ( is_wp_error( $tags ) ) {
			return array();
		}

		$result = array();
		foreach ( $tags as $tag ) {
			$result[] = array(
				'id'         => $tag->term_id,
				'name'       => $tag->name,
				'slug'       => $tag->slug,
				'post_count' => $tag->count,
			);
		}
		return $result;
	}

	/**
	 * Collect recent posts with excerpts for writing style analysis.
	 *
	 * @return array Posts data.
	 */
	private function collect_recent_posts() {
		$posts = get_posts( array(
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => 10,
			'orderby'        => 'date',
			'order'          => 'DESC',
		) );

		$result = array();
		foreach ( $posts as $post ) {
			$categories = wp_get_post_categories( $post->ID, array( 'fields' => 'names' ) );
			$tags       = wp_get_post_tags( $post->ID, array( 'fields' => 'names' ) );

			// Get post meta for SEO data.
			$seo_data = $this->collect_post_seo_data( $post->ID );

			// Get content excerpt (first 500 chars of stripped content).
			$content_text = wp_strip_all_tags( $post->post_content );
			$excerpt      = mb_substr( $content_text, 0, 500 );

			$result[] = array(
				'id'            => $post->ID,
				'title'         => $post->post_title,
				'slug'          => $post->post_name,
				'excerpt'       => $excerpt,
				'word_count'    => str_word_count( $content_text ),
				'categories'    => $categories,
				'tags'          => is_wp_error( $tags ) ? array() : $tags,
				'date'          => $post->post_date,
				'author'        => get_the_author_meta( 'display_name', $post->post_author ),
				'has_thumbnail' => has_post_thumbnail( $post->ID ),
				'seo'           => $seo_data,
			);
		}
		return $result;
	}

	/**
	 * Collect SEO data from popular SEO plugins for a post.
	 *
	 * @param int $post_id Post ID.
	 * @return array SEO data.
	 */
	private function collect_post_seo_data( $post_id ) {
		$seo = array();

		// Yoast SEO.
		$yoast_title   = get_post_meta( $post_id, '_yoast_wpseo_title', true );
		$yoast_desc    = get_post_meta( $post_id, '_yoast_wpseo_metadesc', true );
		$yoast_keyword = get_post_meta( $post_id, '_yoast_wpseo_focuskw', true );
		if ( $yoast_title || $yoast_desc || $yoast_keyword ) {
			$seo['yoast'] = array_filter( array(
				'title'         => $yoast_title,
				'meta_desc'     => $yoast_desc,
				'focus_keyword' => $yoast_keyword,
			) );
		}

		// Rank Math.
		$rm_title   = get_post_meta( $post_id, 'rank_math_title', true );
		$rm_desc    = get_post_meta( $post_id, 'rank_math_description', true );
		$rm_keyword = get_post_meta( $post_id, 'rank_math_focus_keyword', true );
		if ( $rm_title || $rm_desc || $rm_keyword ) {
			$seo['rankmath'] = array_filter( array(
				'title'         => $rm_title,
				'meta_desc'     => $rm_desc,
				'focus_keyword' => $rm_keyword,
			) );
		}

		// All in One SEO.
		$aioseo_title = get_post_meta( $post_id, '_aioseo_title', true );
		$aioseo_desc  = get_post_meta( $post_id, '_aioseo_description', true );
		if ( $aioseo_title || $aioseo_desc ) {
			$seo['aioseo'] = array_filter( array(
				'title'     => $aioseo_title,
				'meta_desc' => $aioseo_desc,
			) );
		}

		return $seo;
	}

	/**
	 * Collect active plugins (filtered to relevant ones).
	 *
	 * @return array Active plugins info.
	 */
	private function collect_active_plugins() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$active_plugins = get_option( 'active_plugins', array() );
		$all_plugins    = get_plugins();
		$result         = array();

		// Categories of plugins relevant to blog generation.
		$relevant_keywords = array( 'seo', 'schema', 'sitemap', 'cache', 'woocommerce', 'multilingual', 'polylang', 'wpml', 'translate', 'amp', 'analytics', 'social' );

		foreach ( $active_plugins as $plugin_path ) {
			if ( ! isset( $all_plugins[ $plugin_path ] ) ) {
				continue;
			}

			$plugin = $all_plugins[ $plugin_path ];
			$name   = $plugin['Name'];
			$slug   = dirname( $plugin_path );

			// Check if this plugin is relevant.
			$name_lower = strtolower( $name . ' ' . $slug );
			$is_relevant = false;
			foreach ( $relevant_keywords as $keyword ) {
				if ( false !== strpos( $name_lower, $keyword ) ) {
					$is_relevant = true;
					break;
				}
			}

			if ( $is_relevant ) {
				$result[] = array(
					'name'    => $name,
					'slug'    => $slug,
					'version' => $plugin['Version'],
				);
			}
		}

		return $result;
	}

	/**
	 * Collect active theme information.
	 *
	 * @return array Theme data.
	 */
	private function collect_theme_info() {
		$theme = wp_get_theme();
		return array(
			'name'        => $theme->get( 'Name' ),
			'version'     => $theme->get( 'Version' ),
			'template'    => $theme->get_template(),
			'is_child'    => is_child_theme(),
			'parent_name' => is_child_theme() ? $theme->parent()->get( 'Name' ) : null,
		);
	}

	/**
	 * Collect site settings relevant to content generation.
	 *
	 * @return array Site settings.
	 */
	private function collect_site_settings() {
		$settings = array(
			'show_on_front' => get_option( 'show_on_front' ),
			'page_on_front' => (int) get_option( 'page_on_front' ),
			'page_for_posts' => (int) get_option( 'page_for_posts' ),
		);

		// Detect multilingual setup.
		if ( function_exists( 'pll_languages_list' ) ) {
			$settings['multilingual'] = array(
				'plugin'    => 'polylang',
				'languages' => pll_languages_list( array( 'fields' => 'locale' ) ),
				'default'   => pll_default_language( 'locale' ),
			);
		} elseif ( defined( 'ICL_SITEPRESS_VERSION' ) ) {
			$settings['multilingual'] = array(
				'plugin'  => 'wpml',
				'version' => ICL_SITEPRESS_VERSION,
			);
		}

		return $settings;
	}

	/**
	 * Collect WooCommerce products data.
	 *
	 * @return array Products data.
	 */
	private function collect_products() {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return array();
		}

		$products = get_posts( array(
			'post_type'      => 'product',
			'post_status'    => 'publish',
			'posts_per_page' => 30,
			'orderby'        => 'date',
			'order'          => 'DESC',
		) );

		$result = array();
		foreach ( $products as $product_post ) {
			$product = wc_get_product( $product_post->ID );
			if ( ! $product ) {
				continue;
			}

			$categories = wp_get_post_terms( $product_post->ID, 'product_cat', array( 'fields' => 'names' ) );

			$result[] = array(
				'id'          => $product_post->ID,
				'name'        => $product->get_name(),
				'slug'        => $product_post->post_name,
				'price'       => $product->get_price(),
				'description' => mb_substr( wp_strip_all_tags( $product->get_short_description() ?: $product->get_description() ), 0, 300 ),
				'categories'  => is_wp_error( $categories ) ? array() : $categories,
				'permalink'   => get_permalink( $product_post->ID ),
				'in_stock'    => $product->is_in_stock(),
			);
		}

		return $result;
	}

	/**
	 * Disconnect endpoint.
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error Response.
	 */
	public function disconnect( $request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found -- Required by REST API callback signature.
		// Get stored connection data.
		$api_key  = get_option( 'blogbotz_api_key' );
		$site_id  = get_option( 'blogbotz_site_id' );
		$site_key = get_option( 'blogbotz_site_key' );

		// If we have connection data, notify BlogBotz API.
		if ( $api_key && $site_id && $site_key ) {
			$response = wp_remote_post(
				BLOGBOTZ_APP_URL . '/api/wordpress/disconnect',
				array(
					'timeout' => 15,
					'headers' => array(
						'X-BlogBotz-Key' => $api_key,
						'Content-Type'   => 'application/json',
					),
					'body'    => wp_json_encode(
						array(
							'site_id'  => intval( $site_id ),
							'site_key' => $site_key,
						)
					),
				)
			);

			// Log the response for debugging (even if it fails, we still disconnect locally).
			if ( is_wp_error( $response ) ) {
				blogbotz_log( 'BlogBotz disconnect API error: ' . $response->get_error_message() );
			} else {
				$response_code = wp_remote_retrieve_response_code( $response );
				$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

				if ( 200 !== $response_code ) {
					blogbotz_log( 'BlogBotz disconnect API failed with code ' . $response_code . ': ' . wp_remote_retrieve_body( $response ) );
				}
			}
		}

		// Clear local WordPress options (but keep site_key - it's a permanent identifier).
		delete_option( 'blogbotz_api_key' );
		delete_option( 'blogbotz_site_id' );
		// Note: blogbotz_site_key is NOT deleted - it's a permanent site identifier needed for reconnection.
		delete_option( 'blogbotz_connected_at' );
		delete_option( 'blogbotz_user_email' );

		// Ensure site_key exists (generate if missing - should not happen, but safety check).
		$site_key = get_option( 'blogbotz_site_key' );
		if ( ! $site_key ) {
			update_option( 'blogbotz_site_key', wp_generate_uuid4() );
			blogbotz_log( 'BlogBotz: Site key was missing after disconnect, regenerated it.' );
		}

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Disconnected from BlogBotz successfully', 'blogbotz-ai-blog-automation' ),
			)
		);
	}

	// Removed helper methods: get_categories_data, get_recent_posts_data, set_featured_image.
	// These are no longer needed as BlogBotz uses standard WordPress REST API.
}
