<?php
/**
 * BlogBotz - Admin Interface Class
 *
 * Handles admin UI and settings page
 *
 * @package BlogBotz
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin interface class.
 */
class BlogBotz_Admin {

	/**
	 * Connection instance.
	 *
	 * @var BlogBotz_Connection
	 */
	private $connection;

	/**
	 * Constructor.
	 *
	 * @param BlogBotz_Connection $connection Connection instance.
	 */
	public function __construct( $connection ) {
		$this->connection = $connection;
		$this->init();
	}

	/**
	 * Initialize admin hooks.
	 */
	private function init() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'admin_init', array( $this, 'handle_oauth_callback' ) );

		// Easy Connect AJAX handler.
		add_action( 'wp_ajax_blogbotz_easy_connect', array( $this, 'ajax_easy_connect' ) );

		// Dashboard token refresh AJAX handler.
		add_action( 'wp_ajax_blogbotz_refresh_dashboard_token', array( $this, 'ajax_refresh_dashboard_token' ) );
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'BlogBotz', 'blogbotz-ai-blog-automation' ),
			__( 'BlogBotz', 'blogbotz-ai-blog-automation' ),
			'manage_options',
			'blogbotz-ai-blog-automation',
			array( $this, 'render_admin_page' ),
			BLOGBOTZ_PLUGIN_URL . 'assets/images/icon-20.png',
			30
		);
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'toplevel_page_blogbotz-ai-blog-automation' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'blogbotz-admin',
			BLOGBOTZ_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			BLOGBOTZ_VERSION
		);

		wp_enqueue_script(
			'blogbotz-admin',
			BLOGBOTZ_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			BLOGBOTZ_VERSION,
			true
		);

		wp_localize_script(
			'blogbotz-admin',
			'blogbotzData',
			array(
				'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
				'restUrl'        => rest_url( 'blogbotz/v1/' ),
				'nonce'          => wp_create_nonce( 'blogbotz_nonce' ),
				'restNonce'      => wp_create_nonce( 'wp_rest' ),
				'siteUrl'        => get_site_url(),
				'siteName'       => get_bloginfo( 'name' ),
				'appUrl'         => BLOGBOTZ_APP_URL,
				'isConnected'    => $this->connection->is_connected(),
				'connectionData' => $this->connection->get_connection_data(),
			)
		);
	}

	/**
	 * Handle OAuth callback from BlogBotz.
	 */
	public function handle_oauth_callback() {
		if ( ! isset( $_GET['page'] ) || 'blogbotz-ai-blog-automation' !== $_GET['page'] ) {
			return;
		}

		if ( ! isset( $_GET['blogbotz_callback'] ) ) {
			return;
		}

		// Log callback received.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce is verified below at line 415.
		blogbotz_log( 'BlogBotz: OAuth callback received for page: ' . ( isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : 'unknown' ) );

		// Verify nonce.
		if ( ! isset( $_GET['state'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['state'] ) ), 'blogbotz_connect' ) ) {
			blogbotz_log( 'BlogBotz: Security check failed - invalid nonce' );
			wp_die( esc_html__( 'Security check failed', 'blogbotz-ai-blog-automation' ) );
		}

		$api_key     = isset( $_GET['api_key'] ) ? sanitize_text_field( wp_unslash( $_GET['api_key'] ) ) : '';
		$user_email  = isset( $_GET['user_email'] ) ? sanitize_email( wp_unslash( $_GET['user_email'] ) ) : '';
		$status      = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : '';
		$site_id     = isset( $_GET['site_id'] ) ? intval( $_GET['site_id'] ) : 0;
		$site_key    = isset( $_GET['site_key'] ) ? sanitize_text_field( wp_unslash( $_GET['site_key'] ) ) : '';
		$author_name = isset( $_GET['author_name'] ) ? sanitize_text_field( wp_unslash( $_GET['author_name'] ) ) : '';

		blogbotz_log( 'BlogBotz: Processing callback - status: ' . $status . ', api_key: ' . ( $api_key ? 'present' : 'missing' ) . ', site_id: ' . $site_id . ', author_name: ' . ( $author_name ? $author_name : 'not provided' ) );

		if ( 'success' === $status && $api_key ) {
			// Store author name if provided.
			if ( $author_name ) {
				update_option( 'blogbotz_author_name', $author_name );
			}

			// Ensure API user exists and get credentials.
			$credentials = $this->connection->get_api_user_credentials();
			if ( ! $credentials ) {
				// Create user if it doesn't exist (with author name if provided).
				$user_result = $this->connection->create_api_user( $author_name );
				if ( is_wp_error( $user_result ) ) {
					wp_safe_redirect( admin_url( 'admin.php?page=blogbotz-ai-blog-automation&error=user_creation_failed' ) );
					exit;
				}
				$credentials = $this->connection->get_api_user_credentials();
			} elseif ( $author_name && $credentials['user_id'] ) {
				// Update existing user's display name if author name is provided.
				$user = get_user_by( 'id', $credentials['user_id'] );
				if ( $user ) {
					wp_update_user(
						array(
							'ID'           => $user->ID,
							'display_name' => $author_name,
							'nickname'     => $author_name,
						)
					);
				}
			}

			// Send credentials to BlogBotz API.
			if ( $credentials && $site_id && $site_key ) {
				$response = wp_remote_post(
					BLOGBOTZ_APP_URL . '/api/wordpress/store-credentials',
					array(
						'timeout' => 15,
						'headers' => array(
							'X-BlogBotz-Key' => $api_key,
							'Content-Type'   => 'application/json',
						),
						'body'    => wp_json_encode(
							array(
								'site_id'                => intval( $site_id ),
								'site_key'               => $site_key,
								'wordpress_username'     => $credentials['username'],
								'wordpress_app_password' => $credentials['app_password'],
							)
						),
					)
				);

				// Log response for debugging (even if it fails, we still store connection locally).
				if ( is_wp_error( $response ) ) {
					blogbotz_log( 'BlogBotz: Failed to send credentials: ' . $response->get_error_message() );
				} else {
					$response_code = wp_remote_retrieve_response_code( $response );
					if ( 200 !== $response_code ) {
						blogbotz_log( 'BlogBotz: Credentials API returned code ' . $response_code . ': ' . wp_remote_retrieve_body( $response ) );
					}
				}
			}

			// Store connection data.
			update_option( 'blogbotz_api_key', $api_key );
			update_option( 'blogbotz_connected_at', current_time( 'mysql' ) );
			update_option( 'blogbotz_user_email', $user_email );

			// Store site_id and site_key if provided (for disconnect functionality).
			if ( $site_id ) {
				update_option( 'blogbotz_site_id', $site_id );
			}
			if ( $site_key ) {
				update_option( 'blogbotz_site_key', $site_key );
			}

			blogbotz_log( 'BlogBotz: Connection successful, redirecting...' );
			// Redirect to clean URL with success message.
			wp_safe_redirect( admin_url( 'admin.php?page=blogbotz-ai-blog-automation&connected=1' ) );
			exit;
		} else {
			blogbotz_log( 'BlogBotz: Connection failed - status: ' . $status . ', api_key: ' . ( $api_key ? 'present' : 'missing' ) );
			// Redirect with error.
			wp_safe_redirect( admin_url( 'admin.php?page=blogbotz-ai-blog-automation&error=connection_failed' ) );
			exit;
		}
	}

	/**
	 * Generate connect URL.
	 *
	 * @return string Connect URL or empty string on error.
	 */
	public function get_connect_url() {
		blogbotz_log( 'BlogBotz: Generating connect URL...' );

		// Check system requirements first.
		$requirements = $this->connection->check_requirements();
		if ( ! $requirements['all_ok'] ) {
			$error_parts = array();
			if ( ! $requirements['wp_version_ok'] ) {
				/* translators: %s: Current WordPress version number. */
				$error_parts[] = sprintf( __( 'WordPress 5.6+ required (you have %s)', 'blogbotz-ai-blog-automation' ), $requirements['wp_version'] );
			}
			if ( ! $requirements['app_passwords_available'] ) {
				$error_parts[] = __( 'Application Passwords not available', 'blogbotz-ai-blog-automation' );
			}
			if ( ! $requirements['site_key_exists'] ) {
				$error_parts[] = __( 'Site key missing - please deactivate and reactivate the plugin', 'blogbotz-ai-blog-automation' );
			}
			$error_msg = implode( '. ', $error_parts );
			set_transient( 'blogbotz_user_creation_error', $error_msg, 30 );
			blogbotz_log( 'BlogBotz: Requirements check failed: ' . $error_msg );
			return '';
		}

		// Ensure API user exists (safety fallback if not created during activation).
		$credentials = $this->connection->get_api_user_credentials();
		if ( ! $credentials ) {
			// Create user if it doesn't exist.
			blogbotz_log( 'BlogBotz: Creating API user...' );
			// Get author name from option if set.
			$author_name = get_option( 'blogbotz_author_name', 'BlogBotz' );
			$user_result = $this->connection->create_api_user( $author_name );
			if ( is_wp_error( $user_result ) ) {
				// Store error in transient to display on next page load.
				$error_msg = $user_result->get_error_message();
				set_transient( 'blogbotz_user_creation_error', $error_msg, 30 );
				blogbotz_log( 'BlogBotz: Failed to create API user: ' . $error_msg );
				return '';
			}
			$credentials = $this->connection->get_api_user_credentials();
			if ( ! $credentials ) {
				$error_msg = __( 'Failed to create API user. User was created but Application Password is missing. Please check WordPress error logs.', 'blogbotz-ai-blog-automation' );
				set_transient( 'blogbotz_user_creation_error', $error_msg, 30 );
				blogbotz_log( 'BlogBotz: User created but credentials not available' );
				return '';
			}
			blogbotz_log( 'BlogBotz: API user created successfully' );
		}

		$site_key = get_option( 'blogbotz_site_key' );
		if ( ! $site_key ) {
			// Generate site key if missing (should not happen, but safety fallback).
			$site_key = wp_generate_uuid4();
			update_option( 'blogbotz_site_key', $site_key );
			blogbotz_log( 'BlogBotz: Site key was missing, generated new one: ' . $site_key );
		}

		$state        = wp_create_nonce( 'blogbotz_connect' );
		$callback_url = admin_url( 'admin.php?page=blogbotz-ai-blog-automation&blogbotz_callback=1' );

		$params = array(
			'site_url'       => get_site_url(),
			'site_name'      => get_bloginfo( 'name' ),
			'site_key'       => $site_key,
			'callback_url'   => $callback_url,
			'state'          => $state,
			'is_woocommerce' => class_exists( 'WooCommerce' ) ? '1' : '0',
		);

		$connect_url = BLOGBOTZ_APP_URL . '/connect/wordpress?' . http_build_query( $params );
		blogbotz_log( 'BlogBotz: Connect URL generated successfully' );
		return $connect_url;
	}

	/**
	 * Render admin page.
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check for error transients and display them.
		$user_creation_error = get_transient( 'blogbotz_user_creation_error' );
		if ( $user_creation_error ) {
			delete_transient( 'blogbotz_user_creation_error' );
		}

		$connection_error = get_transient( 'blogbotz_connection_error' );
		if ( $connection_error ) {
			delete_transient( 'blogbotz_connection_error' );
		}

		// Pass errors to template.
		$error_message = $user_creation_error ? $user_creation_error : $connection_error;

		include BLOGBOTZ_PLUGIN_DIR . 'templates/admin-page.php';
	}

	/**
	 * Get connection instance.
	 *
	 * @return BlogBotz_Connection Connection instance.
	 */
	public function get_connection() {
		return $this->connection;
	}

	/**
	 * AJAX handler for Easy Connect.
	 * Handles one-click connection from WordPress without browser redirect.
	 */
	public function ajax_easy_connect() {
		check_ajax_referer( 'blogbotz_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Insufficient permissions', 'blogbotz-ai-blog-automation' ) );
		}

		$result = $this->easy_connect();

		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			// wp_send_json_error passes data as response.data, so send the error message directly.
			$error_message = isset( $result['error'] ) ? $result['error'] : __( 'Connection failed. Please try again.', 'blogbotz-ai-blog-automation' );
			wp_send_json_error( $error_message );
		}
	}

	/**
	 * AJAX handler for refreshing dashboard token.
	 * Generates a fresh auto-login token each time user clicks "Open Dashboard".
	 */
	public function ajax_refresh_dashboard_token() {
		check_ajax_referer( 'blogbotz_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Insufficient permissions', 'blogbotz-ai-blog-automation' ) );
		}

		$site_key = get_option( 'blogbotz_site_key' );
		if ( ! $site_key ) {
			wp_send_json_error( array(
				'message'            => __( 'Site key not found. Please reconnect.', 'blogbotz-ai-blog-automation' ),
				'reconnect_required' => true,
			) );
		}

		// Request a fresh dashboard token from BlogBotz server.
		$response = wp_remote_post(
			BLOGBOTZ_APP_URL . '/api/wordpress/refresh-dashboard-token',
			array(
				'timeout' => 15,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode(
					array(
						'site_key' => $site_key,
						'site_url' => get_site_url(),
					)
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			blogbotz_log( 'BlogBotz Dashboard: Token refresh failed - ' . $response->get_error_message() );
			wp_send_json_error( array(
				'message' => __( 'Could not connect to BlogBotz server. Please try again.', 'blogbotz-ai-blog-automation' ),
			) );
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$body          = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( ! is_array( $body ) ) {
			wp_send_json_error( array(
				'message' => __( 'Invalid response from server. Please try again.', 'blogbotz-ai-blog-automation' ),
			) );
		}

		// If site/user not found on server, trigger reconnect via Easy Connect.
		if ( ! empty( $body['reconnect_required'] ) ) {
			blogbotz_log( 'BlogBotz Dashboard: Reconnect required - ' . ( $body['error'] ?? 'unknown' ) );

			// Auto-reconnect: trigger Easy Connect to re-establish connection.
			$reconnect_result = $this->easy_connect();

			if ( $reconnect_result['success'] ) {
				// Easy Connect succeeded — the dashboard_url is now fresh.
				$dashboard_url = $reconnect_result['dashboard_url'];
				update_option( 'blogbotz_dashboard_url', $dashboard_url );

				wp_send_json_success( array(
					'dashboard_url' => $dashboard_url,
					'reconnected'   => true,
				) );
			} else {
				wp_send_json_error( array(
					'message'            => $reconnect_result['error'] ?? __( 'Reconnection failed. Please try again.', 'blogbotz-ai-blog-automation' ),
					'reconnect_required' => true,
				) );
			}
			return;
		}

		if ( 200 !== $response_code || empty( $body['success'] ) ) {
			$error = $body['error'] ?? $body['message'] ?? __( 'Failed to refresh dashboard token.', 'blogbotz-ai-blog-automation' );
			wp_send_json_error( array( 'message' => $error ) );
		}

		// Update the stored dashboard URL with the fresh token.
		$dashboard_url = $body['dashboard_url'];
		update_option( 'blogbotz_dashboard_url', $dashboard_url );

		wp_send_json_success( array(
			'dashboard_url' => $dashboard_url,
		) );
	}

	/**
	 * Easy Connect - one-click connection with HMAC signing.
	 * Auto-registers or auto-logins user on BlogBotz server.
	 *
	 * @return array Result with success status and data.
	 */
	public function easy_connect() {
		// Check system requirements first.
		$requirements = $this->connection->check_requirements();
		if ( ! $requirements['all_ok'] ) {
			$error_parts = array();
			if ( ! $requirements['wp_version_ok'] ) {
				/* translators: %s: Current WordPress version number. */
				$error_parts[] = sprintf( __( 'WordPress 5.6+ required (you have %s)', 'blogbotz-ai-blog-automation' ), $requirements['wp_version'] );
			}
			if ( ! $requirements['app_passwords_available'] ) {
				$error_parts[] = __( 'Application Passwords not available', 'blogbotz-ai-blog-automation' );
			}
			if ( ! $requirements['site_key_exists'] ) {
				$error_parts[] = __( 'Site key missing - please deactivate and reactivate the plugin', 'blogbotz-ai-blog-automation' );
			}
			return array(
				'success' => false,
				'error'   => implode( '. ', $error_parts ),
			);
		}

		// Get current admin user info.
		$current_user = wp_get_current_user();
		$admin_email  = $current_user->user_email;
		$wp_user_id   = $current_user->ID;

		// Get site info.
		$site_url       = get_site_url();
		$site_name      = get_bloginfo( 'name' );
		$site_key       = get_option( 'blogbotz_site_key' );
		$is_woocommerce = class_exists( 'WooCommerce' );
		$platform       = $is_woocommerce ? 'woocommerce' : 'wordpress';
		$timestamp      = time();

		// Get author name from settings or current user.
		$author_name = get_option( 'blogbotz_author_name', $current_user->display_name );

		// Generate HMAC signature.
		$secret    = $this->get_hmac_secret();
		$payload   = implode(
			'|',
			array(
				$site_url,
				$admin_email,
				$wp_user_id,
				$platform,
				$timestamp,
			)
		);
		$signature = hash_hmac( 'sha256', $payload, $secret );

		// Ensure API user exists and get credentials.
		$credentials = $this->connection->get_api_user_credentials();
		if ( ! $credentials ) {
			$user_result = $this->connection->create_api_user( $author_name );
			if ( is_wp_error( $user_result ) ) {
				return array(
					'success' => false,
					'error'   => $user_result->get_error_message(),
				);
			}
			$credentials = $this->connection->get_api_user_credentials();
			if ( ! $credentials ) {
				return array(
					'success' => false,
					'error'   => __( 'Failed to create API user credentials', 'blogbotz-ai-blog-automation' ),
				);
			}
		}

		// Make API request to Easy Connect endpoint.
		$response = wp_remote_post(
			BLOGBOTZ_APP_URL . '/api/wordpress/easy-connect',
			array(
				'timeout' => 30,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode(
					array(
						'site_url'    => $site_url,
						'admin_email' => $admin_email,
						'wp_user_id'  => $wp_user_id,
						'site_name'   => $site_name,
						'platform'    => $platform,
						'timestamp'   => $timestamp,
						'signature'   => $signature,
						'site_key'    => $site_key,
						'author_name' => $author_name,
					)
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			blogbotz_log( 'BlogBotz Easy Connect: Request failed - ' . $response->get_error_message() );
			return array(
				'success' => false,
				'error'   => $response->get_error_message(),
			);
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$body          = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( ! is_array( $body ) ) {
			blogbotz_log( 'BlogBotz Easy Connect: Invalid JSON response (HTTP ' . $response_code . ')' );
			return array(
				'success' => false,
				'error'   => __( 'Invalid response from server. Please try again.', 'blogbotz-ai-blog-automation' ),
			);
		}

		if ( 200 !== $response_code || empty( $body['success'] ) ) {
			// Check multiple error formats: our custom 'error' key, Laravel's 'message' key, or validation errors
			$error = __( 'Connection failed', 'blogbotz-ai-blog-automation' );
			if ( ! empty( $body['error'] ) ) {
				$error = $body['error'];
			} elseif ( ! empty( $body['message'] ) ) {
				$error = $body['message'];
			} elseif ( ! empty( $body['errors'] ) && is_array( $body['errors'] ) ) {
				$error = implode( ' ', array_map( function ( $msgs ) {
					return is_array( $msgs ) ? implode( ' ', $msgs ) : $msgs;
				}, $body['errors'] ) );
			}
			blogbotz_log( 'BlogBotz Easy Connect: API returned error (HTTP ' . $response_code . ') - ' . $error );
			return array(
				'success' => false,
				'error'   => $error,
				'code'    => isset( $body['code'] ) ? $body['code'] : null,
			);
		}

		// Store connection data.
		update_option( 'blogbotz_api_key', $body['token'] );
		update_option( 'blogbotz_site_id', $body['site']['id'] );
		update_option( 'blogbotz_user_email', $body['user']['email'] );
		update_option( 'blogbotz_connected_at', current_time( 'mysql' ) );
		update_option( 'blogbotz_dashboard_url', $body['dashboard_url'] );

		// Send WordPress credentials to BlogBotz server.
		$cred_response = wp_remote_post(
			$body['credentials_endpoint'],
			array(
				'timeout' => 15,
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'body'    => wp_json_encode(
					array(
						'site_id'                => $body['site']['id'],
						'site_key'               => $site_key,
						'wordpress_username'     => $credentials['username'],
						'wordpress_app_password' => $credentials['app_password'],
					)
				),
			)
		);

		if ( is_wp_error( $cred_response ) ) {
			blogbotz_log( 'BlogBotz Easy Connect: Failed to send credentials - ' . $cred_response->get_error_message() );
		} else {
			$cred_code = wp_remote_retrieve_response_code( $cred_response );
			if ( 200 !== $cred_code ) {
				blogbotz_log( 'BlogBotz Easy Connect: Credentials API returned code ' . $cred_code );
			}
		}

		blogbotz_log( 'BlogBotz Easy Connect: Connection successful for ' . $admin_email );

		return array(
			'success'       => true,
			'user'          => $body['user'],
			'site'          => $body['site'],
			'is_new_user'   => $body['is_new_user'],
			'dashboard_url' => $body['dashboard_url'],
			'limits'        => isset( $body['limits'] ) ? $body['limits'] : null,
		);
	}

	/**
	 * Get HMAC secret for signing Easy Connect requests.
	 *
	 * This shared secret authenticates plugin-to-server communication.
	 * It can be overridden via a constant in wp-config.php for self-hosted setups.
	 *
	 * @return string The HMAC secret.
	 */
	protected function get_hmac_secret() {
		// Allow override via wp-config.php constant for self-hosted or custom deployments.
		if ( defined( 'BLOGBOTZ_HMAC_SECRET' ) && BLOGBOTZ_HMAC_SECRET ) {
			return BLOGBOTZ_HMAC_SECRET;
		}

		// Default shared secret — must match the EASY_CONNECT_HMAC_SECRET on the BlogBotz server.
		return 'bb_ec_2026_f8a3d9c7e2b1a4f6d8e9c3b5a7f2e4d6c8b0a1e3f5d7c9b2a4f6e8d0c2b4a6f8';
	}
}
