/**
 * BlogBotz Promotional Content Protection
 * Prevents third-party promotional content injection into plugin admin pages.
 */

(function() {
    var promotionalKeywords = [
        "WordPress Theme Bundle",
        "120+ Templates",
        "theme bundle",
        "bundle 120",
        "$99",
        "Buy Now",
        "Check Out Premium Templates"
    ];

    function isPromotionalContent(element) {
        if (!element) return false;
        var className = element.className || "";
        var id = element.id || "";
        var textContent = element.textContent || element.innerText || "";

        if (className.toLowerCase().indexOf("promo") !== -1 ||
            className.toLowerCase().indexOf("theme-bundle") !== -1 ||
            className.toLowerCase().indexOf("theme-bundle") !== -1 ||
            id.toLowerCase().indexOf("promo") !== -1 ||
            id.toLowerCase().indexOf("theme") !== -1 ||
            id.toLowerCase().indexOf("bundle") !== -1) {
            return true;
        }

        return promotionalKeywords.some(function(keyword) {
            return textContent.toLowerCase().indexOf(keyword.toLowerCase()) !== -1;
        });
    }

    function removePromoFromPluginContainers() {
        var headerSection = document.querySelector("[data-blogbotz-header-section]");
        if (headerSection) {
            var allowedElements = headerSection.querySelectorAll(".blogbotz-logo, h1, p, svg");
            var allowedIds = [];
            allowedElements.forEach(function(el) {
                if (el.id) allowedIds.push(el.id);
            });

            Array.from(headerSection.children).forEach(function(child) {
                if (!child.classList.contains("blogbotz-logo") &&
                    child.tagName !== "H1" &&
                    child.tagName !== "P" &&
                    child.tagName !== "SVG" &&
                    !allowedIds.includes(child.id)) {
                    try {
                        if (isPromotionalContent(child) || child.tagName === "IFRAME" || child.tagName === "SCRIPT") {
                            child.remove();
                        }
                    } catch (e) {}
                }
            });
        }

        var pluginContainers = document.querySelectorAll("[data-blogbotz-plugin-container], .blogbotz-wrap");
        pluginContainers.forEach(function(container) {
            var allElements = container.querySelectorAll("div, section, aside, article, iframe");
            allElements.forEach(function(element) {
                if (element.closest("[data-blogbotz-header-section]") === headerSection) {
                    return;
                }
                if (element.classList.contains("blogbotz-") ||
                    element.closest(".blogbotz-card") ||
                    element.closest(".blogbotz-content") ||
                    element.closest(".blogbotz-features") ||
                    element.closest(".blogbotz-footer")) {
                    return;
                }
                if (isPromotionalContent(element)) {
                    try {
                        element.remove();
                    } catch (e) {}
                }
            });
        });
    }

    function setupProtection() {
        var headerSection = document.querySelector("[data-blogbotz-header-section]");
        if (headerSection) {
            var observer = new MutationObserver(function(mutations) {
                mutations.forEach(function(mutation) {
                    if (mutation.addedNodes && mutation.addedNodes.length > 0) {
                        mutation.addedNodes.forEach(function(node) {
                            if (node.nodeType === 1) {
                                if (isPromotionalContent(node) ||
                                    node.tagName === "IFRAME" ||
                                    node.tagName === "SCRIPT" ||
                                    (node.className && typeof node.className === "string" &&
                                     (node.className.indexOf("promo") !== -1 ||
                                      node.className.indexOf("theme") !== -1 ||
                                      node.className.indexOf("bundle") !== -1))) {
                                    try {
                                        node.remove();
                                    } catch (e) {}
                                }
                            }
                        });
                    }
                });
            });
            observer.observe(headerSection, {
                childList: true,
                subtree: true,
                attributes: false
            });
        }
    }

    removePromoFromPluginContainers();

    if (document.readyState === "loading") {
        document.addEventListener("DOMContentLoaded", function() {
            removePromoFromPluginContainers();
            setupProtection();
        });
    } else {
        setupProtection();
    }

    setTimeout(removePromoFromPluginContainers, 100);
    setTimeout(removePromoFromPluginContainers, 500);
    setTimeout(removePromoFromPluginContainers, 1000);
    setTimeout(removePromoFromPluginContainers, 2000);
})();
