/**
 * BlogBotz Notice Protection
 * Prevents WordPress from moving plugin notices to header area.
 */

(function() {
    function preventNoticeMovement() {
        var contentArea = document.querySelector('.blogbotz-content');
        if (!contentArea) return;

        var notices = contentArea.querySelectorAll('.notice');
        notices.forEach(function(notice) {
            notice.setAttribute('data-blogbotz-notice', 'true');
            notice.style.position = 'relative';
        });
    }

    function watchForNoticeMovement() {
        var contentArea = document.querySelector('.blogbotz-content');
        if (!contentArea) return;

        var observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1 && node.classList && node.classList.contains('notice')) {
                            var originalParent = node.getAttribute('data-blogbotz-original-parent');
                            if (originalParent === 'blogbotz-content') {
                                var targetArea = document.querySelector('.blogbotz-content');
                                if (targetArea && node.parentNode !== targetArea) {
                                    var connectionCard = targetArea.querySelector('.blogbotz-card');
                                    if (connectionCard) {
                                        targetArea.insertBefore(node, connectionCard);
                                    } else {
                                        targetArea.insertBefore(node, targetArea.firstChild);
                                    }
                                }
                            }
                        }
                    });
                }
            });
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    function initializeNotices() {
        var contentArea = document.querySelector('.blogbotz-content');
        if (!contentArea) return;

        var notices = contentArea.querySelectorAll('.notice');
        notices.forEach(function(notice) {
            notice.setAttribute('data-blogbotz-notice', 'true');
            notice.setAttribute('data-blogbotz-original-parent', 'blogbotz-content');
            notice.style.position = 'relative';
        });
    }

    initializeNotices();

    if (document.readyState === "loading") {
        document.addEventListener("DOMContentLoaded", function() {
            initializeNotices();
            preventNoticeMovement();
            watchForNoticeMovement();
        });
    } else {
        preventNoticeMovement();
        watchForNoticeMovement();
    }

    if (typeof jQuery !== 'undefined') {
        jQuery(document).ready(function($) {
            var originalAppend = $.fn.append;
            $.fn.append = function() {
                var result = originalAppend.apply(this, arguments);
                var contentArea = $('.blogbotz-content');
                if (contentArea.length) {
                    $('.notice[data-blogbotz-original-parent="blogbotz-content"]').each(function() {
                        var $notice = $(this);
                        if (!$notice.closest('.blogbotz-content').length) {
                            var $connectionCard = contentArea.find('.blogbotz-card').first();
                            if ($connectionCard.length) {
                                $connectionCard.before($notice);
                            } else {
                                contentArea.prepend($notice);
                            }
                        }
                    });
                }
                return result;
            };
        });
    }

    setInterval(function() {
        var contentArea = document.querySelector('.blogbotz-content');
        if (!contentArea) return;

        var notices = document.querySelectorAll('.notice[data-blogbotz-original-parent="blogbotz-content"]');
        notices.forEach(function(notice) {
            if (!contentArea.contains(notice)) {
                var connectionCard = contentArea.querySelector('.blogbotz-card');
                if (connectionCard) {
                    contentArea.insertBefore(notice, connectionCard);
                } else {
                    contentArea.insertBefore(notice, contentArea.firstChild);
                }
            }
        });
    }, 500);
})();
