/**
 * BlogBotz Admin JavaScript
 */

(function($) {
    'use strict';

    var BlogBotzAdmin = {
        init: function() {
            this.cleanupHeaderContainer();
            this.bindEvents();
        },

        /**
         * Remove any injected content from the header container.
         * Some themes/plugins inject promotional content into admin pages.
         */
        cleanupHeaderContainer: function() {
            var $headerContainer = $('#blogbotz-header-container');
            if ($headerContainer.length) {
                // Remove any elements that are not the official BlogBotz header
                $headerContainer.children().not('.blogbotz-header').remove();
                // Also clean any injected content before the header container
                $headerContainer.prevAll().not('.notice, .updated, .error').each(function() {
                    var $el = $(this);
                    // Only remove if it's not a WordPress admin notice
                    if (!$el.hasClass('notice') && !$el.hasClass('updated') && !$el.hasClass('error')) {
                        // Check if it looks like injected promotional content
                        if ($el.find('a[href*="theme"], a[href*="template"], a[href*="premium"]').length > 0) {
                            $el.remove();
                        }
                    }
                });
            }
        },

        bindEvents: function() {
            var self = this;

            // Disconnect button
            $(document).on('click', '#blogbotz-disconnect', function(e) {
                self.handleDisconnect(e);
            });

            // Easy Connect button
            $(document).on('click', '#blogbotz-easy-connect', function(e) {
                self.handleEasyConnect(e);
            });

            // Open Dashboard button - fetches fresh token before redirecting
            $(document).on('click', '#blogbotz-open-dashboard', function(e) {
                self.handleOpenDashboard(e);
            });
        },

        handleEasyConnect: function(e) {
            e.preventDefault();
            e.stopPropagation();

            var $btn = $('#blogbotz-easy-connect');
            var $btnText = $btn.find('.btn-text');
            var $btnLoading = $btn.find('.btn-loading');
            var $errorDiv = $('#blogbotz-connect-error');

            // Show loading state
            $btn.prop('disabled', true);
            $btnText.hide();
            $btnLoading.show();
            $errorDiv.hide();

            $.ajax({
                url: blogbotzData.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'blogbotz_easy_connect',
                    nonce: blogbotzData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Connection successful - reload page to show connected state
                        window.location.href = window.location.pathname + '?page=blogbotz-ai-blog-automation&connected=1';
                    } else {
                        // Show error
                        var errorMsg = response.data || 'Connection failed. Please try again.';
                        $errorDiv.find('.error-message').text(errorMsg);
                        $errorDiv.css('display', 'block');
                        $btn.prop('disabled', false);
                        $btnText.show();
                        $btnLoading.hide();
                    }
                },
                error: function(xhr, status, error) {
                    var errorMsg = 'Connection failed. Please try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data) {
                        errorMsg = xhr.responseJSON.data;
                    } else if (xhr.status === 0) {
                        errorMsg = 'Network error. Please check your internet connection.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please try again later.';
                    }
                    $errorDiv.find('.error-message').text(errorMsg);
                    $errorDiv.css('display', 'block');
                    $btn.prop('disabled', false);
                    $btnText.show();
                    $btnLoading.hide();
                }
            });
        },

        handleOpenDashboard: function(e) {
            e.preventDefault();
            e.stopPropagation();

            var $btn = $('#blogbotz-open-dashboard');
            var $btnText = $btn.find('.btn-text');
            var $btnLoading = $btn.find('.btn-loading');
            var $errorDiv = $('#blogbotz-dashboard-error');

            // Show loading state
            $btn.prop('disabled', true);
            $btnText.hide();
            $btnLoading.show();
            $errorDiv.hide();

            $.ajax({
                url: blogbotzData.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'blogbotz_refresh_dashboard_token',
                    nonce: blogbotzData.nonce
                },
                success: function(response) {
                    if (response.success && response.data && response.data.dashboard_url) {
                        // Open the fresh dashboard URL in a new tab
                        window.open(response.data.dashboard_url, '_blank');

                        // Reset button state
                        $btn.prop('disabled', false);
                        $btnText.show();
                        $btnLoading.hide();
                    } else {
                        var errorData = response.data || {};
                        var errorMsg = errorData.message || 'Failed to open dashboard. Please try again.';

                        if (errorData.reconnect_required) {
                            // Connection lost — show error and suggest reconnecting
                            errorMsg += ' Click "Disconnect" and then reconnect to re-establish the connection.';
                        }

                        $errorDiv.find('.error-message').text(errorMsg);
                        $errorDiv.css('display', 'block');
                        $btn.prop('disabled', false);
                        $btnText.show();
                        $btnLoading.hide();
                    }
                },
                error: function(xhr) {
                    var errorMsg = 'Failed to open dashboard. Please try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data) {
                        var errorData = xhr.responseJSON.data;
                        errorMsg = errorData.message || errorData || errorMsg;
                    }
                    $errorDiv.find('.error-message').text(errorMsg);
                    $errorDiv.css('display', 'block');
                    $btn.prop('disabled', false);
                    $btnText.show();
                    $btnLoading.hide();
                }
            });
        },

        handleDisconnect: function(e) {
            e.preventDefault();

            // Create modal
            var modal = $('<div class="blogbotz-modal-overlay">' +
                '<div class="blogbotz-modal">' +
                '<h3>Disconnect from BlogBotz?</h3>' +
                '<p>This will remove the connection between your WordPress site and BlogBotz. You can reconnect at any time.</p>' +
                '<div class="blogbotz-modal-actions">' +
                '<button type="button" class="button button-secondary" id="blogbotz-cancel-disconnect">Cancel</button>' +
                '<button type="button" class="button button-primary" id="blogbotz-confirm-disconnect">Disconnect</button>' +
                '</div>' +
                '</div>' +
                '</div>');

            $('body').append(modal);

            // Bind modal events
            $('#blogbotz-cancel-disconnect').on('click', function() {
                modal.remove();
            });

            modal.on('click', function(e) {
                if ($(e.target).hasClass('blogbotz-modal-overlay')) {
                    modal.remove();
                }
            });

            $('#blogbotz-confirm-disconnect').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).html('<span class="blogbotz-loading"><span class="blogbotz-spinner"></span> Disconnecting...</span>');

                $.ajax({
                    url: blogbotzData.restUrl + 'disconnect',
                    method: 'POST',
                    beforeSend: function(xhr) {
                        xhr.setRequestHeader('X-WP-Nonce', blogbotzData.restNonce);
                    },
                    success: function(response) {
                        if (response.success) {
                            var cleanUrl = window.location.pathname + '?page=blogbotz-ai-blog-automation&disconnected=1';
                            window.location.href = cleanUrl;
                        } else {
                            alert('Failed to disconnect: ' + (response.message || 'Unknown error'));
                            $btn.prop('disabled', false).text('Disconnect');
                        }
                    },
                    error: function(xhr) {
                        var errorMsg = 'Failed to disconnect. Please try again.';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMsg = xhr.responseJSON.message;
                        } else if (xhr.status === 403) {
                            errorMsg = 'Permission denied. Please ensure you have administrator access.';
                        }
                        alert(errorMsg);
                        $btn.prop('disabled', false).text('Disconnect');
                    }
                });
            });
        }
    };

    $(document).ready(function() {
        BlogBotzAdmin.init();
    });

})(jQuery);
