<?php

if ( ! defined( 'ABSPATH' ) ) exit;

class BlogPostReadingTime_Core {

    public function __construct() {
        add_filter( 'the_title', [ $this, 'add_reading_time_to_title' ], 10, 2 );
        add_filter( 'the_content', [ $this, 'add_reading_time_to_content' ] );
    }

    public function get_reading_time( $post_id ) {
        $post = get_post( $post_id );
        $content = $post->post_content;
        $word_count = str_word_count( wp_strip_all_tags( $content ) );
        $wpm = intval( get_option( 'bprt_words_per_minute', 200 ) );
        $minutes = max(1, ceil( $word_count / ($wpm ?: 200) ));

        $show_icon = get_option( 'bprt_show_clock_icon', true );
        $icon = $show_icon ? '⏱ ' : '';
        $label = get_option( 'bprt_custom_label', 'Estimated reading time:' );
        $font_size = intval( get_option( 'bprt_font_size', 14 ) );
        $postfix = get_option( 'bprt_time_postfix', 'min read' );

        return "<p class='bprt-reading-time' style='font-weight:bold; font-size: {$font_size}px;'>{$icon}{$label} {$minutes} {$postfix}</p>";
    }

    public function add_reading_time_to_title( $title, $post_id ) {
        if ( is_singular( 'post' ) && in_the_loop() && is_main_query() ) {
            if ( get_option( 'bprt_display_mode', 'automatic' ) !== 'automatic' ) return $title;
            if ( get_option( 'bprt_display_position' ) === 'after_title' ) {
                $title .= $this->get_reading_time( $post_id );
            }
        }
        return $title;
    }

    public function add_reading_time_to_content( $content ) {
        if ( ! is_singular( 'post' ) || ! is_main_query() ) return $content;

        $display_mode = get_option( 'bprt_display_mode', 'automatic' );
        if ( $display_mode !== 'automatic' ) return $content;

        $post_id = get_the_ID();
        $reading_time = $this->get_reading_time( $post_id );
        $listen_btn   = $this->get_listen_button( $post_id );

        $position = get_option( 'bprt_display_position', 'before_content' );

        switch ( $position ) {
            case 'before_content':
                return $reading_time . $listen_btn . $content;
            case 'after_featured_image':
                $thumb = has_post_thumbnail( $post_id ) ? get_the_post_thumbnail( $post_id, 'full' ) : '';
                return $thumb . $reading_time . $listen_btn . $content;
            case 'after_title':
                // handled in the_title filter, but also append listen button
                return $listen_btn . $content;
            default:
                return $content;
        }
    }

    public function get_listen_button( $post_id ) {
        if ( ! get_option('bprt_enable_listen') ) {
            return '';
        }

        ob_start(); ?>
        <p>
            <button id="bprt-listen-btn-<?php echo esc_attr( $post_id ); ?>" class="bprt-listen-btn">
            <i class="fas fa-play"></i>
            </button>
        </p>
        <script>
        document.addEventListener("DOMContentLoaded", function() {
            const btn = document.getElementById("bprt-listen-btn-<?php echo esc_attr( $post_id ); ?>");
            if (!btn) return;

            const text = document.querySelector(".entry-content")?.innerText || "";
            let synth = window.speechSynthesis;
            let utterance;

            btn.addEventListener("click", function() {
                if (synth.speaking) {
                    synth.cancel();
                    btn.innerHTML = "<i class='fas fa-play'></i>";
                } else {
                    utterance = new SpeechSynthesisUtterance(text);
                    utterance.lang = "en-US";
                    synth.speak(utterance);
                    btn.innerHTML = "<i class='fa fa-stop'></i>";
                }
            });

            // Stop speech when page is unloaded or refreshed
            window.addEventListener("beforeunload", function() {
                if (synth.speaking) {
                    synth.cancel();
                    btn.innerHTML = '<i class="fas fa-stop"></i> Listen to this article';
                }
            });
        });
        </script>
        <?php
        return ob_get_clean();
    }
    

}
