<?php

/*
*   @since 1.0.0

*   @package black-desk
*   @subpackage black-desk/public
*/

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'BDDS_Public' ) ) {

    /**
     * Public class for frontend functionalities.
     *
     * Adds shortcodes for displaying data from created tables.
     *
     * @since 1.0.0
     */
    class BDDS_Public
    {

        private $plugin_name;
        private $version;

        /**
         * BDDS_Public class constructor.
         *
         * Initiates shortcodes.
         *
         *
         * @since 1.0.0
         *
         * @var String $plugin_name Plugin name.
         * @var Int $version Plugin version.
         * @return void
         */
        public function __construct( $plugin_name, $version)
        {
            $this->plugin_name = $plugin_name;
            $this->version     = $version;

            add_shortcode( BDDS_PLUGIN_PREFIX . '-view-table', array( $this, BDDS_PLUGIN_PREFIX . '_view_table_shortcode' ));
            add_shortcode( BDDS_PLUGIN_PREFIX . '-view-field', array( $this, BDDS_PLUGIN_PREFIX . '_view_field_shortcode' ));
            add_shortcode( BDDS_PLUGIN_PREFIX . '-view-repeat', array( $this, BDDS_PLUGIN_PREFIX . '_view_repeat_shortcode' ));
            add_shortcode( BDDS_PLUGIN_PREFIX . '-form-start', array( $this, BDDS_PLUGIN_PREFIX . '_form_field_start_shortcode' ) );
            add_shortcode( BDDS_PLUGIN_PREFIX . '-form-field', array( $this, BDDS_PLUGIN_PREFIX . '_form_field_shortcode' ) );
            add_shortcode( BDDS_PLUGIN_PREFIX . '-form-stop', array( $this, BDDS_PLUGIN_PREFIX . '_form_field_stop_shortcode' ) );
        }

        /**
         * Sets shortcode attributes and includes a view.
         *
         * Sets attributes used for shortcode that displays data from table or query.
         * Enqueues style for the table and includes table view.
         *
         * @since 1.0.0
         *
         * @param array $atts Attributes passed in shortcode
         */
        public function bdds_view_table_shortcode ( $atts )
        {
            $table_id    = isset( $atts['id'] ) ? $atts['id'] : null;
            $table_name  = isset( $atts['table'] ) ? $atts['table'] : null;
            $query       = isset( $atts['query'] ) ? $atts['query'] : null;
            $fields      = isset( $atts['fields'] ) ? $atts['fields'] : null;
            $fieldnames  = isset( $atts['fieldnames'] ) ? $atts['fieldnames'] : null;
            $sort        = isset( $atts['sort'] ) ? $atts['sort'] : null;
            $search      = isset( $atts['search'] ) ? $atts['search'] : null;
            $rownumbers  = isset( $atts['rownumbers'] ) ? $atts['rownumbers'] : null;
            $paginate    = isset( $atts['paginate'] ) ? $atts['paginate'] : null;


            wp_enqueue_style( 'public_table_view_style', plugin_dir_url(__FILE__) . 'css/table_view.css', array(), '1.0' );

            ob_start();
            include __DIR__ . "/views/table_view.php";
            return ob_get_clean();
        }

        /**
         * Sets shortcode attributes and includes a view.
         *
         * Sets attributes used for shortcode that displays only one field from the table.
         * Includes file that returns one field depending on attributes used.
         *
         * @since 1.0.0
         *
         * @param array $atts Attributes passed in shortcode
         */
        public function bdds_view_field_shortcode ( $atts )
        {
            $id         = isset( $atts['id'] ) ? $atts['id'] : null;
            $table_name = isset( $atts['table'] ) ? $atts['table'] : null;
            $query      = isset( $atts['query'] ) ? $atts['query'] : null;
            $field      = isset( $atts['field'] ) ? $atts['field'] : null;
            $rowid      = isset( $atts['rowid'] ) ? $atts['rowid'] : null;

            // We need to call this to set global variables used in table_view_field
            // $bdds_queried_data, $bdds_table_fields, $bdds_table_aliases, $bdds_attachment_id_fields
            if (!empty($query)) $this->get_query_data($query);

            ob_start();
            include __DIR__ . "/views/table_view_field.php";
            return ob_get_clean();
        }

        /**
         * Retrieves all records from the table.
         *
         *
         * @since 1.0.0
         *
         * @param string $table_name Name of the table in database.
         * @return array|boolean $data Array that contains all records from table || false if table does not exist
         */
        public function get_data( $table_name )
        {
            global $wpdb;
            $table_name   = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_" . strtolower( str_replace( ' ', '_', $table_name ) );

            $data = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare( "SELECT * FROM `$table_name` WHERE `deleted_at` IS NULL;" ) // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            );

            return empty($data) ? false : $data;

        }

        /**
         * Exectues query and returns records.
         *
         *
         * @since 1.0.0
         *
         * @param string $query_name Name of an existing query from database.
         * @return array $data Array that contains all records.
         */
        public function get_query_data( $query_name ) {
            global $wpdb;
            global $bdds_table_fields, $bdds_table_aliases, $bdds_queried_data;
            global $bdds_attachment_id_fields;
            $queries_table_name = $wpdb->prefix . BDDS_PLUGIN_PREFIX . '_queries';

            $query_db = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                $wpdb->prepare("SELECT query,query_object FROM `$queries_table_name` WHERE query_name=%s", $query_name) // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            );

            if (empty($query_db)) {
                return false;
            } else {
                $query = $query_db[0]->query;
                $query_object = unserialize($query_db[0]->query_object);
                $fields = $query_object->getFieldsObject()->getFields();
                $aliases = $query_object->getFieldsObject()->getAliases();
                $bdds_table_fields = $fields;
                $bdds_table_aliases = $aliases;
                $bdds_attachment_id_fields = $query_object->getAttachmentIdAliasesOrFields();
                $append_id = false;
                $table_name_for_id = '';
                $field_name = '';
                $fields_query = '';
                $id_count = 0;

                foreach($fields as $table_field) {
                    $table_field = explode('.', $table_field);
                    $table_name = $table_field[0];
                    $field_name = $table_field[1];

                    if ($field_name == "id") $id_count++;
                }

                foreach($fields as $table_field) {
                    $table_field = explode('.', $table_field);
                    $table_name = isset($table_field[0]) ? $table_field[0] : '';
                    $field_name = isset($table_field[1]) ? $table_field[1] : '';
                    $aliases_key = "field_alias_$table_name" . "_$field_name";

                    if( !in_array($table_name . '.' . "id", $fields) ) {
                        $append_id = true;
                        $table_name_for_id = $table_name;
                    }

                    if( $aliases[$aliases_key] != $table_name . '_' . $field_name ) {
                        $fields_query .= "`$aliases[$aliases_key]`" . ',';

                    } else {
                        $fields_query .= ($id_count > 1) ? $table_name . "_id" . ',' : $field_name . ',';
                    }
                }

                if( $append_id && $id_count < 1) {
                    $field_id = count($fields);
                    $fields_query .= "id";
                    $query_object->getFieldsObject()->setFieldsWithKeys( "field_$field_id", $table_name_for_id . ".id" );
                    $query_object->getFieldsObject()->setAliasesWithKeys("field_alias_$table_name_for_id" . "_id", '');
                    $query = $query_object->createQuery();
                }

                $fields_query = rtrim($fields_query, ',');
                $data = $wpdb->get_results( $wpdb->prepare( "SELECT $fields_query FROM ($query) as `alias`;" ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

                $bdds_queried_data = $data;

                return $data;
            }

        }

        /**
         * Sets shortcode attributes for self-closing shortcode.
         *
         * Sets attributes used in shortcode for repeating fields.
         * Calls shortcode that displays only one field and appends attributes to it.
         * Executes shortcode for every record returned from the table.
         *
         * @since 1.0.0
         *
         * @param array $atts Attributes passed in shortcode
         * @param string $content Content inside a closing shortcode
         * @return string $rv Returns data from fields.
         */
        public function bdds_view_repeat_shortcode ( $atts, $content = null )
        {
            global $bdds_queried_data;
            $id         = isset($atts['id']) ? $atts['id'] : null;
            $table_name = isset($atts['table']) ? $atts['table'] : null;
            $query_name = isset($atts['query']) ? $atts['query'] : null;
            $rv = '';

            if (!empty($table_name) && !empty($query_name) && is_user_logged_in()) {
                $id_text = !empty($id) ? " with the id $id" : '';
                return "There is an error in shortcode$id_text. You need to use query or table name";

            } else if (!empty($table_name) && empty($query_name)) {
                $data = $this->get_data( $table_name );

                if ($data == false && is_user_logged_in()) {
                    return "Table with name \"$table_name\" does not exist!";

                } else if ($data == false && !is_user_logged_in()) {
                    return "There is an error with retrieving data.";

                } else {
                    foreach( $data as $row ) {
                        $nc = str_replace( '[' . BDDS_PLUGIN_PREFIX . '-view-field', "[" . BDDS_PLUGIN_PREFIX .  "-view-field table='$table_name' rowid='$row->id'", $content );
                        $rv .= $nc;
                    }
                }

            } else if (!empty($query_name) && empty($table_name)) {
                $data = $this->get_query_data( $query_name );

                if ($data == false && is_user_logged_in()) {
                    return "Query with name \"$query_name\" does not exist!";

                } else if ($data == false && !is_user_logged_in()) {
                    return "There is an error with retrieving data.";

                } else {
                    foreach($data as $key => $row) {
                        $nc = str_replace( '[' . BDDS_PLUGIN_PREFIX . '-view-field', "[" . BDDS_PLUGIN_PREFIX .  "-view-field query='$query_name' key='$key'", $content );
                        $rv .= $nc;
                    }
                }

            } else return "Unexpected error occurred";

            $bdds_queried_data = $data;

            return do_shortcode( $rv );

        }

        /**
         * Sets attributes for form shortcode.
         *
         * Sets attributes for form shortcode and opens form HTML element.
         *
         * @since 1.0.0
         *
         * @param array $atts Attributes passed in shortcode
         * @return string Returns a view
         */
        public function bdds_form_field_start_shortcode ( $atts )
        {
            global $wpdb;
            global $bdds_table_names;
            $form_id    = isset($atts['id']) ? $atts['id'] : null;
            $table_name = isset($atts['table']) ? $atts['table'] : null;
            $query      = isset($atts['query']) ? $atts['query'] : null;

            if(!empty($bdds_table_names)) {
                array_push($bdds_table_names, $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_" . $table_name);
            } else {
                $bdds_table_names = array();
                array_push($bdds_table_names, $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_" . $table_name);
            }

            ob_start();
            include __DIR__ . "/views/forms/form_start_view.php";
            return ob_get_clean();
        }

        /**
         * Sets attributes for form field shortcode.
         *
         * Sets attributes for form field shortcode and returns inputs for passed fields.
         *
         * @since 1.0.0
         *
         * @param array $atts Attributes passed in shortcode
         * @return string Returns a view
         */
        public function bdds_form_field_shortcode ( $atts )
        {
            $id         = isset($atts['id']) ? $atts['id'] : null;
            $field_name = isset($atts['field']) ? $atts['field'] : null;
            $allow_edit = isset($atts['allowedit']) ? $atts['allowedit'] : null;
            $required   = isset($atts['required']) ? $atts['required'] : null;

            ob_start();
            include __DIR__ . "/views/forms/form_field_view.php";
            return ob_get_clean();
        }

        /**
         * Sets attributes for closing form shortcode.
         *
         * Sets attributes for closing form shortcode.
         *
         * @since 1.0.0
         *
         * @param array $atts Attributes passed in shortcode
         * @return string Returns a view
         */

        public function bdds_form_field_stop_shortcode ( $atts )
        {
            $id = isset($atts['id']) ? $atts['id'] : null;

            ob_start();
            include __DIR__ . "/views/forms/form_stop_view.php";
            return ob_get_clean();
        }
    }

} else {
    echo esc_html(BDDS_PLUGIN_PREFIX) . " public class already exists!";
}