<?php
if (!defined('ABSPATH')) {
    exit;
}

require_once BDDS_PLUGIN_SERVICES_DIR . "BDDS_TableStructureService.php";

class BDDS_TableService
{
    /**
     * Extract the table basename from the full table name.
     *
     * @param string $table_name Full table name (including prefix).
     * @return string The basename of the table.
     */
    public static function get_basename(string $table_name): string
    {
        return str_replace(BDDS_TABLE_PREFIX . '_', '', $table_name);
    }


    /**
     * Return the structure of a specified database table.
     *
     * @param string $table_name Full table name, including prefix.
     * @return stdClass[] The structure of the table.
     *
     * @example Array
     * (
     *     [0] => stdClass Object
     *         (
     *             [Field] => id
     *             [Type] => bigint
     *             [Null] => NO
     *             [Key] => PRI
     *             [Default] =>
     *             [Extra] => auto_increment
     *         )
     *
     *     [2] => stdClass Object
     *         (
     *             [Field] => column_1
     *             [Type] => longtext
     *             [Null] => NO
     *             [Key] =>
     *             [Default] =>
     *             [Extra] =>
     *         )
     *  )
     */
    public static function get_structure(string $table_name): array
    {
        global $wpdb;

        $table_structure = $wpdb->get_results($wpdb->prepare(
            "DESC `$table_name`"), 'OBJECT'
        );

        return $table_structure;
    }


    /**
     * Return an array of names of attachment ID fields for the table by the provided name.
     *
     * @param string $table_name Full table name (including prefix).
     * @return string[] Array of attachment ID field names.
     */
    public static function get_attachment_id_fields(string $table_name): array
    {
        $table_structure = BDDS_TableService::get_structure($table_name);

        return BDDS_TableStructureService::get_attachment_id_fields($table_structure);
    }
}
