<?php
if ( ! defined( 'ABSPATH' ) ) exit;

require_once BDDS_PLUGIN_SERVICES_DIR . "BDDS_TableService.php";

if ( ! class_exists( 'BDDS_Query' ) )
{
    class BDDS_Query
    {
        private $query;
        private $fields_object;
        private $tables_object;
        private $conditions_object;

        public function __construct( BDDS_Select $fields_object, BDDS_From $tables_object, BDDS_Where $conditions_object )
        {
            $this->fields_object = $fields_object;
            $this->tables_object   = $tables_object;
            $this->conditions_object  = $conditions_object;
        }

        public function getTablesObject()
        {
            return $this->tables_object;
        }

        public function getFieldsObject()
        {
            return $this->fields_object;
        }

        public function getConditionsObject()
        {
            return $this->conditions_object;
        }

        public function setConditionsObject( $conditions_object )
        {
            $this->conditions_object = $conditions_object;
        }

        /**
         * Creates a raw query.
         *
         * Gets tables, fields, aliases and conditions objects and combaines them into a raw query.
         *
         * @since 1.0.0
         *
         * @return string query_raw MySQL data type as string.
         */
        public function createQuery()
        {
            global $wpdb;

            $tables            = $this->tables_object->getTables();
            $fields            = $this->fields_object->getFields();
            $aliases           = $this->fields_object->getAliases();
            $conditions        = $this->conditions_object->getConditions();

            foreach($conditions as $key=>$condition) {
                if (isset($condition["field_condition_0"])) {
                    $conditions_operators = array(
                        "Equal to" => "=",
                        "Greater than" => ">",
                        "Less than" => "<",
                        "Greater than or equal to" => ">=",
                        "Less than or equal to" => "<=",
                        "Not Equal to" => "<>",
                        "Like"  => "LIKE",
                    );

                    if (array_key_exists($condition["field_condition_0"], $conditions_operators)) {
                        $condition["field_condition_0"] = $conditions_operators[$condition["field_condition_0"]];
                    }
                }

                $conditions[$key] = $condition;
            }

            $tables_string     = $this->getTablesString($tables);
            $fields_string     = $this->getFieldsString($fields, $aliases);
            $conditions_string = $this->getConditionsString($conditions);
            $conditions_string = !empty($conditions_string) ? "WHERE " . $conditions_string : '';

            $query_raw         = "SELECT $fields_string FROM $tables_string $conditions_string";

            return $query_raw;
        }

        /**
         * Converts array into a string divided by commas.
         *
         *
         * @since 1.0.0
         *
         * @param array $tables Array of table names used in query.
         * @return string $tables_string Table names divided by commas.
         */
        public function getTablesString( $tables )
        {
            $tables_string = '';
            foreach($tables as $table) $tables_string .= $table . ',';

            return rtrim($tables_string, ',');
        }

        /**
         * Returns string used in SELECT statement.
         *
         * Returns a string that contains field names and aliases used in a query to get requested fields.
         *
         * @since 1.0.0
         *
         * @param array $fields Array of fields used in query.
         * @param array $aliases Array of aliases used in query.
         * @return string $fields_string Selected fields with aliases used in query.
         */
        public function getFieldsString( $fields, $aliases )
        {
            global $wpdb;
            $fields_string    = '';
            $wp_tables_prefix = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_";

            foreach($fields as $field) {
                $table_and_field = $wp_tables_prefix . $field;
                $field_name = str_replace('.', '_', $field);
                $table_name = explode('.', $field)[0];
                $column     = end(explode('.', $field));

                $alias = ($field_name != $aliases["field_alias_$field_name"]) ? $aliases["field_alias_$field_name"] : null;

                // check for duplicate column names
                if(empty($alias)) {
                    foreach($fields as $field) {
                        // tn => table_name, c => column
                        $tn   = explode('.', $field)[0];
                        $c    = end(explode('.', $field));
                        // if two same column names exist with different table name, add table prefix to column
                        if($table_name != $tn && $column == $c) $alias = $table_name . '_' . $column;
                    }
                }

                if(!empty($alias)) {
                    $fields_string .= $table_and_field . " AS \"" . $alias . '",';
                } else {
                    $fields_string .= $table_and_field . ',';
                }

            }

            return rtrim($fields_string, ',');
        }

        /**
         * Returns conditions string for query.
         *
         *
         * @since 1.0.0
         *
         * @param array $conditions Array of conditions.
         * @return string $conditions_string String representing WHERE clause in a query.
         */
        public function getConditionsString( $conditions )
        {
            global $wpdb;
            $conditions_string = '';
            $wp_tables_prefix  = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_";
            $conditions = (isset($conditions)) ? $conditions : array();

            foreach($conditions as $condition_array) {
                $operator     = isset($condition_array["condition_operator_0"]) ? $condition_array["condition_operator_0"] : '';
                $field_name   = isset($condition_array["condition_field_name_0"]) ? $condition_array["condition_field_name_0"] : '';
                $condition    = isset($condition_array["field_condition_0"]) ? $condition_array["field_condition_0"] : '';
                $type         = isset($condition_array["field_value_type_0"]) ? $condition_array["field_value_type_0"] : '';
                $value_select = !empty($condition_array["field_value_select_0"]) ? $condition_array["field_value_select_0"] : '';
                $value_text   = !empty($condition_array["field_value_text_0"]) ? $condition_array["field_value_text_0"] : '';
                $value        = ($type == "var") ? '"' . $value_text . '"' : $value_select;

                $field_name   = $wp_tables_prefix . $field_name;

                $conditions_string .= strtoupper($operator) . ' (' . $field_name . ' ' . $condition . ' ' . $value . ') ';

            }

            return $conditions_string;
        }


        /**
         * Return an array of all the aliases that contain attachment IDs. For
         * an alias that is not set, field name is returned.
         *
         * @return string[] Array of aliases (or field names) containing attachment IDs.
         */
        public function getAttachmentIdAliasesOrFields(): array
        {
            $attachment_id_fields = [];
            $query_tables = $this->getTablesObject()->getTables();
            $query_fields = $this->getFieldsObject();

            /* For each table in the query:
                get names of all the fields that contain attachment IDs,
                get alias if set, otherwise field name, for each field containing
                attachment ID */
            foreach ($query_tables as $table_name) {
                $table_attachment_id_fields =
                    BDDS_TableService::get_attachment_id_fields($table_name);
                $table_basename = BDDS_TableService::get_basename($table_name);

                foreach ($table_attachment_id_fields as $field) {
                    $field_display_name =
                        $query_fields->getAliasOrFieldName($table_basename, $field);
                    $attachment_id_fields[] = $field_display_name;
                }
            }

            return $attachment_id_fields;
        }

    }
}



////////////////////////////////////////////////////////
//////////////////// OTHER CLASSES ////////////////////
//////////////////////////////////////////////////////

if ( ! class_exists( 'BDDS_Select' ) )
{
    class BDDS_Select
    {
        private $fields;
        private $aliases;

        public function __construct( $fields, $aliases = array() )
        {
            $this->fields = $fields;
            $this->aliases = $aliases;
        }

        public function setFields( $fields )
        {
            $this->fields = $fields;
        }

        public function setFieldsWithKeys( $key, $field )
        {
            $this->fields[$key] = $field;
        }

        public function setAliases( $aliases )
        {
            $this->aliases = $aliases;
        }

        public function getFields()
        {
            return $this->fields;
        }

        public function setAliasesWithKeys( $key, $alias )
        {
            $this->aliases[$key] = $alias;
        }

        /**
         * Get a map of the query aliases, or column names, if the aliases are not set.
         *
         * @return array
         *
         * @example ` Array
         *(
         *    [field_alias_product_id] => Product ID
         *    [field_alias_product_name] => Product Name
         *    [field_alias_review_id] => Review ID
         *    [field_alias_review_product_id] => FK @review (product ID)
         *    [field_alias_review_user_id] => User ID
         *    [field_alias_review_grade] => Grade
         *    [field_alias_review_comment] => Comment
         * )
         *
         * @example ` (aliases not set) Array
         * (
         *     [field_alias_content_id] => content_id
         *     [field_alias_content_attachment] => content_attachment
         *     [field_alias_content_column_1] => content_column_1
         * )
         */
        public function getAliases()
        {
            return $this->aliases;
        }

        /**
         * Get the query alias for a specific field if set, otherwise return the field name.
         *
         * @param string $table_basename The basename of the table.
         * @param string $field_name The name of the field.
         * @return string The alias if set, or the field name if no alias is set.
         *
         * @example ` $display_name = $query_fields->getAliasOrFieldName('users', 'email');
         */
        public function getAliasOrFieldName($table_basename, $field_name): string
        {
            if (empty($table_basename) || empty($field_name)) {
                error_log(
                    __METHOD__ . '(), ln: ' . __LINE__
                    . ' - table_basename or field_name is empty. table_basename: '
                    . $table_basename . ', field_name: ' . $field_name
                );

                return '-';
            }

            $alias_key = BDDS_ALIAS_PREFIX . $table_basename . '_' . $field_name;
            $alias = $this->aliases[$alias_key];

            if (empty($alias)) {
                return '-';
            }

            /* Value format is different depending on whether alias is set or not.
                Examples:
                    - alias set: $alias === 'Email' => return as is;
                    - alias not set: $alias === 'users_email'
                        (table basename + field name) => return just the field name. */
            if ($alias === $table_basename . '_' . $field_name) {
                return $field_name;
            }
            return $alias;
        }

    }
}

if ( ! class_exists( 'BDDS_From' ) )
{
    class BDDS_From
    {
        private $tables;

        public function __construct( $tables )
        {
            $this->tables = $tables;
        }

        public function setTables( $tables )
        {
            $this->tables = $tables;
        }

        public function getTables()
        {
            return $this->tables;
        }

        public function addTable( $table_name )
        {
            array_push($this->tables, $table_name);
        }
    }
}

if ( ! class_exists( 'BDDS_Where' ) )
{
    class BDDS_Where
    {
        private $fields;
        private $conditions;

        public function __construct()
        {
            $this->conditions = array();
        }

        public function setFields( $fields )
        {
            $this->fields = $fields;
        }

        public function setConditions( $conditions )
        {
            $this->conditions = $conditions;
        }

        public function setConditionsWithKeys( $conditions_arr )
        {
            $ids = array();
            $conditions = isset($this->conditions) ? $this->conditions : array();

            foreach($conditions as $key=>$condition) {
                $id = str_replace("condition_", '', $key);
                array_push($ids, $id);
            }

            $condition_id = !empty($ids) ? max($ids) + 1 : 0;

            $this->conditions["condition_$condition_id"] =  $conditions_arr;
        }

        public function getFields()
        {
            return $this->fields;
        }

        public function getConditions()
        {
            return $this->conditions;
        }

        public function addFieldName( $key, $field_name )
        {
            $this->fields[$key] =  $field_name;
        }
    }
}