<?php
    if ( ! defined( 'ABSPATH' ) ) exit;

    class BDDS_Helpers
    {
        /**
         * Returns MySQL data type.
         *
         * Depending on column type provided function returns corresponding MySQL data type.
         *
         * @since 1.0.0
         *
         * @param string $column_type Column type.
         * @return string MySQL data type as string.
         */
        public static function switchTypeForSQL( $column_type )
        {
            $types = array(
                "Short text" => "varchar(255)",
                "Number"     => "int(11)",
                "Big number" => "bigint(20)",
                "Currency"   => "decimal(15,2)",
                "Date time"  => "datetime",
                "Yes no"     => "bit(1)",
                "Lookup"     => "varchar(100)",
                "Rich text"  => "mediumtext",
                "Long text"  => "longtext",
                /* !!! IMPORTANT: The whole uploaded file displaying logic revolves
                    around column type being 'longblob' (BDDS_ATTACHMENT_COLUMN_TYPE)
                    because of legacy reasons.
                    DO NOT add more 'longblob' types before considering this! */
                "Attachment" => "longblob",
                "Hyperlink"  => "varchar(1024)",
                "ENUM"       => "ENUM"
            );

            if(array_key_exists($column_type, $types)) return $types[$column_type];
            else return "Unexpected type was passed!";
        }

        /**
         * Returns data type.
         *
         * Depending on MySQL data type provided function returns corresponding data type as string.
         *
         * @since 1.0.0
         *
         * @param string $sql_column_type MySQL column type.
         * @return string Data type as string.
         */
        public static function getColumnTypeByName( $sql_column_type ) {
            $types = array(
                "varchar(255)" => "Short text",
                "int(11)" => "Number",
                "bigint(20)" =>  "Big number",
                "decimal(15,2)" => "Currency",
                "datetime" => "Date time",
                "bit(1)" => "Yes no",
                "varchar(100)" => "Lookup",
                "mediumtext" => "Rich text",
                "longtext" => "Long text",
                "longblob" => "Attachment",
                "varchar(1024)" => "Hyperlink",
            );

            if(array_key_exists($sql_column_type, $types)) return $types[$sql_column_type];
            else if (str_starts_with($sql_column_type, "enum")) return "ENUM";
            else return "Unexpected type was passed!";
        }

        /**
         * Checks if value is of valid type.
         *
         *
         * @since 1.0.0
         *
         * @param string $type Data type.
         * @param string $value Value provided by user.
         * @param string $column_name Name of the column where value is going to be inserted.
         * @return string $err_msg If value is invalid a proper error message is returned.
         */
        public static function validateData($type, $value, $column_name) {
            global $wpdb;
            $err_msg      = '';
            $decimal_reg  = "/^-?\d{0,15}(,\d+)*(\.\d{0,2}(e\d+)?)?$/";
            $lookup_reg   = "/[a-zA-Z0-9]\.[a-zA-Z0-9]/";

            if (($type == "Number" || $type == "Big number" || str_contains($type, "int") ||
                str_contains($type, "bigint") ) && !is_numeric($value) && !empty($value)) {
                    /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be a number', 'black-desk'), $column_name);

            } else if (($type == "Currency" || str_contains($type, "decimal")) && !preg_match($decimal_reg, $value) && !empty($value)) {
                /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be a decimal number {15,2}', 'black-desk'), $column_name);

            } else if($type == "bit" && ($value != "Yes" && $value != "No") && !empty($value)) {
                /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be "Yes" or "No"', 'black-desk'), $column_name);

            } else if($type == "datetime" && !empty($value) && !BDDS_Helpers::validateDate($value)) {
                /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be formatted as Y-m-d H:i:s', 'black-desk'), $column_name);

            } else if ($type == "ENUM" ) {

            }

            return (empty($err_msg)) ? "valid" : $err_msg;
        }

        /**
         * Validates date format.
         *
         * Validates if date is formatted right.
         *
         * @since 1.0.0
         *
         * @param string $date Date that should be validated.
         * @param string $format Format of date, if none is passed function uses default format 'Y-m-d H:m:s'
         * @return boolean true if date is formatted right.
         */
        public static function validateDate($date, $format = "Y-m-d H:i:s" ) {
            $d = DateTime::createFromFormat($format, $date);
            return $d && $d->format($format) === $date;
        }
    }
