<?php
/*
*   @since 1.0.0

*   @package black-desk
*   @subpackage black-desk/admin
*/

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'BDDS_Admin' ) ) {
    require_once "classes/queryRouter.php";
    require_once BDDS_PLUGIN_DIR . "admin/classes/tables/BDDS_Column.php";
    require_once BDDS_PLUGIN_DIR . "admin/classes/tables/BDDS_Columns.php";
    require_once BDDS_PLUGIN_DIR . "admin/classes/BDDS_Type_Constants.php";

    /**
     * Admin class for backend functionalities.
     *
     * Adds menu and submenu pages to WordPress dashboard.
     * Routing trough menu and submenu pages and data validation.
     *
     * @since 1.0.0
     */
    class BDDS_Admin {

        private static $instance;
        private $plugin_name;
        private $version;
        private $admin_notices = array(); // Store multiple notices


        /**
         * Get the singleton instance of the BDDS_Admin class.
         *
         * @since 1.1.1
         *
         * @return BDDS_Admin The singleton instance.
         */
        public static function get_instance() {
            if (null === self::$instance) {
                self::$instance = new self(BDDS_PLUGIN_SLUG, BDDS_VERSION);
            }
            return self::$instance;
        }


        /**
         * BDDS_Admin class constructor.
         *
         * Initiates menu and submenu pages.
         * Initiates route method for pages and starts a session if it is not already defined.
         *
         * @since 1.0.0
         *
         * @var String $plugin_name Plugin name.
         * @var Int $version Plugin version.
         * @return void
         */
        public function __construct( $plugin_name, $version )
        {
            if (null !== self::$instance) {
                throw new \RuntimeException('Cannot instantiate singleton class multiple times. Use get_instance() instead.');
            }

            self::$instance = $this;
            $this->plugin_name = $plugin_name;
            $this->version = $version;

            if( !session_id() ) session_start();

            add_action('admin_menu', array( $this, 'bdds_admin_tables_menu_page' ));
            add_action('admin_menu', array( $this, 'bdds_admin_show_tables_submenu_page' ));
            add_action('admin_menu', array( $this, 'bdds_admin_add_table_name' ));
            add_action('admin_menu', array( $this, 'bdds_admin_clean_table_submenu_page' ));
            add_action('admin_menu', array( $this, 'bdds_admin_query_all_tables'));
            add_action('admin_menu', array( $this, 'bdds_admin_add_table_columns' ));
            add_action('admin_menu', array( $this, 'bdds_admin_preview_table' ));
            add_action('admin_menu', array( $this, 'bdds_admin_remove_column' ));
            add_action('admin_menu', array( $this, 'bdds_admin_query_submenu_page'));
            add_action('admin_menu', array( $this, 'bdds_admin_query_clean_submenu_page'));
            add_action('admin_menu', array( $this, 'bdds_admin_add_query_fields' ));
            add_action('admin_menu', array( $this, 'bdds_admin_add_query_aliases' ));
            add_action('admin_menu', array( $this, 'bdds_admin_query_preview' ));
            add_action('admin_menu', array( $this, 'bdds_admin_delete_query' ));
            add_action('admin_menu', array( $this, 'bdds_admin_shortcodes' ));
            add_action('admin_menu', array( $this, 'bdds_admin_settings' ));
            add_action('admin_menu', array( $this, 'bdds_admin_add_conditions' ));
            add_action('admin_menu', array( $this, 'bdds_admin_delete_condition' ));

            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-clean-table-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-add-table-columns-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-preview-table-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-remove-column-slug');
            });

            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-add-query-fields-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-add-query-aliases-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-query-preview-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-query-name-and-tables-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-query-clean-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-query-delete-query-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-add-query-conditions-slug');
            });
            add_action('admin_head', function(){
                remove_submenu_page('bdds-tables-menu-slug', 'bdds-query-delete-condition-slug');
            });

            // Actions for redirections and form handling
            add_action('init', array( $this, 'route' ));
            add_action('init', array( $this, 'bdds_handle_drop_table_form' ));
            add_action('init', array( $this, 'bdds_handle_truncate_table_form' ));
            add_action('init', array( $this, 'bdds_handle_table_view_form' ));
            add_action('init', array( $this, 'bdds_handle_view_restore_trash_form' ));
            add_action('admin_init', array( $this, 'bdds_handle_clean_table_check' ));
            add_action('admin_init', array( $this, 'bdds_handle_clean_query_check' ));

        }

        /**
         * Adds a top-level menu.
         *
         * Adds a top-level menu page that holds three submenu pages.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix.
         */
        public function  bdds_admin_tables_menu_page () {
            add_menu_page (
                'Tables',
                __('Tables', 'black-desk'),
                'manage_options',
                'bdds-tables-menu-slug',
                '',
                'dashicons-editor-table'
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page that shows create tables.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_show_tables_submenu_page () {
            add_submenu_page (
                'bdds-tables-menu-slug',
                'All Tables',
                __('All Tables', 'black-desk'),
                'manage_options',
                'bdds-tables-menu-slug',
                array( $this, 'bdds_show_tables_view' )
            );
        }

        /**
         * Callback for submenu page that displays all created tables.
         *
         * Callback used in method that creates a submenu page to display all tables.
         * Enqueues JS script and CSS file.
         * Includes a view for submenu page.
         *
         * @since 1.0.0
         *
         * @return string View for all tables submenu page.
         */
        public function bdds_show_tables_view () {
            wp_enqueue_script( 'admin_show_tables', BDDS_PLUGIN_DIR_URL . 'admin/js/showTables.js', array(), '1.0', true );
            wp_enqueue_style( 'admin_show_tables_style', BDDS_PLUGIN_DIR_URL . 'admin/css/show_tables_style.css', array(), '1.0');
            include __DIR__  . "/views/show_tables.php";
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view for defining table name, description and collation.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_add_table_name() {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Add table', 'black-desk' ),
                __( 'Add table', 'black-desk' ),
                'manage_options',
                'bdds-add-table-name-slug',
                function () {
                    include "views/add_table_views/add_table_name.php";
                }
            );
        }

        /**
         * Allows for a separate slug for starting to create a table in order to
         * clear the table session variables beforehand.
         *
         *  Common view is used to:
         *   - start creating a new table,
         *   - go back a step in table creation.
         *
         * (accessible through bdds-add-table-name-slug).
         *
         * Going back a step needs session variables, while starting a new table
         * creation needs the session variables cleared beforehand.
         *
         * Therefore - separate slug to clear the session and then redirect to
         * a common table slug.
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_clean_table_submenu_page()
        {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __('Clean Table', 'black-desk'),
                __('Clean Table', 'black-desk'),
                'manage_options',
                'bdds-clean-table-slug',
                array($this, 'bdds_handle_clean_table')
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view for defining table columns.
         * Enqueues JS script for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_add_table_columns() {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Add table columns', 'black-desk' ),
                __( 'Add table columns', 'black-desk' ),
                'manage_options',
                'bdds-add-table-columns-slug',
                function () {
                    include "views/add_table_views/add_table_columns.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/addColumns.js', array(), '1.0', true );
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view for preview page.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_preview_table() {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Preview table', 'black-desk' ),
                __( 'Preview table', 'black-desk' ),
                'manage_options',
                'bdds-preview-table-slug',
                function () {
                    include "views/add_table_views/preview_table.php";
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view for column removal.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_remove_column() {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Remove column', 'black-desk' ),
                __( 'Remove column', 'black-desk' ),
                'manage_options',
                'bdds-remove-column-slug',
                function () {
                    include "views/add_table_views/remove_column.php";
                }
            );
        }

        /* Query menu and pages */

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view for defining query name and tables that are going to be used in the query.
         * Enqueues JS script and CSS style for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_query_submenu_page () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Queries', 'black-desk' ),
                __( 'Queries', 'black-desk' ),
                'manage_options',
                'bdds-query-name-and-tables-slug',
                function () {
                    include "views/query_views/query_name.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                    wp_enqueue_style( 'admin_show_query_tables', BDDS_PLUGIN_DIR_URL . 'admin/css/query_tables_style.css', array(), '1.0' );
                }
            );
        }


        /**
         * Allows for a separate slug for starting to create or edit a query
         * in order to clear the query session variables beforehand.
         *
         * Common view is used to:
         *   - start creating a new query,
         *   - go back a step in query creation,
         *   - start editing an existing query,
         *   - go back a step in query editing.
         *
         * (accessible through bdds-query-name-and-tables-slug).
         *
         * Going back a step needs session variables, while starting a new query
         * creation or editing needs the session variables cleared beforehand.
         *
         * Therefore - separate slug to clear the session and then redirect to
         * a common query slug.
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_query_clean_submenu_page ()
        {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __('Clean Query', 'black-desk'),
                __('Clean Query', 'black-desk'),
                'manage_options',
                'bdds-query-clean-slug',
                array($this, 'bdds_handle_clean_query')
            );
        }


        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view that displays all created queries.
         * Enqueues JS script and CSS style for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_query_all_tables() {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'All queries', 'black-desk' ),
                __( 'All queries', 'black-desk' ),
                'manage_options',
                'bdds-all-queries-slug',
                function () {
                    include "views/query_views/all_queries.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                    wp_enqueue_style( 'admin_show_query_tables', BDDS_PLUGIN_DIR_URL . 'admin/css/query_tables_style.css', array(), '1.0' );
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view to choose columns from selected tables that are going to be used in a query.
         * Enqueues JS script and CSS style for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_add_query_fields () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Query fields', 'black-desk' ),
                __( 'Query fields', 'black-desk' ),
                'manage_options',
                'bdds-add-query-fields-slug',
                function () {
                    include "views/query_views/query_fields.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                    wp_enqueue_style( 'admin_show_query_tables', BDDS_PLUGIN_DIR_URL . 'admin/css/query_tables_style.css', array(), '1.0' );
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view to add column aliases for selected columns.
         * Enqueues JS script for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_add_query_aliases () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Query aliases', 'black-desk' ),
                __( 'Query aliases', 'black-desk' ),
                'manage_options',
                'bdds-add-query-aliases-slug',
                function () {
                    include "views/query_views/query_aliases.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view to add query conditions for selected columns and tables.
         * Enqueues JS script and CSS style for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_add_conditions () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Query conditions', 'black-desk' ),
                __( 'Query conditions', 'black-desk' ),
                'manage_options',
                'bdds-add-query-conditions-slug',
                function () {
                    include "views/query_views/query_conditions.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                    wp_enqueue_style( 'admin_query_style', BDDS_PLUGIN_DIR_URL . "admin/css/query_tables_style.css", array(), '1.0', true);

                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view that previews the query.
         * Enqueues JS script for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_query_preview () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Query preview', 'black-desk' ),
                __( 'Query preview', 'black-desk' ),
                'manage_options',
                'bdds-query-preview-slug',
                function () {
                    include "views/query_views/query_preview.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view to remove created query.
         * Enqueues JS script for the view.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_delete_query () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Query delete', 'black-desk' ),
                __( 'Query delete', 'black-desk' ),
                'manage_options',
                'bdds-query-delete-query-slug',
                function () {
                    include "views/query_views/query_delete.php";
                    wp_enqueue_script( 'admin_add_table_script', BDDS_PLUGIN_DIR_URL . 'admin/js/queryTable.js', array(), '1.0', true );
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view to remove an added condition.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_delete_condition () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Condition delete', 'black-desk' ),
                __( 'Condition delete', 'black-desk' ),
                'manage_options',
                'bdds-query-delete-condition-slug',
                function () {
                    include "views/query_views/query_remove_condition.php";
                }
            );
        }

        /**
         * Adds a submenu page.
         *
         * Adds a submenu page and includes a view for settings page.
         *
         * @since 1.0.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_settings () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Settings', 'black-desk' ),
                __( 'Settings', 'black-desk' ),
                'manage_options',
                'bdds-tables-settings-slug',
                function () {
                    include "views/settings.php";
                }
            );
        }

        /**
         * Adds a submenu page for displaying available shortcodes.
         *
         * Adds a submenu page and includes a view with information about plugin shortcodes.
         * Enqueues CSS files for styling the shortcodes page.
         *
         * @since 1.1.0
         *
         * @return string The resulting page's hook_suffix, or false if the user does not have the capability required.
         */
        public function bdds_admin_shortcodes () {
            add_submenu_page(
                'bdds-tables-menu-slug',
                __( 'Shortcodes', 'black-desk' ),
                __( 'Shortcodes', 'black-desk' ),
                'manage_options',
                'bdds-tables-shortcodes-slug',
                function () {
                    include "views/shortcodes.php";
                    wp_enqueue_style('admin_info_style', BDDS_PLUGIN_DIR_URL . 'admin/css/info_style.css', array(), '1.0');
                    wp_enqueue_style('admin_shortcodes_style', BDDS_PLUGIN_DIR_URL . 'admin/css/shortcodes_style.css', array(), '1.0');
                }
            );
        }


        ///// Submenu Pages Functions

        /**
         * Handles the redirection for the clean table page
         */
        public function bdds_handle_clean_table()
        {
            // This function is just a placeholder for the admin page
            // The actual redirect happens in bdds_handle_clean_table_check
        }

        /**
         * Checks and performs the redirect during admin_init for table cleaning
         */
        public function bdds_handle_clean_table_check()
        {
            if (!is_admin()) {
                return;
            }

            // Check if we're on our specific page
            if (isset($_GET['page']) && $_GET['page'] === 'bdds-clean-table-slug') {

                unset($_SESSION['bdds_variables']);

                $redirect_url = add_query_arg(
                    [
                        'page' => 'bdds-add-table-name-slug',
                        '_wpnonce' => wp_create_nonce('bdds-add-table-nonce')
                    ],
                    admin_url('admin.php')
                );

                wp_safe_redirect($redirect_url);
                exit;
            }
        }


        /**
         * Handles the redirection for the clean query page
         */
        public function bdds_handle_clean_query()
        {
            // This function is just a placeholder for the admin page
            // The actual redirect happens in bdds_handle_clean_query_check
        }

        /**
         * Checks and performs the redirect during admin_init
         */
        public function bdds_handle_clean_query_check()
        {
            if (!is_admin()) {
                return;
            }

            // Check if we're on our specific page
            if (isset($_GET['page']) && $_GET['page'] === 'bdds-query-clean-slug') {

                unset($_SESSION['bdds_query_variables']);
                unset($_SESSION['bdds_edit_variables']);

                $query_id = isset($_GET['query_id']) ? intval($_GET['query_id']) : null;
                $action = isset($_GET['action'])
                    ? sanitize_text_field(wp_unslash($_GET['action']))
                    : null;

                $redirect_url = add_query_arg(
                    [
                        'page' => 'bdds-query-name-and-tables-slug',
                        'query_id' => $query_id,
                        'action' => $action,
                        '_wpnonce' => wp_create_nonce('create-select-query-nonce')
                    ],
                    admin_url('admin.php')
                );

                wp_safe_redirect($redirect_url);
                exit;
            }
        }

        ///// END of Submenu Pages Functions


        /**
         * Validates whether the value is of an acceptable type.
         *
         *
         * @since 1.0.0
         *
         * @param int|float|string $value Value that should be validated.
         * @param string $table Name of the table where value is going to be inserted.
         * @param string $field Name of the field where value is going to be inserted.
         * @return string $err_msg If value is invalid, proper error message is returned.
         */
        public static function validate_data($type, $value, $column_name) {
            $err_msg      = '';
            $decimal_reg  = "/^-?\d{0,15}(,\d+)*(\.\d{0,2}(e\d+)?)?$/";
            $lookup_reg   = "/[a-zA-Z0-9]\.[a-zA-Z0-9]/";

            if (($type == "Number" || $type == "Big number" || str_contains($type, "int") ||
                str_contains($type, "bigint") ) && !is_numeric($value) && !empty($value)) {
                /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be a number', 'black-desk'), $column_name);

            } else if (($type == "Currency" || str_contains($type, "decimal")) && !preg_match($decimal_reg, $value) && !empty($value)) {
                /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be a decimal number {15,2}', 'black-desk'), $column_name);

            } else if($type == "bit" && ($value != "Yes" && $value != "No") && !empty($value)) {
                /* translators: %s: name of the column */
                $err_msg = sprintf(__('Value for "%s" must be "Yes" or "No"', 'black-desk'), $column_name);

            } else if($type == "Lookup" && !preg_match($lookup_reg, $value) && !empty($value)) {
                $err_msg = __('Lookup table value should be formatted as "table_name.field_name"', 'black-desk');
            }

            return (empty($err_msg)) ? "valid" : $err_msg;
        }

        /**
         * Validates date format.
         *
         * Validates if date is formatted right.
         *
         * @since 1.0.0
         *
         * @param string $date Date that should be validated.
         * @param string $format Format of date, if none is passed function uses default format 'Y-m-d H:m:s'
         * @return boolean true if date is formatted right.
         */
        public static function validateDate($date, $format = 'Y-m-d H:m:s') {
            $d = DateTime::createFromFormat($format, $date);

            return $d && $d->format($format) == $date;
        }

        /**
         * Redirects user to specific url.
         *
         * Changes header and redirects user to provided url.
         *
         * @since 1.0.0
         *
         * @param string $url Path to the page where user should be redirected.
         */
        public static function redirectTo($url) {
            header("Location: $url", true);
        }

        /**
         * Routes and session variables declaration.
         *
         * Sets session variables depending on POST values and redirects users.
         *
         * @since 1.0.0
         *
         */
        public function route() {
            global $wpdb;
            $nonce = isset($_GET["_wpnonce"]) && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET["_wpnonce"])) , "bdds-add-table-nonce") ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;

            // Create custom table: Add table name, description and collation
            if( isset($_POST['table_name_next']) && check_admin_referer('table-name-nonce')) {
                global $wpdb;
                $table_name = !empty($_POST['bdds_table_name']) ? $wpdb->prefix . "bdds_" . wp_kses(wp_unslash($_POST['bdds_table_name']), array()) : null;

                $show_table_sql = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

                if(( $show_table_sql == $table_name ) || ($table_name == $wpdb->prefix . "bdds_") ) {

                } else if(preg_match("/^[a-zA-Z0-9_]+$/", $table_name)) {
                    $url = add_query_arg("page", "bdds-add-table-columns-slug");

                    $_SESSION["bdds_variables"]["bdds_table_name"]        = isset($_POST['bdds_table_name']) ? sanitize_text_field( wp_unslash($_POST['bdds_table_name']) ) : null;
                    $_SESSION["bdds_variables"]["bdds_table_description"] = isset($_POST['bdds-table-description']) ? sanitize_text_field( wp_unslash($_POST['bdds-table-description']) ) : null;
                    $_SESSION["bdds_variables"]["bdds_table_collation"]   = isset($_POST['bdds-table-collation']) ? sanitize_text_field( wp_unslash($_POST['bdds-table-collation']) ) : null;
                    $_SESSION["bdds_variables"]["bdds_create_started"]    = 1;

                    BDDS_Admin::redirectTo($url);
                }
            }

            /*
                This is called when user goes directly on 'Add table' from menu so we cannot pass nonce to it
                So we refresh the page and we add nonce to the url so it can be processed
            */
            $page_slug = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : null; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            if ($page_slug == 'bdds-add-table-name-slug' && !isset($_GET["_wpnonce"])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $url_with_nonce = add_query_arg("_wpnonce", wp_create_nonce("bdds-add-table-nonce"));
                BDDS_Admin::redirectTo($url_with_nonce);
            }

            // redirect from page that add's table column's to starting page
            if( isset($_POST['table_name_back']) ) {
                $url = add_query_arg("page", "bdds-add-table-name-slug");

                BDDS_Admin::redirectTo($url);
            }

            // Create custom table: Add new column when creating a new table
            if (
                isset($_POST[BDDS_PLUGIN_PREFIX . "-column-name"]) &&
                isset($_POST[BDDS_PLUGIN_PREFIX . "-column-type"]) &&
                isset($_POST["add_new_column"]) &&
                (
                    wp_verify_nonce($nonce, "bdds-add-table-nonce") ||
                    check_admin_referer("bdds-table-columns-nonce")
                )
            ) {
                $post_array    = map_deep($_POST, 'sanitize_text_field');
                $columns = isset($_SESSION[BDDS_PLUGIN_PREFIX . "_variables"]["columns"])
                    ? unserialize(sanitize_text_field($_SESSION[BDDS_PLUGIN_PREFIX . "_variables"]["columns"]))
                    : new BDDS_Columns();

                $new_default_value = isset($_POST[BDDS_PLUGIN_PREFIX . "-default-value"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-default-value"])) : null;
                $col_name          = isset($_POST[BDDS_PLUGIN_PREFIX . "-column-name"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-column-name"])) : null;
                $duplicate         = false;
                $invalid_input_msg = false;

                // validate if default value user entered is valid
                $column_type_to_check = sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-column-type"]));
                $column_name_to_check = sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-column-name"]));
                $is_valid_input    = BDDS_Admin::validate_data($column_type_to_check, $new_default_value , $column_name_to_check);

                $created_columns = (null != $columns->getColumns()) ? $columns->getColumns() : array();
                // check if new column already exists
                if(count($created_columns) > 0) {
                    foreach($created_columns as $key => $created_column) {
                        $created_column_name = $created_column->getColumnName();
                        $new_column_name     = $col_name;

                        if($created_column_name == $new_column_name) $duplicate = true;

                    }
                }

                if ($duplicate) {
                    /* translators: %s: name of the column */
                    $this->bdds_set_admin_notice(sprintf(
                        esc_html__('Column name "%s" already exists!', 'black-desk'),
                        esc_html($col_name)
                    ), 'error');

                } else if($is_valid_input != "valid"  && isset($_POST["add_new_column"])) {
                    // user input is invalid, set variable to true and later display error above form
                    $invalid_input_msg = true;
                    $this->bdds_set_admin_notice($is_valid_input, 'error');

                } else if(!preg_match("/^[a-zA-Z0-9_]+$/", $col_name) && isset($_POST["add_new_column"])) {
                    // column name entered by user is wrong, show error
                    /* translators: %s: name of the column */
                    $this->bdds_set_admin_notice(sprintf(
                        esc_html__('Wrong column name "%s"', 'black-desk'), esc_html($col_name)
                    ), 'error');

                } else if ( isset($_POST["add_new_column"] ) && $is_valid_input == "valid") {
                    $column = new BDDS_Column();
                    $arr = array();
                    // user input is good, iterate trough post array and save variables in session array
                    foreach( $post_array as $post_key=>$post_value ) {
                        $column_name         = sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-column-name"]));
                        $column_type         = sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-column-type"]));
                        $column_null         = isset($_POST[BDDS_PLUGIN_PREFIX . "-is-null"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-is-null"])) : null;
                        $column_default      = isset($_POST[BDDS_PLUGIN_PREFIX . "-default-value"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-default-value"])) : null;
                        $column_table_lookup = isset($_POST[BDDS_PLUGIN_PREFIX . "-lookup-table"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-lookup-table"])) : null;
                        $column_field_lookup = isset($_POST[BDDS_PLUGIN_PREFIX . "-column-lookup"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-column-lookup"])) : null;
                        // Enum values are sanitized a little bit later in the code
                        $column_enum_values  = isset($_POST[BDDS_PLUGIN_PREFIX . "-enum-values"]) ? $_POST[BDDS_PLUGIN_PREFIX . "-enum-values"] : null; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

                        if( !empty($column_name) && !empty($column_type) ) {
                            $column->setColumnName( $column_name );
                            $column->setColumnType( $column_type );
                            $column->setColumnNull( $column_null );

                            if(!empty($column_default)) {
                                $column->setColumnDefault( $column_default );
                            }

                            if(!empty($column_table_lookup) && !empty($column_field_lookup)) {
                                $column->setLookup( $column_table_lookup, $column_field_lookup );
                            }

                            if($column_type == "ENUM" && !empty($column_enum_values)) {
                                $enum_values = explode("\r\n", trim($column_enum_values));
                                $sanitized_enum_values = array();

                                // Here we sanitize enum values
                                foreach($enum_values as $enum_value) {
                                    array_push($sanitized_enum_values, sanitize_text_field(wp_unslash($enum_value)));
                                }

                                $column->setEnumValues($sanitized_enum_values);
                            }

                        }
                    }

                    $columns->addNewColumn( $column );

                    $_SESSION[BDDS_PLUGIN_PREFIX . "_variables"]["columns"] = serialize( $columns );

                    $url = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
                    BDDS_Admin::redirectTo($url);
                }

                // user submitted incorrect values, print error
                if(isset($invalid_input_msg) && $invalid_input_msg) {
                    $this->bdds_set_admin_notice(esc_html($is_valid_input), 'error');
                }

            }

            // redirect to preview table page
            if( isset($_POST['preview_table']) ) {
                $url = add_query_arg("page", "bdds-preview-table-slug");

                $columns_object = isset($_SESSION["bdds_variables"]["columns"])
                    ? unserialize(sanitize_text_field(wp_unslash($_SESSION["bdds_variables"]["columns"])))
                    : null;

                $columns = isset($columns_object) ? $columns_object->getColumns() : array();

                // count added column's
                $number_of_columns = count($columns);

                // if there is at least one added column we redirect user to preview table
                // else user stays on the page
                if($number_of_columns > 0) {
                    BDDS_Admin::redirectTo($url);
                }
            }

            // redirect from preview page to add column's page
            if( isset($_POST['add_more_columns']) ) {
                $url = add_query_arg("page", "bdds-add-table-columns-slug");

                BDDS_Admin::redirectTo($url);
            }

            // if table was successfully created redirect to 'All tabels' page
            if( isset($_POST['preview_save_columns']) ) {
                include "views/add_table_views/save_table.php";

                if ($query_success) {
                    $url = add_query_arg("page", "bdds-tables-menu-slug" );

                    BDDS_Admin::redirectTo($url);
                } else {
                    $wpdb->show_errors();
                    if( !empty( $wpdb->last_error ) ) return $wpdb->last_error;
                    $wpdb->hide_errors();
                }
            }

            if( isset($_POST['yes-delete-column']) && check_admin_referer("table-delete-column") ) {
                $columns_object = isset($_SESSION["bdds_variables"]["columns"])
                                ? unserialize(sanitize_text_field(wp_unslash($_SESSION["bdds_variables"]["columns"])))
                                : null;
                $columns   = isset($columns_object) ? $columns_object->getColumns() : null;
                $column_id = isset($_POST["column_id"]) ? sanitize_text_field(wp_unslash($_POST["column_id"])) : null;

                unset($columns["column_$column_id"]);

                $columns_object->setColumns($columns);
                $_SESSION["bdds_variables"]["columns"] = serialize($columns_object);

                $url = remove_query_arg("column_id");
                $url = str_replace("page=bdds-remove-column-slug", "page=bdds-add-table-columns-slug", $url);
                BDDS_Admin::redirectTo($url);
            }

            if( isset($_POST['no-delete-column']) && check_admin_referer("table-delete-column") ) {
                $url = remove_query_arg("column_name");
                $url = str_replace("page=bdds-remove-column-slug", "page=bdds-add-table-columns-slug", $url);

                BDDS_Admin::redirectTo($url);
            }

            // drop column for edit page
            if( isset($_POST["bdds-drop-column"]) ) {
                global $wpdb;

                $column_name     = isset($_GET["column_name"]) ? sanitize_text_field(wp_unslash($_GET["column_name"])) : null;
                $table_name      = isset($_GET["table_name"]) ? sanitize_text_field(wp_unslash($_GET["table_name"])) : null;

                $wpdb->query( $wpdb->prepare("ALTER TABLE `$table_name` DROP COLUMN `$column_name`") ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

                $edit_page = remove_query_arg("column_name");
                $edit_page = str_replace("action=drop", "action=edit", $edit_page);

                BDDS_Admin::redirectTo($edit_page);
            }

            if( isset($_POST["bdds-no-drop-column"]) ) {
                $edit_page = remove_query_arg("column_name");
                $edit_page = str_replace("action=drop", "action=edit", $edit_page);

                BDDS_Admin::redirectTo($edit_page);
            }

            // redirection from add new column to edit page
            if( isset($_POST["bdds-back"]) ) {
                $edit_page = add_query_arg("action", "edit");

                BDDS_Admin::redirectTo($edit_page);
            }

            if( isset($_POST["bdds-trash-view-back"]) ) {
                $view_page = remove_query_arg("trash");

                BDDS_Admin::redirectTo($view_page);
            }

            // Renaming a table when action view is edit
            if ( isset( $_POST[ BDDS_PLUGIN_PREFIX . '-new-table-name' ] ) && !empty( $_POST[ BDDS_PLUGIN_PREFIX . '-new-table-name' ] ) ) {
                $old_table_name      = isset($_GET[ 'table_name' ]) ? sanitize_text_field( wp_unslash($_GET[ 'table_name' ]) ) : null;
                $new_table_name      = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_" . sanitize_text_field( wp_unslash($_POST[ BDDS_PLUGIN_PREFIX . '-new-table-name' ]) );
                $metadata_table_name = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_table_metadata";

                $wpdb->query( $wpdb->prepare( "RENAME TABLE `$old_table_name` TO `$new_table_name`;" ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

                $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                    $metadata_table_name,
                    array(
                        "table_name" => $new_table_name
                    ),
                    array(
                        "table_name" => $old_table_name
                    )
                );

                $new_url = add_query_arg( array( "table_name" => $new_table_name ) );
                BDDS_Admin::redirectTo(esc_url_raw($new_url));
            }

            // Change structure when action view is edit
            if ( isset( $_POST['column_name'] ) && !empty( $_POST['column_name'] ) && $_GET[ 'column_name' ] != $_POST[ 'column_name' ] || //phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
            isset( $_POST[ 'column_type' ] ) || isset( $_POST[ 'hidden_null' ] ) || isset( $_POST[ 'column_default' ] ) ||
            isset( $_POST[ BDDS_PLUGIN_PREFIX . '-new-lookup-table' ] ) || isset( $_POST[ 'new-enum-values' ] ))
            {
                // $_GET[ 'column_name' ]  => old column name
                // $_POST[ 'column_name' ] => new column name
                $nonce = isset($_GET["_wpnonce"]) ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;
                $table_name         = isset( $_GET['table_name'] ) && wp_verify_nonce($nonce, "bdds-change-structure-nonce") ? sanitize_text_field(wp_unslash($_GET['table_name'])) : "No table name";
                $column_name        = isset( $_GET[ 'column_name' ] ) ? sanitize_text_field(wp_unslash($_GET[ 'column_name' ])) : null;
                $new_column_name    = isset( $_POST[ 'column_name' ] ) ? sanitize_text_field(wp_unslash($_POST[ 'column_name' ])) : '';
                $new_column_type    = isset( $_POST[ 'column_type' ] ) ? sanitize_text_field(wp_unslash($_POST[ 'column_type' ])) : null;
                $new_null_value     = isset( $_POST[ 'hidden_null' ] ) ? sanitize_text_field(wp_unslash($_POST[ 'hidden_null' ])) : null;
                $column_new_default = isset( $_POST[ 'column_default' ] ) ? sanitize_text_field(wp_unslash($_POST[ 'column_default' ])) : null;
                $number_types       = array( "int", "bigint", "decimal" );
                $string_types       = array( "varchar", "mediumtext", "longtext", "longblob" );
                $other_types        = array( "datetime", "bit" );
                $type = ''; $nullable = ''; $default = '';
                $enum_values_string = '';
                $metadata_table_name  = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_table_metadata";
                $table_metadata   = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->prepare(
                        "SELECT `metadata` FROM `$metadata_table_name` WHERE table_name = %s", $table_name // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    ), ARRAY_A
                );
                $table_meta       = isset($table_metadata[0]["metadata"]) ? $table_metadata[0]["metadata"] : null;
                $metadata         = json_decode($table_meta, true);
                $lookup_tables    = isset($metadata["lookup_tables"]) ? $metadata["lookup_tables"] : null;

                $data_type = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                    $wpdb->prepare(
                        "SELECT DATA_TYPE,CHARACTER_MAXIMUM_LENGTH,COLUMN_DEFAULT,IS_NULLABLE FROM INFORMATION_SCHEMA.COLUMNS WHERE table_name = %s AND COLUMN_NAME = %s;",
                        array( $table_name, $column_name )
                    )
                );
                $column_type     = ( !empty( $new_column_type ) ) ? $new_column_type : $data_type[0]->DATA_TYPE;
                $column_length   = isset($data_type[0]->CHARACTER_MAXIMUM_LENGTH) ? $data_type[0]->CHARACTER_MAXIMUM_LENGTH : "255";

                if($column_type == "decimal") {
                    $column_length = "15,2";
                }

                if(isset($column_new_default)) {
                    $column_default = $column_new_default;

                } else if (empty($column_new_default) && !empty($data_type[0]->COLUMN_DEFAULT)) {
                    $column_default = $data_type[0]->COLUMN_DEFAULT;

                } else {
                    $column_default = '';
                    $default = "DEAFAULT ''";

                }

                $column_nullable = $new_null_value;

                if($column_type == "datetime") $column_new_default = str_replace('T', ' ', $column_new_default);
                $is_valid_default = BDDS_Admin::validate_data($column_type, $column_default, $column_name);

                if (!preg_match("/^[a-zA-Z0-9_]+$/", $column_name) || !preg_match("/^[a-zA-Z0-9_]+$/", $new_column_name)) {
                    // column name entered by user is wrong, show error
                    $this->bdds_set_admin_notice(
                        esc_html__('Invalid column name!', 'black-desk'), 'error'
                    );
                    $column_name_err = "Invalid column name!";

                } else if( $is_valid_default != "valid") {
                    $this->bdds_set_admin_notice(esc_html($is_valid_default), 'error');
                    $invalid_default_value = true;

                } else {
                    if ( $column_type == "varchar" || $column_type == "decimal" ) {
                        $column_length = ( $column_type == "varchar" ) ? "255" : "15,2";
                        $type = $column_type . '(' . $column_length . ')';
                    } else $type = $column_type;

                    if ( $column_nullable == "YES" ) {
                        $nullable = "NULL";
                    } else $nullable = "NOT NULL";

                    if ( str_contains($column_type, "bit") ) {
                        $column_default = ($column_default != "empty") ? (int)$column_default : "''";
                        $default = "DEFAULT $column_default";

                    } else if ( $column_default != '' ) {
                        $default = "DEFAULT '$column_default'";
                    } else $default = '';

                    if( in_array( $column_type, $number_types ) && !is_numeric( $column_default ) ) {
                        $default = '';
                    }

                    // varchar(100) means it is a Lookup type of field
                    if( $column_type == "varchar(100)" ) {
                        $new_lookup_field = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-lookup-field"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-lookup-field"])) : null;
                        $default = "DEFAULT \"$column_default\"";

                        $column_id = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                            $wpdb->prepare(
                                "SELECT ORDINAL_POSITION FROM INFORMATION_SCHEMA.COLUMNS WHERE table_name = %s AND COLUMN_NAME = %s;",
                                array($table_name, $column_name)
                            )
                        );

                        $lookup_table_field    = explode('.', array_column($lookup_tables, "column_$column_id")[0]);
                        $lookup_table_field[1] = $new_lookup_field;
                        $new_lookup            = implode('.', $lookup_table_field);

                        foreach($lookup_tables as $key=>$lookup_table) {
                            foreach($lookup_table as $lookup_key=>$lookup) {
                                // replace value for current column in metadata, lookup_tables array
                                if($lookup_key == "column_$column_id") $lookup_tables[$key] = array($lookup_key => $new_lookup);
                            }
                        }

                        $metadata["lookup_tables"] = $lookup_tables;
                        $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                            $metadata_table_name,
                            array(
                                "metadata" => wp_json_encode($metadata)
                            ),
                            array(
                                "table_name" => $table_name
                            )
                        );
                    }

                    if( $column_type == "ENUM" ) {
                        $new_enum_values = isset($_POST["new-enum-values"]) ? sanitize_text_field(wp_unslash($_POST["new-enum-values"])) : null;
                        $new_enum_values_arr = explode("\r\n", trim($new_enum_values));

                        foreach($new_enum_values_arr as $enum_value) {
                            if(str_contains($enum_value, "'")) {
                                $enum_value = stripcslashes($enum_value);
                                $enum_value = str_replace("'", "\'", $enum_value);
                            }
                            if(str_contains('"', $enum_value)) {
                                $enum_value = stripcslashes($enum_value);
                                $enum_value = str_replace('"', '\"', $enum_value);
                            }

                            $enum_values_string .= "'" . $enum_value . "',";
                        }

                        $enum_values_string = "ENUM(" . rtrim($enum_values_string, ',') . ')';
                        $type = $enum_values_string;
                    }

                    if($column_type == "mediumtext" || $column_type == "longtext" || $column_type == "longblob" || $column_default == "empty" || $column_type == "ENUM") {
                        $default = '';
                    }

                    if( isset( $_POST['column_name'] ) && !empty( $_POST['column_name'] ) && $_GET[ 'column_name' ] != $_POST[ 'column_name' ] ) {
                        $wpdb->query( $wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                            "ALTER TABLE `$table_name` CHANGE `$column_name` $new_column_name $type $nullable $default;", // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                            )
                        );
                    } else {
                        $wpdb->query( $wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                            "ALTER TABLE `$table_name` MODIFY `$column_name` $type $nullable $default;", // phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                            )
                        );
                    }

                    $wpdb->show_errors();
                    if (empty( $wpdb->last_error )) {
                        // query was successfully exectued, change action parametar to edit and open it
                        $back_url = remove_query_arg( "column_name" );
                        $back_url = str_replace( "action=change", "action=edit", $back_url );

                        BDDS_Admin::redirectTo(esc_url_raw($back_url));

                    } else {
                        // query failed, display DB error
                        if(str_contains($wpdb->last_error, "Duplicate column name ")) {
                            /* translators: %s: name of the new column */
                            $column_name_err = sprintf(esc_html__('Column name "%s" already exists', 'black-desk'), $new_column_name);
                            $this->bdds_set_admin_notice(esc_html($column_name_err), 'error');
                        } else {
                            $this->bdds_set_admin_notice(esc_html($wpdb->last_error), 'error');
                        }
                    }

                    $wpdb->hide_errors();
                }
            }

            // Add new column when action view is edit
            if(isset($_POST[BDDS_PLUGIN_PREFIX . "-save-new-column"]) && $_POST[BDDS_PLUGIN_PREFIX . "-save-new-column"] == __("Add new column", 'black-desk')) {
                global $wpdb;
                $nonce                  = isset($_GET["_wpnonce"]) ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;
                $table_name             = (isset($_GET["table_name"]) && wp_verify_nonce($nonce, "bdds-edit-add-column-nonce")) ? sanitize_text_field(wp_unslash($_GET["table_name"])) : '';
                $table_prefix           = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_";
                $metadata_table_name    = $table_prefix . "table_metadata";

                $column_name         = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-name"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-name"])) : null;
                $column_type         = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-type"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-type"])) : null;
                $column_null         = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-null"]) ? "NULL" : "NOT NULL";
                $column_default      = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-default"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-default"])) : null;
                $insert_after_column = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-insert-after"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-insert-after"])) : null;
                $default = '';

                if($column_type == "varchar(100)") {
                    $lookup_table     = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-lookup-table"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-lookup-table"])) : null;
                    $lookup_field     = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-lookup-field"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-lookup-field"])) : null;
                    $lookup           = $lookup_table . '.' . $lookup_field;
                    $table_metadata   = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->prepare("SELECT `metadata` FROM `$metadata_table_name` WHERE table_name = %s", $table_name), // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        ARRAY_A
                    );

                    $metadata         = json_decode($table_metadata[0]["metadata"], true);
                    $lookup_tables    = isset($metadata["lookup_tables"]) ? $metadata["lookup_tables"] : null;

                    // here we take the id of a column that goes before our new column
                    // column id is used as a key "column_ID" for metadata[lookup_tables]
                    $column_id = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->prepare(
                            "SELECT ORDINAL_POSITION FROM information_schema.columns WHERE table_name = %s AND column_name = %s",
                            array($table_name, $insert_after_column)
                        )
                    );

                    // iterate trough metadata[lookup_tables] to increment column id's of lookup tables
                    foreach($lookup_tables as $key=>$lookup_table) {
                        foreach($lookup_table as $lookup_key=>$old_lookup) {
                            $lookup_table_id = explode('_', $lookup_key)[1];

                            // if lookup table id is greater than current column id, unset that column and add new with incremented id
                            if($lookup_table_id > $column_id) {
                                unset($lookup_tables[array_search($lookup_table, $lookup_tables)]);
                                array_push($lookup_tables, array("column_" . ($lookup_table_id + 1) => $old_lookup));
                            }
                        }
                    }

                    // increment column id so it matches our column when it's inserted in database table
                    $column_id = $column_id + 1;

                    if(isset($lookup_tables)) {
                        array_push($lookup_tables, array("column_" . $column_id => $lookup));
                    }

                    $metadata["lookup_tables"] = $lookup_tables;
                    //update existing JSON metadata in metadata table
                    $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        $metadata_table_name,
                        array(
                            "metadata" => wp_json_encode($metadata)
                        ),
                        array(
                            "table_name" => $table_name
                        )
                    );

                    $default = "DEFAULT \"$column_default\"";

                } else if ($column_type == "bit") {
                    $default = ($column_default == "Yes") ? "DEFAULT 1" : "DEFAULT 0";

                } else if ($column_type == "datetime" && !empty($default)) {
                    $default = "DEFAULT \"" . str_replace('T', ' ', $column_default) . "\"";

                } else if ($column_type == "enum") {
                    $post_enum_values = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-enum-values"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-enum-values"])) : null;
                    $enum_values_array = explode("\r\n", trim($post_enum_values));
                    $enum_values_string = '';

                    foreach($enum_values_array as $enum_value) {
                        if(str_contains($enum_value, "'")) {
                            $enum_value = stripcslashes($enum_value);
                            $enum_value = str_replace("'", "\'", $enum_value);
                        }
                        if(str_contains('"', $enum_value)) {
                            $enum_value = stripcslashes($enum_value);
                            $enum_value = str_replace('"', '\"', $enum_value);
                        }

                        $enum_values_string .= "'" . $enum_value . "',";
                    }

                    $enum_values_string = "ENUM(" . rtrim($enum_values_string, ',') . ')';
                    $column_type        = $enum_values_string;
                    $default            = '';

                }

                // $add_column_query = "ALTER TABLE %i ADD COLUMN %i $column_type $column_null $default AFTER %i;";
                $is_valid_default = BDDS_Admin::validate_data($column_type, $column_default, $column_name);
                $new_column_name = isset($_POST[BDDS_PLUGIN_PREFIX . "-new-column-name"]) ? sanitize_text_field(wp_unslash($_POST[BDDS_PLUGIN_PREFIX . "-new-column-name"])) : null;

                $wpdb->show_errors();
                // Data validation
                if(isset($new_column_name) && !preg_match( "/^[a-zA-Z0-9_]+$/", $new_column_name)) {
                    // validate column name
                    $column_name_err = "Invalid column name!";

                } else if (!empty( $wpdb->last_error )) {
                    // unexpected error returned by wpdb
                    $wpdb_err = $wpdb->last_error;

                } else if ($is_valid_default != "valid") {
                    // validate column default value
                    $default_value_error = $is_valid_default;

                } else {
                    // all good execute query
                    $wpdb->query( $wpdb->prepare("ALTER TABLE `$table_name` ADD COLUMN `$column_name` $column_type $column_null $default AFTER `$insert_after_column`;") ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    $_SESSION[BDDS_PLUGIN_PREFIX . "_column_added_successfully"] = true;

                    $edit_page = add_query_arg("action", "edit");
                    // if error occurred after query execution print error, else redirect user to edit page
                    if(!empty( $wpdb->last_error )) {
                        $this->bdds_set_admin_notice(esc_html($wpdb->last_error), 'error');
                    } else {
                        BDDS_Admin::redirectTo($edit_page);
                    }
                }
                $wpdb->hide_errors();
            }

            // Redirections for queries
            BDDS_Query_Router::query_router();

            // delete record from table when action is view
            if( isset($_POST["bdds-drop-row"]) && check_admin_referer("bdds-drop-row-nonce") ) {
                global $wpdb;
                $table_name = isset($_GET["table_name"]) ? sanitize_text_field(wp_unslash($_GET["table_name"])) : null;
                $record_id_to_delete = isset($_GET["record_id"]) ? sanitize_text_field(wp_unslash($_GET["record_id"])) : null;

                $wpdb->query( $wpdb->prepare(  // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        "UPDATE `$table_name` SET deleted_at = CURRENT_TIMESTAMP WHERE id=%d;", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        $record_id_to_delete
                    )
                );

                $view_page = remove_query_arg("record_id");
                $view_page = str_replace("action=drop_row", "action=view", $view_page);
                $view_page = add_query_arg("_wpnonce", wp_create_nonce("bdds-yes-drop-row-nonce"), $view_page);

                BDDS_Admin::redirectTo($view_page);
            }

            if( isset($_POST["bdds-no-drop-row"]) && check_admin_referer("bdds-drop-row-nonce")) {
                global $wpdb;
                $view_page = remove_query_arg("record_id");
                $view_page = str_replace("action=drop_row", "action=view", $view_page);
                $view_page = add_query_arg("_wpnonce", wp_create_nonce("bdds-no-drop-row-nonce"), $view_page);

                BDDS_Admin::redirectTo($view_page);
            }

            // delete query
            if( isset($_POST["yes-delete-query"]) ) {
                global $wpdb;

                $queries_table_name = $wpdb->prefix . "bdds_queries";
                $query_id_to_delete = isset($_POST["query-id-delete"]) ? sanitize_text_field(wp_unslash($_POST["query-id-delete"])) : null;

                $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                    $queries_table_name,
                    array(
                        'id' => $query_id_to_delete
                    )
                );

                $query_tables_page = remove_query_arg("query_id");
                $query_tables_page = str_replace("page=bdds-query-delete-query-slug", "page=bdds-all-queries-slug", $query_tables_page);

                BDDS_Admin::redirectTo($query_tables_page);
            }

            if( isset($_POST["no-delete-query"]) ) {
                $query_tables_page = remove_query_arg("query_id");
                $query_tables_page = str_replace("page=bdds-query-delete-query-slug", "page=bdds-all-queries-slug", $query_tables_page);

                BDDS_Admin::redirectTo($query_tables_page);
            }

        }

        /**
         * Returns data type, regex pattern and error message.
         *
         *
         * @since 1.0.0
         *
         * @param string $field_type Value that should be validated.
         * @return array $options_info Array which includes data type, regex pattern and error message for passed field type.
         */
        public static function getFieldType($field_type) {
            $pattern = '';
            $title   = '';
            $string_regex = "^[a-zA-Z0-9_]+$";
            $number_regex = "^([0-9])+$";
            $decimal_reg  = "^-?\d{0,15}(,\d+)*(\.\d{0,2}(e\d+)?)?$";
            $lookup_reg   = "^[a-zA-Z0-9]+\.[a-zA-Z0-9]+$";

            switch( $field_type ) {
                case str_contains($field_type, "varchar(255)"):
                    $option_type = BDDS_Type_Constants::SHORT_TEXT;
                    $pattern = $string_regex;
                    $title = esc_attr__('Default value can only have uppercase and lowercase Latin letters, underscore and numbers', 'black-desk');
                    break;
                case str_starts_with($field_type, "int"):
                    $option_type = BDDS_Type_Constants::NUMBER;
                    $pattern = $number_regex;
                    $title = esc_attr__('Default value must be a number', 'black-desk');
                    break;
                case str_contains($field_type, "bigint"):
                    $option_type = BDDS_Type_Constants::BIG_NUMBER;
                    $pattern = $number_regex;
                    $title = esc_attr__('Default value must be a number', 'black-desk');
                    break;
                case str_contains($field_type, "decimal"):
                    $option_type = BDDS_Type_Constants::CURRENCY;
                    $pattern = $decimal_reg;
                    $title = esc_attr__('Default value must be a decimal number with max precision {15,2}', 'black-desk');
                    break;
                case str_contains($field_type, "datetime"):
                    $option_type = BDDS_Type_Constants::DATE_TIME;
                    break;
                case str_contains($field_type, "bit"):
                    $option_type = BDDS_Type_Constants::YES_NO;
                    break;
                case str_contains($field_type, "varchar(100)"):
                    $option_type = BDDS_Type_Constants::LOOKUP;
                    $pattern = $lookup_reg;
                    $title = esc_attr__('Lookup table default value should be formatted as "table_name.field_name"', 'black-desk');
                    break;
                case str_contains($field_type, "mediumtext"):
                    $option_type = BDDS_Type_Constants::RICH_TEXT;
                    break;
                case str_contains($field_type, "longtext"):
                    $option_type = BDDS_Type_Constants::LONG_TEXT;
                    break;
                case str_contains($field_type, "longblob"):
                    $option_type = BDDS_Type_Constants::ATTACHMENT;
                    break;
                case str_contains($field_type, "varchar(1024)"):
                    $option_type = BDDS_Type_Constants::HYPERLINK;
                    break;
                case str_starts_with($field_type, "enum"):
                    $option_type = BDDS_Type_Constants::ENUM;
                    break;
                default:
                    echo "Unexpected value was passed!";
                    break;
            }

            $options_info = array();
            array_push($options_info, $option_type, $pattern, $title);

            return $options_info;
        }

        /* Methods used for redirections and form handling*/

        // method handles a form from: admin/views/action_views/delete.php
        public function bdds_handle_drop_table_form() {
            global $wpdb;
            $nonce = isset($_GET["_wpnonce"]) ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;
            $table_name = (isset( $_GET[ 'table_name' ] ) && wp_verify_nonce($nonce, "bdds-drop-table-nonce")) ? sanitize_text_field(wp_unslash($_GET[ 'table_name' ])) : null;
            $metadata_table_name = $wpdb->prefix . BDDS_PLUGIN_PREFIX . "_table_metadata";

            $back_url = remove_query_arg( array( "action", "table_name" ), false );

            if( isset( $_POST[ 'yes-delete' ] ) && check_admin_referer("bdds-drop-table") ) {
                // $delete_sql = "DROP TABLE $table_name;";
                // delete table from database
                $wpdb->query( $wpdb->prepare( "DROP TABLE `$table_name`" ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

                // we also need to delete it from metadata table
                $wpdb->delete( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                    $metadata_table_name,
                    array("table_name" => $table_name),
                );

                BDDS_Admin::redirectTo(esc_url_raw($back_url));
            }
        }

        // method handles a form from truncate table: admin/views/action_views/turncate.php
        public function bdds_handle_truncate_table_form() {
            global $wpdb;
            $nonce = isset($_GET["_wpnonce"]) ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;
            $table_name = isset( $_GET[ 'table_name' ] ) &&
                (wp_verify_nonce($nonce, "bdds-truncate-table-nonce") || wp_verify_nonce($nonce, "view-truncate-table-nonce")) ? sanitize_text_field(wp_unslash($_GET[ 'table_name' ])) : null;

            $from_page = isset( $_GET[ 'from_page' ] ) ? sanitize_text_field(wp_unslash($_GET[ 'from_page' ])) : null;
            $back_url = '';

            switch ( $from_page ) {
                case BDDS_PLUGIN_PREFIX . "_show_tables":
                    $back_url = remove_query_arg( array( "action", "table_name", "_wpnonce", "from_page" ), false );

                    break;
                case BDDS_PLUGIN_PREFIX . "_table_view":
                    $back_url = remove_query_arg( "from_page", false );
                    $back_url = str_replace("action=turncate", "action=view", $back_url);
                    break;
            }

            if ( isset( $_POST[ 'yes-input' ] ) && check_admin_referer("bdds-truncate-table")) {
                // delete all records from database table
                $wpdb->query( $wpdb->prepare( "TRUNCATE TABLE `$table_name`;" ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared

                BDDS_Admin::redirectTo(esc_url_raw($back_url));

            } else if ( isset( $_POST[ 'no-input' ] ) && check_admin_referer("bdds-truncate-table")) {
                BDDS_Admin::redirectTo(esc_url_raw($back_url));
            }
        }

        // Method handles a functionalities when action view is view
        public function bdds_handle_table_view_form() {
            global $wpdb;
            $nonce = isset($_GET["_wpnonce"]) ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;
            $table_name  = (isset( $_GET[ 'table_name' ] ) && (wp_verify_nonce($nonce, "bdds-table-view-nonce")) || wp_verify_nonce($nonce, "bdds-yes-drop-row-nonce")
            || wp_verify_nonce($nonce, "view-truncate-table-nonce") || wp_verify_nonce($nonce, "bdds-no-drop-row-nonce")) ? sanitize_text_field(wp_unslash($_GET[ 'table_name' ])) : null;

            // bulk action delete
            if ( ( isset( $_POST[ 'doaction' ] ) && $_POST[ 'doaction' ] == "delete_action_1" ) ||
            ( isset( $_POST[ 'action2' ] ) && $_POST[ 'action2' ] == "delete_action_2" ) ) {

                $post_arr = $_POST;
                $row_id = '';

                // records are not deleted, we only set "deleted_at" column to CURRENT_TIMESTAMP
                // we do not display records that have date in "deleted_at" column
                foreach( $post_arr as $post_key=>$post_value ) {
                    if ( str_contains( $post_key, 'column_id' ) ) {
                        $row_id         = $post_value;

                        $wpdb->query( $wpdb->prepare( "UPDATE `$table_name` SET deleted_at = CURRENT_TIMESTAMP WHERE id = %d;", $row_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    }
                }

                $url = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
                BDDS_Admin::redirectTo($url);
            }

            // user submited new record, "Add record" button
            if ( !empty( $_POST[ 'add-new-column' ] ) && (check_admin_referer("bdds-table-view") || wp_verify_nonce($nonce, "bdds-table-view-nonce")) ) {
                global $wpdb;
                include "views/action_views/view_table/add_new_column.php";
            }

            // inline record edit, new value is submitted by pressing Enter
            if ( isset( $_POST[ 'edit_input' ] )  && isset( $_POST[ 'submit_yes' ] ) && isset( $_POST[ 'data_info' ] ) ) {
                $break = 0;
                if( !empty( $_POST[ 'edit_input' ] ) ) {
                    $new_value   = sanitize_text_field(wp_unslash($_POST[ 'edit_input' ]));
                    $post_data_info = isset($_POST[ 'data_info' ]) ? sanitize_text_field(wp_unslash($_POST[ 'data_info' ])) : null;
                    $data_info   = explode( '-', $post_data_info);
                    $column_name = !empty($data_info[0]) ? $data_info[0] : null;
                    $column_id   = !empty($data_info[1]) ? $data_info[1] : null;

                    $column_type = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->prepare(
                            "SELECT DATA_TYPE FROM INFORMATION_SCHEMA.COLUMNS WHERE table_name = %s AND COLUMN_NAME = %s;",
                            array($table_name, $column_name)
                        )
                    );
                    $is_input_valid = BDDS_Admin::validate_data($column_type, $new_value, $column_name);

                    if( !BDDS_Admin::validateDate($new_value) && $column_type == "datetime" ) {
                        $this->bdds_set_admin_notice(sprintf(
                            esc_html__('Value for %s must be formatted as "YYYY-MM-DD hh:mm:ss"', 'black-desk'),
                            esc_html($column_name)
                        ), 'error');
                        $break = 1;
                    }

                    if($is_input_valid != "valid") {
                        $this->bdds_set_admin_notice($is_input_valid, 'error');

                    } else {
                        if($column_type != "bit") {
                            // update record in table
                            $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                                $table_name,
                                array(
                                    $column_name => $new_value,
                                ),
                                array(
                                    'id' => $column_id,
                                )
                            );

                        } else if ($column_type == "bit") {
                            $new_value = ($new_value == "Yes") ? 1 : 0;

                            $wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                                $wpdb->prepare(
                                    "UPDATE `$table_name` SET $column_name = $new_value WHERE id=%d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                                    $column_id
                                )
                            );
                        }

                        // set current timestamp for updated_at column
                        $wpdb->query( $wpdb->prepare( "UPDATE `$table_name` SET updated_at = CURRENT_TIMESTAMP WHERE id = %d;", $column_id ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared

                        $url = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
                        BDDS_Admin::redirectTo($url);
                    }

                } else {
                    $this->bdds_set_admin_notice(
                        __('Please provide a new value to update!', 'black-desk'),
                        'warning'
                    );

                }
            }
        }

        // Method handles a form for restoring deleted data: admin/views/action_views/view_table/trash.php
        public function bdds_handle_view_restore_trash_form() {
            global $wpdb;
            $nonce = isset($_GET["_wpnonce"]) ? sanitize_text_field(wp_unslash($_GET["_wpnonce"])) : null;
            $table_name  = (
                isset( $_GET[ 'table_name' ] ) && (wp_verify_nonce($nonce, "bdds-table-view-nonce"))
                || wp_verify_nonce($nonce, "bdds-yes-drop-row-nonce") || wp_verify_nonce($nonce, "view-truncate-table-nonce")
                || wp_verify_nonce($nonce, "bdds-no-drop-row-nonce")
            ) ? sanitize_text_field(wp_unslash($_GET[ 'table_name' ])) : null;

            if( isset( $_POST['doaction_trash'] )  && $_POST['doaction_trash'] == "restore_action_trash"
                && check_admin_referer("bdds-doaction-trash-nonce")
            ){
                $post_values = $_POST;

                // to restore a record from "Trash" we set deleted_at to NULL
                // we only need id of a row that should be restored, and we pass it as ID in query WHERE clause
                foreach( $post_values as $key=>$post_value ) {
                    if( str_contains( $key, "restore_deleted_column_" ) ) {
                        $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                            $table_name,
                            array(
                                "deleted_at" => NULL
                            ),
                            array(
                                "id" => $post_value
                            )
                        );
                    }
                }

                $back_to_table_view = remove_query_arg( "trash" );

                BDDS_Admin::redirectTo(esc_url_raw($back_to_table_view));
            }
        }


        /* Patch: using self::instance() that checks if admin_notices action was already
            taken works outside the context of this class, but using it in this
            class causes charge_structure.php content to double on notice display
            (edit column); tried but cannot go into further research for the moment,
            so this is the working patch - use bdds_set_and_display_admin_notice
            outside the context of this class.
            * For multiple messages, use bdds_set_admin_notice multiple times and
            bdds_display_admin_notice in the end.
        */
        /**
         * Sets and displays an admin notice.
         *
         * For use outside the context of BDDS_Admin class.
         *
         * @since 1.1.1
         *
         * @param string $message The notice message
         * @param string $type The notice type (error, warning, success, info)
         */
        public function bdds_set_and_display_admin_notice($message, $type = 'error') {
            $this->bdds_set_admin_notice($message, $type);
            $this->bdds_display_admin_notice();
        }


        /**
         * Sets an admin notice to be displayed.
         *
         * @since 1.1.1
         *
         * @param string $message The notice message
         * @param string $type The notice type (error, warning, success, info)
         */
        public function bdds_set_admin_notice($message, $type = 'error') {
            $this->admin_notices[] = array('message' => $message, 'type' => $type);

            add_action('admin_notices', array($this, 'bdds_display_admin_notice'));
        }


        /**
         * Displays the admin notice.
         *
         * @since 1.1.1
         */
        public function bdds_display_admin_notice() {
            foreach ($this->admin_notices as $notice) {
                printf(
                    '<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
                    esc_attr($notice['type']),
                    wp_kses_post($notice['message'])
                );
            }
            // Clear notices after displaying
            $this->admin_notices = array();
        }

    }
} else {
    echo "WP admin class already exists!";
}