<?php
/**
 * Status class - handles custom order status
 *
 * @package Bizzorder
 */

defined( 'ABSPATH' ) || exit;

/**
 * BIZZORDER_Status class
 */
class BIZZORDER_Status {

    /**
     * Status slug
     */
    const STATUS_SLUG = 'wc-on-bizzorder';

    /**
     * Constructor
     */
    public function __construct() {
        // Register custom status
        add_action( 'init', array( $this, 'register_status' ) );

        // Add to WooCommerce status list
        add_filter( 'wc_order_statuses', array( $this, 'add_to_status_list' ) );

        // Add status to bulk actions
        add_filter( 'bulk_actions-edit-shop_order', array( $this, 'add_bulk_action' ) );
        add_filter( 'bulk_actions-woocommerce_page_wc-orders', array( $this, 'add_bulk_action' ) );

        // Schedule cron for auto-completion
        add_action( 'init', array( $this, 'schedule_cron' ) );
        add_action( 'bizzorder_check_preorders', array( $this, 'check_preorders' ) );
    }

    /**
     * Register custom order status
     */
    public function register_status() {
        $label = bizzorder_get_option( 'status_label', __( 'Pre-Ordered', 'bizzorder' ) );

        register_post_status( self::STATUS_SLUG, array(
            'label'                     => $label,
            'public'                    => true,
            'exclude_from_search'       => false,
            'show_in_admin_all_list'    => true,
            'show_in_admin_status_list' => true
        ) );
    }

    /**
     * Add custom status to WooCommerce status list
     *
     * @param array $statuses Order statuses.
     * @return array
     */
    public function add_to_status_list( $statuses ) {
        $label = bizzorder_get_option( 'status_label', __( 'Pre-Ordered', 'bizzorder' ) );

        // Add after on-hold
        $new_statuses = array();
        foreach ( $statuses as $key => $value ) {
            $new_statuses[ $key ] = $value;
            if ( 'wc-on-hold' === $key ) {
                $new_statuses[ self::STATUS_SLUG ] = $label;
            }
        }

        // If on-hold wasn't found, add at the end
        if ( ! isset( $new_statuses[ self::STATUS_SLUG ] ) ) {
            $new_statuses[ self::STATUS_SLUG ] = $label;
        }

        return $new_statuses;
    }

    /**
     * Add bulk action for pre-ordered status
     *
     * @param array $actions Bulk actions.
     * @return array
     */
    public function add_bulk_action( $actions ) {
        $label = bizzorder_get_option( 'status_label', __( 'Pre-Ordered', 'bizzorder' ) );

        $action_key = str_replace( 'wc-', 'mark_', self::STATUS_SLUG );

        $actions[ $action_key ] = sprintf(
            /* translators: %s: status name */
            __( 'Change status to %s', 'bizzorder' ),
            $label
        );
        return $actions;
    }


    /**
     * Schedule cron job for checking pre-orders
     */
    public function schedule_cron() {
        if ( ! wp_next_scheduled( 'bizzorder_check_preorders' ) ) {
            wp_schedule_event( time(), 'twicedaily', 'bizzorder_check_preorders' );
        }
    }

    /**
     * Check pre-orders and update status if date has passed
     */
    public function check_preorders() {
        $args = array(
            'status' => self::STATUS_SLUG,
            'limit'  => -1,
        );

        $orders = wc_get_orders( $args );

        foreach ( $orders as $order ) {
            $preorder_date = $order->get_meta( '_bizzorder_preorder_date' );
            
            if ( empty( $preorder_date ) ) {
                continue;
            }

            $preorder_time = strtotime( $preorder_date );
            
            if ( $preorder_time < time() ) {
                // Date has passed, update to processing/completed
                $new_status = apply_filters( 'bizzorder_preorder_complete_status', 'wc-processing' );
                $order->update_status( $new_status, __( '[Bizz Pre-Order] Pre-order date has passed. ', 'bizzorder' ) );
            }
        }
    }

    /**
     * Deactivation - clear cron
     */
    public static function deactivate() {
        wp_clear_scheduled_hook( 'bizzorder_check_preorders' );
    }
}

// Clear cron on deactivation
register_deactivation_hook( BIZZORDER_PLUGIN_DIR . '/bizzorder.php', array( 'BIZZORDER_Status', 'deactivate' ) );
