<?php
/**
 * Shop class - handles shop and product page display
 *
 * @package Bizzorder
 */

defined( 'ABSPATH' ) || exit;

/**
 * BIZZORDER_Shop class
 */
class BIZZORDER_Shop {

    /**
     * Constructor
     */
    public function __construct() {
        // Change button text
        add_filter( 'woocommerce_product_add_to_cart_text', array( $this, 'change_button_text' ), 10, 2 );
        add_filter( 'woocommerce_product_single_add_to_cart_text', array( $this, 'change_button_text' ), 10, 2 );

        // Variable product button text data
        add_filter( 'woocommerce_available_variation', array( $this, 'variation_data' ), 10, 3 );

        // Display available date
        add_action( 'woocommerce_before_add_to_cart_form', array( $this, 'display_available_date' ) );
        add_action( 'woocommerce_after_shop_loop_item_title', array( $this, 'display_available_date_loop' ), 15 );

        // Display pre-order badge
        add_action( 'woocommerce_before_shop_loop_item_title', array( $this, 'display_badge_loop' ), 10 );
        add_action( 'woocommerce_before_single_product_summary', array( $this, 'display_badge_single' ), 10 );

        // Change stock availability text
        add_filter( 'woocommerce_get_availability_text', array( $this, 'stock_availability_text' ), 10, 2 );
    }

    /**
     * Change add to cart button text for pre-order products
     *
     * @param string     $text    Button text.
     * @param WC_Product $product Product object.
     * @return string
     */
    public function change_button_text( $text, $product = null ) {
        if ( ! $product ) {
            global $product;
        }

        if ( $product && bizzorder_is_preorder( $product ) ) {
            return bizzorder_get_option( 'button_text', __( 'Pre Order Now!', 'bizzorder' ) );
        }

        return $text;
    }

    /**
     * Add pre-order flag to variation data
     *
     * @param array                $data      Variation data.
     * @param WC_Product           $product   Parent product.
     * @param WC_Product_Variation $variation Variation product.
     * @return array
     */
    public function variation_data( $data, $product, $variation ) {
        $data['is_bizzorder_preorder'] = bizzorder_is_preorder( $variation );
        return $data;
    }

    /**
     * Display available date on single product page
     */
    public function display_available_date() {
        if ( bizzorder_get_option( 'show_date', '1' ) !== '1' ) {
            return;
        }

        global $product;

        if ( ! $product || ! bizzorder_is_preorder( $product ) ) {
            return;
        }

        $date = bizzorder_get_preorder_date( $product );
        if ( ! $date ) {
            return;
        }

        $formatted_date = date_i18n( get_option( 'date_format' ), strtotime( $date ) );
        $text_template  = bizzorder_get_option( 'available_date_text', __( 'Available on {date_format}', 'bizzorder' ) );
        $text           = str_replace( '{date_format}', $formatted_date, $text_template );

        echo '<div class="bizzorder-available-date">' . esc_html( $text ) . '</div>';
    }

    /**
     * Display available date on shop loop
     */
    public function display_available_date_loop() {
        if ( bizzorder_get_option( 'show_date', '1' ) !== '1' ) {
            return;
        }

        global $product;

        if ( ! $product || ! bizzorder_is_preorder( $product ) ) {
            return;
        }

        $date = bizzorder_get_preorder_date( $product );
        if ( ! $date ) {
            return;
        }

        $formatted_date = date_i18n( get_option( 'date_format' ), strtotime( $date ) );
        $text_template  = bizzorder_get_option( 'available_date_text', __( 'Available on {date_format}', 'bizzorder' ) );
        $text           = str_replace( '{date_format}', $formatted_date, $text_template );

        echo '<div class="bizzorder-available-date-loop">' . esc_html( $text ) . '</div>';
    }

    /**
     * Display pre-order badge on shop loop
     */
    public function display_badge_loop() {
        if ( bizzorder_get_option( 'show_badge', '1' ) !== '1' ) {
            return;
        }

        global $product;

        if ( ! $this->product_has_preorder( $product ) ) {
            return;
        }

        echo '<span class="bizzorder-badge">' . esc_html__( 'Pre-Order', 'bizzorder' ) . '</span>';
    }

    /**
     * Display pre-order badge on single product page
     */
    public function display_badge_single() {
        if ( bizzorder_get_option( 'show_badge', '1' ) !== '1' ) {
            return;
        }

        global $product;

        if ( ! $this->product_has_preorder( $product ) ) {
            return;
        }

        echo '<span class="bizzorder-badge bizzorder-badge-single">' . esc_html__( 'Pre-Order', 'bizzorder' ) . '</span>';
    }

    /**
     * Check if product or any of its variations is pre-order
     *
     * @param WC_Product $product Product object.
     * @return bool
     */
    private function product_has_preorder( $product ) {
        if ( ! $product ) {
            return false;
        }

        // Check simple product
        if ( bizzorder_is_preorder( $product ) ) {
            return true;
        }

        // Check variable product variations
        if ( $product->is_type( 'variable' ) ) {
            $variations = $product->get_available_variations();
            foreach ( $variations as $variation ) {
                if ( bizzorder_is_preorder( $variation['variation_id'] ) ) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Change stock availability text for pre-order products
     *
     * @param string     $availability Availability text.
     * @param WC_Product $product      Product object.
     * @return string
     */
    public function stock_availability_text( $availability, $product ) {
        if ( ! $product instanceof WC_Product ) {
            return $availability;
        }

        if ( bizzorder_is_preorder( $product ) && $product->get_stock_quantity() > 0 ) {
            return sprintf(
                /* translators: %d is the stock quantity */
                _n(
                    '%d pre-order available',
                    '%d pre-orders available',
                    $product->get_stock_quantity(),
                    'bizzorder'
                ),
                $product->get_stock_quantity()
            );
        }

        return $availability;
    }
}
