<?php
/**
 * Product class - handles product edit meta fields
 *
 * @package Bizzorder
 */

defined( 'ABSPATH' ) || exit;

/**
 * BIZZORDER_Product class
 */
class BIZZORDER_Product {

    /**
     * Constructor
     */
    public function __construct() {
        // Simple product fields in Inventory tab
        add_action( 'woocommerce_product_options_stock_status', array( $this, 'add_simple_fields' ) );
        add_action( 'woocommerce_process_product_meta', array( $this, 'save_simple_fields' ) );

        // Variable product fields
        add_action( 'woocommerce_product_after_variable_attributes', array( $this, 'add_variation_fields' ), 10, 3 );
        add_action( 'woocommerce_save_product_variation', array( $this, 'save_variation_fields' ), 10, 2 );
    }

    /**
     * Add pre-order fields to simple products
     */
    public function add_simple_fields() {
        global $post;

        echo '<div class="options_group bizzorder-preorder-fields hide_if_variable">';

        woocommerce_wp_checkbox( array(
            'id'          => '_bizzorder_is_preorder',
            'label'       => __( 'Pre-Order Product', 'bizzorder' ),
            'description' => __( 'Enable this to mark the product as a pre-order item.', 'bizzorder' ),
            'value'       => get_post_meta( $post->ID, '_bizzorder_is_preorder', true ),
        ) );

        woocommerce_wp_text_input( array(
            'id'          => '_bizzorder_preorder_date',
            'label'       => __( 'Available Date', 'bizzorder' ),
            'type'        => 'date',
            'placeholder' => 'YYYY-MM-DD',
            'desc_tip'    => true,
            'description' => __( 'The date when the product will be available for shipping.', 'bizzorder' ),
            'value'       => get_post_meta( $post->ID, '_bizzorder_preorder_date', true ),
        ) );

        echo '</div>';

        // Notice for variable products
        echo '<div class="options_group bizzorder-preorder-fields hide_if_simple">';
        echo '<p class="form-field">';
        echo '<span class="description">' . esc_html__( 'Pre-Order options are available under each variation.', 'bizzorder' ) . '</span>';
        echo '</p>';
        echo '</div>';
    }

    /**
     * Save simple product pre-order fields
     *
     * @param int $post_id Product ID.
     */
    public function save_simple_fields( $post_id ) {
        $product = wc_get_product( $post_id );

        if ( ! $product || $product->is_type( 'variable' ) ) {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $is_preorder = isset( $_POST['_bizzorder_is_preorder'] ) ? 'yes' : 'no';
        $product->update_meta_data( '_bizzorder_is_preorder', $is_preorder );

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( 'yes' === $is_preorder && ! empty( $_POST['_bizzorder_preorder_date'] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $preorder_date = sanitize_text_field( wp_unslash( $_POST['_bizzorder_preorder_date'] ) );
            $product->update_meta_data( '_bizzorder_preorder_date', $preorder_date );
        } else {
            $product->delete_meta_data( '_bizzorder_preorder_date' );
        }

        $product->save();
    }

    /**
     * Add pre-order fields to product variations
     *
     * @param int     $loop           Variation loop index.
     * @param array   $variation_data Variation data.
     * @param WP_Post $variation      Variation post object.
     */
    public function add_variation_fields( $loop, $variation_data, $variation ) {
        echo '<div class="bizzorder-variation-fields options_group form-row form-row-full">';

        woocommerce_wp_checkbox( array(
            'id'          => '_bizzorder_is_preorder_' . $variation->ID,
            'label'       => __( 'Pre-Order Variation', 'bizzorder' ),
            'description' => __( 'Enable pre-order for this variation.', 'bizzorder' ),
            'value'       => get_post_meta( $variation->ID, '_bizzorder_is_preorder', true ),
        ) );

        woocommerce_wp_text_input( array(
            'id'          => '_bizzorder_preorder_date_' . $variation->ID,
            'label'       => __( 'Available Date', 'bizzorder' ),
            'type'        => 'date',
            'placeholder' => 'YYYY-MM-DD',
            'desc_tip'    => true,
            'description' => __( 'The date when this variation will be available.', 'bizzorder' ),
            'value'       => get_post_meta( $variation->ID, '_bizzorder_preorder_date', true ),
        ) );

        echo '</div>';
    }

    /**
     * Save variation pre-order fields
     *
     * @param int $variation_id Variation ID.
     * @param int $loop         Variation loop index.
     */
    public function save_variation_fields( $variation_id, $loop ) {
        $variation = wc_get_product( $variation_id );

        if ( ! $variation ) {
            return;
        }

        $checkbox_key = '_bizzorder_is_preorder_' . $variation_id;
        $date_key     = '_bizzorder_preorder_date_' . $variation_id;

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $is_preorder = isset( $_POST[ $checkbox_key ] ) ? 'yes' : 'no';
        $variation->update_meta_data( '_bizzorder_is_preorder', $is_preorder );

        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        if ( 'yes' === $is_preorder && ! empty( $_POST[ $date_key ] ) ) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            $preorder_date = sanitize_text_field( wp_unslash( $_POST[ $date_key ] ) );
            $variation->update_meta_data( '_bizzorder_preorder_date', $preorder_date );
        } else {
            $variation->delete_meta_data( '_bizzorder_preorder_date' );
        }

        $variation->save();
    }
}
