<?php
/**
 * Order class - handles order list customization
 *
 * @package Bizzorder
 */

defined( 'ABSPATH' ) || exit;

/**
 * BIZZORDER_Order class
 */
class BIZZORDER_Order {

    /**
     * Constructor
     */
    public function __construct() {
        // Add custom column to orders list
        add_filter( 'manage_edit-shop_order_columns', array( $this, 'add_order_column' ), 20 );
        add_filter( 'manage_woocommerce_page_wc-orders_columns', array( $this, 'add_order_column' ), 20 );

        // Populate custom column
        add_action( 'manage_shop_order_posts_custom_column', array( $this, 'order_column_content' ), 20, 2 );
        add_action( 'manage_woocommerce_page_wc-orders_custom_column', array( $this, 'order_column_content' ), 20, 2 );

        // Show pre-order date in order item meta
        add_action( 'woocommerce_order_item_meta_end', array( $this, 'show_preorder_date_in_order' ), 10, 3 );

        // Add pre-order date to order details (admin)
        add_action( 'woocommerce_admin_order_data_after_order_details', array( $this, 'admin_order_preorder_info' ) );
    }

    /**
     * Add pre-order date column to orders list
     *
     * @param array $columns Order columns.
     * @return array
     */
    public function add_order_column( $columns ) {
        $new_columns = array();

        foreach ( $columns as $key => $value ) {
            $new_columns[ $key ] = $value;
            if ( 'order_total' === $key ) {
                $new_columns['bizzorder_preorder_date'] = __( 'Pre-Order Date', 'bizzorder' );
            }
        }

        return $new_columns;
    }

    /**
     * Display content in pre-order date column
     *
     * @param string $column Column name.
     * @param mixed  $order  Order ID or object.
     */
    public function order_column_content( $column, $order ) {
        if ( 'bizzorder_preorder_date' !== $column ) {
            return;
        }

        // Get order object
        if ( ! $order instanceof WC_Order ) {
            if ( is_numeric( $order ) ) {
                $order = wc_get_order( $order );
            } elseif ( $order instanceof WP_Post ) {
                $order = wc_get_order( $order->ID );
            }
        }

        if ( ! $order ) {
            return;
        }

        $preorder_date = $order->get_meta( '_bizzorder_preorder_date' );

        if ( empty( $preorder_date ) ) {
            echo '—';
            return;
        }

        $preorder_time = strtotime( $preorder_date );
        $now           = time();
        $diff          = round( ( $preorder_time - $now ) / DAY_IN_SECONDS );

        if ( $diff > 0 ) {
            printf(
                /* translators: %d: number of days */
                esc_html__( 'Available in %d days', 'bizzorder' ),
                esc_html( $diff )
            );
        } elseif ( 0 === $diff ) {
            esc_html_e( 'Available today', 'bizzorder' );
        } else {
            esc_html_e( 'Already available', 'bizzorder' );
        }
    }

    /**
     * Show pre-order date in order item meta
     *
     * @param int      $item_id Item ID.
     * @param WC_Order_Item $item    Order item.
     * @param WC_Order $order   Order object.
     */
    public function show_preorder_date_in_order( $item_id, $item, $order ) {
        $preorder_date = $order->get_meta( '_bizzorder_preorder_date' );

        if ( empty( $preorder_date ) ) {
            return;
        }

        $formatted_date = date_i18n( get_option( 'date_format' ), strtotime( $preorder_date ) );
        
        echo '<br><small class="bizzorder-order-item-date">';
        printf(
            /* translators: %s: pre-order date */
            esc_html__( 'Pre-Order Date: %s', 'bizzorder' ),
            esc_html( $formatted_date )
        );
        echo '</small>';
    }

    /**
     * Show pre-order info in admin order details
     *
     * @param WC_Order $order Order object.
     */
    public function admin_order_preorder_info( $order ) {
        $preorder_date = $order->get_meta( '_bizzorder_preorder_date' );

        if ( empty( $preorder_date ) ) {
            return;
        }

        $formatted_date = date_i18n( get_option( 'date_format' ), strtotime( $preorder_date ) );
        ?>
        <div class="form-field form-field-wide bizzorder-admin-preorder-date">
            <div class="bizzorder-admin-date-wrapper">
                <label><?php esc_html_e( 'Pre-Order Available Date:', 'bizzorder' ); ?></label>
                <strong><?php echo esc_html( $formatted_date ); ?></strong>
            </div>
        </div>
        <?php
    }
}
