<?php
/**
 * Checkout class - handles checkout process and order meta
 *
 * @package Bizzorder
 */

defined( 'ABSPATH' ) || exit;

/**
 * BIZZORDER_Checkout class
 */
class BIZZORDER_Checkout {

    /**
     * Constructor
     */
    public function __construct() {
        // Save pre-order date to order meta
        add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'save_preorder_date' ), 10, 2 );
        add_action( 'woocommerce_store_api_checkout_update_order_meta', array( $this, 'store_api_save_preorder_date' ) );

        // Set order status to pre-ordered after payment
        add_filter( 'woocommerce_payment_complete_order_status', array( $this, 'set_preorder_status' ), 10, 3 );

        // Handle COD orders
        add_filter( 'woocommerce_cod_process_payment_order_status', array( $this, 'set_preorder_status_cod' ), 10, 2 );

        // Send emails on status change
        add_action( 'woocommerce_order_status_changed', array( $this, 'send_email_notifications' ), 10, 4 );
    }

    /**
     * Save pre-order date to order meta
     *
     * @param int   $order_id Order ID.
     * @param array $data     Posted data.
     */
    public function save_preorder_date( $order_id, $data ) {
        if ( ! BIZZORDER_Cart::cart_has_preorder() ) {
            return;
        }

        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        $oldest_date = BIZZORDER_Cart::get_oldest_preorder_date();
        if ( $oldest_date ) {
            $order->update_meta_data( '_bizzorder_preorder_date', sanitize_text_field( $oldest_date ) );
            $order->save();
        }
    }

    /**
     * Store API version of saving pre-order date to order meta
     *
     * @param array $data Checkout data.
     * @return array
     */
    public function store_api_save_preorder_date( $order ) {

        $oldest_date = BIZZORDER_Cart::get_oldest_preorder_date();
        if ( $oldest_date ) {
            $order->update_meta_data( '_bizzorder_preorder_date', sanitize_text_field( $oldest_date ) );
        }
    }

    /**
     * Set order status to pre-ordered after payment
     *
     * @param string   $status   Order status.
     * @param int      $order_id Order ID.
     * @param WC_Order $order    Order object.
     * @return string
     */
    public function set_preorder_status( $status, $order_id, $order ) {
        if ( ! $order instanceof WC_Order ) {
            return $status;
        }

        $preorder_date = $order->get_meta( '_bizzorder_preorder_date' );
        
        if ( ! empty( $preorder_date ) ) {
            return 'bizzorder-preordered';
        }

        return $status;
    }

    /**
     * Set order status to pre-ordered for COD orders
     *
     * @param string   $status Order status.
     * @param WC_Order $order  Order object.
     * @return string
     */
    public function set_preorder_status_cod( $status, $order ) {
        if ( ! $order instanceof WC_Order ) {
            return $status;
        }

        $preorder_date = $order->get_meta( '_bizzorder_preorder_date' );
        
        if ( ! empty( $preorder_date ) ) {
            return 'bizzorder-preordered';
        }

        return $status;
    }

    /**
     * Send email notifications when order status changes to pre-ordered
     *
     * @param int      $order_id   Order ID.
     * @param string   $old_status Old status.
     * @param string   $new_status New status.
     * @param WC_Order $order      Order object.
     */
    public function send_email_notifications( $order_id, $old_status, $new_status, $order ) {
        // Only send on transition to pre-ordered
        if ( 'bizzorder-preordered' !== $new_status ) {
            return;
        }

        // Only from pending, on-hold, or failed
        $valid_from = array( 'pending', 'on-hold', 'failed' );
        if ( ! in_array( $old_status, $valid_from, true ) ) {
            return;
        }

        // Check if already sent
        $email_sent = $order->get_meta( '_bizzorder_email_sent' );
        if ( $email_sent ) {
            return;
        }

        // Send customer email (processing email as fallback)
        $mailer = WC()->mailer();
        $emails = $mailer->get_emails();

        if ( isset( $emails['WC_Email_Customer_Processing_Order'] ) ) {
            $emails['WC_Email_Customer_Processing_Order']->trigger( $order_id );
        }

        // Send admin email
        if ( isset( $emails['WC_Email_New_Order'] ) ) {
            $emails['WC_Email_New_Order']->trigger( $order_id );
        }

        // Mark as sent
        $order->update_meta_data( '_bizzorder_email_sent', true );
        $order->save();
    }

    /**
     * Check if order has pre-order products
     *
     * @param WC_Order $order Order object.
     * @return bool
     */
    public static function order_has_preorder( $order ) {
        if ( ! $order instanceof WC_Order ) {
            return false;
        }

        foreach ( $order->get_items() as $item ) {
            $product_id = $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
            if ( bizzorder_is_preorder( $product_id ) ) {
                return true;
            }
        }

        return false;
    }
}
