<?php
/**
 * Cart class - handles cart validation and notices
 *
 * @package Bizzorder
 */

defined( 'ABSPATH' ) || exit;

/**
 * BIZZORDER_Cart class
 */
class BIZZORDER_Cart {

    /**
     * Constructor
     */
    public function __construct() {
        // Either mode - prevent mixing pre-order and regular products
        if ( bizzorder_get_option( 'either_mode', '1' ) === '1' ) {
            add_filter( 'woocommerce_add_to_cart_validation', array( $this, 'validate_cart_mix' ), 99, 2 );
        }

        // Add pre-order notice to cart items
        add_action( 'woocommerce_after_cart_item_name', array( $this, 'add_cart_item_notice' ), 10, 2 );

        // Mini cart notice
        add_filter( 'woocommerce_widget_cart_item_quantity', array( $this, 'mini_cart_notice' ), 10, 3 );
    }

    /**
     * Validate cart to prevent mixing pre-order and regular products
     *
     * @param bool $passed     Whether validation passed.
     * @param int  $product_id Product ID being added.
     * @return bool
     */
    public function validate_cart_mix( $passed, $product_id ) {
        // Get the product being added
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
        $variation_id = isset( $_POST['variation_id'] ) ? absint( $_POST['variation_id'] ) : 0;
        $check_id     = $variation_id ? $variation_id : $product_id;
        $is_preorder  = bizzorder_is_preorder( $check_id );

        // Check existing cart items
        $cart_has_preorder = false;
        $cart_items        = WC()->cart->get_cart_contents();

        if ( empty( $cart_items ) ) {
            return $passed;
        }

        foreach ( $cart_items as $cart_item ) {
            $item_id = $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
            if ( bizzorder_is_preorder( $item_id ) ) {
                $cart_has_preorder = true;
                break;
            }
        }

        // If cart is empty or same type, allow
        if ( empty( $cart_items ) ) {
            return $passed;
        }

        // Check for mix
        if ( ( $is_preorder && ! $cart_has_preorder ) || ( ! $is_preorder && $cart_has_preorder ) ) {
            if ( $is_preorder ) {
                wc_add_notice(
                    __( 'You cannot add pre-order products to a cart with regular products. Please complete your current order first or remove the items from your cart.', 'bizzorder' ),
                    'error'
                );
            } else {
                wc_add_notice(
                    __( 'You cannot add regular products to a cart with pre-order products. Please complete your current order first or remove the items from your cart.', 'bizzorder' ),
                    'error'
                );
            }
            return false;
        }

        return $passed;
    }

    /**
     * Add pre-order notice to cart items
     *
     * @param array  $cart_item     Cart item data.
     * @param string $cart_item_key Cart item key.
     */
    public function add_cart_item_notice( $cart_item, $cart_item_key ) {
        $product_id = $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
        
        if ( ! bizzorder_is_preorder( $product_id ) ) {
            return;
        }

        $preorder_date = bizzorder_get_preorder_date( $product_id );
        if ( ! $preorder_date ) {
            return;
        }

        $available_from = new DateTime( $preorder_date );
        $now            = new DateTime();

        if ( $available_from <= $now ) {
            return;
        }

        $diff = $now->diff( $available_from )->days;

        $notice_template = bizzorder_get_option( 
            'cart_notice', 
            __( 'This item will be available for shipping in {days_left} days', 'bizzorder' ) 
        );
        $notice = str_replace( '{days_left}', $diff, $notice_template );

        echo '<br><small class="bizzorder-cart-notice">' . esc_html( $notice ) . '</small>';
    }

    /**
     * Add pre-order notice to mini cart
     *
     * @param string $output    Quantity output.
     * @param array  $cart_item Cart item data.
     * @param string $cart_item_key Cart item key.
     * @return string
     */
    public function mini_cart_notice( $output, $cart_item, $cart_item_key ) {
        $product_id = $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
        
        if ( ! bizzorder_is_preorder( $product_id ) ) {
            return $output;
        }

        $preorder_date = bizzorder_get_preorder_date( $product_id );
        if ( ! $preorder_date ) {
            return $output;
        }

        $available_from = new DateTime( $preorder_date );
        $now            = new DateTime();

        if ( $available_from <= $now ) {
            return $output;
        }

        $diff = $now->diff( $available_from )->days;

        $notice_template = bizzorder_get_option( 
            'cart_notice', 
            __( 'This item will be available for shipping in {days_left} days', 'bizzorder' ) 
        );
        $notice = str_replace( '{days_left}', $diff, $notice_template );

        return $output . '<br><small class="bizzorder-mini-cart-notice">' . esc_html( $notice ) . '</small>';
    }

    /**
     * Check if cart has pre-order products
     *
     * @return bool
     */
    public static function cart_has_preorder() {
        if ( ! WC()->cart ) {
            return false;
        }

        $cart_items = WC()->cart->get_cart_contents();

        foreach ( $cart_items as $cart_item ) {
            $product_id = $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
            if ( bizzorder_is_preorder( $product_id ) ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get the oldest (furthest) pre-order date from cart
     *
     * @return string|false
     */
    public static function get_oldest_preorder_date() {
        if ( ! WC()->cart ) {
            return false;
        }

        $cart_items   = WC()->cart->get_cart_contents();
        $oldest_date  = false;
        $oldest_time  = 0;

        foreach ( $cart_items as $cart_item ) {
            $product_id = $cart_item['variation_id'] ? $cart_item['variation_id'] : $cart_item['product_id'];
            
            if ( ! bizzorder_is_preorder( $product_id ) ) {
                continue;
            }

            $preorder_date = bizzorder_get_preorder_date( $product_id );
            if ( ! $preorder_date ) {
                continue;
            }

            $preorder_time = strtotime( $preorder_date );
            if ( $preorder_time > $oldest_time ) {
                $oldest_time = $preorder_time;
                $oldest_date = $preorder_date;
            }
        }

        return $oldest_date;
    }
}
