<?php
/**
 * Plugin Name: Bizzorder - Pre Order for WooCommerce
 * Requires Plugins: woocommerce
 * Plugin URI: https://wordpress.org/plugins/bizzorder
 * Description: A simple and lightweight pre-order plugin for WooCommerce that allows customers to pre-order products before they are available.
 * Author: Saiful Islam
 * Author URI: https://profiles.wordpress.org/codersaiful/
 * Tags: woocommerce, pre-order, products, availability
 * 
 * 
 * Version: 1.0.2
 * Requires at least: 6.2
 * Tested up to: 6.9
 * Requires PHP: 8.0
 * 
 * WC requires at least: 5.0
 * WC tested up to: 10.4.2
 * 
 * Text Domain: bizzorder
 * Domain Path: /languages
 * prefix: bizzorder_
 * 
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;

// Define plugin constants
if ( ! defined( 'BIZZORDER_PLUGIN_DIR' ) ) {
    define( 'BIZZORDER_PLUGIN_DIR', __DIR__ );
}
if ( ! defined( 'BIZZORDER_PLUGIN_URL' ) ) {
    define( 'BIZZORDER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}
if ( ! defined( 'BIZZORDER_PLUGIN_BASE' ) ) {
    define( 'BIZZORDER_PLUGIN_BASE', plugin_basename( __FILE__ ) );
}
if ( ! defined( 'BIZZORDER_VERSION' ) ) {
    define( 'BIZZORDER_VERSION', '1.0.2.1' );
}

/**
 * Main plugin class using Singleton pattern
 */
final class BIZZORDER_Loader {

    /**
     * Plugin instance
     *
     * @var BIZZORDER_Loader|null
     */
    private static $instance = null;

    /**
     * Get plugin instance
     *
     * @return BIZZORDER_Loader
     */
    public static function init() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_plugin();
    }


    /**
     * Initialize the plugin
     */
    private function init_plugin() {
        // Check if WooCommerce is active
        if ( ! class_exists( 'WooCommerce' ) ) {
            add_action( 'admin_notices', array( $this, 'woocommerce_missing_notice' ) );
            return;
        }

        // Declare WooCommerce compatibility
        add_action( 'before_woocommerce_init', function() {
            if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
                \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
            }
        } );
        // Load dependencies
        $this->load_dependencies();

        // Initialize components
        $this->init_components();

        // Admin hooks
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );
    }

    /**
     * Load required files
     */
    private function load_dependencies() {
        require_once BIZZORDER_PLUGIN_DIR . '/includes/class-product.php';
        require_once BIZZORDER_PLUGIN_DIR . '/includes/class-shop.php';
        require_once BIZZORDER_PLUGIN_DIR . '/includes/class-cart.php';
        require_once BIZZORDER_PLUGIN_DIR . '/includes/class-checkout.php';
        require_once BIZZORDER_PLUGIN_DIR . '/includes/class-status.php';
        require_once BIZZORDER_PLUGIN_DIR . '/includes/class-order.php';

        // Load admin files
        if ( is_admin() ) {
            require_once BIZZORDER_PLUGIN_DIR . '/admin/framework-loader.php';
            require_once BIZZORDER_PLUGIN_DIR . '/admin/class-admin-loader.php';
        }
    }

    /**
     * Initialize plugin components
     */
    private function init_components() {
        new BIZZORDER_Product();
        new BIZZORDER_Shop();
        new BIZZORDER_Cart();
        new BIZZORDER_Checkout();
        new BIZZORDER_Status();
        new BIZZORDER_Order();

        // Initialize admin components
        if ( is_admin() ) {
            BizzOrder_Framework_Settings::instance();
            BizzOrder_Admin_Loader::instance();
        }
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function admin_enqueue_scripts( $hook ) {
        $screen = get_current_screen();
        // Only load on product edit, settings pages and orders page
        if ( $screen && ( $screen->id === 'product' || strpos( $hook, 'bizzorder-settings' ) !== false || strpos( $hook, 'wc-orders' ) !== false ) ) {
            wp_enqueue_style(
                'bizzorder-admin-css',
                BIZZORDER_PLUGIN_URL . 'assets/css/admin.css',
                array(),
                BIZZORDER_VERSION
            );

            wp_enqueue_script(
                'bizzorder-admin-js',
                BIZZORDER_PLUGIN_URL . 'assets/js/admin.js',
                array( 'jquery' ),
                BIZZORDER_VERSION,
                true
            );
        }
    }

    /**
     * Enqueue frontend scripts and styles
     */
    public function frontend_enqueue_scripts() {
        wp_enqueue_style(
            'bizzorder-frontend-css',
            BIZZORDER_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            BIZZORDER_VERSION
        );

        wp_enqueue_script(
            'bizzorder-frontend-js',
            BIZZORDER_PLUGIN_URL . 'assets/js/frontend.js',
            array( 'jquery' ),
            BIZZORDER_VERSION,
            true
        );

        wp_localize_script( 'bizzorder-frontend-js', 'BIZZORDER_DATA', array(
            'default_button_text'   => __( 'Add to cart', 'bizzorder' ),
            'preorder_button_text'  => bizzorder_get_option( 'button_text', __( 'Pre Order Now!', 'bizzorder' ) ),
        ) );
    }

    /**
     * Show WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="notice notice-error">
            <p>
                <?php
                printf(
                    /* translators: %1$s and %2$s are opening and closing strong tags respectively */
                    esc_html__( '%1$sBizzorder%2$s requires WooCommerce to be installed and activated.', 'bizzorder' ),
                    '<strong>',
                    '</strong>'
                );
                ?>
            </p>
        </div>
        <?php
    }

}

/**
 * Helper function to get plugin option
 *
 * @param string $key     Option key.
 * @param mixed  $default Default value.
 * @return mixed
 */
function bizzorder_get_option( $key = '', $default = null ) {
    $options = get_option( 'bizzorder_settings', array() );
    
    if ( empty( $key ) ) {
        return $options;
    }

    return isset( $options[ $key ] ) ? $options[ $key ] : $default;
}

/**
 * Check if product is pre-order
 *
 * @param int|WC_Product $product Product ID or object.
 * @return bool
 */
function bizzorder_is_preorder( $product ) {
    if ( is_numeric( $product ) ) {
        $product = wc_get_product( $product );
    }

    if ( ! $product ) {
        return false;
    }

    $is_preorder   = $product->get_meta( '_bizzorder_is_preorder' );
    $preorder_date = $product->get_meta( '_bizzorder_preorder_date' );

    if ( $is_preorder !== 'yes' || empty( $preorder_date ) ) {
        return false;
    }

    return strtotime( $preorder_date ) > time();
}

/**
 * Get pre-order date
 *
 * @param int|WC_Product $product Product ID or object.
 * @return string|false
 */
function bizzorder_get_preorder_date( $product ) {
    if ( is_numeric( $product ) ) {
        $product = wc_get_product( $product );
    }

    if ( ! $product ) {
        return false;
    }

    return $product->get_meta( '_bizzorder_preorder_date' );
}

/**
 * Initialize the plugin
 */
function bizzorder_init() {
    return BIZZORDER_Loader::init();
}



// Initialize plugin on plugins_loaded
add_action( 'plugins_loaded', 'bizzorder_init' );

// Activation hook for default options
register_activation_hook( __FILE__, function() {
    $default_options = array(
        'button_text'         => __( 'Pre Order Now!', 'bizzorder' ),
        'status_label'        => __( 'Pre-Ordered', 'bizzorder' ),
        'cart_notice'         => __( 'This item will be available for shipping in {days_left} days', 'bizzorder' ),
        'available_date_text' => __( 'Available on {date_format}', 'bizzorder' ),
        'show_badge'          => '1',
        'show_date'           => '1',
        'either_mode'         => '1',
    );

    if ( ! get_option( 'bizzorder_settings' ) ) {
        update_option( 'bizzorder_settings', $default_options );
    }

    // Flush rewrite rules
    flush_rewrite_rules();
    add_option( BIZZORDER_PLUGIN_BASE, true );
} );
