/**
 * Bizzorder - Frontend JavaScript
 * 
 * Object Oriented Module Pattern
 *
 * @package Bizzorder
 */

(function($) {
    'use strict';

    /**
     * BPOW Frontend Module
     */
    var Bizzorder_Frontend = (function() {
        
        /**
         * Module configuration
         */
        var config = {
            selectors: {
                variationWrapper: '.single_variation_wrap',
                addToCartButton: '.single_add_to_cart_button',
                variationForm: '.variations_form'
            },
            data: {
                defaultButtonText: BIZZORDER_DATA.default_button_text || 'Add to cart',
                preorderButtonText: BIZZORDER_DATA.preorder_button_text || 'Pre Order Now!'
            }
        };

        /**
         * Cache DOM elements
         */
        var $elements = {};

        /**
         * Initialize the module
         */
        function init() {
            cacheElements();
            bindEvents();
        }

        /**
         * Cache DOM elements for better performance
         */
        function cacheElements() {
            $elements.variationWrapper = $(config.selectors.variationWrapper);
            $elements.addToCartButton = $(config.selectors.addToCartButton);
            $elements.variationForm = $(config.selectors.variationForm);
        }

        /**
         * Bind event handlers
         */
        function bindEvents() {
            // Handle variation change on product page
            $elements.variationWrapper.on('show_variation', handleVariationChange);
            
            // Handle variation reset
            $elements.variationForm.on('reset_data', handleVariationReset);
        }

        /**
         * Handle variation change event
         * 
         * @param {Event} event Event object
         * @param {Object} variation Variation data
         */
        function handleVariationChange(event, variation) {
            var $button = $(config.selectors.addToCartButton);

            if (variation && variation.is_bizzorder_preorder === true) {
                updateButtonText($button, config.data.preorderButtonText);
                $button.addClass('bizzorder-preorder-button');
            } else {
                updateButtonText($button, config.data.defaultButtonText);
                $button.removeClass('bizzorder-preorder-button');
            }
        }

        /**
         * Handle variation reset event
         */
        function handleVariationReset() {
            var $button = $(config.selectors.addToCartButton);
            
            // Reset to default or check if parent product is pre-order
            if ($button.hasClass('bizzorder-preorder-button')) {
                $button.removeClass('bizzorder-preorder-button');
            }
        }

        /**
         * Update button text
         * 
         * @param {jQuery} $button Button element
         * @param {string} text New button text
         */
        function updateButtonText($button, text) {
            if ($button.length) {
                $button.text(text);
            }
        }

        /**
         * Utility: Check if element exists
         * 
         * @param {jQuery} $element jQuery element
         * @return {boolean}
         */
        function elementExists($element) {
            return $element.length > 0;
        }

        /**
         * Public API
         */
        return {
            init: init,
            updateButtonText: updateButtonText
        };

    })();

    /**
     * BPOW Cart Module
     */
    var Bizzorder_Cart = (function() {
        
        /**
         * Module configuration
         */
        var config = {
            selectors: {
                cartNotice: '.bizzorder-cart-notice',
                miniCartNotice: '.bizzorder-mini-cart-notice'
            }
        };

        /**
         * Initialize the module
         */
        function init() {
            // Cart specific initializations can be added here
        }

        /**
         * Public API
         */
        return {
            init: init
        };

    })();

    /**
     * Initialize modules on document ready
     */
    $(document).ready(function() {
        Bizzorder_Frontend.init();
        Bizzorder_Cart.init();
    });

    /**
     * Expose modules globally for external access if needed
     */
    window.Bizzorder_Frontend = Bizzorder_Frontend;
    window.Bizzorder_Cart = Bizzorder_Cart;

})(jQuery);
