<?php
/**
 * BizzPlugin Options Framework - Welcome Page Class
 * 
 * Handles the welcome/setup wizard page that appears on plugin activation.
 * 
 * @package BizzPlugin_Options_Framework
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Welcome Page Class
 * 
 * Creates a full-screen welcome/setup wizard that guides users through
 * initial plugin configuration using fields marked with setup_wizard => true.
 */
class BizzPlugin_Setup_Wizard {
    
    public $panel_id;
    public $is_active = false;
    public $setup_slug;
    public $setup_url;
    public $setup_wizard_url;
    public $setup_title;

    public $option_name;
    public $wizard_status_data = array();

    private $assets_url;
    private $version;

    private $redirect_key;
    
    /**
     * Configuration array
     * @var array
     */
    private $config = array();
    
    /**
     * Panel instance
     * @var BizzPlugin_Panel
     */
    private $panel_args;
    
    /**
     * Welcome steps extracted from sections
     * @var array
     */
    private $steps = array();
    
    private $panel;
    private $framework;

    private static $instance;
    public static function get_instance( $new_config, $panel_args, $framework ) {
        if ( null === self::$instance ) {
            self::$instance = new self( $new_config, $panel_args, $framework );
        }
        return self::$instance;
    }
    /**
     * Constructor
     * 
     * @param BizzPlugin_Panel $panel Panel instance
     * @param array $new_config Welcome page configuration
     * @param BizzPlugin_Framework $framework Framework instance
     */
    public function __construct( $new_config, $panel_args, $framework ) {

        
        $this->panel_args = $panel_args;

        $defaults = array(
            'title'             => '',
            'redirect_key'      => '',
            'skip_button'       => true,
            'skip_button_text'  => 'Skip Setup',
            'progress'          => true,
            'next_text'         => 'Next',
            'prev_text'         => 'Previous',
            'finish_button'     => 'Finish',
            'thank_you_title'   => 'Setup Complete!',
            'thank_you_message' => 'Thank you for setting up the plugin. Your settings have been saved successfully.',
            'settings_link_text' => 'Go to Settings',
            'visit_site_text'   => 'Visit Site',
            'close_text'        => 'Close',
            'edit_text'        => 'Edit',
            'links'             => array(

                // array(
                //     'text'  => 'Documentation',
                //     'url'   => 'https://example.com/docs',
                //     'target'=> '_blank',
                //     'icon'  => 'dashicons-book',
                //     'class' => 'custom-class',
                // ),
                // array(
                //     'text'  => 'Support',
                //     'url'   => 'https://example.com/support',
                //     'target'=> '_blank',
                //     'icon'  => 'dashicons-book',
                //     'class' => 'custom-class',
                // ),
            ),
        );
        
        $this->config = wp_parse_args($new_config, $defaults);
        $this->is_active = $this->is_active();

        $this->panel_id = isset($this->panel_args['id']) ? $this->panel_args['id'] : '';
        $this->option_name = isset($this->panel_args['option_name']) ? $this->panel_args['option_name'] . '-setup' : '';
        
        $this->setup_title = ! empty( $this->config['title'] ) ? $this->config['title'] : $this->panel_args['title'];
        $this->redirect_key = ! empty( $this->config['redirect_key'] ) ? $this->config['redirect_key'] : '';

        $menu_slug = isset($this->panel_args['menu_slug']) ? $this->panel_args['menu_slug'] : '';
        
        $this->setup_slug = ! empty( $menu_slug )  ? $menu_slug : '';
        $this->setup_url = admin_url('admin.php?page=' . $this->setup_slug);
        $this->setup_wizard_url = $this->setup_url . '&setup-wizard=welcome';

        

        if(  ! is_array( $new_config ) ||  empty( $new_config ) ) return;

        if( ! $this->is_active) return;

        

        $this->framework = $framework;
        $this->panel = $this->framework->get_panel( $this->panel_id );
        
        if( $this->is_active ) {
            $this->wizard_status_data = get_option( $this->option_name, array() );
            $this->assets_url = $this->framework->get_framework_url() . 'assets/';
            $this->version = $this->framework::VERSION;
            
            //add content only on setup wizard page
            add_action( 'bizzplugin_content_before' . $this->panel_id, array( $this, 'render_setup_wizard_content_before' ), 20, 1 );
            add_action( 'bizzplugin_content_after' . $this->panel_id, array( $this, 'render_setup_wizard_content_after' ), 0, 1 );

            add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        }
        $this->init_hooks();
    }

    public function get_localize_data(){
        return array(
            'isActive'          => $this->is_active,
            'panelId'           => $this->panel_id,
            'setupTitle'        => $this->setup_title,
            'settingsUrl'       => $this->setup_url,
            'siteUrl'           => home_url(),
            'config'            => array(
                'skipButton'        => $this->config['skip_button'],
                'skipButtonText'    => $this->config['skip_button_text'],
                'showProgress'      => $this->config['progress'],
                'nextText'          => $this->config['next_text'],
                'prevText'          => $this->config['prev_text'],
                'finishText'        => $this->config['finish_button'],
                'thankYouTitle'     => $this->config['thank_you_title'],
                'thankYouMessage'   => $this->config['thank_you_message'],
                'settingsLinkText'  => $this->config['settings_link_text'],
                'visitSiteText'     => $this->config['visit_site_text'],
                'closeText'         => $this->config['close_text'],
                'editText'          => $this->config['edit_text'],
                'links'             => $this->config['links'],
            ),
        );
    }

    /**
     * Enable / On Redirection on this plugin
     *
     * @return void
     */
    public function on_redirect() {
        add_action( 'admin_init', array( $this, 'redirect_on_activate' ) );
        return $this;
    }   

    public function redirect_on_activate() {
        if( empty( $this->redirect_key ) ) return;

        if ( ! get_option( $this->redirect_key, false ) ) {
            return;
        }

        delete_option( $this->redirect_key );
        // Multisite bulk activation এ redirect না করাই ভালো
        if ( isset( $_GET['activate-multi'] ) ) {
            return;
        }

        wp_safe_redirect( $this->setup_wizard_url );
        exit;
    }

    public function get_framework() {
        return $this->framework;
    }

    //panel
    public function get_panel() {
        return $this->panel;
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Register activation hook if plugin file is provided
        // if (!empty($this->config['plugin_file'])) {
        //     register_activation_hook($this->config['plugin_file'], array($this, 'set_redirect_transient'));
        // }
        
    }
    
    /**
     * Check if we're on the welcome page
     * 
     * @return bool
     */
    public function is_active() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is a read-only navigation parameter
        if (!isset($_GET['setup-wizard'])) {
            return false;
        }
        // Sanitize and validate against whitelist
        $setup_wizard = sanitize_key(wp_unslash($_GET['setup-wizard']));
        if( $setup_wizard === 'welcome' ) {
            return true;
        }
        return false;
    }
    

    public function render_setup_wizard_content_before( $target_panel_id ) {
        if( $target_panel_id !== $this->panel_id ) return;
        //.bizzplugin-temp-name - for initial hidden state to prevent flash of unstyled content
        ?>
            <div class="bizzplugin-wizard-wrapper bizzplugin-temp-name" id="bizzplugin-wizard-wrapper-box">
                <div class="bizzplugin-wizard-step-container">
                       
        <?php 
    }
    public function render_setup_wizard_content_after( $target_panel_id ) {
        if( $target_panel_id !== $this->panel_id ) return;
        ?>
                </div><!-- .bizzplugin-wizard-step-container -->
            
            
            </div><!-- .bizzplugin-wizard-wrapper -->
        
        <?php 
    }

    /**
     * Enqueue scripts and styles for the welcome page
     */
    public function enqueue_scripts() {
        if( ! $this->is_active ) return;

        // Enqueue your styles and scripts here
        wp_enqueue_style(
            'bizzplugin-framework-setup-wizard',
            $this->assets_url . 'css/setup-wizard.css',
            array(),
            $this->version
        );

        wp_enqueue_script(
            'bizzplugin-framework-setup-wizard',
            $this->assets_url . 'js/setup-wizard.js',
            array( 'jquery', 'bizzplugin-framework-script' ),
            $this->version,
            true
        );

    }
    
    /**
     * Get configuration value
     * 
     * @param string $key Configuration key
     * @return mixed
     */
    public function get_config( $key = null ) {
        if ( $key === null ) {
            return $this->config;
        }
        return isset( $this->config[ $key ] ) ? $this->config[ $key ] : null;
    }
}