<?php
/**
 * BizzPlugin Options Framework - Field Renderer
 * 
 * Shared field rendering class used by both BizzPlugin_Panel and BizzPlugin_Metabox.
 * 
 * @package BizzPlugin_Options_Framework
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Field Renderer Class
 * 
 * Provides static methods for rendering various field types.
 * Used by both Panel (options page) and Metabox contexts.
 */
class BizzPlugin_Render_Field {

    /**
     * Render input field (text, email, url, number, password)
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param string $type     Input type (text, email, url, number, password)
     * @param bool   $disabled Whether field is disabled
     */
    public static function input($field, $value, $name = null, $type = 'text', $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        
        $attrs = '';
        if (isset($field['placeholder'])) {
            $attrs .= sprintf(' placeholder="%s"', esc_attr($field['placeholder']));
        }
        if (isset($field['min']) && $type === 'number') {
            $attrs .= sprintf(' min="%s"', esc_attr($field['min']));
        }
        if (isset($field['max']) && $type === 'number') {
            $attrs .= sprintf(' max="%s"', esc_attr($field['max']));
        }
        if (isset($field['step']) && $type === 'number') {
            $attrs .= sprintf(' step="%s"', esc_attr($field['step']));
        }
        if ($disabled) {
            $attrs .= ' disabled="disabled"';
        }
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-' . $type;
        ?>
        <input 
            type="<?php echo esc_attr($type); ?>" 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>" 
            value="<?php echo esc_attr($value); ?>" 
            class="bizzplugin-input <?php echo esc_attr($class); ?>"
            <?php echo $attrs; ?>
        />
        <?php
    }

    /**
     * Render textarea field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function textarea($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $rows = isset($field['rows']) ? $field['rows'] : 5;
        
        $attrs = '';
        if (isset($field['placeholder'])) {
            $attrs .= sprintf(' placeholder="%s"', esc_attr($field['placeholder']));
        }
        if ($disabled) {
            $attrs .= ' disabled="disabled"';
        }
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-textarea';
        ?>
        <textarea 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>" 
            rows="<?php echo esc_attr($rows); ?>"
            class="bizzplugin-textarea <?php echo esc_attr($class); ?>"
            <?php echo $attrs; ?>
        ><?php echo esc_textarea($value); ?></textarea>
        <?php
    }

    /**
     * Render select field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function select($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $options = isset($field['options']) ? $field['options'] : array();
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-select';
        ?>
        <select 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>"
            class="bizzplugin-select <?php echo esc_attr($class); ?>"
            <?php echo $attrs; ?>
        >
            <?php foreach ($options as $opt_value => $opt_label) : ?>
                <option value="<?php echo esc_attr($opt_value); ?>" <?php selected($value, $opt_value); ?>>
                    <?php echo esc_html($opt_label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Render multi select field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function multi_select($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $options = isset($field['options']) ? $field['options'] : array();
        $value = is_array($value) ? $value : array();
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-multi-select';
        ?>
        <select 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>[]"
            class="bizzplugin-multi-select <?php echo esc_attr($class); ?>"
            multiple="multiple"
            <?php echo $attrs; ?>
        >
            <?php foreach ($options as $opt_value => $opt_label) : ?>
                <option value="<?php echo esc_attr($opt_value); ?>" <?php selected(in_array($opt_value, $value), true); ?>>
                    <?php echo esc_html($opt_label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Render checkbox field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function checkbox($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $label = isset($field['label']) ? $field['label'] : '';
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-checkbox';
        ?>
        <label class="bizzplugin-checkbox-label">
            <input 
                type="checkbox" 
                id="<?php echo esc_attr($field_id); ?>" 
                name="<?php echo esc_attr($input_name); ?>" 
                value="1"
                class="bizzplugin-checkbox <?php echo esc_attr($class); ?>"
                <?php checked($value, '1'); ?>
                <?php echo $attrs; ?>
            />
            <span class="bizzplugin-checkbox-text"><?php echo esc_html($label); ?></span>
        </label>
        <?php
    }

    /**
     * Render checkbox group field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function checkbox_group($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $options = isset($field['options']) ? $field['options'] : array();
        $value = is_array($value) ? $value : array();
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-checkbox-group';
        ?>
        <div class="bizzplugin-checkbox-group <?php echo esc_attr($class); ?>">
            <?php foreach ($options as $opt_value => $opt_label) : ?>
                <label class="bizzplugin-checkbox-label">
                    <input 
                        type="checkbox" 
                        name="<?php echo esc_attr($input_name); ?>[]" 
                        value="<?php echo esc_attr($opt_value); ?>"
                        class="bizzplugin-checkbox"
                        <?php checked(in_array($opt_value, $value), true); ?>
                        <?php echo $attrs; ?>
                    />
                    <span class="bizzplugin-checkbox-text"><?php echo esc_html($opt_label); ?></span>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Render radio field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function radio($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $options = isset($field['options']) ? $field['options'] : array();
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-radio';
        ?>
        <div class="bizzplugin-radio-group <?php echo esc_attr($class); ?>">
            <?php foreach ($options as $opt_value => $opt_label) : ?>
                <label class="bizzplugin-radio-label">
                    <input 
                        type="radio" 
                        name="<?php echo esc_attr($input_name); ?>" 
                        value="<?php echo esc_attr($opt_value); ?>"
                        class="bizzplugin-radio"
                        <?php checked($value, $opt_value); ?>
                        <?php echo $attrs; ?>
                    />
                    <span class="bizzplugin-radio-text"><?php echo esc_html($opt_label); ?></span>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Render switch toggle field (also known as on_off)
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function switch_toggle($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $on_label = isset($field['on_label']) ? $field['on_label'] : 'On';
        $off_label = isset($field['off_label']) ? $field['off_label'] : 'Off';
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-toggle';
        ?>
        <div class="bizzplugin-toggle-wrap <?php echo esc_attr($class); ?>">
            <label class="bizzplugin-toggle">
                <input 
                    type="checkbox" 
                    id="<?php echo esc_attr($field_id); ?>" 
                    name="<?php echo esc_attr($input_name); ?>" 
                    value="1"
                    class="bizzplugin-toggle-input"
                    <?php checked($value, '1'); ?>
                    <?php echo $attrs; ?>
                />
                <span class="bizzplugin-toggle-slider"></span>
                <span class="bizzplugin-toggle-on"><?php echo esc_html($on_label); ?></span>
                <span class="bizzplugin-toggle-off"><?php echo esc_html($off_label); ?></span>
            </label>
        </div>
        <?php
    }

    /**
     * Render color picker field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function color($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $default = isset($field['default']) ? $field['default'] : '';
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-color';
        ?>
        <input 
            type="text" 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>" 
            value="<?php echo esc_attr($value); ?>"
            class="bizzplugin-color-picker <?php echo esc_attr($class); ?>"
            data-default-color="<?php echo esc_attr($default); ?>"
            <?php echo $attrs; ?>
        />
        <?php
    }
    
    /**
     * Render alpha color picker field (with opacity support)
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function alpha_color($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $default = isset($field['default']) ? $field['default'] : '';
        $alpha_step = isset($field['alpha_step']) ? floatval($field['alpha_step']) : 0.01;
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-alpha-color';
        ?>
        <input 
            type="text" 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>" 
            value="<?php echo esc_attr($value); ?>"
            class="bizzplugin-alpha-color-picker <?php echo esc_attr($class); ?>"
            data-default-color="<?php echo esc_attr($default); ?>"
            data-alpha="1"
            data-alpha-step="<?php echo esc_attr($alpha_step); ?>"
            <?php echo $attrs; ?>
        />
        <?php
    }

    /**
     * Render date picker field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function date($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        
        $attrs = '';
        if (isset($field['placeholder'])) {
            $attrs .= sprintf(' placeholder="%s"', esc_attr($field['placeholder']));
        }
        if ($disabled) {
            $attrs .= ' disabled="disabled"';
        }
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-date';
        ?>
        <input 
            type="text" 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?>" 
            value="<?php echo esc_attr($value); ?>"
            class="bizzplugin-date-picker <?php echo esc_attr($class); ?>"
            autocomplete="off"
            <?php echo $attrs; ?>
        />
        <?php
    }

    /**
     * Render image upload field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function image($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $preview = '';
        if (!empty($value)) {
            $preview = wp_get_attachment_image_url($value, 'thumbnail');
        }
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-image';
        ?>
        <div class="bizzplugin-image-upload <?php echo esc_attr($class); ?>">
            <input 
                type="hidden" 
                id="<?php echo esc_attr($field_id); ?>" 
                name="<?php echo esc_attr($input_name); ?>" 
                value="<?php echo esc_attr($value); ?>"
                class="bizzplugin-image-input"
            />
            <div class="bizzplugin-image-preview" <?php echo empty($preview) ? 'style="display:none;"' : ''; ?>>
                <img src="<?php echo esc_url($preview); ?>" alt="" />
                <button type="button" class="bizzplugin-image-remove" <?php echo $attrs; ?>>&times;</button>
            </div>
            <button type="button" class="button bizzplugin-image-select" <?php echo $attrs; ?>>
                Select Image
            </button>
        </div>
        <?php
    }

    /**
     * Render file upload field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function file($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $file_name = '';
        if (!empty($value)) {
            $file_path = get_attached_file($value);
            if ($file_path) {
                $file_name = basename($file_path);
            }
        }
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-file';
        ?>
        <div class="bizzplugin-file-upload <?php echo esc_attr($class); ?>">
            <input 
                type="hidden" 
                id="<?php echo esc_attr($field_id); ?>" 
                name="<?php echo esc_attr($input_name); ?>" 
                value="<?php echo esc_attr($value); ?>"
                class="bizzplugin-file-input"
            />
            <span class="bizzplugin-file-name"><?php echo esc_html($file_name); ?></span>
            <button type="button" class="button bizzplugin-file-select" <?php echo $attrs; ?>>
                Select File
            </button>
            <button type="button" class="button bizzplugin-file-remove" <?php echo $attrs; ?> <?php echo empty($value) ? 'style="display:none;"' : ''; ?>>
                Remove
            </button>
        </div>
        <?php
    }

    /**
     * Render image select field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function image_select($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $options = isset($field['options']) ? $field['options'] : array();
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-image-select';
        ?>
        <div class="bizzplugin-image-select-wrap <?php echo esc_attr($class); ?>">
            <?php foreach ($options as $opt_value => $opt_image) : ?>
                <label class="bizzplugin-image-select-item <?php echo ($value === $opt_value) ? 'selected' : ''; ?>">
                    <input 
                        type="radio" 
                        name="<?php echo esc_attr($input_name); ?>" 
                        value="<?php echo esc_attr($opt_value); ?>"
                        class="bizzplugin-image-select-input"
                        <?php checked($value, $opt_value); ?>
                        <?php echo $disabled ? ' disabled="disabled"' : ''; ?>
                    />
                    <img src="<?php echo esc_url($opt_image); ?>" alt="<?php echo esc_attr($opt_value); ?>" />
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Render option select field (text-based option selection similar to image_select)
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function option_select($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $options = isset($field['options']) ? $field['options'] : array();
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-option-select';
        ?>
        <div class="bizzplugin-option-select-wrap <?php echo esc_attr($class); ?>">
            <?php foreach ($options as $opt_value => $opt_label) : ?>
                <label class="bizzplugin-option-select-item <?php echo ($value === $opt_value) ? 'selected' : ''; ?>">
                    <input 
                        type="radio" 
                        name="<?php echo esc_attr($input_name); ?>" 
                        value="<?php echo esc_attr($opt_value); ?>"
                        class="bizzplugin-option-select-input"
                        <?php checked($value, $opt_value); ?>
                        <?php echo $disabled ? ' disabled="disabled"' : ''; ?>
                    />
                    <span class="bizzplugin-option-select-label"><?php echo esc_html($opt_label); ?></span>
                </label>
            <?php endforeach; ?>
        </div>
        <?php
    }

    /**
     * Render post select field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function post_select($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $post_type = isset($field['post_type']) ? $field['post_type'] : 'post';
        $multiple = isset($field['multiple']) ? $field['multiple'] : false;
        $posts_per_page = isset($field['posts_per_page']) ? intval($field['posts_per_page']) : -1;
        $value = $multiple ? (is_array($value) ? array_map('intval', $value) : array()) : intval($value);
        
        $posts = get_posts(array(
            'post_type' => $post_type,
            'posts_per_page' => $posts_per_page,
            'orderby' => 'title',
            'order' => 'ASC',
            'post_status' => 'publish'
        ));
        
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($field);
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-post-select';
        ?>
        <select 
            id="<?php echo esc_attr($field_id); ?>" 
            name="<?php echo esc_attr($input_name); ?><?php echo $multiple ? '[]' : ''; ?>"
            class="bizzplugin-post-select bizzplugin-select <?php echo esc_attr($class); ?>"
            <?php echo $multiple ? 'multiple="multiple"' : ''; ?>
            <?php echo $attrs; ?>
        >
            <?php if (!$multiple) : ?>
                <option value="">Select...</option>
            <?php endif; ?>
            <?php foreach ($posts as $post) : ?>
                <?php
                $selected = $multiple ? in_array($post->ID, $value, true) : ($value === $post->ID);
                ?>
                <option value="<?php echo esc_attr($post->ID); ?>" <?php selected($selected, true); ?>>
                    <?php echo esc_html($post->post_title); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <?php
    }

    /**
     * Render HTML field (for custom content)
     * 
     * @param array $field Field configuration
     */
    public static function html($field) {
        $html = isset($field['html']) ? $field['html'] : '';
        $html .= isset($field['content']) ? $field['content'] : '';
        $html .= isset($field['default']) ? $field['default'] : '';
        $html .= isset($field['text']) ? $field['text'] : '';

        $allowed_tags = wp_kses_allowed_html( 'post' );


        $allowed_tags['input'] = [
            'type'    => true,
            'name'    => true,
            'value'   => true,
            'checked' => true,
            'id'      => true,
            'class'   => true,
            'placeholder' => true,
            'required' => true,
            'min' => true,
            'max' => true,
            'step' => true,
            'pattern' => true,
            'disabled' => true,
            'readonly' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,

        ];

        $allowed_tags['textarea'] = [
            'name'  => true,
            'rows'  => true,
            'cols'  => true,
            'id'    => true,
            'class' => true,
            'placeholder' => true,
            'required' => true,
            'disabled' => true,
            'readonly' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,

        ];

        $allowed_tags['select'] = [
            'name'  => true,
            'id'    => true,
            'class' => true,
            'required' => true,
            'disabled' => true,
            'multiple' => true,
            'readonly' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,

        ];

        $allowed_tags['option'] = [
            'value'    => true,
            'selected' => true,
            'disabled' => true,
            'readonly' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,
        ];

        $allowed_tags['button'] = [
            'type'  => true,
            'name'  => true,
            'value' => true,
            'id'    => true,
            'class' => true,
            'disabled' => true,
            'readonly' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,

        ];

        $allowed_tags['label'] = [
            'for'   => true,
            'class' => true,
            'style' => true,
            'id'    => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,
        ];

        $allowed_tags['fieldset'] = [
            'class' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,
        ];

        $allowed_tags['legend'] = [
            'class' => true,
            'style' => true,
            'onclick' => true,
            'data-*' => true,
            'aria-*' => true,
        ];

        // dd($allowed_tags);
        if ( ! empty( $html ) ) {
            echo wp_kses( $html, $allowed_tags );
        }
    }

    /**
     * Render slider/range field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether field is disabled
     */
    public static function slider($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        $min = isset($field['min']) ? $field['min'] : 0;
        $max = isset($field['max']) ? $field['max'] : 100;
        $step = isset($field['step']) ? $field['step'] : 1;
        $unit = isset($field['unit']) ? $field['unit'] : '';
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-field' : '';
        $class .= ' bizzplugin-input-slider';
        ?>
        <div class="bizzplugin-slider-wrap <?php echo esc_attr($class); ?>">
            <div class="bizzplugin-slider-container">
                <input 
                    type="range" 
                    id="<?php echo esc_attr($field_id); ?>" 
                    name="<?php echo esc_attr($input_name); ?>" 
                    value="<?php echo esc_attr($value); ?>"
                    min="<?php echo esc_attr($min); ?>"
                    max="<?php echo esc_attr($max); ?>"
                    step="<?php echo esc_attr($step); ?>"
                    class="bizzplugin-slider"
                    <?php disabled($disabled, true); ?>
                />
                <div class="bizzplugin-slider-value">
                    <span class="bizzplugin-slider-value-number"><?php echo esc_html($value); ?></span>
                    <?php if (!empty($unit)) : ?>
                        <span class="bizzplugin-slider-value-unit"><?php echo esc_html($unit); ?></span>
                    <?php endif; ?>
                </div>
            </div>
            <div class="bizzplugin-slider-labels">
                <span class="bizzplugin-slider-min"><?php echo esc_html($min); ?><?php echo esc_html($unit); ?></span>
                <span class="bizzplugin-slider-max"><?php echo esc_html($max); ?><?php echo esc_html($unit); ?></span>
            </div>
        </div>
        <?php
    }

    /**
     * Render repeater field
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value (array of items)
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether the field is disabled
     */
    public static function repeater($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        
        $sub_fields = isset($field['fields']) ? $field['fields'] : array();
        $button_text = isset($field['button_text']) ? $field['button_text'] : 'Add Item';
        $max_items = isset($field['max_items']) ? intval($field['max_items']) : 0;
        $min_items = isset($field['min_items']) ? intval($field['min_items']) : 0;
        $sortable = isset($field['sortable']) ? (bool) $field['sortable'] : true;
        $allow_add = isset($field['allow_add']) ? (bool) $field['allow_add'] : true;
        $header_template = isset($field['header_template']) ? $field['header_template'] : '';
        if( empty( $sub_fields) || !is_array($sub_fields) ){
            return; // No sub-fields defined, do not render
        }
        // Ensure value is an array
        if (!is_array($value) || empty($value)) {
            $value = array();
        }
        
        // Ensure minimum items
        while (count($value) < $min_items) {
            $value[] = array();
        }
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-repeater-wrap' : '';
        ?>
        <div class="bizzplugin-repeater-wrap <?php echo esc_attr($class); ?>" 
             data-field-id="<?php echo esc_attr($input_name); ?>"
             data-max-items="<?php echo esc_attr($max_items); ?>"
             data-min-items="<?php echo esc_attr($min_items); ?>"
             data-sortable="<?php echo esc_attr($sortable ? '1' : '0'); ?>"
             data-allow-add="<?php echo esc_attr($allow_add ? '1' : '0'); ?>"
             <?php if ($header_template) : ?>data-header-template="<?php echo esc_attr($header_template); ?>"<?php endif; ?>>
            
            <div class="bizzplugin-repeater-items">
                <?php 
                if (!empty($value)) {
                    foreach ($value as $index => $item_value) {
                        // dd($item_value);
                        self::repeater_item($field, $sub_fields, $index, $item_value, $input_name, $disabled, $min_items, count($value), false, $allow_add);
                    }
                }
                ?>
            </div>
            
            <?php if ($allow_add) : ?>
            <div class="bizzplugin-repeater-footer">
                <button type="button" 
                        class="button button-primary bizzplugin-repeater-add" 
                        <?php echo $disabled ? ' disabled="disabled"' : ''; ?>
                        <?php echo ($max_items > 0 && count($value) >= $max_items) ? ' style="display:none;"' : ''; ?>>
                    <span class="dashicons dashicons-plus-alt2"></span>
                    <?php echo esc_html($button_text); ?>
                </button>
            </div>
            
            <!-- Template for new items (hidden div container) -->
            <div class="bizzplugin-repeater-template" style="display:none !important;" aria-hidden="true">
                <?php self::repeater_item($field, $sub_fields, '{{INDEX}}', array(), $input_name, $disabled, $min_items, 0, true, $allow_add); ?>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render a single repeater item
     * 
     * @param array  $field         Parent field configuration
     * @param array  $sub_fields    Sub-fields configuration
     * @param mixed  $index         Item index (numeric or {{INDEX}} for template)
     * @param array  $item_value    Current item values
     * @param string $name          Base input name
     * @param bool   $disabled      Whether the field is disabled
     * @param int    $min_items     Minimum items allowed
     * @param int    $current_count Current number of items
     * @param bool   $is_template   Whether this is a template render
     * @param bool   $allow_add     Whether adding/removing items is allowed
     */
    public static function repeater_item($field, $sub_fields, $index, $item_value, $name, $disabled, $min_items, $current_count, $is_template = false, $allow_add = true) {
        $can_remove = $allow_add && ($current_count > $min_items || $is_template);
        $header_template = isset($field['header_template']) ? $field['header_template'] : '';
        $class = $field['class'] ?? '';
        $class = ! empty( $class ) ? $class . '-item' : '';
        ?>
        <div class="bizzplugin-repeater-item <?php echo esc_attr($class); ?>" data-index="<?php echo esc_attr($index); ?>">
            <div class="bizzplugin-repeater-item-header">
                <span class="bizzplugin-repeater-item-handle dashicons dashicons-move" title="Drag to reorder"></span>
                <span class="bizzplugin-repeater-item-title">
                    <?php 
                    $item_title = '';
                    
                    if (!$is_template && !empty($sub_fields)) {
                        // If header_template is set, use it
                        if (!empty($header_template)) {
                            $item_title = $header_template;
                            // Replace placeholders with actual values
                            foreach ($sub_fields as $sf) {
                                $sf_id = $sf['id'];
                                $sf_type = isset($sf['type']) ? $sf['type'] : 'text';
                                $sf_value = isset($item_value[$sf_id]) ? $item_value[$sf_id] : '';
                                
                                // For select fields, get the option label instead of value
                                if ($sf_type === 'select' && !empty($sf['options']) && isset($sf['options'][$sf_value])) {
                                    $sf_value = $sf['options'][$sf_value];
                                }
                                
                                $item_title = str_replace('{' . $sf_id . '}', $sf_value, $item_title);
                            }
                            // Clean up any unreplaced placeholders
                            $item_title = preg_replace('/\{[^}]+\}/', '', $item_title);
                            $item_title = trim($item_title);
                        } else {
                            // Try to get title from any field value (not just text fields)
                            foreach ($sub_fields as $sf) {
                                $sf_type = isset($sf['type']) ? $sf['type'] : 'text';
                                $sf_id = $sf['id'];
                                $sf_value = isset($item_value[$sf_id]) ? $item_value[$sf_id] : '';
                                
                                if (!empty($sf_value)) {
                                    // For select fields, get the option label
                                    if ($sf_type === 'select' && !empty($sf['options']) && isset($sf['options'][$sf_value])) {
                                        $item_title = $sf['options'][$sf_value];
                                    } elseif (is_string($sf_value)) {
                                        $item_title = $sf_value;
                                    }
                                    
                                    if (!empty($item_title)) {
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    
                    if (empty($item_title)) {
                        $display_index = $is_template ? '{{DISPLAY_INDEX}}' : ($index + 1);
                        echo sprintf('Item #%s', $display_index);
                    } else {
                        echo esc_html($item_title);
                    }
                    ?>
                </span>
                <div class="bizzplugin-repeater-item-actions">
                    <button type="button" class="bizzplugin-repeater-item-toggle" title="Toggle">
                        <span class="dashicons dashicons-arrow-down-alt2"></span>
                    </button>
                    <?php if ($allow_add) : ?>
                    <button type="button" 
                            class="bizzplugin-repeater-item-remove" 
                            title="Remove"
                            <?php echo (!$can_remove && !$is_template) ? ' style="display:none;"' : ''; ?>
                            <?php echo $disabled ? ' disabled="disabled"' : ''; ?>>
                        <span class="dashicons dashicons-trash"></span>
                    </button>
                    <?php endif; ?>
                </div>
            </div>
            <div class="bizzplugin-repeater-item-content">
                <?php
                foreach ($sub_fields as $sub_field) {
                    $sub_field_id = $sub_field['id'];
                    $sub_field_type = isset($sub_field['type']) ? $sub_field['type'] : 'text';
                    $sub_field_default = isset($sub_field['default']) ? $sub_field['default'] : '';
                    $sub_field_value = isset($item_value[$sub_field_id]) ? $item_value[$sub_field_id] : $sub_field_default;
                    
                    // Build the input name: fieldname[index][subfield_id]
                    $input_name = $name . '[' . $index . '][' . $sub_field_id . ']';
                    $input_id = str_replace(array('[', ']'), '_', $name) . '_' . $index . '_' . $sub_field_id;
                    
                    // Disable inputs in template to prevent form submission.
                    // JavaScript will re-enable them when cloning the template to create new items.
                    $field_disabled = $disabled || $is_template;
                    ?>
                    <div class="bizzplugin-repeater-subfield bizzplugin-repeater-subfield-<?php echo esc_attr($sub_field_type); ?>"<?php echo $is_template ? ' data-template-field="1"' : ''; ?>>
                        <label class="bizzplugin-repeater-subfield-label" for="<?php echo esc_attr($input_id); ?>">
                            <?php echo esc_html(isset($sub_field['title']) ? $sub_field['title'] : $sub_field_id); ?>
                        </label>
                        <div class="bizzplugin-repeater-subfield-input">
                            <?php

                            self::repeater_subfield($sub_field, $sub_field_value, $input_name, $input_id, $field_disabled);
                            ?>
                        </div>
                        <?php if (!empty($sub_field['description'])) : ?>
                            <p class="bizzplugin-repeater-subfield-desc"><?php echo esc_html($sub_field['description']); ?></p>
                        <?php endif; ?>
                    </div>
                    <?php
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render a subfield inside repeater item
     * 
     * @param array  $sub_field Subfield configuration
     * @param mixed  $value     Current value
     * @param string $name      Input name attribute
     * @param string $id        Input id attribute
     * @param bool   $disabled  Whether the field is disabled
     */
    public static function repeater_subfield($sub_field, $value, $name, $id, $disabled) {
        $sub_field['id'] = $id;
        $type = isset($sub_field['type']) ? $sub_field['type'] : 'text';
        $attrs = $disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($sub_field);
        switch ($type) {
            case 'text':
            case 'email':
            case 'url':
            case 'number':
            case 'password':
                self::input($sub_field, $value, $name, $type, $disabled);
                break;

            case 'toggle':
            case 'on_off':
            case 'switch':
                self::switch_toggle($sub_field, $value, $name, $disabled);
                break;
                
            case 'textarea':
                if(!isset($sub_field['rows']) || !is_numeric($sub_field['rows']) || intval($sub_field['rows']) <= 0 ){
                    $sub_field['rows'] = 3;
                } 
                self::textarea($sub_field, $value, $name, $disabled);
                break;
                
            case 'select':
                self::select($sub_field, $value, $name, $disabled);
                break;
                
            case 'checkbox':
                self::checkbox($sub_field, $value, $name, $disabled);
                break;
                
            case 'color':
                self::color($sub_field, $value, $name, $disabled);
                break;
            
            case 'alpha_color':
                if(! isset($sub_field['alpha_step']) || !is_numeric($sub_field['alpha_step'])  ){
                    $sub_field['alpha_step'] = 0.01;
                }
                self::alpha_color($sub_field, $value, $name, $disabled);
                break;
                
            case 'image':
                self::image($sub_field, $value, $name, false);
                break;

            case 'info':
            case 'notice':
            case 'content':
            case 'html':
                self::html($value);
                break;

            default:
                // Fallback to text input
                $sub_field['type'] = 'text';
                self::input($sub_field, $value, $name, $type, $disabled);
                break;
        }
    }

    /**
     * Get framework URL
     */
    private static function get_framework_url() {
        if (defined('BIZZPLUGIN_FRAMEWORK_URL')) {
            return BIZZPLUGIN_FRAMEWORK_URL;
        }
        
        return plugins_url('/', dirname(__FILE__));
    }
    
    /**
     * Render plugins field for recommended plugins
     */
    public static function plugins($field) {
        
        $plugins = isset($field['plugins']) ? $field['plugins'] : array();
        
        if (empty($plugins)) {
            return;
        }
        
        // Include plugin functions if not available
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $installed_plugins = get_plugins();

        wp_enqueue_style(
            'bizzplugin-plugins-field',
            self::get_framework_url() . 'assets/css/fields/plugins.css',
            array(),
            BizzPlugin_Framework::VERSION
        );

        
        ?>
        <div class="bizzplugin-plugins-field" data-field-id="<?php echo esc_attr($field['id'] ?? ''); ?>">
            <div class="bizzplugin-plugins-grid">
                <?php foreach ($plugins as $plugin) : 
                    $slug = isset($plugin['slug']) ? $plugin['slug'] : '';
                    $name = isset($plugin['name']) ? $plugin['name'] : $slug;
                    $description = isset($plugin['description']) ? $plugin['description'] : '';
                    $thumbnail = isset($plugin['thumbnail']) ? $plugin['thumbnail'] : '';
                    $plugin_file = isset($plugin['file']) ? $plugin['file'] : $slug . '/' . $slug . '.php';
                    $author = isset($plugin['author']) ? $plugin['author'] : '';
                    $url = isset($plugin['url']) ? $plugin['url'] : 'https://wordpress.org/plugins/' . $slug . '/';
                    
                    // Check plugin status - first check if installed, then if active
                    $is_installed = array_key_exists($plugin_file, $installed_plugins);
                    $is_active = $is_installed && is_plugin_active($plugin_file);
                    
                    // Determine status
                    if ($is_active) {
                        $status = 'active';
                        $status_text = 'Active';
                        $status_class = 'bizzplugin-plugin-status-active';
                    } elseif ($is_installed) {
                        $status = 'installed';
                        $status_text = 'Installed (Inactive)';
                        $status_class = 'bizzplugin-plugin-status-installed';
                    } else {
                        $status = 'not_installed';
                        $status_text = 'Not Installed';
                        $status_class = 'bizzplugin-plugin-status-not-installed';
                    }
                ?>
                <div class="bizzplugin-plugin-card" data-slug="<?php echo esc_attr($slug); ?>" data-file="<?php echo esc_attr($plugin_file); ?>">
                    <?php if (!empty($thumbnail)) : ?>
                    <div class="bizzplugin-plugin-thumbnail">
                        <img src="<?php echo esc_url($thumbnail); ?>" alt="<?php echo esc_attr($name); ?>">
                    </div>
                    <?php endif; ?>
                    
                    <div class="bizzplugin-plugin-info">
                        <h4 class="bizzplugin-plugin-name">
                            <?php if (!empty($url)) : ?>
                                <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer">
                                    <?php echo esc_html($name); ?>
                                </a>
                            <?php else : ?>
                                <?php echo esc_html($name); ?>
                            <?php endif; ?>
                        </h4>
                        
                        <?php if (!empty($author)) : ?>
                        <p class="bizzplugin-plugin-author">
                            By <?php echo esc_html($author); ?>
                        </p>
                        <?php endif; ?>
                        
                        <?php if (!empty($description)) : ?>
                        <p class="bizzplugin-plugin-description"><?php echo esc_html($description); ?></p>
                        <?php endif; ?>
                        
                        <div class="bizzplugin-plugin-footer">
                            <span class="bizzplugin-plugin-status <?php echo esc_attr($status_class); ?>">
                                <?php echo esc_html($status_text); ?>
                            </span>
                            
                            <div class="bizzplugin-plugin-actions">
                                <?php if ($status === 'not_installed') : ?>
                                    <button type="button" class="button button-primary bizzplugin-install-plugin" 
                                            data-slug="<?php echo esc_attr($slug); ?>" 
                                            data-file="<?php echo esc_attr($plugin_file); ?>">
                                        <span class="dashicons dashicons-download"></span>
                                        Install
                                    </button>
                                <?php elseif ($status === 'installed') : ?>
                                    <button type="button" class="button button-primary bizzplugin-activate-plugin" 
                                            data-slug="<?php echo esc_attr($slug); ?>" 
                                            data-file="<?php echo esc_attr($plugin_file); ?>">
                                        <span class="dashicons dashicons-yes-alt"></span>
                                        Activate
                                    </button>
                                <?php else : ?>
                                    <span class="bizzplugin-plugin-activated">
                                        <span class="dashicons dashicons-yes"></span>
                                        Activated
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render additional HTML attributes for a field
     * getting from 'attr' and 'attributes' keys
     * 
     * @param array $field Field configuration
     * @return null|string Generated HTML attributes string
     */
    protected static function render_attr($field) {
        if(empty($field) || !is_array($field)) {
            return '';
        }
        $attrs = '';

        if(isset($field['premium']) && $field['premium'] === true) {
            $attrs .= ' data-is_premium="true"';
        }


        $attributes = isset($field['attributes']) ? $field['attributes'] : array();
        $attributes = isset($field['attr']) ? $field['attr'] : $attributes;
        $attributes = isset($field['attrs']) ? $field['attrs'] : $attributes;
        
        $attrs .= self::generate_additional_attributes($attributes);
        return $attrs;
    }

    /**
     * Generate additional HTML attributes from associative array
     * 
     * @param array $attributes Associative array of attributes (name => value)
     * @return null|string Generated HTML attributes string
     */
    protected static function generate_additional_attributes($attributes) {

        if(empty($attributes) || !is_array($attributes)) {
            return '';
        }

        $attrs = '';
        foreach ($attributes as $attr_name => $attr_value) {
            if(is_bool($attr_value)) {
                if ($attr_value) {
                    $attrs .= sprintf(' %s="%s"', esc_attr($attr_name), esc_attr($attr_name));
                }
                continue;
            }else{
                $attrs .= sprintf(' %s="%s"', esc_attr($attr_name), esc_attr($attr_value));
            }
            
        }
        return $attrs;
    }

    /**
     * Render single repeater field (repeater with only one field type)
     * 
     * @param array  $field    Field configuration
     * @param mixed  $value    Current field value (array of items)
     * @param string $name     Input name attribute (optional, defaults to $field['id'])
     * @param bool   $disabled Whether the field is disabled
     */
    public static function single_repeater($field, $value, $name = null, $disabled = false) {
        $field_id = isset($field['id']) ? $field['id'] : '';
        $input_name = ($name !== null) ? $name : $field_id;
        
        // Get the single field configuration
        $sub_field = isset($field['field']) ? $field['field'] : array();
        $sub_field_type = isset($sub_field['type']) ? $sub_field['type'] : 'text';
        $sub_field_id = isset($sub_field['id']) ? $sub_field['id'] : 'value';
        
        $button_text = isset($field['button_text']) ? $field['button_text'] : 'Add Item';
        $max_items = isset($field['max_items']) ? intval($field['max_items']) : 0;
        $min_items = isset($field['min_items']) ? intval($field['min_items']) : 0;
        $sortable = isset($field['sortable']) ? (bool) $field['sortable'] : true;
        $allow_add = isset($field['allow_add']) ? (bool) $field['allow_add'] : true;
        $placeholder = isset($sub_field['placeholder']) ? $sub_field['placeholder'] : '';
        
        if (empty($sub_field) || !is_array($sub_field)) {
            error_log('Error: Single repeater field requires field configuration. Field ID: ' . esc_html($field_id));
            return;
        }
        
        // Ensure value is an array
        if (!is_array($value) || empty($value)) {
            $value = array();
        }
        
        // Ensure minimum items
        while (count($value) < $min_items) {
            $value[] = '';
        }
        ?>
        <div class="bizzplugin-single-repeater-wrap" 
             data-field-id="<?php echo esc_attr($input_name); ?>"
             data-sub-field-id="<?php echo esc_attr($sub_field_id); ?>"
             data-sub-field-type="<?php echo esc_attr($sub_field_type); ?>"
             data-max-items="<?php echo esc_attr($max_items); ?>"
             data-min-items="<?php echo esc_attr($min_items); ?>"
             data-sortable="<?php echo esc_attr($sortable ? '1' : '0'); ?>"
             data-allow-add="<?php echo esc_attr($allow_add ? '1' : '0'); ?>"
             data-placeholder="<?php echo esc_attr($placeholder); ?>">
            
            <div class="bizzplugin-single-repeater-items">
                <?php 
                if (!empty($value)) {
                    foreach ($value as $index => $item_value) {
                        self::single_repeater_item($field, $sub_field, $index, $item_value, $input_name, $disabled, $min_items, count($value), false, $allow_add);
                    }
                }
                ?>
            </div>
            
            <?php if ($allow_add) : ?>
            <div class="bizzplugin-single-repeater-footer">
                <button type="button" 
                        class="button button-primary bizzplugin-single-repeater-add" 
                        <?php echo $disabled ? ' disabled="disabled"' : ''; ?>
                        <?php echo ($max_items > 0 && count($value) >= $max_items) ? ' style="display:none;"' : ''; ?>>
                    <span class="dashicons dashicons-plus-alt2"></span>
                    <?php echo esc_html($button_text); ?>
                </button>
            </div>
            
            <!-- Template for new items (hidden div container) -->
            <div class="bizzplugin-single-repeater-template" style="display:none !important;" aria-hidden="true">
                <?php self::single_repeater_item($field, $sub_field, '{{INDEX}}', '', $input_name, $disabled, $min_items, 0, true, $allow_add); ?>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render a single repeater item
     * 
     * @param array  $field         Parent field configuration
     * @param array  $sub_field     Sub-field configuration
     * @param mixed  $index         Item index (numeric or {{INDEX}} for template)
     * @param mixed  $item_value    Current item value
     * @param string $name          Base input name
     * @param bool   $disabled      Whether the field is disabled
     * @param int    $min_items     Minimum items allowed
     * @param int    $current_count Current number of items
     * @param bool   $is_template   Whether this is a template render
     * @param bool   $allow_add     Whether adding/removing items is allowed
     */
    public static function single_repeater_item($field, $sub_field, $index, $item_value, $name, $disabled, $min_items, $current_count, $is_template = false, $allow_add = true) {
        $can_remove = $allow_add && ($current_count > $min_items || $is_template);
        $sub_field_type = isset($sub_field['type']) ? $sub_field['type'] : 'text';
        $sub_field_id = isset($sub_field['id']) ? $sub_field['id'] : 'value';
        $placeholder = isset($sub_field['placeholder']) ? $sub_field['placeholder'] : '';
        
        // Build the input name: fieldname[index]
        $input_name = $name . '[' . $index . ']';
        $input_id = str_replace(array('[', ']'), '_', $name) . '_' . $index;
        $sub_field['id'] = $input_id;
        
        // Disable inputs in template to prevent form submission
        $input_disabled = $disabled || $is_template;
        $attrs = $input_disabled ? ' disabled="disabled"' : '';
        $attrs .= self::render_attr($sub_field);
        $repeater_sub_class = ! empty($sub_field['class']) && is_string($sub_field['class'])  ? $sub_field['class'] : '';
        ?>
        <div class="bizzplugin-single-repeater-item" data-index="<?php echo esc_attr($index); ?>"<?php echo $is_template ? ' data-template="1"' : ''; ?>>
            <?php if ($allow_add) : ?>
            <span class="bizzplugin-single-repeater-handle dashicons dashicons-move" title="Drag to reorder"></span>
            <?php endif; ?>
            
            <div class="bizzplugin-single-repeater-input">
                <?php
                switch ($sub_field_type) {
                    case 'text':
                    case 'email':
                    case 'url':
                    case 'number':
                    case 'password':
                        self::input($sub_field, $item_value, $input_name, $sub_field_type, $input_disabled);
                        break;
                        
                    case 'textarea':
                        if(!isset($sub_field['rows']) || !is_numeric($sub_field['rows']) || intval($sub_field['rows']) <= 0 ){
                            $sub_field['rows'] = 3;
                        }
                        self::textarea($sub_field, $item_value, $input_name, $input_disabled); 
                        break;
                        
                    case 'select':
                        self::select($sub_field, $item_value, $input_name, $input_disabled);
                        break;
                        
                    case 'color':
                        self::color($sub_field, $item_value, $input_name, $input_disabled);
                        break;
                        
                    default:
                        // Fallback to text input
                        $sub_field['type'] = 'text';
                        self::input($sub_field, $item_value, $input_name, $sub_field_type, $input_disabled);
                        break;
                }
                ?>
            </div>
            
            <?php if ($allow_add) : ?>
            <button type="button" 
                    class="bizzplugin-single-repeater-remove" 
                    title="Remove"
                    <?php echo (!$can_remove && !$is_template) ? ' style="display:none;"' : ''; ?>
                    <?php echo $disabled ? ' disabled="disabled"' : ''; ?>>
                <span class="dashicons dashicons-no-alt"></span>
            </button>
            <?php endif; ?>
        </div>
        <?php
    }
}