<?php
/**
 * BizzPlugin Options Framework - Panel Class
 * 
 * @package BizzPlugin_Options_Framework
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Panel Class
 */
class BizzPlugin_Panel {
    
    /**
     * Panel ID
     */
    private $id;
    
    /**
     * Panel title
     */
    private $title;
    
    /**
     * Menu title
     */
    private $menu_title;
    
    /**
     * Menu slug
     */
    private $menu_slug;
    
    /**
     * Parent slug for submenu
     */
    private $parent_slug;

    /**
     * Theme mode (light or dark)
     */
    private $theme_mode;

    /**
     * Show theme switcher
     */
    private $show_theme_switcher;

    /**
     * Route namespace for REST API
     */
    private $route_namespace;
    
    /**
     * Required capability
     */
    private $capability;
    
    /**
     * Menu icon
     */
    private $icon;
    
    /**
     * Menu position
     */
    private $position;
    
    /**
     * Option name in database
     */
    private $option_name;
    
    /**
     * Is premium active
     */
    private $is_premium;
    
    /**
     * Sections
     */
    private $sections = array();
    
    /**
     * Framework instance
     */
    private $framework;
    
    /**
     * Current page hook
     */
    private $page_hook;

    /**
     * Current section
     */
    private $current_section;

    
    
    /**
     * Current subsection
     */
    private $current_subsection;
    
    /**
     * Panel configuration (logo, version, footer_text, etc.)
     */
    private $panel_config = array();
    
    /**
     * Enable search functionality
     */
    private $enable_search = true;
    
    /**
     * Enable Export/Import section
     */
    private $show_export_import = true;
    
    /**
     * Enable API & Webhook section
     */
    private $show_api = true;
    
    /**
     * Resource links for sidebar
     */
    private $resources = array();
    
    /**
     * Recommended plugins for sidebar
     */
    private $recommended_plugins = array();

    private $setup_wizard;
    private $setup_wizard_args = array();
    private $is_setup_wizard = false;

    /**
     * Topbar configuration
     */
    private $topbar = array();

    /**
     * Whether to show topbar
     */
    private $show_topbar = true;
    
    /**
     * Constructor
     */
    public function __construct($args, $framework) {
        $this->id = $args['id'];
        $this->title = $args['title'];
        $this->menu_title = !empty($args['menu_title']) ? $args['menu_title'] : $args['title'];
        $this->menu_slug = !empty($args['menu_slug']) ? $args['menu_slug'] : sanitize_title($args['id']);
        $this->parent_slug = $args['parent_slug'];
        $this->capability = $args['capability'];
        $this->icon = $args['icon'];
        $this->position = $args['position'];
        $this->option_name = $args['option_name'];
        $this->is_premium = $args['is_premium'];
        $this->route_namespace = $args['route_namespace'];
        $this->enable_search = isset($args['enable_search']) ? (bool) $args['enable_search'] : true;
        $this->show_export_import = isset($args['show_export_import']) ? (bool) $args['show_export_import'] : true;
        $this->show_api = isset($args['show_api']) ? (bool) $args['show_api'] : true;
        $this->theme_mode = isset($args['theme_mode']) ? $args['theme_mode'] : 'light';
        $this->show_theme_switcher = isset($args['show_theme_switcher']) ? (bool) $args['show_theme_switcher'] : true;
        $this->show_topbar = isset($args['topbar']) ? (bool) $args['topbar'] : true;
        
        $this->framework = $framework;

        //setup wizard 
        $this->setup_wizard = $args['setup_wizard'] ?? array();
        
        // Add sections if provided
        if (!empty($args['sections'])) {
            foreach ($args['sections'] as $section) {
                $this->add_section($section);
            }
        }
        
        // Register admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_body_class', array($this, 'admin_body_class'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        if (!empty($this->parent_slug)) {
            $this->page_hook = add_submenu_page(
                $this->parent_slug,
                $this->title,
                $this->menu_title,
                $this->capability,
                $this->menu_slug,
                array($this, 'render_page')
            );
        } else {
            $this->page_hook = add_menu_page(
                $this->title,
                $this->menu_title,
                $this->capability,
                $this->menu_slug,
                array($this, 'render_page'),
                $this->icon,
                $this->position
            );
        }
    }
    public function admin_body_class($classes) {
        if ($this->is_current_page(get_current_screen()->id)) {
            $classes .= ' bizzplugin-panel-page bizzplugin-theme-' . esc_attr($this->theme_mode);

            if( $this->is_setup_wizard ) {
                $classes .= ' bizzplugin-setup-wizard-active-body';
            }
        }
        return $classes;
    }
    /**
     * Check if current page
     */
    public function is_current_page($hook) {
        return $hook === $this->page_hook || strpos($hook, $this->menu_slug) !== false;
    }
    
    /**
     * Add section
     */
    public function add_section($args) {
        $defaults = array(
            'id' => '',
            'title' => '',
            'description' => '',
            'icon' => '',
            'fields' => array(),
            'subsections' => array(),
            'hide_reset_button' => false,
            'dependency' => array(), // Section-level dependency support
        );
        
        $section = wp_parse_args($args, $defaults);
        
        if (empty($section['id'])) {
            return false;
        }
        
        // Apply filter for adding fields from other plugins
        $section['fields'] = apply_filters('bizzplugin_section_fields', $section['fields'], $section['id'], $this->id);
        // Apply panel-specific filter for individual panel control
        $section['fields'] = apply_filters('bizzplugin_section_fields_' . $this->id, $section['fields'], $section['id'], $this->id);
        
        $this->sections[$section['id']] = $section;
        
        return $this;
    }
    
    /**
     * Get sections
     */
    public function get_sections() {
        // $sections = apply_filters('bizzplugin_panel_sections', $this->sections, $this->id);
        // Apply panel-specific filter for individual panel control
        return apply_filters('bizzplugin_panel_sections_' . $this->id, $this->sections, $this->id);
    }
    
    /**
     * Get section by ID
     */
    public function get_section($section_id) {
        return isset($this->sections[$section_id]) ? $this->sections[$section_id] : null;
    }
    
    /**
     * Get option name
     */
    public function get_option_name() {
        return $this->option_name;
    }
    
    /**
     * Get panel ID
     */
    public function get_id() {
        return $this->id;
    }
    
    /**
     * Is premium active
     */
    public function is_premium() {
        // $is_premium = apply_filters('bizzplugin_is_premium', $this->is_premium, $this->id);
        // Apply panel-specific filter for individual panel control
        return apply_filters('bizzplugin_is_premium_' . $this->id, $this->is_premium, $this->id);
    }
    
    /**
     * Get all fields
     */
    public function get_all_fields() {
        $fields = array();
        
        foreach ($this->sections as $section) {
            if (!empty($section['fields'])) {
                foreach ($section['fields'] as $field) {
                    $fields[$field['id']] = $field;
                }
            }
            
            // Get fields from subsections
            if (!empty($section['subsections'])) {
                foreach ($section['subsections'] as $subsection) {
                    if (!empty($subsection['fields'])) {
                        foreach ($subsection['fields'] as $field) {
                            $fields[$field['id']] = $field;
                        }
                    }
                }
            }
        }
        
        return $fields;
    }
    
    /**
     * GET SECTION DEFAULTS on SCREEN 
     * **********************************************
     * Get section defaults (specific section without subsections)
     * only sections fields, not subsections
     * If on subsection, only that subsection fields of specific subsection by id
     * 
     * @param string $section_id The ID of the section
     * @param string|null $subsection_id Optional ID of the subsection
     * 
     * @return array Default values array of the section fields or subsection fields
     */
    public function get_section_defaults($section_id, $subsection_id = null) {
        $defaults = array();
        $section = $this->get_section($section_id);
        
        if (!$section) {
            return $defaults;
        }
        
        
        
        // // Include subsection fields
        if ( ! empty( $subsection_id ) && !empty($section['subsections'])) {
            foreach ($section['subsections'] as $subsection) {
                if ( $subsection['id'] !== $subsection_id ) {
                    continue;
                }
                if (!empty($subsection['fields'])) {
                    foreach ($subsection['fields'] as $field) {
                        $defaults[$field['id']] = isset($field['default']) ? $field['default'] : '';
                    }
                }
            }
        }else if (!empty($section['fields'])) {
            foreach ($section['fields'] as $field) {
                $defaults[$field['id']] = isset($field['default']) ? $field['default'] : '';
            }
        }
        
        return $defaults;
    }
    
    /**
     * Get all defaults
     */
    public function get_all_defaults() {
        $defaults = array();
        
        foreach ($this->sections as $section_id => $section) {
            $defaults = array_merge($defaults, $this->get_section_defaults($section_id));
        }
        
        return $defaults;
    }
    
    /**
     * Check if section/subsection has editable fields (not just html, info, plugins types)
     */
    private function has_editable_fields($fields) {
        if (empty($fields)) {
            return false;
        }
        
        $non_editable_types = array('html', 'info', 'plugins', 'link', 'heading', 'divider', 'notice');
        
        foreach ($fields as $field) {
            $field_type = isset($field['type']) ? $field['type'] : 'text';
            if (!in_array($field_type, $non_editable_types, true)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Build search keywords from title, description and ID
     * 
     * @param string $title Item title
     * @param string $description Item description
     * @param string $id Item ID
     * @param null| string | array  $extra Optional extra keywords
     * @return string Lowercase search keywords string
     */
    private function build_search_keywords($title, $description, $id, $extra = array()) {
        $string = strtolower(
            (isset($title) ? $title : '') . ' ' . 
            (isset($description) ? $description : '') . ' ' . 
            (isset($id) ? $id : '')
        );
        if ( empty($extra) ) {
            return $string;
        }
        $string .= ' ' . $this->recursive_to_lowercase_string( $extra );
        return $string;
    }

    /**
     * recursive - if need,
     * can be array
     * or can be string
     * 
     * convert to lowercase string
     */
    private function recursive_to_lowercase_string($string_array ) {
        if(empty($string_array)) return '';
        $result = '';
        if (is_array($string_array)) {
            foreach ($string_array as $item) {
                if (is_array($item)) {
                    $result .= ' ' . $this->recursive_to_lowercase_string($item);
                } else {
                    $result .= ' ' . strtolower($item);
                }
            }
        } else {
            $result = strtolower($string_array);
        }
        return trim($result);
    }
    
    /**
     * Build search keywords for fields array
     * 
     * @param array $fields Array of field configurations
     * @return string Lowercase search keywords string
     */
    private function build_fields_search_keywords($fields) {
        $keywords = '';
        if (!empty($fields)) {
            foreach ($fields as $field) {
                $search_extra = array( 
                    $field['default'] ?? '',
                    $field['placeholder'] ?? '',
                    $field['class'] ?? '',
                    $field['value'] ?? '',
                    isset($field['is_premium']) ? 'premium' : 'free'
                );
                $keywords .= ' ' . $this->build_search_keywords(
                    isset($field['title']) ? $field['title'] : '',
                    isset($field['description']) ? $field['description'] : '',
                    isset($field['id']) ? $field['id'] : '',
                    $search_extra
                );
            }
        }
        return $keywords;
    }
    
    /**
     * Get panel args
     */
    public function get_args() {
        return array(
            'id' => $this->id,
            'title' => $this->title,
            'menu_title' => $this->menu_title,
            'menu_slug' => $this->menu_slug,
            'parent_slug' => $this->parent_slug,
            'capability' => $this->capability,
            'icon' => $this->icon,
            'position' => $this->position,
            'option_name' => $this->option_name,
            'is_premium' => $this->is_premium,
        );
    }
    
    /**
     * Add a subsection to an existing section
     * 
     * @param string $section_id The ID of the parent section
     * @param array  $subsection Subsection configuration array
     * @return $this For method chaining
     */
    public function add_subsection($section_id, $subsection) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        $defaults = array(
            'id' => '',
            'title' => '',
            'description' => '',
            'icon' => '',
            'fields' => array(),
            'hide_reset_button' => false,
        );
        
        $subsection = wp_parse_args($subsection, $defaults);
        
        if (empty($subsection['id'])) {
            return $this;
        }
        
        if (!isset($this->sections[$section_id]['subsections'])) {
            $this->sections[$section_id]['subsections'] = array();
        }
        
        // Check for duplicate subsection ID and replace if exists
        $found = false;
        foreach ($this->sections[$section_id]['subsections'] as $key => $existing) {
            if ($existing['id'] === $subsection['id']) {
                $this->sections[$section_id]['subsections'][$key] = $subsection;
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            $this->sections[$section_id]['subsections'][] = $subsection;
        }
        
        return $this;
    }
    
    /**
     * Add a field to a section
     * 
     * @param string $section_id The ID of the section
     * @param array  $field Field configuration array
     * @return $this For method chaining
     */
    public function add_field($section_id, $field) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        if (empty($field['id'])) {
            return $this;
        }
        
        $this->sections[$section_id]['fields'][] = $field;
        
        return $this;
    }
    
    /**
     * Add a field to a subsection
     * 
     * @param string $section_id The ID of the parent section
     * @param string $subsection_id The ID of the subsection
     * @param array  $field Field configuration array
     * @return $this For method chaining
     */
    public function add_subsection_field($section_id, $subsection_id, $field) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        if (empty($field['id'])) {
            return $this;
        }
        
        if (!isset($this->sections[$section_id]['subsections'])) {
            return $this;
        }
        
        foreach ($this->sections[$section_id]['subsections'] as &$subsection) {
            if ($subsection['id'] === $subsection_id) {
                $subsection['fields'][] = $field;
                break;
            }
        }
        
        return $this;
    }
    
    /**
     * Set resource links for the sidebar
     * 
     * @param array $resources Array of resource links
     * @return $this For method chaining
     */
    public function add_resources($resources) {
        return $this->set_resources($resources);
    }
    public function set_resources($resources) {
        $this->resources = array_merge($this->resources, $resources);
        return $this;
    }
    
    /**
     * Add a single resource link
     * 
     * @param array $resource Resource link configuration (icon, title, url)
     * @return $this For method chaining
     */
    public function add_resource($resource) {
        $defaults = array(
            'icon' => 'dashicons-admin-links',
            'title' => '',
            'url' => '#',
        );
        $this->resources[] = wp_parse_args($resource, $defaults);
        return $this;
    }
    
    /**
     * Get resource links
     * 
     * @return array Resource links array
     */
    public function get_resources() {
        return $this->resources;
    }
    
    /**
     * Set recommended plugins for the sidebar
     * 
     * @param array $plugins Array of recommended plugins
     * @return $this For method chaining
     */
    public function add_recommended_plugins($plugins) {
        return $this->set_recommended_plugins($plugins);
    }
    public function set_recommended_plugins($plugins) {
        $this->recommended_plugins = array_merge($this->recommended_plugins, $plugins);
        return $this;
    }
    
    /**
     * Add a single recommended plugin
     * 
     * @param array $plugin Plugin configuration array
     * @return $this For method chaining
     */
    public function add_recommended_plugin($plugin) {
        $defaults = array(
            'slug' => '',
            'name' => '',
            'description' => '',
            'thumbnail' => '',
            'author' => '',
            'file' => '',
            'url' => '',
        );
        $this->recommended_plugins[] = wp_parse_args($plugin, $defaults);
        return $this;
    }
    
    /**
     * Get recommended plugins
     * 
     * @return array Recommended plugins array
     */
    public function get_recommended_plugins() {
        return $this->recommended_plugins;
    }
    
    /**
     * Set panel configuration value
     * 
     * @param string $key Configuration key (logo, version, footer_text, title, is_premium)
     * @param mixed  $value Configuration value
     * @return $this For method chaining
     */
    public function set_config($key, $value) {
        $this->panel_config[$key] = $value;
        return $this;
    }
    
    /**
     * set route namespace
     *
     * @param [type] $config
     * @return void
     */
    public function set_route_namespace( $namespace ) {
        $this->route_namespace = $namespace;
        return $this;
    }
    public function add_route_namespace( $namespace ) {
        $this->route_namespace = $namespace;
        return $this;
    }
    public function get_route_namespace() {
        return $this->route_namespace;
    }

    
    public function add_setup_wizard( $setup_wizard ) {
        if(  ! is_array( $setup_wizard ) ||  empty( $setup_wizard ) ) return $this;
        $this->setup_wizard_args = $setup_wizard;
        // $this->setup_wizard = BizzPlugin_Setup_Wizard::get_instance( $this->setup_wizard_args, $this->get_args(), $this->framework );
        $this->setup_wizard = new BizzPlugin_Setup_Wizard( $this->setup_wizard_args, $this->get_args(), $this->framework );
        $this->setup_wizard->on_redirect();
        $this->is_setup_wizard = $this->setup_wizard->is_active;
        return $this;
    }
    public function set_setup_wizard( $setup_wizard ) {
        return $this->add_setup_wizard( $setup_wizard );
    }
    public function get_setup_wizard() {
        return $this->setup_wizard;// BizzPlugin_Setup_Wizard::get_instance( $this->setup_wizard_args, $this->get_args(), $this->framework );
    }

    /**
     * Set multiple panel configuration values at once
     * 
     * @param array $config Configuration array
     * @return $this For method chaining
     */
    public function set_panel_config($config) {
        $this->panel_config = array_merge($this->panel_config, $config);
        return $this;
    }
    
    /**
     * Get panel configuration
     * 
     * @param string|null $key Optional specific key to get
     * @return mixed Panel configuration array or specific value
     */
    public function get_panel_config($key = null) {
        if ($key !== null) {
            return isset($this->panel_config[$key]) ? $this->panel_config[$key] : null;
        }
        return $this->panel_config;
    }
    
    /**
     * Set the panel logo
     * 
     * @param string $logo_url URL to the logo image
     * @return $this For method chaining
     */
    public function set_logo($logo_url) {
        return $this->set_config('logo', $logo_url);
    }
    
    /**
     * Set the panel version
     * 
     * @param string $version Version string
     * @return $this For method chaining
     */
    public function set_version($version) {
        return $this->set_config('version', $version);
    }
    
    /**
     * Set the panel title (for header display)
     * 
     * @param string $title Title string
     * @return $this For method chaining
     */
    public function set_panel_title($title) {
        return $this->set_config('title', $title);
    }
    
    /**
     * Set premium status
     * 
     * Updates both the instance property (used by is_premium() method and filter)
     * and the panel config (used in render_page() display).
     * 
     * @param bool $is_premium Premium status
     * @return $this For method chaining
     */
    public function set_premium($is_premium) {
        // Update property for is_premium() method
        $this->is_premium = $is_premium;
        // Also store in config for render_page() panel_config merge
        return $this->set_config('is_premium', $is_premium);
    }
    
    /**
     * Set footer text
     * 
     * @param string $text Footer text
     * @return $this For method chaining
     */
    public function set_footer_text($text) {
        return $this->set_config('footer_text', $text);
    }

    /**
     * Set topbar configuration
     * 
     * Topbar is displayed at the top of the panel before the notice area.
     * It can contain custom links with icons (e.g., Live Demo, Get Premium).
     * 
     * @param array $topbar Topbar configuration array with the following structure:
     *   - 'left_links'  => array of link items for left side
     *   - 'right_links' => array of link items for right side (e.g., Live Demo, Get Premium)
     *   Each link item can have:
     *     - 'title'  => Link text
     *     - 'url'    => Link URL
     *     - 'icon'   => Dashicons class (e.g., 'dashicons-external')
     *     - 'target' => Link target (e.g., '_blank')
     *     - 'class'  => Additional CSS class
     * @return $this For method chaining
     */
    public function set_topbar($topbar) {
        if (!is_array($topbar)) {
            return $this;
        }
        
        $defaults = array(
            'logo'        => '',
            'left_links'  => array(),
            'right_links' => array(),
        );
        
        $this->topbar = wp_parse_args($topbar, $defaults);
        $this->show_topbar = true;
        
        return $this;
    }

    /**
     * Add a topbar link
     * 
     * @param array  $link Link configuration (title, url, icon, target, class)
     * @param string $position Position ('left' or 'right', default: 'right')
     * @return $this For method chaining
     */
    public function add_topbar_link($link, $position = 'right') {
        $defaults = array(
            'title'  => '',
            'url'    => '#',
            'icon'   => '',
            'target' => '_blank',
            'class'  => '',
        );
        
        $link = wp_parse_args($link, $defaults);
        
        if ($position === 'left') {
            $this->topbar['left_links'][] = $link;
        } else {
            $this->topbar['right_links'][] = $link;
        }
        
        return $this;
    }

    /**
     * Get topbar configuration
     * 
     * @return array Topbar configuration array
     */
    public function get_topbar() {
        return $this->topbar;
    }

    /**
     * Check if topbar is enabled
     * 
     * @return bool Whether topbar is enabled
     */
    public function is_topbar_enabled() {
        return $this->show_topbar && !empty($this->topbar);
    }
    
    /**
     * Enable search functionality
     * 
     * @param bool $enable Whether to enable search (default: true)
     * @return $this For method chaining
     */
    public function enable_search($enable = true) {
        $this->enable_search = (bool) $enable;
        return $this;
    }
    
    /**
     * Disable search functionality
     * 
     * @return $this For method chaining
     */
    public function disable_search() {
        $this->enable_search = false;
        return $this;
    }
    
    /**
     * Check if search is enabled
     * 
     * @return bool Whether search is enabled
     */
    public function is_search_enabled() {
        return $this->enable_search;
    }
    
    /**
     * Enable Export/Import section
     * 
     * @param bool $enable Whether to enable Export/Import section (default: true)
     * @return $this For method chaining
     */
    public function enable_export_import($enable = true) {
        $this->show_export_import = (bool) $enable;
        return $this;
    }
    
    /**
     * Disable Export/Import section
     * 
     * @return $this For method chaining
     */
    public function disable_export_import() {
        $this->show_export_import = false;
        return $this;
    }
    
    /**
     * Check if Export/Import section is enabled
     * 
     * @return bool Whether Export/Import section is enabled
     */
    public function is_export_import_enabled() {
        return $this->show_export_import;
    }
    
    /**
     * Enable API & Webhook section
     * 
     * @param bool $enable Whether to enable API section (default: true)
     * @return $this For method chaining
     */
    public function enable_api($enable = true) {
        $this->show_api = (bool) $enable;
        return $this;
    }
    
    /**
     * Disable API & Webhook section
     * 
     * @return $this For method chaining
     */
    public function disable_api() {
        $this->show_api = false;
        return $this;
    }
    
    /**
     * Check if API section is enabled
     * 
     * @return bool Whether API section is enabled
     */
    public function is_api_enabled() {
        return $this->show_api;
    }
    
    /**
     * Remove a section by ID
     * 
     * @param string $section_id Section ID to remove
     * @return $this For method chaining
     */
    public function remove_section($section_id) {
        if (isset($this->sections[$section_id])) {
            unset($this->sections[$section_id]);
        }
        return $this;
    }
    
    /**
     * Remove a field from a section
     * 
     * @param string $section_id Section ID
     * @param string $field_id Field ID to remove
     * @return $this For method chaining
     */
    public function remove_field($section_id, $field_id) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        foreach ($this->sections[$section_id]['fields'] as $key => $field) {
            if ($field['id'] === $field_id) {
                array_splice($this->sections[$section_id]['fields'], $key, 1);
                break;
            }
        }
        
        return $this;
    }
    
    /**
     * Remove a subsection from a section
     * 
     * @param string $section_id Section ID
     * @param string $subsection_id Subsection ID to remove
     * @return $this For method chaining
     */
    public function remove_subsection($section_id, $subsection_id) {
        if (!isset($this->sections[$section_id]) || !isset($this->sections[$section_id]['subsections'])) {
            return $this;
        }
        
        foreach ($this->sections[$section_id]['subsections'] as $key => $subsection) {
            if ($subsection['id'] === $subsection_id) {
                array_splice($this->sections[$section_id]['subsections'], $key, 1);
                break;
            }
        }
        
        return $this;
    }
    
    /**
     * Render the settings page
     */
    public function render_page() {
        $options = get_option($this->option_name, array());
        $sections = $this->get_sections();
        
        // Get current section and subsection from URL parameter (sanitized)
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is a read-only navigation parameter, not a state-changing action
        $current_section = isset($_GET['section']) ? sanitize_key(wp_unslash($_GET['section'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is a read-only navigation parameter, not a state-changing action
        $current_subsection = isset($_GET['subsection']) ? sanitize_key(wp_unslash($_GET['subsection'])) : '';

        $this->current_section = $current_section;
        $this->current_subsection = $current_subsection;
        
        // Validate that the section exists
        if (!empty($current_section) && !isset($sections[$current_section]) && $current_section !== 'api' && $current_section !== 'export_import') {
            $current_section = '';
        }
        
        if (empty($current_section) && !empty($sections)) {
            $first_section = reset($sections);
            $current_section = $first_section['id'];
            
            // If section has no fields but has subsections, auto-select first subsection
            if (empty($first_section['fields']) && !empty($first_section['subsections'])) {
                $first_subsection = reset($first_section['subsections']);
                $current_subsection = $first_subsection['id'];
            }
        }
        
        // Auto-select first subsection if section has no fields
        if (!empty($current_section) && isset($sections[$current_section])) {
            $section_data = $sections[$current_section];
            if (empty($section_data['fields']) && !empty($section_data['subsections']) && empty($current_subsection)) {
                $first_subsection = reset($section_data['subsections']);
                $current_subsection = $first_subsection['id'];
            }
        }
        
        // Allow custom rendering
        if (has_action('bizzplugin_render_panel_' . $this->id)) {
            do_action('bizzplugin_render_panel_' . $this->id, $this, $options, $sections, $current_section, $current_subsection);
            return;
        }
        
        // Build default panel config
        $default_config = array(
            'logo' => $this->framework->get_url() . 'assets/images/logo.png',
            'version' => defined('BIZZPLUGIN_PLUGIN_VERSION') ? BIZZPLUGIN_PLUGIN_VERSION : '1.0.0',
            'is_premium' => $this->is_premium(),
            'resources' => array(
                // array(
                //     'icon' => 'dashicons-media-document',
                //     'title' => 'Documentation',
                //     'url' => '#',
                // ),
            ),
            'recommended_plugins' => array(),
        );
        
        // Merge with instance-level panel config (set via chainable methods)
        $instance_config = $this->panel_config;
        if( ! is_array( $this->panel_config ) ){
            $this->panel_config = $default_config;
        }
        
        $instance_config = array_merge($default_config, $this->panel_config);
        
        // Add instance-level resources if set
        if (! empty($this->resources) && is_array( $this->resources ) && is_array( $instance_config['resources'] ) ) {
            $instance_config['resources'] = array_merge($instance_config['resources'], $this->resources);
        }else if( is_array( $this->resources ) && ! empty( $this->resources ) ){
            $instance_config['resources'] = $this->resources;
        }
        
        // Add instance-level recommended plugins if set
        if (!empty($this->recommended_plugins) && is_array( $this->recommended_plugins ) && is_array( $instance_config['recommended_plugins'] ) ) {
            $instance_config['recommended_plugins'] = array_merge($instance_config['recommended_plugins'], $this->recommended_plugins);
        }else if( is_array( $this->recommended_plugins ) && ! empty( $this->recommended_plugins ) ){
            $instance_config['recommended_plugins'] = $this->recommended_plugins;
        }
        
        // Merge instance config with defaults (instance config takes priority)
        $panel_config = array_merge($default_config, $instance_config);
        
        // Get panel config for branding (filter allows additional modifications)
        // $panel_config = apply_filters('bizzplugin_panel_config', $panel_config, $this->id);
        // Apply panel-specific filter for individual panel control
        $panel_config = apply_filters('bizzplugin_panel_config_' . $this->id, $panel_config, $this->id);
        
        // Determine theme mode class
        $theme_mode = $this->theme_mode;
        $theme_class = ($theme_mode === 'dark') ? 'bizzplugin-dark-mode' : '';
        $show_theme_switcher = $this->show_theme_switcher;

        $setup_wizard = $this->get_setup_wizard();
        $setup_class = isset( $setup_wizard->is_active ) && $setup_wizard->is_active ? 'bizzplugin-setup-wizard-active' : 'bizzplugin-dashboard-active';
        $setup_wise_hide_style = isset( $setup_wizard->is_active ) && $setup_wizard->is_active ? 'style=display:none;' : '';

        $topbar_enabled = $this->is_topbar_enabled();
        $topbar_class = $topbar_enabled ? 'bizzplugin-has-topbar' : 'bizzplugin-no-topbar';
        ?>
        <div class="wrap bizzplugin-framework-wrap <?php echo esc_attr($setup_class); ?> <?php echo esc_attr($theme_class); ?> <?php echo esc_attr($topbar_class); ?>" 
        data-panel-id="<?php echo esc_attr($this->id); ?>" 
        data-option-name="<?php echo esc_attr($this->option_name); ?>" 
        data-theme-mode="<?php echo esc_attr($theme_mode); ?>"
        <?php if( isset( $setup_wizard->is_active ) && $setup_wizard->is_active ){ ?>
        data-setup_wizard_localize='<?php echo wp_json_encode( $setup_wizard->get_localize_data() ); ?>'
        <?php } ?>
        >
            <?php $this->render_topbar(); ?>
            <div class="wp-notice-area-wrapper" <?php echo esc_attr($setup_wise_hide_style); ?> >
                <h2><!-- For Notice Area --></h2>
            </div>           
            <div class="bizzplugin-framework-container">
                <!-- Navigation Sidebar -->
                <div class="bizzplugin-nav" <?php echo esc_attr($setup_wise_hide_style); ?>>
                    <!-- Logo and Branding -->
                    <div class="bizzplugin-nav-header">
                        <?php if (!empty($panel_config['logo'])) : ?>
                            <img src="<?php echo esc_url($panel_config['logo']); ?>" alt="<?php echo esc_attr($this->title); ?>" class="bizzplugin-nav-logo" />
                        <?php endif; if (!empty($panel_config['title'])) : ?>
                            <h4 class="bizzplugin-nav-title-main"><?php echo esc_html( $panel_config['title'] ); ?></h4>
                        <?php endif; ?>
                        <div class="bizzplugin-nav-meta">
                            <?php if ($show_theme_switcher) : ?>
                            <button type="button" class="bizzplugin-theme-switcher" title="Toggle theme">
                                <span class="dashicons dashicons-admin-appearance"></span>
                            </button>
                            <?php endif; ?>
                            <?php if ($panel_config['is_premium']) : ?>
                                <span class="bizzplugin-badge bizzplugin-badge-pro">Premium</span>
                            <?php else : ?>
                                <span class="bizzplugin-badge bizzplugin-badge-free">Free</span>
                            <?php endif; ?>

                            <?php
                            $version = $panel_config['version'] ?? '0.0.0';
                            $version_premium = $panel_config['version_premium'] ?? '';
                            ?>
                            <span class="bizzplugin-version ">
                                <i title="Base Version - <?php echo esc_attr($version); ?>" class="bizzplugin-v-regular">v<?php echo esc_html($version); ?></i>
                            <?php if( ! empty( $version_premium ) && $this->is_premium ){ ?>
                                 | <i title="Extended Premium Version - <?php echo esc_attr($version_premium) ?>" class="bizzplugin-v-premium"> v<?php echo esc_html($version_premium); ?></i>
                            <?php } ?>
                            </span>
                        </div>
                    </div>
                    
                    <?php if ($this->is_search_enabled()) : ?>
                    <!-- Search Field -->
                    <div class="bizzplugin-search-wrap">
                        <div class="bizzplugin-search-field">
                            <!-- <span class="dashicons dashicons-search bizzplugin-search-icon"></span> -->
                            <input type="text" 
                                   id="bizzplugin-settings-search" 
                                   class="bizzplugin-search-input" 
                                   placeholder="Search settings..." 
                                   autocomplete="off" />
                            <button type="button" class="bizzplugin-search-clear" style="display:none;" aria-label="Clear search">
                                <span class="dashicons dashicons-no-alt"></span>
                            </button>
                        </div>
                        <div class="bizzplugin-search-results-info" style="display:none;"></div>
                    </div>
                    <?php endif; ?>
                    
                    <?php do_action('bizzplugin_nav_before_menu', $this->id); ?>
                    
                    <ul class="bizzplugin-nav-menu">
                        <?php foreach ($sections as $section) : 
                            $has_fields = !empty($section['fields']);
                            $has_subsections = !empty($section['subsections']);
                            $is_active_section = ($current_section === $section['id']);
                            $is_expanded = $is_active_section && $has_subsections;
                            
                            // Section dependency support
                            $section_dependency_attr = BizzPlugin_Helper_Common::get_dependency_attr($section);
                            
                            
                            // Build search keywords for section using helper methods
                            $section_search_keywords = $this->build_search_keywords(
                                $section['title'],
                                isset($section['description']) ? $section['description'] : '',
                                $section['id']
                            );
                            if ($has_fields) {
                                $section_search_keywords .= $this->build_fields_search_keywords($section['fields']);
                            }
                            if ($has_subsections) {
                                foreach ($section['subsections'] as $subsection) {
                                    $section_search_keywords .= ' ' . $this->build_search_keywords(
                                        isset($subsection['title']) ? $subsection['title'] : '',
                                        isset($subsection['description']) ? $subsection['description'] : '',
                                        isset($subsection['id']) ? $subsection['id'] : ''
                                    );
                                    if (!empty($subsection['fields'])) {
                                        $section_search_keywords .= $this->build_fields_search_keywords($subsection['fields']);
                                    }
                                }
                            }
                        ?>
                            <li class="bizzplugin-nav-item <?php echo $is_active_section ? 'active' : ''; ?> <?php echo $has_subsections ? 'has-subsections' : ''; ?> <?php echo $is_expanded ? 'expanded' : ''; ?>" data-search-keywords="<?php echo esc_attr($section_search_keywords); ?>" <?php echo $section_dependency_attr; ?>>
                                <a href="#" 
                                   data-section="<?php echo esc_attr($section['id']); ?>" 
                                   data-has-fields="<?php echo $has_fields ? '1' : '0'; ?>"
                                   data-has-subsections="<?php echo $has_subsections ? '1' : '0'; ?>"
                                   class="bizzplugin-nav-link <?php echo ($is_active_section && empty($current_subsection)) ? 'current' : ''; ?>">
                                    <?php if (!empty($section['icon'])) : ?>
                                        <span class="bizzplugin-nav-icon <?php echo esc_attr($section['icon']); ?>"></span>
                                    <?php endif; ?>
                                    <span class="bizzplugin-nav-title"><?php echo esc_html($section['title']); ?></span>
                                    <?php if ($has_subsections) : ?>
                                        <span class="bizzplugin-nav-arrow dashicons dashicons-arrow-down-alt2"></span>
                                    <?php endif; ?>
                                </a>
                                
                                <?php if ($has_subsections) : ?>
                                    <ul class="bizzplugin-nav-submenu" <?php echo $is_expanded ? '' : 'style="display:none;"'; ?>>
                                        <?php foreach ($section['subsections'] as $subsection) : 
                                            $is_active_subsection = ($is_active_section && $current_subsection === $subsection['id']);
                                            

                                            // Build search keywords for subsection using helper methods
                                            $subsection_search_keywords = $this->build_search_keywords(
                                                isset($subsection['title']) ? $subsection['title'] : '',
                                                isset($subsection['description']) ? $subsection['description'] : '',
                                                isset($subsection['id']) ? $subsection['id'] : '',
                                            );
                                            if (!empty($subsection['fields'])) {
                                                $subsection_search_keywords .= $this->build_fields_search_keywords($subsection['fields']);
                                            }

                                            // Section dependency support
                                            $subsection_dependency_attr = BizzPlugin_Helper_Common::get_dependency_attr($subsection);
                                            
                                        ?>
                                            <li class="bizzplugin-nav-subitem <?php echo $is_active_subsection ? 'active' : ''; ?>" data-search-keywords="<?php echo esc_attr($subsection_search_keywords); ?>" <?php echo $subsection_dependency_attr; ?>>
                                                <a href="#" 
                                                   data-section="<?php echo esc_attr($section['id']); ?>" 
                                                   data-subsection="<?php echo esc_attr($subsection['id']); ?>" 
                                                   class="bizzplugin-nav-sublink <?php echo $is_active_subsection ? 'current' : ''; ?>">
                                                    <?php if (!empty($subsection['icon'])) : ?>
                                                        <span class="bizzplugin-nav-subicon <?php echo esc_attr($subsection['icon']); ?>"></span>
                                                    <?php endif; ?>
                                                    <?php echo esc_html($subsection['title']); ?>
                                                </a>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                        
                        <?php if ($this->show_export_import) : ?>
                        <!-- Export/Import Section -->
                        <li class="bizzplugin-nav-item <?php echo ($current_section === 'export_import') ? 'active' : ''; ?>">
                            <a href="#" data-section="export_import" class="bizzplugin-nav-link <?php echo ($current_section === 'export_import') ? 'current' : ''; ?>">
                                <span class="bizzplugin-nav-icon dashicons dashicons-database-export"></span>
                                <span class="bizzplugin-nav-title">Export & Import</span>
                            </a>
                        </li>
                        <?php endif; ?>
                        
                        <?php if ($this->show_api) : ?>
                        <!-- API Section -->
                        <li class="bizzplugin-nav-item <?php echo ($current_section === 'api') ? 'active' : ''; ?>">
                            <a href="#" data-section="api" class="bizzplugin-nav-link <?php echo ($current_section === 'api') ? 'current' : ''; ?>">
                                <span class="bizzplugin-nav-icon dashicons dashicons-rest-api"></span>
                                <span class="bizzplugin-nav-title">API & Webhook</span>
                            </a>
                        </li>
                        <?php endif; ?>
                    </ul>
                    
                    <?php do_action('bizzplugin_nav_after_menu', $this->id); ?>
                    <?php do_action('bizzplugin_nav_after_menu' . $this->id, $this->id); ?>
                    
                    <!-- Footer Branding -->
                    <div class="bizzplugin-nav-footer">
                        <span class="bizzplugin-nav-by">By</span>
                        <img src="<?php echo esc_url($this->framework->get_url() . 'assets/images/logo.png'); ?>" alt="BizzPlugin" class="bizzplugin-nav-footer-logo" />
                    </div>
                </div>
                
                <!-- Content -->
                <div class="bizzplugin-content">
                        <!-- Extra Save Button, when click, then trigger to save options #bizzplugin-save-options -->
                         <button type="button" 
                         title="Click: Ctrl + S"
                         id="bizzplugin-save-options-top" 
                         onclick="document.getElementById('bizzplugin-save-options').click();"
                         class="button">
                          🖫 Save
                        </button>
                

                    <?php do_action('bizzplugin_content_before' . $this->id, $this->id); ?>
                    <?php do_action('bizzplugin_content_before', $this->id); ?>
                    <div class="bizzplugin-save-status"></div>
                    
                    <form id="bizzplugin-options-form" class="bizzplugin-form">
                        <?php wp_nonce_field('bizzplugin_framework_nonce', 'bizzplugin_nonce'); ?>
                        
                        <?php foreach ($sections as $section) : 
                            $has_fields = !empty($section['fields']);
                            $has_subsections = !empty($section['subsections']);
                            $show_section = ($current_section === $section['id']) && (empty($current_subsection) || !$has_subsections);
                            $section_has_editable_fields = $has_fields && $this->has_editable_fields($section['fields']);
                            $hide_reset = isset($section['hide_reset_button']) && $section['hide_reset_button'];
                            $show_reset_button = $section_has_editable_fields && !$hide_reset;
                        ?>
                            <!-- Main Section Content (only fields, not subsections) -->
                            <?php if ($has_fields) : ?>
                            <div class="bizzplugin-section" 
                                 id="section-<?php echo esc_attr($section['id']); ?>" 
                                 data-section="<?php echo esc_attr($section['id']); ?>" 
                                 style="<?php echo !$show_section ? 'display:none;' : ''; ?>">
                                <div class="bizzplugin-section-header">
                                    
                                    <h2><?php echo esc_html($section['title']); ?></h2>
                                    <?php if (!empty($section['description'])) : ?>
                                        <p class="bizzplugin-section-desc"><?php echo esc_html($section['description']); ?></p>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="bizzplugin-section-content">
                                    <?php $this->render_fields($section['fields'], $options); ?>
                                </div>
                                
                                <?php if ($show_reset_button) : ?>
                                <div class="bizzplugin-section-footer">
                                    <button type="button" class="button bizzplugin-reset-section" data-section="<?php echo esc_attr($section['id']); ?>">
                                        Reset Section
                                    </button>
                                </div>
                                <?php endif; ?>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Subsections as separate content areas -->
                            <?php if ($has_subsections) : ?>
                                <?php foreach ($section['subsections'] as $subsection) : 
                                    $show_subsection = ($current_section === $section['id'] && $current_subsection === $subsection['id']);
                                    $subsection_has_editable_fields = !empty($subsection['fields']) && $this->has_editable_fields($subsection['fields']);
                                    $subsection_hide_reset = isset($subsection['hide_reset_button']) && $subsection['hide_reset_button'];
                                    $show_subsection_reset = $subsection_has_editable_fields && !$subsection_hide_reset;
                                ?>
                                <div class="bizzplugin-section bizzplugin-subsection-content" 
                                     id="subsection-<?php echo esc_attr($subsection['id']); ?>" 
                                     data-section="<?php echo esc_attr($section['id']); ?>"
                                     data-subsection="<?php echo esc_attr($subsection['id']); ?>" 
                                     style="<?php echo !$show_subsection ? 'display:none;' : ''; ?>">
                                    <div class="bizzplugin-section-header">
                                        <h2><?php echo esc_html($subsection['title']); ?></h2>
                                        <?php if (!empty($subsection['description'])) : ?>
                                            <p class="bizzplugin-section-desc"><?php echo esc_html($subsection['description']); ?></p>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="bizzplugin-section-content">
                                        <?php 
                                        if (!empty($subsection['fields'])) {
                                            $this->render_fields($subsection['fields'], $options);
                                        }
                                        ?>
                                    </div>
                                    
                                    <?php if ($show_subsection_reset) : ?>
                                    <div class="bizzplugin-section-footer">
                                        <button type="button" class="button bizzplugin-reset-section" data-section="<?php echo esc_attr($section['id']); ?>" data-subsection="<?php echo esc_attr($subsection['id']); ?>">
                                            Reset Section
                                        </button>
                                    </div>
                                    <?php endif; ?>
                                </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        <?php endforeach; ?>
                        
                        <?php if ($this->show_export_import) : ?>
                        <!-- Export/Import Section -->
                        <?php $this->render_export_import_section(); ?>
                        <?php endif; ?>
                        
                        <?php if ($this->show_api) : ?>
                        <!-- API Section -->
                        <?php $this->render_api_section(); ?>
                        <?php endif; ?>
                    </form>
                    
                    <!-- Footer with Buttons -->
                    <div class="bizzplugin-footer">
                        <button type="button" class="button bizzplugin-reset-all">
                            RESET ALL
                        </button>
                        <button type="button" id="bizzplugin-save-options" class="button button-primary button-large">
                            SAVE
                        </button>
                    </div>

                    <?php do_action('bizzplugin_content_after', $this->id); ?>
                </div>
                
                <!-- Right Sidebar -->
                <div class="bizzplugin-sidebar" <?php echo esc_attr($setup_wise_hide_style); ?>>
                    <?php do_action('bizzplugin_sidebar_top', $this->id); ?>
                    
                    <!-- Resources Box -->
                    <?php if (!empty($panel_config['resources'])) : ?>
                    <div class="bizzplugin-sidebar-box bizzplugin-resources-box">
                        
                        <ul class="bizzplugin-resource-list">
                            <?php foreach ($panel_config['resources'] as $resource) : ?>
                            <li class="bizzplugin-resource-item">
                                <a href="<?php echo esc_url($resource['url']); ?>" target="_blank" rel="noopener noreferrer">
                                    <span class="dashicons <?php echo esc_attr($resource['icon']); ?>"></span>
                                    <?php echo esc_html($resource['title']); ?>
                                </a>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <?php do_action('bizzplugin_sidebar_middle', $this->id); ?>

                    <!-- Recommended Plugins Section / user render plugin -->
                    <?php
                    if (! empty($panel_config['recommended_plugins'])) {
                        $plugins = $panel_config['recommended_plugins'] ?? array();
                        $this->sidebar_recommended_plugins($plugins);
                    }
                    ?>
                    
                    <?php do_action('bizzplugin_sidebar_bottom', $this->id); ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render topbar
     * 
     * Renders the topbar with left and right links.
     */
    private function render_topbar() {
        if (!$this->is_topbar_enabled()) {
            return;
        }

        $topbar = $this->get_topbar();
        $logo = isset($topbar['logo']) ? $topbar['logo'] : '';
        $left_links = isset($topbar['left_links']) ? $topbar['left_links'] : array();
        $right_links = isset($topbar['right_links']) ? $topbar['right_links'] : array();

        // Don't render if no logo and no links
        if (empty($logo) && empty($left_links) && empty($right_links)) {
            return;
        }

        $setup_wizard = $this->get_setup_wizard();
        $setup_wise_hide_style = isset( $setup_wizard->is_active ) && $setup_wizard->is_active ? 'style=display:none;' : '';
        ?>
        <div class="bizzplugin-topbar" <?php echo esc_attr($setup_wise_hide_style); ?>>
            <div class="bizzplugin-topbar-inner">
                <div class="bizzplugin-topbar-left">
                    <?php if (!empty($logo)) : ?>
                        <div class="bizzplugin-topbar-logo">
                            <img src="<?php echo esc_url($logo); ?>" alt="<?php echo esc_attr($this->title); ?>" />
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($left_links)) : ?>
                        <nav class="bizzplugin-topbar-menu">
                            <?php foreach ($left_links as $link) : ?>
                                <?php $this->render_topbar_menu_link($link); ?>
                            <?php endforeach; ?>
                        </nav>
                    <?php endif; ?>
                </div>
                <div class="bizzplugin-topbar-right">
                    <?php foreach ($right_links as $link) : ?>
                        <?php $this->render_topbar_button_link($link); ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render a menu-style topbar link (for left side)
     * 
     * @param array $link Link configuration
     */
    private function render_topbar_menu_link($link) {
        $title = isset($link['title']) ? $link['title'] : '';
        $url = isset($link['url']) ? $link['url'] : '#';
        $icon = isset($link['icon']) ? $link['icon'] : '';
        $target = isset($link['target']) ? $link['target'] : '_blank';

        if (empty($title)) {
            return;
        }
        ?>
        <a href="<?php echo esc_url($url); ?>" class="bizzplugin-topbar-menu-link" target="<?php echo esc_attr($target); ?>">
            <?php if (!empty($icon)) : ?>
                <span class="dashicons <?php echo esc_attr($icon); ?>"></span>
            <?php endif; ?>
            <span><?php echo esc_html($title); ?></span>
        </a>
        <?php
    }

    /**
     * Render a button-style topbar link (for right side)
     * 
     * @param array $link Link configuration
     */
    private function render_topbar_button_link($link) {
        $title = isset($link['title']) ? $link['title'] : '';
        $url = isset($link['url']) ? $link['url'] : '#';
        $icon = isset($link['icon']) ? $link['icon'] : '';
        $target = isset($link['target']) ? $link['target'] : '_blank';
        $class = isset($link['class']) ? $link['class'] : '';

        if (empty($title)) {
            return;
        }

        $link_class = 'bizzplugin-topbar-btn';
        if (!empty($class)) {
            $link_class .= ' ' . $class;
        }
        ?>
        <a href="<?php echo esc_url($url); ?>" class="<?php echo esc_attr($link_class); ?>" target="<?php echo esc_attr($target); ?>">
            <?php if (!empty($icon)) : ?>
                <span class="dashicons <?php echo esc_attr($icon); ?>"></span>
            <?php endif; ?>
            <span><?php echo esc_html($title); ?></span>
        </a>
        <?php
    }
    
    /**
     * Render subsection
     */
    private function render_subsection($subsection, $options) {
        ?>
        <div class="bizzplugin-subsection" id="subsection-<?php echo esc_attr($subsection['id']); ?>" data-subsection="<?php echo esc_attr($subsection['id']); ?>">
            <h3 class="bizzplugin-subsection-title"><?php echo esc_html($subsection['title']); ?></h3>
            <?php if (!empty($subsection['description'])) : ?>
                <p class="bizzplugin-subsection-desc"><?php echo esc_html($subsection['description']); ?></p>
            <?php endif; ?>
            
            <div class="bizzplugin-subsection-fields">
                <?php
                if (!empty($subsection['fields'])) {
                    $this->render_fields($subsection['fields'], $options);
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render fields
     */
    private function render_fields($fields, $options) {
        foreach ($fields as $field) {
            $this->render_field($field, $options);
        }
    }
    
    /**
     * Render single field
     */
    private function render_field($field, $options) {
        $field_id = $field['id'];
        $field_type = isset($field['type']) ? $field['type'] : 'text';
        $field_title = isset($field['title']) ? $field['title'] : '';
        $field_desc = isset($field['description']) ? $field['description'] : '';
        $field_default = isset($field['default']) ? $field['default'] : '';
        $field_value = isset($options[$field_id]) ? $options[$field_id] : $field_default;
        $field_class = isset($field['class']) ? $field['class'] : '';
        $field_premium = isset($field['premium']) ? $field['premium'] : false;
        $is_premium_active = $this->is_premium();
        
        $search_extra = array(
            $field_value, $field_default,$is_premium_active ? 'premium' : 'free'
        );
        // Build search keywords for this field using helper method
        $search_keywords = $this->build_search_keywords($field_title, $field_desc, $field_id, $search_extra);
        
        // Check for conditional field
        $dependency = isset($field['dependency']) ? $field['dependency'] : array();
        $dependency_attr = '';
        if (!empty($dependency) && is_array($dependency) && isset($dependency['field']) && isset($dependency['value'])) {
            $dependency_attr = sprintf(
                'data-dependency="%s" data-dependency-value="%s"',
                esc_attr($dependency['field']),
                esc_attr($dependency['value'])
            );
        }
        
        // Premium field styling
        $premium_class = '';
        $is_disabled = false;
        if ($field_premium && !$is_premium_active) {
            $premium_class = 'bizzplugin-field-premium-locked';
            $is_disabled = true;
        }
        
        $setup_wizard_class = $this->is_setup_wizard && isset( $field['setup_wizard'] ) ? 'setup-wizard' : '';
        ?>
        <div class="bizzplugin-field bizzplugin-field-<?php echo esc_attr($field_type); ?> <?php echo esc_attr($field_class); ?> <?php echo esc_attr($premium_class); ?> <?php echo esc_attr($setup_wizard_class); ?>" <?php echo $dependency_attr; ?> 
            id="bizzplugin-field-<?php echo esc_attr($field_id); ?>" 
            data-setup_wizard="<?php echo $this->is_setup_wizard && isset( $field['setup_wizard'] ) ? '1' : '0'; ?>"
            data-field-id="<?php echo esc_attr($field_id); ?>" 
            data-field-type="<?php echo esc_attr($field_type); ?>"
            data-search-keywords="<?php echo esc_attr($search_keywords); ?>">
            <div class="bizzplugin-field-header">
                <label for="<?php echo esc_attr($field_id); ?>" class="bizzplugin-field-title">
                    <?php echo esc_html($field_title); ?>
                    <?php if ($field_premium && !$is_premium_active) : ?>
                        <span class="bizzplugin-premium-badge">Blocked</span>
                    <?php endif; ?>
                </label>
            </div>
            
            <div class="bizzplugin-field-content">
                <?php
                // Allow custom field rendering
                if (has_action('bizzplugin_render_field_' . $field_type)) {
                    do_action('bizzplugin_render_field_' . $field_type, $field, $field_value, $is_disabled);
                } else {
                    // Render field based on type using shared BizzPlugin_Render_Field class
                    switch ($field_type) {
                        case 'text':
                        case 'email':
                        case 'url':
                        case 'number':
                        case 'password':
                            BizzPlugin_Render_Field::input($field, $field_value, null, $field_type, $is_disabled);
                            break;
                        case 'textarea':
                            BizzPlugin_Render_Field::textarea($field, $field_value, null, $is_disabled);
                            break;
                        case 'select':
                            BizzPlugin_Render_Field::select($field, $field_value, null, $is_disabled);
                            break;
                        case 'multi_select':
                            BizzPlugin_Render_Field::multi_select($field, $field_value, null, $is_disabled);
                            break;
                        case 'checkbox':
                            BizzPlugin_Render_Field::checkbox($field, $field_value, null, $is_disabled);
                            break;
                        case 'checkbox_group':
                            BizzPlugin_Render_Field::checkbox_group($field, $field_value, null, $is_disabled);
                            break;
                        case 'radio':
                            BizzPlugin_Render_Field::radio($field, $field_value, null, $is_disabled);
                            break;
                        case 'toggle':
                        case 'on_off':
                        case 'switch':
                            BizzPlugin_Render_Field::switch_toggle($field, $field_value, null, $is_disabled);
                            break;
                        case 'color':
                            BizzPlugin_Render_Field::color($field, $field_value, null, $is_disabled);
                            break;
                        case 'alpha_color':
                            BizzPlugin_Render_Field::alpha_color($field, $field_value, null, $is_disabled);
                            break;
                        case 'date':
                            BizzPlugin_Render_Field::date($field, $field_value, null, $is_disabled);
                            break;
                        case 'image':
                            BizzPlugin_Render_Field::image($field, $field_value, null, $is_disabled);
                            break;
                        case 'file':
                            BizzPlugin_Render_Field::file($field, $field_value, null, $is_disabled);
                            break;
                        case 'image_select':
                            BizzPlugin_Render_Field::image_select($field, $field_value, null, $is_disabled);
                            break;
                        case 'option_select':
                            BizzPlugin_Render_Field::option_select($field, $field_value, null, $is_disabled);
                            break;
                        case 'post_select':
                            BizzPlugin_Render_Field::post_select($field, $field_value, null, $is_disabled);
                            break;
                        case 'html':
                            BizzPlugin_Render_Field::html($field);
                            break;
                        case 'callback':
                            $this->render_callback_field($field, $field_value, $is_disabled);
                            break;
                        case 'plugins':
                            BizzPlugin_Render_Field::plugins($field);
                            break;
                        case 'slider':
                        case 'range':
                            BizzPlugin_Render_Field::slider($field, $field_value, null, $is_disabled);
                            break;
                        case 'repeater':
                            BizzPlugin_Render_Field::repeater($field, $field_value, null, $is_disabled);
                            break;
                        case 'single_repeater':
                            BizzPlugin_Render_Field::single_repeater($field, $field_value, null, $is_disabled);
                            break;
                        default:
                            // Allow other plugins to add custom field types
                            do_action('bizzplugin_render_custom_field', $field, $field_value, $is_disabled);
                            break;
                    }
                }
                
                // Check for callback
                if (isset($field['callback']) && is_callable($field['callback'])) {

                    /**
                     * callback for Each fieldd
                     * 
                     * @param array $field
                     * @param mixed $field_value
                     */
                    call_user_func($field['callback'], $field, $field_value, $options);
                }
                ?>
                
                <?php if (!empty($field_desc)) : ?>
                    <p class="bizzplugin-field-desc"><?php echo wp_kses_post($field_desc); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render callback field
     */
    private function render_callback_field($field, $value, $disabled = false) {
        if (isset($field['render_callback']) && is_callable($field['render_callback'])) {
            call_user_func($field['render_callback'], $field, $value, $disabled);
        }
    }
    
    /**
     * Render plugins field for recommended plugins
     */
    private function render_plugins_field($field) {
        
        $plugins = isset($field['plugins']) ? $field['plugins'] : array();
        
        if (empty($plugins)) {
            return;
        }
        
        // Include plugin functions if not available
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $installed_plugins = get_plugins();



        ?>
        <div class="bizzplugin-plugins-field" data-field-id="<?php echo esc_attr($field['id'] ?? ''); ?>">
            <div class="bizzplugin-plugins-grid">
                <?php foreach ($plugins as $plugin) : 
                    $slug = isset($plugin['slug']) ? $plugin['slug'] : '';
                    $name = isset($plugin['name']) ? $plugin['name'] : $slug;
                    $description = isset($plugin['description']) ? $plugin['description'] : '';
                    $thumbnail = isset($plugin['thumbnail']) ? $plugin['thumbnail'] : '';
                    $plugin_file = isset($plugin['file']) ? $plugin['file'] : $slug . '/' . $slug . '.php';
                    $author = isset($plugin['author']) ? $plugin['author'] : '';
                    $url = isset($plugin['url']) ? $plugin['url'] : 'https://wordpress.org/plugins/' . $slug . '/';
                    
                    // Check plugin status - first check if installed, then if active
                    $is_installed = array_key_exists($plugin_file, $installed_plugins);
                    $is_active = $is_installed && is_plugin_active($plugin_file);
                    
                    // Determine status
                    if ($is_active) {
                        $status = 'active';
                        $status_text = 'Active';
                        $status_class = 'bizzplugin-plugin-status-active';
                    } elseif ($is_installed) {
                        $status = 'installed';
                        $status_text = 'Installed (Inactive)';
                        $status_class = 'bizzplugin-plugin-status-installed';
                    } else {
                        $status = 'not_installed';
                        $status_text = 'Not Installed';
                        $status_class = 'bizzplugin-plugin-status-not-installed';
                    }
                ?>
                <div class="bizzplugin-plugin-card" data-slug="<?php echo esc_attr($slug); ?>" data-file="<?php echo esc_attr($plugin_file); ?>">
                    <?php if (!empty($thumbnail)) : ?>
                    <div class="bizzplugin-plugin-thumbnail">
                        <img src="<?php echo esc_url($thumbnail); ?>" alt="<?php echo esc_attr($name); ?>">
                    </div>
                    <?php endif; ?>
                    
                    <div class="bizzplugin-plugin-info">
                        <h4 class="bizzplugin-plugin-name">
                            <?php if (!empty($url)) : ?>
                                <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer">
                                    <?php echo esc_html($name); ?>
                                </a>
                            <?php else : ?>
                                <?php echo esc_html($name); ?>
                            <?php endif; ?>
                        </h4>
                        
                        <?php if (!empty($author)) : ?>
                        <p class="bizzplugin-plugin-author">
                            By <?php echo esc_html($author); ?>
                        </p>
                        <?php endif; ?>
                        
                        <?php if (!empty($description)) : ?>
                        <p class="bizzplugin-plugin-description"><?php echo esc_html($description); ?></p>
                        <?php endif; ?>
                        
                        <div class="bizzplugin-plugin-footer">
                            <span class="bizzplugin-plugin-status <?php echo esc_attr($status_class); ?>">
                                <?php echo esc_html($status_text); ?>
                            </span>
                            
                            <div class="bizzplugin-plugin-actions">
                                <?php if ($status === 'not_installed') : ?>
                                    <button type="button" class="button button-primary bizzplugin-install-plugin" 
                                            data-slug="<?php echo esc_attr($slug); ?>" 
                                            data-file="<?php echo esc_attr($plugin_file); ?>">
                                        <span class="dashicons dashicons-download"></span>
                                        Install
                                    </button>
                                <?php elseif ($status === 'installed') : ?>
                                    <button type="button" class="button button-primary bizzplugin-activate-plugin" 
                                            data-slug="<?php echo esc_attr($slug); ?>" 
                                            data-file="<?php echo esc_attr($plugin_file); ?>">
                                        <span class="dashicons dashicons-yes-alt"></span>
                                        Activate
                                    </button>
                                <?php else : ?>
                                    <span class="bizzplugin-plugin-activated">
                                        <span class="dashicons dashicons-yes"></span>
                                        Activated
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render plugins field for recommended plugins
     */
    private function sidebar_recommended_plugins($plugins) {
        if (empty($plugins)) {
            return;
        }
        
        // Include plugin functions if not available
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        $installed_plugins = get_plugins();
        ?>
        <div class="sidebar-recommended-plugins-area bizzplugin-plugins-field">
            <div class="bizzplugin-plugins-grid">
                <?php foreach ($plugins as $plugin) : 
                    $slug = isset($plugin['slug']) ? $plugin['slug'] : '';
                    $name = isset($plugin['name']) ? $plugin['name'] : $slug;
                    $description = isset($plugin['description']) ? $plugin['description'] : '';
                    $thumbnail = isset($plugin['thumbnail']) ? $plugin['thumbnail'] : '';
                    $plugin_file = isset($plugin['file']) ? $plugin['file'] : $slug . '/' . $slug . '.php';
                    $author = isset($plugin['author']) ? $plugin['author'] : '';
                    $url = isset($plugin['url']) ? $plugin['url'] : 'https://wordpress.org/plugins/' . $slug . '/';
                    
                    // Check plugin status - first check if installed, then if active
                    $is_installed = array_key_exists($plugin_file, $installed_plugins);
                    $is_active = $is_installed && is_plugin_active($plugin_file);
                    
                    // Determine status
                    if ($is_active) {
                        $status = 'active';
                        $status_text = 'Active';
                        $status_class = 'bizzplugin-plugin-status-active';
                    } elseif ($is_installed) {
                        $status = 'installed';
                        $status_text = 'Installed (Inactive)';
                        $status_class = 'bizzplugin-plugin-status-installed';
                    } else {
                        $status = 'not_installed';
                        $status_text = 'Not Installed';
                        $status_class = 'bizzplugin-plugin-status-not-installed';
                    }
                ?>
                <div class="bizzplugin-plugin-card" data-slug="<?php echo esc_attr($slug); ?>" data-file="<?php echo esc_attr($plugin_file); ?>">
                    <?php if (!empty($thumbnail)) : ?>
                    <div class="bizzplugin-plugin-thumbnail">
                        <img src="<?php echo esc_url($thumbnail); ?>" alt="<?php echo esc_attr($name); ?>">
                    </div>
                    <?php endif; ?>
                    
                    <div class="bizzplugin-plugin-info">
                        <h4 class="bizzplugin-plugin-name">
                            <?php if (!empty($url)) : ?>
                                <a href="<?php echo esc_url($url); ?>" target="_blank" rel="noopener noreferrer">
                                    <?php echo esc_html($name); ?>
                                </a>
                            <?php else : ?>
                                <?php echo esc_html($name); ?>
                            <?php endif; ?>
                        </h4>
                        
                        <?php if (!empty($author)) : ?>
                        <p class="bizzplugin-plugin-author">
                            <?php echo 'By ' . esc_html($author); ?>
                        </p>
                        <?php endif; ?>
                        
                        <?php if (!empty($description)) : ?>
                        <p class="bizzplugin-plugin-description"><?php echo esc_html($description); ?></p>
                        <?php endif; ?>
                        
                        <div class="bizzplugin-plugin-footer">
                            <span class="bizzplugin-plugin-status <?php echo esc_attr($status_class); ?>">
                                <?php echo esc_html($status_text); ?>
                            </span>
                            
                            <div class="bizzplugin-plugin-actions">
                                <?php if ($status === 'not_installed') : ?>
                                    <button type="button" class="button button-primary bizzplugin-install-plugin" 
                                            data-slug="<?php echo esc_attr($slug); ?>" 
                                            data-file="<?php echo esc_attr($plugin_file); ?>">
                                        <span class="dashicons dashicons-download"></span>
                                        Install
                                    </button>
                                <?php elseif ($status === 'installed') : ?>
                                    <button type="button" class="button button-primary bizzplugin-activate-plugin" 
                                            data-slug="<?php echo esc_attr($slug); ?>" 
                                            data-file="<?php echo esc_attr($plugin_file); ?>">
                                        <span class="dashicons dashicons-yes-alt"></span>
                                        Activate
                                    </button>
                                <?php else : ?>
                                    <span class="bizzplugin-plugin-activated">
                                        <span class="dashicons dashicons-yes"></span>
                                        Activated
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Export/Import section
     */
    private function render_export_import_section() {
        $show = ( $this->current_section === 'export_import' ) ? '' : 'display:none;';
        ?>
        <div class="bizzplugin-section bizzplugin-export-import-section" id="section-export_import" data-section="export_import" style="<?php echo esc_attr( $show ); ?>">
            <div class="bizzplugin-section-header">
                <h2>Export & Import Settings</h2>
                <p class="bizzplugin-section-desc">Export your current settings to a JSON file or import settings from a previously exported file.</p>
            </div>
            
            <div class="bizzplugin-section-content">
                <!-- Export Card -->
                <div class="bizzplugin-api-card">
                    <div class="bizzplugin-api-card-header">
                        <span class="dashicons dashicons-database-export"></span>
                        <h3>Export Settings</h3>
                    </div>
                    <div class="bizzplugin-api-card-body">
                        <p class="bizzplugin-field-desc" style="margin-bottom: 15px;">
                            Export your current settings to a JSON file. This file can be used to restore your settings later or transfer them to another site.
                        </p>
                        <button type="button" id="bizzplugin-export-options" class="button button-primary">
                            <span class="dashicons dashicons-download" style="margin-right: 5px; margin-top: 3px;"></span>
                            Export Settings
                        </button>
                    </div>
                </div>
                
                <!-- Import Card -->
                <div class="bizzplugin-api-card">
                    <div class="bizzplugin-api-card-header">
                        <span class="dashicons dashicons-database-import"></span>
                        <h3>Import Settings</h3>
                    </div>
                    <div class="bizzplugin-api-card-body">
                        <p class="bizzplugin-field-desc" style="margin-bottom: 15px;">
                            Import settings from a previously exported JSON file. After importing, click the Save button to apply the imported settings.
                        </p>
                        <input type="file" id="bizzplugin-import-file" accept=".json" style="display: none;" />
                        <button type="button" id="bizzplugin-import-trigger" class="button button-secondary">
                            <span class="dashicons dashicons-upload" style="margin-right: 5px; margin-top: 3px;"></span>
                            Import Settings
                        </button>
                        <p class="bizzplugin-api-note" style="margin-top: 15px;">
                            <span class="dashicons dashicons-warning"></span>
                            Warning: Importing will overwrite your current settings. Make sure to export your current settings first if you want to keep them.
                        </p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render API section
     */
    private function render_api_section() {
        $rest_url = rest_url( $this->get_route_namespace() . '/' . $this->option_name);
        $webhook_url = get_option('bizzplugin_webhook_' . $this->option_name, '');
        $webhook_secret = get_option('bizzplugin_webhook_secret_' . $this->option_name, '');
        $api_key = BizzPlugin_API_Handler::get_api_key($this->id);

        $show = ( $this->current_section === 'api' ) ? '' : 'display:none;';
        ?>
        <div class="bizzplugin-section bizzplugin-api-section" id="section-api" data-section="api" style="<?php echo esc_attr( $show ); ?>">
            <div class="bizzplugin-section-header">
                <h2>API & Webhook Settings</h2>
                <p class="bizzplugin-section-desc">Configure API access and webhook notifications for this settings panel.</p>
            </div>
            
            <div class="bizzplugin-section-content">
                <!-- API Key Card -->
                <div class="bizzplugin-api-card">
                    <div class="bizzplugin-api-card-header">
                        <span class="dashicons dashicons-admin-network"></span>
                        <h3>API Key Authentication</h3>
                    </div>
                    <div class="bizzplugin-api-card-body">
                        <p class="bizzplugin-field-desc" style="margin-bottom: 15px;">
                            Generate an API key to authenticate REST API requests. This key is specific to this panel only. Include the API key in the <code>x-api-key</code> header of your requests.
                        </p>
                        
                        <div class="bizzplugin-api-key-field">
                            <label class="bizzplugin-field-title">
                                API Key
                            </label>
                            <div class="bizzplugin-api-key-display">
                                <?php if (!empty($api_key)) : ?>
                                <div class="bizzplugin-api-key-row">
                                    <code class="bizzplugin-api-key-code" id="bizzplugin-api-key-value"><?php echo esc_html($api_key); ?></code>
                                    <button type="button" class="button bizzplugin-copy-btn" data-copy="<?php echo esc_attr($api_key); ?>">
                                        <span class="dashicons dashicons-admin-page"></span>
                                    </button>
                                </div>
                                <?php else : ?>
                                <div class="bizzplugin-api-key-row" id="bizzplugin-no-api-key">
                                    <span class="bizzplugin-no-key-text">
                                        No API key generated yet.
                                    </span>
                                </div>
                                <?php endif; ?>
                            </div>
                            <div style="margin-top: 15px;">
                                <button type="button" id="bizzplugin-generate-api-key" class="button button-primary">
                                    <span class="dashicons dashicons-update" style="margin-right: 5px; margin-top: 3px;"></span>
                                    <?php echo empty($api_key) ? 'Generate API Key' : 'Regenerate API Key'; ?>
                                </button>
                            </div>
                        </div>
                        
                        <p class="bizzplugin-api-note">
                            <span class="dashicons dashicons-warning"></span>
                            Keep your API key secure. Regenerating will invalidate the previous key for this panel only.
                        </p>
                    </div>
                </div>
                
                <!-- REST API Info Card -->
                <div class="bizzplugin-api-card">
                    <div class="bizzplugin-api-card-header">
                        <span class="dashicons dashicons-rest-api"></span>
                        <h3>REST API Endpoints</h3>
                    </div>
                    <div class="bizzplugin-api-card-body">
                        <div class="bizzplugin-api-endpoint-row">
                            <span class="bizzplugin-method bizzplugin-method-get">GET</span>
                            <code class="bizzplugin-endpoint-url"><?php echo esc_url($rest_url); ?></code>
                            <button type="button" class="button bizzplugin-copy-btn" data-copy="<?php echo esc_attr($rest_url); ?>">
                                <span class="dashicons dashicons-admin-page"></span>
                            </button>
                        </div>
                        <div class="bizzplugin-api-endpoint-row">
                            <span class="bizzplugin-method bizzplugin-method-get">GET</span>
                            <code class="bizzplugin-endpoint-url"><?php echo esc_url($rest_url); ?>/{field_id}</code>
                            <button type="button" class="button bizzplugin-copy-btn" data-copy="<?php echo esc_attr($rest_url); ?>">
                                <span class="dashicons dashicons-admin-page"></span>
                            </button>
                        </div>
                        <div class="bizzplugin-api-endpoint-row">
                            <span class="bizzplugin-method bizzplugin-method-post">POST</span>
                            <code class="bizzplugin-endpoint-url"><?php echo esc_url($rest_url); ?></code>
                            <button type="button" class="button bizzplugin-copy-btn" data-copy="<?php echo esc_attr($rest_url); ?>">
                                <span class="dashicons dashicons-admin-page"></span>
                            </button>
                        </div>
                        <div class="bizzplugin-api-endpoint-row">
                            <span class="bizzplugin-method bizzplugin-method-post">POST</span>
                            <code class="bizzplugin-endpoint-url"><?php echo esc_url($rest_url); ?>/{field_id}</code>
                            <button type="button" class="button bizzplugin-copy-btn" data-copy="<?php echo esc_attr($rest_url); ?>">
                                <span class="dashicons dashicons-admin-page"></span>
                            </button>
                        </div>
                        <p class="bizzplugin-api-note">
                            <span class="dashicons dashicons-info-outline"></span>
                            Authentication required: Include x-api-key header with your API key in all requests.
                        </p>
                    </div>
                </div>
                
                <!-- Webhook Settings Card -->
                <div class="bizzplugin-api-card bizzplugin-webhook-card">
                    <div class="bizzplugin-api-card-header">
                        <span class="dashicons dashicons-randomize"></span>
                        <h3>Webhook Configuration</h3>
                    </div>
                    <div class="bizzplugin-api-card-body">
                        <?php $this->render_advanced_webhooks_ui(); ?>
                        
                        <?php if (!empty($webhook_secret)) : ?>
                        <div class="bizzplugin-webhook-secret" style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #ddd;">
                            <label class="bizzplugin-field-title">
                                Webhook Secret (for signature verification)
                                <span class="bizzplugin-field-info dashicons dashicons-info-outline" title="Use this secret to verify webhook signatures in X-BizzPlugin-Signature header"></span>
                            </label>
                            <div class="bizzplugin-secret-display">
                                <code class="bizzplugin-secret-code"><?php echo esc_html(substr($webhook_secret, 0, 8) . '...' . substr($webhook_secret, -8)); ?></code>
                                <button type="button" class="button bizzplugin-copy-btn" data-copy="<?php echo esc_attr($webhook_secret); ?>">
                                    <span class="dashicons dashicons-admin-page"></span>
                                </button>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <!-- Webhook Test Response -->
                        <div id="bizzplugin-webhook-response" class="bizzplugin-webhook-response" style="display:none;">
                            <h4>Test Response</h4>
                            <pre class="bizzplugin-code-block"></pre>
                        </div>
                    </div>
                </div>
                
                <!-- API Documentation Card -->
                <div class="bizzplugin-api-card bizzplugin-api-docs-card">
                    <div class="bizzplugin-api-card-header">
                        <span class="dashicons dashicons-media-document"></span>
                        <h3>Quick Reference</h3>
                    </div>
                    <div class="bizzplugin-api-card-body">
                        <div class="bizzplugin-docs-tabs">
                            <button type="button" class="bizzplugin-docs-tab active" data-tab="get">GET Request</button>
                            <button type="button" class="bizzplugin-docs-tab" data-tab="post">POST Request</button>
                            <button type="button" class="bizzplugin-docs-tab" data-tab="webhook">Webhook Payload</button>
                        </div>
                        
                        <div class="bizzplugin-docs-content active" data-tab-content="get">
                            <pre class="bizzplugin-code-block">curl -X GET "<?php echo esc_url($rest_url); ?>" \
  -H "x-api-key: YOUR_API_KEY"</pre><br>
  <pre class="bizzplugin-code-block">curl -X GET "<?php echo esc_url($rest_url); ?>/{field_id}" \
  -H "x-api-key: YOUR_API_KEY"</pre>
                        </div>
                        
                        <div class="bizzplugin-docs-content" data-tab-content="post">
                            <pre class="bizzplugin-code-block">curl -X POST "<?php echo esc_url($rest_url); ?>" \
  -H "x-api-key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{"field_name": "new_value"}'</pre>
                        <br>
                        <pre class="bizzplugin-code-block">curl -X POST "<?php echo esc_url($rest_url); ?>/{field_id}" \
  -H "x-api-key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{"value": "new_value"}'</pre>
                        </div>
                        
                        <div class="bizzplugin-docs-content" data-tab-content="webhook">
                            <pre class="bizzplugin-code-block">{
  "event": "settings_saved",
  "option_name": "<?php echo esc_js($this->option_name); ?>",
  "timestamp": "2024-01-01T12:00:00Z",
  "site_url": "<?php echo esc_url(get_site_url()); ?>",
  "data": { "field_name": "value", ... },
  "changed_fields": { "field_name": { "old": "...", "new": "..." } }
}</pre>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render advanced webhooks UI
     * 
     * Displays multiple webhook configuration with custom JSON payload,
     * authentication options, and available shortcodes.
     */
    private function render_advanced_webhooks_ui() {
        // Get saved webhooks
        $webhooks = get_option('bizzplugin_webhooks_' . $this->option_name, array());
        
        // Get legacy webhook URL for backward compatibility
        $legacy_url = get_option('bizzplugin_webhook_' . $this->option_name, '');
        if (!empty($legacy_url) && empty($webhooks)) {
            $webhooks = array(
                array(
                    'url' => $legacy_url,
                    'enabled' => true,
                    'auth_type' => 'none',
                    'custom_payload' => '',
                ),
            );
        }
        
        // Get available shortcodes
        $shortcodes = BizzPlugin_Webhook_Handler::get_available_shortcodes();
        ?>
        <div class="bizzplugin-webhooks-container" data-option-name="<?php echo esc_attr($this->option_name); ?>">
            <p class="bizzplugin-field-desc" style="margin-bottom: 15px;">
                Configure multiple webhooks to receive POST notifications when settings are saved. Each webhook can have its own URL, authentication, and custom payload format.
            </p>
            
            <div class="bizzplugin-webhooks-list">
                <?php 
                if (!empty($webhooks)) {
                    foreach ($webhooks as $index => $webhook) {
                        $this->render_webhook_item($index, $webhook);
                    }
                }
                ?>
            </div>
            
            <div class="bizzplugin-webhooks-footer" style="margin-top: 15px;">
                <button type="button" class="button button-primary bizzplugin-add-webhook">
                    <span class="dashicons dashicons-plus-alt2" style="margin-top: 3px;"></span>
                    Add Webhook
                </button>
            </div>
            
            <!-- Webhook Template (hidden) -->
            <div class="bizzplugin-webhook-template" style="display: none !important;" aria-hidden="true">
                <?php $this->render_webhook_item('{{INDEX}}', array(), true); ?>
            </div>
            
            <!-- Available Shortcodes Reference -->
            <div class="bizzplugin-shortcodes-reference" style="margin-top: 20px; padding: 15px; background: #f0f0f1; border-radius: 4px;">
                <h4 style="margin-top: 0; margin-bottom: 10px;">
                    <span class="dashicons dashicons-shortcode" style="margin-right: 5px;"></span>
                    Available Shortcodes for Custom Payload
                </h4>
                <div class="bizzplugin-shortcodes-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(280px, 1fr)); gap: 8px;">
                    <?php foreach ($shortcodes as $shortcode => $description) : ?>
                        <div class="bizzplugin-shortcode-item" style="display: flex; align-items: center; gap: 8px;">
                            <code style="background: #fff; padding: 2px 6px; border-radius: 3px; font-size: 12px;"><?php echo esc_html($shortcode); ?></code>
                            <span style="color: #646970; font-size: 12px;"><?php echo esc_html($description); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render a single webhook configuration item
     * 
     * @param int|string $index   Item index (number or {{INDEX}} for template)
     * @param array      $webhook Webhook configuration
     * @param bool       $is_template Whether this is a template render
     */
    private function render_webhook_item($index, $webhook = array(), $is_template = false) {
        $defaults = array(
            'url' => '',
            'enabled' => true,
            'auth_type' => 'none',
            'auth_token' => '',
            'auth_username' => '',
            'auth_password' => '',
            'auth_header_name' => 'X-API-Key',
            'auth_api_key' => '',
            'custom_payload' => '',
        );
        $webhook = wp_parse_args($webhook, $defaults);
        
        $name_prefix = 'bizzplugin_webhooks[' . $index . ']';
        $id_prefix = 'bizzplugin_webhook_' . $index;
        ?>
        <div class="bizzplugin-webhook-item" data-index="<?php echo esc_attr($index); ?>" style="border: 1px solid #c3c4c7; border-radius: 4px; padding: 15px; margin-bottom: 15px; background: #fff;">
            <div class="bizzplugin-webhook-item-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                <div style="display: flex; align-items: center; gap: 10px;">
                    <span class="dashicons dashicons-move" style="cursor: move; color: #c3c4c7;" title="Drag to reorder"></span>
                    <strong><?php 
                        if ($is_template) {
                            echo 'Webhook #' . '{{DISPLAY_INDEX}}';
                        } else {
                            echo sprintf('Webhook #%d', $index + 1);
                        }
                    ?></strong>
                </div>
                <div style="display: flex; align-items: center; gap: 10px;">
                    <label class="bizzplugin-webhook-enable" style="display: flex; align-items: center; gap: 5px;">
                        <input 
                            type="checkbox" 
                            name="<?php echo esc_attr($name_prefix); ?>[enabled]" 
                            value="1" 
                            <?php checked($webhook['enabled'], true); ?>
                        />
                        Enabled
                    </label>
                    <button type="button" class="button bizzplugin-remove-webhook" title="Remove">
                        <span class="dashicons dashicons-trash" style="margin-top: 3px;"></span>
                    </button>
                </div>
            </div>
            
            <div class="bizzplugin-webhook-item-body">
                <!-- URL Field -->
                <div class="bizzplugin-webhook-field-row" style="margin-bottom: 15px;">
                    <label for="<?php echo esc_attr($id_prefix); ?>_url" class="bizzplugin-field-title" style="display: block; margin-bottom: 5px;">
                        Webhook URL <span style="color: #dc3232;">*</span>
                    </label>
                    <input 
                        type="url" 
                        id="<?php echo esc_attr($id_prefix); ?>_url"
                        name="<?php echo esc_attr($name_prefix); ?>[url]" 
                        value="<?php echo esc_url($webhook['url']); ?>"
                        class="bizzplugin-input"
                        style="width: 100%;"
                        placeholder="https://example.com/webhook"
                        required
                    />
                </div>
                
                <!-- Authentication -->
                <div class="bizzplugin-webhook-field-row" style="margin-bottom: 15px;">
                    <label class="bizzplugin-field-title" style="display: block; margin-bottom: 5px;">
                        Authentication
                    </label>
                    <select 
                        name="<?php echo esc_attr($name_prefix); ?>[auth_type]" 
                        class="bizzplugin-select bizzplugin-webhook-auth-type"
                        style="width: 200px;"
                    >
                        <option value="none" <?php selected($webhook['auth_type'], 'none'); ?>>No Authentication</option>
                        <option value="bearer" <?php selected($webhook['auth_type'], 'bearer'); ?>>Bearer Token</option>
                        <option value="basic" <?php selected($webhook['auth_type'], 'basic'); ?>>Basic Auth</option>
                        <option value="api_key" <?php selected($webhook['auth_type'], 'api_key'); ?>>API Key Header</option>
                    </select>
                    
                    <!-- Bearer Token Fields -->
                    <div class="bizzplugin-auth-fields bizzplugin-auth-bearer" style="margin-top: 10px; <?php echo $webhook['auth_type'] !== 'bearer' ? 'display:none;' : ''; ?>">
                        <input 
                            type="text" 
                            name="<?php echo esc_attr($name_prefix); ?>[auth_token]" 
                            value="<?php echo esc_attr($webhook['auth_token']); ?>"
                            class="bizzplugin-input"
                            style="width: 100%;"
                            placeholder="Enter Bearer Token"
                        />
                    </div>
                    
                    <!-- Basic Auth Fields -->
                    <div class="bizzplugin-auth-fields bizzplugin-auth-basic" style="margin-top: 10px; display: flex; gap: 10px; <?php echo $webhook['auth_type'] !== 'basic' ? 'display:none;' : ''; ?>">
                        <input 
                            type="text" 
                            name="<?php echo esc_attr($name_prefix); ?>[auth_username]" 
                            value="<?php echo esc_attr($webhook['auth_username']); ?>"
                            class="bizzplugin-input"
                            style="flex: 1;"
                            placeholder="Username"
                        />
                        <input 
                            type="password" 
                            name="<?php echo esc_attr($name_prefix); ?>[auth_password]" 
                            value="<?php echo esc_attr($webhook['auth_password']); ?>"
                            class="bizzplugin-input"
                            style="flex: 1;"
                            placeholder="Password"
                        />
                    </div>
                    
                    <!-- API Key Fields -->
                    <div class="bizzplugin-auth-fields bizzplugin-auth-api_key" style="margin-top: 10px; display: flex; gap: 10px; <?php echo $webhook['auth_type'] !== 'api_key' ? 'display:none;' : ''; ?>">
                        <input 
                            type="text" 
                            name="<?php echo esc_attr($name_prefix); ?>[auth_header_name]" 
                            value="<?php echo esc_attr($webhook['auth_header_name']); ?>"
                            class="bizzplugin-input"
                            style="width: 150px;"
                            placeholder="Header Name"
                        />
                        <input 
                            type="text" 
                            name="<?php echo esc_attr($name_prefix); ?>[auth_api_key]" 
                            value="<?php echo esc_attr($webhook['auth_api_key']); ?>"
                            class="bizzplugin-input"
                            style="flex: 1;"
                            placeholder="API Key Value"
                        />
                    </div>
                </div>
                
                <!-- Custom Payload -->
                <div class="bizzplugin-webhook-field-row">
                    <label for="<?php echo esc_attr($id_prefix); ?>_payload" class="bizzplugin-field-title" style="display: block; margin-bottom: 5px;">
                        Custom JSON Payload (Optional)
                        <span class="bizzplugin-field-info dashicons dashicons-info-outline" title="Leave empty to use default payload. Use shortcodes for dynamic values."></span>
                    </label>
                    <textarea 
                        id="<?php echo esc_attr($id_prefix); ?>_payload"
                        name="<?php echo esc_attr($name_prefix); ?>[custom_payload]" 
                        class="bizzplugin-textarea"
                        rows="6"
                        style="width: 100%; font-family: monospace; font-size: 12px;"
                        placeholder='<?php echo esc_attr('{
  "event": "{{event}}",
  "site": "{{site_url}}",
  "data": {{data}},
  "changed": {{changed_fields}}
}'); ?>'
                    ><?php echo esc_textarea($webhook['custom_payload']); ?></textarea>
                    <p class="bizzplugin-field-desc" style="margin-top: 5px;">
                        Enter valid JSON with shortcodes. For JSON objects like {{data}}, do not wrap in quotes.
                    </p>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function __call($name, $arguments)
    {
        error_log( "Undefined method: " . $name . " has been call from " . __CLASS__ );
        return null;
    }
}
