<?php
/**
 * BizzPlugin Options Framework - Metabox Class
 * 
 * Adds meta box support for posts, pages, and custom post types.
 * Uses the same field types and rendering as the options panel.
 * 
 * @package BizzPlugin_Options_Framework
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Metabox Class
 */
class BizzPlugin_Metabox {
    
    /**
     * Metabox ID
     */
    private $id;
    
    /**
     * Metabox title
     */
    private $title;
    
    /**
     * Post types where metabox should appear
     */
    private $post_types = array();
    
    /**
     * Metabox context (normal, side, advanced)
     */
    private $context = 'normal';
    
    /**
     * Metabox priority (high, core, default, low)
     */
    private $priority = 'default';
    
    /**
     * Meta key for storing all meta values (like option_name in panel)
     */
    private $option_name;
    
    /**
     * Required capability to edit
     */
    private $capability = 'edit_posts';
    
    /**
     * Is premium active
     */
    private $is_premium = false;
    
    /**
     * Sections (tabs)
     */
    private $sections = array();
    
    /**
     * Framework instance
     */
    private $framework;
    
    /**
     * Constructor
     * 
     * @param array             $args      Metabox configuration
     * @param BizzPlugin_Framework $framework Framework instance
     */
    public function __construct($args, $framework) {
        $defaults = array(
            'id'          => '',
            'title'       => '',
            'post_types'  => array('post'),
            'context'     => 'normal',
            'priority'    => 'default',
            'option_name'    => '',  // Main meta key for storing all data
            'capability'  => 'edit_posts',
            'is_premium'  => false,
            'sections'    => array(),
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $this->id          = $args['id'];
        $this->title       = $args['title'];
        $this->post_types  = (array) $args['post_types'];
        $this->context     = $args['context'];
        $this->priority    = $args['priority'];
        $this->option_name    = !empty($args['option_name']) ? $args['option_name'] : '_' . $args['id'] . '_meta';
        $this->capability  = $args['capability'];
        $this->is_premium  = $args['is_premium'];
        $this->framework   = $framework;
        
        // Add sections if provided
        if (!empty($args['sections'])) {
            foreach ($args['sections'] as $section) {
                $this->add_section($section);
            }
        }
        
        // Register hooks
        add_action('add_meta_boxes', array($this, 'register_metabox'));
        add_action('save_post', array($this, 'save_meta'), 10, 3);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
    }
    
    /**
     * Get metabox ID
     * 
     * @return string
     */
    public function get_id() {
        return $this->id;
    }
    
    /**
     * Get meta key
     * 
     * @return string
     */
    public function get_meta_key() {
        return $this->option_name;
    }
    
    /**
     * Get post types
     * 
     * @return array
     */
    public function get_post_types() {
        return $this->post_types;
    }
    
    /**
     * Is premium active
     * 
     * @return bool
     */
    public function is_premium() {
        return apply_filters('bizzplugin_metabox_is_premium_' . $this->id, $this->is_premium, $this->id);
    }
    
    /**
     * Add section (tab)
     * 
     * @param array $args Section configuration
     * @return $this For method chaining
     */
    public function add_section($args) {
        $defaults = array(
            'id'          => '',
            'title'       => '',
            'description' => '',
            'icon'        => '',
            'fields'      => array(),
            'subsections' => array(),
            'option_name' => '',  // Optional separate meta key for this section
        );
        
        $section = wp_parse_args($args, $defaults);
        
        if (empty($section['id'])) {
            return $this;
        }
        
        // Apply filter for adding fields from other plugins
        $section['fields'] = apply_filters('bizzplugin_metabox_section_fields', $section['fields'], $section['id'], $this->id);
        $section['fields'] = apply_filters('bizzplugin_metabox_section_fields_' . $this->id, $section['fields'], $section['id'], $this->id);
        
        $this->sections[$section['id']] = $section;
        
        return $this;
    }
    
    /**
     * Add a subsection to an existing section
     * 
     * @param string $section_id The ID of the parent section
     * @param array  $subsection Subsection configuration array
     * @return $this For method chaining
     */
    public function add_subsection($section_id, $subsection) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        $defaults = array(
            'id'          => '',
            'title'       => '',
            'description' => '',
            'icon'        => '',
            'fields'      => array(),
            'option_name' => '',  // Optional separate meta key for this subsection
        );
        
        $subsection = wp_parse_args($subsection, $defaults);
        
        if (empty($subsection['id'])) {
            return $this;
        }
        
        if (!isset($this->sections[$section_id]['subsections'])) {
            $this->sections[$section_id]['subsections'] = array();
        }
        
        // Check for duplicate subsection ID and replace if exists
        $found = false;
        foreach ($this->sections[$section_id]['subsections'] as $key => $existing) {
            if ($existing['id'] === $subsection['id']) {
                $this->sections[$section_id]['subsections'][$key] = $subsection;
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            $this->sections[$section_id]['subsections'][] = $subsection;
        }
        
        return $this;
    }
    
    /**
     * Add a field to a subsection
     * 
     * @param string $section_id The ID of the parent section
     * @param string $subsection_id The ID of the subsection
     * @param array  $field Field configuration array
     * @return $this For method chaining
     */
    public function add_subsection_field($section_id, $subsection_id, $field) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        if (empty($field['id'])) {
            return $this;
        }
        
        if (!isset($this->sections[$section_id]['subsections'])) {
            return $this;
        }
        
        foreach ($this->sections[$section_id]['subsections'] as &$subsection) {
            if ($subsection['id'] === $subsection_id) {
                $subsection['fields'][] = $field;
                break;
            }
        }
        
        return $this;
    }
    
    /**
     * Get sections
     * 
     * @return array
     */
    public function get_sections() {
        return apply_filters('bizzplugin_metabox_sections_' . $this->id, $this->sections, $this->id);
    }
    
    /**
     * Get section by ID
     * 
     * @param string $section_id Section ID
     * @return array|null
     */
    public function get_section($section_id) {
        return isset($this->sections[$section_id]) ? $this->sections[$section_id] : null;
    }
    
    /**
     * Add field to a section
     * 
     * @param string $section_id Section ID
     * @param array  $field      Field configuration
     * @return $this For method chaining
     */
    public function add_field($section_id, $field) {
        if (!isset($this->sections[$section_id])) {
            return $this;
        }
        
        if (empty($field['id'])) {
            return $this;
        }
        
        $this->sections[$section_id]['fields'][] = $field;
        
        return $this;
    }
    
    /**
     * Get all fields from all sections
     * 
     * @return array
     */
    public function get_all_fields() {
        $fields = array();
        
        foreach ($this->sections as $section) {
            if (!empty($section['fields'])) {
                foreach ($section['fields'] as $field) {
                    $fields[$field['id']] = $field;
                }
            }
            
            // Get fields from subsections
            if (!empty($section['subsections'])) {
                foreach ($section['subsections'] as $subsection) {
                    if (!empty($subsection['fields'])) {
                        foreach ($subsection['fields'] as $field) {
                            $fields[$field['id']] = $field;
                        }
                    }
                }
            }
        }
        
        return $fields;
    }
    
    /**
     * Set premium status
     * 
     * @param bool $is_premium Premium status
     * @return $this For method chaining
     */
    public function set_premium($is_premium) {
        $this->is_premium = (bool) $is_premium;
        return $this;
    }
    
    /**
     * Set post types
     * 
     * @param array|string $post_types Post types
     * @return $this For method chaining
     */
    public function set_post_types($post_types) {
        $this->post_types = (array) $post_types;
        return $this;
    }
    
    /**
     * Add post type
     * 
     * @param string $post_type Post type
     * @return $this For method chaining
     */
    public function add_post_type($post_type) {
        if (!in_array($post_type, $this->post_types, true)) {
            $this->post_types[] = $post_type;
        }
        return $this;
    }
    
    /**
     * Set context
     * 
     * @param string $context Context (normal, side, advanced)
     * @return $this For method chaining
     */
    public function set_context($context) {
        $this->context = $context;
        return $this;
    }
    
    /**
     * Set priority
     * 
     * @param string $priority Priority (high, core, default, low)
     * @return $this For method chaining
     */
    public function set_priority($priority) {
        $this->priority = $priority;
        return $this;
    }
    
    /**
     * Register the metabox
     */
    public function register_metabox() {
        foreach ($this->post_types as $post_type) {
            add_meta_box(
                $this->id,
                $this->title,
                array($this, 'render_metabox'),
                $post_type,
                $this->context,
                $this->priority
            );
        }
    }
    
    /**
     * Enqueue assets for metabox
     * 
     * @param string $hook Current admin page hook
     */
    public function enqueue_assets($hook) {
        global $post_type;
        
        // Only load on edit screens for our post types
        if (!in_array($hook, array('post.php', 'post-new.php'), true)) {
            return;
        }
        
        if (!in_array($post_type, $this->post_types, true)) {
            return;
        }
        
        // Enqueue WordPress media uploader
        wp_enqueue_media();
        
        // Enqueue color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Enqueue jQuery UI datepicker
        wp_enqueue_script('jquery-ui-datepicker');
        
        // Enqueue metabox specific styles
        wp_enqueue_style(
            'bizzplugin-metabox-style',
            $this->framework->get_url() . 'assets/css/metabox.css',
            array(),
            BizzPlugin_Framework::VERSION
        );

        //framework css and date.css include here as datepicker needs it
        wp_enqueue_style(
            'bizzplugin-framework-style',
            $this->framework->get_url() . 'assets/css/framework-common.css',
            array(),
            BizzPlugin_Framework::VERSION
        );
        wp_enqueue_style(
            'bizzplugin-jquery-ui-style',
            $this->framework->get_url() . 'assets/css/fields/date.css',
            array(),
            BizzPlugin_Framework::VERSION
        );
        
        // Enqueue field-specific styles needed by metabox
        $field_css_files = array(
            'repeater'        => 'repeater.css',
            'single_repeater' => 'single-repeater.css',
            'slider'          => 'slider.css',
        );
        foreach ($field_css_files as $field_type => $css_file) {
            wp_enqueue_style(
                'bizzplugin-metabox-field-' . $field_type . '-style',
                $this->framework->get_url() . 'assets/css/fields/' . $css_file,
                array(),
                BizzPlugin_Framework::VERSION
            );
        }
        
        
        // Enqueue common JS (shared functionality)
        wp_enqueue_script(
            'bizzplugin-common-script',
            $this->framework->get_url() . 'assets/js/bizzplugin-common.js',
            array('jquery'),
            BizzPlugin_Framework::VERSION,
            true
        );
        
        // Enqueue metabox specific scripts
        wp_enqueue_script(
            'bizzplugin-metabox-script',
            $this->framework->get_url() . 'assets/js/metabox.js',
            array('jquery', 'wp-color-picker', 'jquery-ui-datepicker', 'bizzplugin-common-script'),
            BizzPlugin_Framework::VERSION,
            true
        );
        
        // Localize script with AJAX URL and nonce for plugin install/activate
        wp_localize_script('bizzplugin-metabox-script', 'bizzpluginMetabox', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('bizzplugin_framework_nonce'),
            'strings' => array(
                'select_image'       => 'Select Image',
                'select_file'        => 'Select File',
                'use_image'          => 'Use this image',
                'use_file'           => 'Use this file',
                'installing'         => 'Installing...',
                'installed_inactive' => 'Installed (Inactive)',
                'activate'           => 'Activate',
                'activating'         => 'Activating...',
                'active'             => 'Active',
                'activated'          => 'Activated',
                'install_error'      => 'Error installing plugin',
                'activate_error'     => 'Error activating plugin',
            )
        ));
    }
    
    /**
     * Render the metabox content
     * 
     * @param WP_Post $post Current post object
     */
    public function render_metabox($post) {
        // Security nonce
        wp_nonce_field('bizzplugin_metabox_' . $this->id, 'bizzplugin_metabox_nonce_' . $this->id);
        
        $sections = $this->get_sections();
        $has_multiple_sections = count($sections) > 1;
        
        // Get first section and subsection for initial display
        $first_section = reset($sections);
        $first_section_id = $first_section ? $first_section['id'] : '';
        $first_subsection_id = '';
        
        // If first section has no fields but has subsections, auto-select first subsection
        if ($first_section && empty($first_section['fields']) && !empty($first_section['subsections'])) {
            $first_sub = reset($first_section['subsections']);
            $first_subsection_id = $first_sub ? $first_sub['id'] : '';
        }
        
        ?>
        <div class="bizzplugin-metabox-wrap" data-metabox-id="<?php echo esc_attr($this->id); ?>">
            <div class="bizzplugin-metabox-container">
                
                <?php if ($has_multiple_sections) : ?>
                <!-- Left Navigation Sidebar -->
                <div class="bizzplugin-metabox-nav">
                    <ul class="bizzplugin-metabox-nav-menu">
                        <?php foreach ($sections as $section) : 
                            $has_fields = !empty($section['fields']);
                            $has_subsections = !empty($section['subsections']);
                            $is_first = ($section['id'] === $first_section_id);
                            $section_dependency_attr = BizzPlugin_Helper_Common::get_dependency_attr($section);

                        ?>
                            <li class="bizzplugin-metabox-nav-item <?php echo $is_first ? 'active' : ''; ?> <?php echo $has_subsections ? 'has-subsections' : ''; ?>" data-section-id="<?php echo esc_attr($section['id']); ?>" <?php echo $section_dependency_attr; ?>>
                                <a href="#<?php echo esc_attr($section['id']); ?>" 
                                   data-section="<?php echo esc_attr($section['id']); ?>" 
                                   data-has-fields="<?php echo $has_fields ? '1' : '0'; ?>"
                                   data-has-subsections="<?php echo $has_subsections ? '1' : '0'; ?>"
                                   class="bizzplugin-metabox-nav-link <?php echo ($is_first && empty($first_subsection_id)) ? 'current' : ''; ?>">
                                    <?php if (!empty($section['icon'])) : ?>
                                        <span class="bizzplugin-metabox-nav-icon <?php echo esc_attr($section['icon']); ?>"></span>
                                    <?php endif; ?>
                                    <span class="bizzplugin-metabox-nav-title"><?php echo esc_html($section['title']); ?></span>
                                    <?php if ($has_subsections) : ?>
                                        <span class="bizzplugin-metabox-nav-arrow dashicons dashicons-arrow-down-alt2"></span>
                                    <?php endif; ?>
                                </a>
                                
                                <?php if ($has_subsections) : ?>
                                    <ul class="bizzplugin-metabox-nav-submenu" style="display:none;">
                                        <?php foreach ($section['subsections'] as $subsection) : 
                                            $is_first_sub = ($subsection['id'] === $first_subsection_id && $section['id'] === $first_section_id);
                                            $subsection_dependency_attr = BizzPlugin_Helper_Common::get_dependency_attr($subsection);
                                        ?>
                                            <li class="bizzplugin-metabox-nav-subitem <?php echo $is_first_sub ? 'active' : ''; ?>" <?php echo $subsection_dependency_attr; ?>>
                                                <a href="#<?php echo esc_attr($section['id']); ?>-<?php echo esc_attr($subsection['id']); ?>" 
                                                   data-section="<?php echo esc_attr($section['id']); ?>" 
                                                   data-subsection="<?php echo esc_attr($subsection['id']); ?>" 
                                                   class="bizzplugin-metabox-nav-sublink <?php echo $is_first_sub ? 'current' : ''; ?>">
                                                    <?php if (!empty($subsection['icon'])) : ?>
                                                        <span class="bizzplugin-metabox-nav-subicon <?php echo esc_attr($subsection['icon']); ?>"></span>
                                                    <?php endif; ?>
                                                    <?php echo esc_html($subsection['title']); ?>
                                                </a>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php endif; ?>
                
                <!-- Content Area -->
                <div class="bizzplugin-metabox-content <?php echo !$has_multiple_sections ? 'full-width' : ''; ?>">
                    <?php foreach ($sections as $section) : 
                        $has_fields = !empty($section['fields']);
                        $has_subsections = !empty($section['subsections']);
                        $is_first = ($section['id'] === $first_section_id);
                        $show_section = $is_first && (empty($first_subsection_id) || !$has_subsections);
                        $section_option_name = !empty($section['option_name']) ? $section['option_name'] : '';
                    ?>
                        <!-- Main Section Content (only fields, not subsections) -->
                        <?php if ($has_fields) : ?>
                        <div class="bizzplugin-metabox-section" 
                             id="section-<?php echo esc_attr($section['id']); ?>" 
                             data-section="<?php echo esc_attr($section['id']); ?>" 
                             style="<?php echo !$show_section ? 'display:none;' : ''; ?>">
                            <div class="bizzplugin-metabox-section-header">
                                <h3><?php echo esc_html($section['title']); ?></h3>
                                <?php if (!empty($section['description'])) : ?>
                                    <p class="bizzplugin-metabox-section-desc"><?php echo esc_html($section['description']); ?></p>
                                <?php endif; ?>
                            </div>
                            
                            <div class="bizzplugin-metabox-section-content">
                                <?php
                                foreach ($section['fields'] as $field) {
                                    $this->render_field($field, $post, $section_option_name);
                                }
                                ?>
                            </div>
                        </div>
                        <?php endif; ?>
                        
                        <!-- Subsections as separate content areas -->
                        <?php if ($has_subsections) : ?>
                            <?php foreach ($section['subsections'] as $subsection) : 
                                $is_first_sub = ($subsection['id'] === $first_subsection_id && $section['id'] === $first_section_id);
                                // Subsection can have its own option_name, otherwise inherit from section
                                $subsection_option_name = !empty($subsection['option_name']) ? $subsection['option_name'] : $section_option_name;
                            ?>
                            <div class="bizzplugin-metabox-section bizzplugin-metabox-subsection-content" 
                                 id="subsection-<?php echo esc_attr($subsection['id']); ?>" 
                                 data-section="<?php echo esc_attr($section['id']); ?>"
                                 data-subsection="<?php echo esc_attr($subsection['id']); ?>" 
                                 style="<?php echo !$is_first_sub ? 'display:none;' : ''; ?>">
                                <div class="bizzplugin-metabox-section-header">
                                    <h3><?php echo esc_html($subsection['title']); ?></h3>
                                    <?php if (!empty($subsection['description'])) : ?>
                                        <p class="bizzplugin-metabox-section-desc"><?php echo esc_html($subsection['description']); ?></p>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="bizzplugin-metabox-section-content">
                                    <?php 
                                    if (!empty($subsection['fields'])) {
                                        foreach ($subsection['fields'] as $field) {
                                            $this->render_field($field, $post, $subsection_option_name);
                                        }
                                    }
                                    ?>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render a single field
     * 
     * @param array   $field              Field configuration
     * @param WP_Post $post               Current post object
     * @param string  $section_option_name Optional section-specific meta key
     */
    private function render_field($field, $post, $section_option_name = '') {
        $field_id      = $field['id'];
        $field_type    = isset($field['type']) ? $field['type'] : 'text';
        $field_title   = isset($field['title']) ? $field['title'] : '';
        $field_desc    = isset($field['description']) ? $field['description'] : '';
        $field_default = isset($field['default']) ? $field['default'] : '';
        $field_premium = isset($field['premium']) ? $field['premium'] : false;
        
        // Determine which meta key to use
        $use_meta_key = !empty($section_option_name) ? $section_option_name : $this->option_name;
        
        // Get meta value from stored array
        $all_meta     = get_post_meta($post->ID, $use_meta_key, true);
        $all_meta     = is_array($all_meta) ? $all_meta : array();
        $meta_value   = isset($all_meta[$field_id]) ? $all_meta[$field_id] : '';
        $field_value  = ($meta_value !== '') ? $meta_value : $field_default;
        
        // The input name should be the option_name[field_id] format
        $input_name = $use_meta_key . '[' . $field_id . ']';
        $input_id   = $use_meta_key . '_' . $field_id;
        
        // Premium field handling
        $is_premium_active = $this->is_premium();
        $premium_class     = '';
        $is_disabled       = false;
        
        if ($field_premium && !$is_premium_active) {
            $premium_class = 'bizzplugin-field-premium-locked';
            $is_disabled   = true;
        }
        
        // Dependency handling
        $dependency      = isset($field['dependency']) ? $field['dependency'] : array();
        $dependency_attr = '';
        if (!empty($dependency) && is_array($dependency) && isset($dependency['field'], $dependency['value'])) {
            $dependency_attr = sprintf(
                'data-dependency="%s" data-dependency-value="%s"',
                esc_attr($use_meta_key . '[' . $dependency['field'] . ']'),
                esc_attr($dependency['value'])
            );
        }
        
        ?>
        <div class="bizzplugin-metabox-field bizzplugin-field-<?php echo esc_attr($field_type); ?> <?php echo esc_attr($premium_class); ?>" <?php echo $dependency_attr; ?>>
            <div class="bizzplugin-metabox-field-header">
                <label for="<?php echo esc_attr($input_id); ?>" class="bizzplugin-metabox-field-title">
                    <?php echo esc_html($field_title); ?>
                    <?php if ($field_premium && !$is_premium_active) : ?>
                        <span class="bizzplugin-premium-badge">Premium</span>
                    <?php endif; ?>
                </label>
            </div>
            
            <div class="bizzplugin-metabox-field-content">
                <?php
                // Allow custom field rendering
                if (has_action('bizzplugin_metabox_render_field_' . $field_type)) {
                    do_action('bizzplugin_metabox_render_field_' . $field_type, $field, $field_value, $input_name, $is_disabled);
                } elseif (has_action('bizzplugin_render_field_' . $field_type)) {
                    // Fallback to panel field rendering hook
                    do_action('bizzplugin_render_field_' . $field_type, $field, $field_value, $is_disabled);
                } else {
                    // Render field based on type using shared BizzPlugin_Render_Field class
                    switch ($field_type) {
                        case 'text':
                        case 'email':
                        case 'url':
                        case 'number':
                        case 'password':
                            BizzPlugin_Render_Field::input($field, $field_value, $input_name, $field_type, $is_disabled);
                            break;
                        case 'textarea':
                            BizzPlugin_Render_Field::textarea($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'select':
                            BizzPlugin_Render_Field::select($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'multi_select':
                            BizzPlugin_Render_Field::multi_select($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'checkbox':
                            BizzPlugin_Render_Field::checkbox($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'checkbox_group':
                            BizzPlugin_Render_Field::checkbox_group($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'radio':
                            BizzPlugin_Render_Field::radio($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'toggle':
                        case 'on_off':
                        case 'switch':
                            BizzPlugin_Render_Field::switch_toggle($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'color':
                            BizzPlugin_Render_Field::color($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'alpha_color':
                            BizzPlugin_Render_Field::alpha_color($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'date':
                            BizzPlugin_Render_Field::date($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'image':
                            BizzPlugin_Render_Field::image($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'file':
                            BizzPlugin_Render_Field::file($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'image_select':
                            BizzPlugin_Render_Field::image_select($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'option_select':
                            BizzPlugin_Render_Field::option_select($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'post_select':
                            BizzPlugin_Render_Field::post_select($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'html':
                            BizzPlugin_Render_Field::html($field);
                            break;
                        case 'plugins':
                            BizzPlugin_Render_Field::plugins($field);
                            break;
                        case 'slider':
                        case 'range':
                            BizzPlugin_Render_Field::slider($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'repeater':
                            BizzPlugin_Render_Field::repeater($field, $field_value, $input_name, $is_disabled);
                            break;
                        case 'single_repeater':
                            BizzPlugin_Render_Field::single_repeater($field, $field_value, $input_name, $is_disabled);
                            break;
                        default:
                            do_action('bizzplugin_metabox_render_custom_field', $field, $field_value, $input_name, $is_disabled);
                            break;
                    }
                }
                ?>
                
                <?php if (!empty($field_desc)) : ?>
                    <p class="bizzplugin-metabox-field-desc"><?php echo wp_kses_post($field_desc); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Save meta data
     * 
     * @param int     $post_id Post ID
     * @param WP_Post $post    Post object
     * @param bool    $update  Whether this is an update
     */
    public function save_meta($post_id, $post, $update) {
        // Check if our nonce is set
        $nonce_key = 'bizzplugin_metabox_nonce_' . $this->id;
        if (!isset($_POST[$nonce_key])) {
            return;
        }
        
        // Verify nonce - wp_unslash is sufficient, sanitize_text_field could alter the nonce
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        if (!wp_verify_nonce(wp_unslash($_POST[$nonce_key]), 'bizzplugin_metabox_' . $this->id)) {
            return;
        }
        
        // Check autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check post type
        if (!in_array($post->post_type, $this->post_types, true)) {
            return;
        }
        
        // Check permissions
        $post_type_obj = get_post_type_object($post->post_type);
        if (!current_user_can($post_type_obj->cap->edit_post, $post_id)) {
            return;
        }
        
        // Collect all meta keys we need to save
        $meta_keys_to_save = array();
        
        // Process each section
        foreach ($this->sections as $section_id => $section) {
            // Determine which meta key to use for this section
            $section_meta_key = !empty($section['option_name']) ? $section['option_name'] : $this->option_name;
            
            // Initialize meta key array if not exists
            if (!isset($meta_keys_to_save[$section_meta_key])) {
                $meta_keys_to_save[$section_meta_key] = array();
            }
            
            // Process section fields
            if (!empty($section['fields'])) {
                foreach ($section['fields'] as $field) {
                    $this->process_field_for_save($field, $section_meta_key, $meta_keys_to_save);
                }
            }
            
            // Process subsections
            if (!empty($section['subsections'])) {
                foreach ($section['subsections'] as $subsection) {
                    // Subsection can have its own option_name
                    $subsection_meta_key = !empty($subsection['option_name']) ? $subsection['option_name'] : $section_meta_key;
                    
                    // Initialize meta key array if not exists
                    if (!isset($meta_keys_to_save[$subsection_meta_key])) {
                        $meta_keys_to_save[$subsection_meta_key] = array();
                    }
                    
                    if (!empty($subsection['fields'])) {
                        foreach ($subsection['fields'] as $field) {
                            $this->process_field_for_save($field, $subsection_meta_key, $meta_keys_to_save);
                        }
                    }
                }
            }
        }
        
        // Save all meta keys
        foreach ($meta_keys_to_save as $option_name => $data) {
            // Get existing data if any
            $existing = get_post_meta($post_id, $option_name, true);
            if (!is_array($existing)) {
                $existing = array();
            }
            
            // Get posted data for this meta key
            if (isset($_POST[$option_name]) && is_array($_POST[$option_name])) {
                // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $posted_data = wp_unslash($_POST[$option_name]);
                
                // Sanitize each field
                foreach ($data['fields'] as $field_id => $field) {
                    $field_type = isset($field['type']) ? $field['type'] : 'text';
                    
                    if (isset($posted_data[$field_id])) {
                        $existing[$field_id] = $this->sanitize_field_value($posted_data[$field_id], $field);
                    } else {
                        // Handle checkboxes and similar fields that may not be set
                        if (in_array($field_type, array('checkbox', 'switch', 'on_off'), true)) {
                            $existing[$field_id] = '';
                        }
                    }
                }
            }
            
            // Save the meta data
            update_post_meta($post_id, $option_name, $existing);
        }
        
        // Allow other plugins to hook into save
        do_action('bizzplugin_metabox_saved', $post_id, $this->id, $this);
    }
    
    /**
     * Process a field for saving
     * 
     * @param array  $field              Field configuration
     * @param string $option_name           Meta key to use
     * @param array  &$meta_keys_to_save Reference to meta keys array
     */
    private function process_field_for_save($field, $option_name, &$meta_keys_to_save) {
        $field_id = $field['id'];
        $field_type = isset($field['type']) ? $field['type'] : 'text';
        
        // Skip non-editable field types
        $non_editable = array('html', 'info', 'heading', 'divider', 'notice');
        if (in_array($field_type, $non_editable, true)) {
            return;
        }
        
        // Skip premium fields if not premium
        if (!empty($field['premium']) && !$this->is_premium()) {
            return;
        }
        
        // Add field to the meta key's fields list
        if (!isset($meta_keys_to_save[$option_name]['fields'])) {
            $meta_keys_to_save[$option_name]['fields'] = array();
        }
        
        $meta_keys_to_save[$option_name]['fields'][$field_id] = $field;
    }
    
    /**
     * Sanitize field value based on field type
     * 
     * @param mixed $value Value to sanitize
     * @param array $field Field configuration
     * @return mixed Sanitized value
     */
    private function sanitize_field_value($value, $field) {
        $type = isset($field['type']) ? $field['type'] : 'text';
        
        // Allow custom sanitization
        if (isset($field['sanitize_callback']) && is_callable($field['sanitize_callback'])) {
            return call_user_func($field['sanitize_callback'], $value, $field);
        }
        
        switch ($type) {
            case 'text':
            case 'password':
                return sanitize_text_field($value);
                
            case 'textarea':
                return sanitize_textarea_field($value);
                
            case 'email':
                return sanitize_email($value);
                
            case 'url':
                return esc_url_raw($value);
                
            case 'number':
            case 'slider':
            case 'range':
                return floatval($value);
                
            case 'checkbox':
            case 'on_off':
            case 'switch':
                return $value === '1' ? '1' : '';
                
            case 'select':
            case 'radio':
            case 'image_select':
                return sanitize_text_field($value);
                
            case 'multi_select':
            case 'checkbox_group':
                if (!is_array($value)) {
                    return array();
                }
                return array_map('sanitize_text_field', $value);
                
            case 'color':
                return sanitize_hex_color($value);
                
            case 'date':
                return sanitize_text_field($value);
                
            case 'image':
            case 'file':
                return absint($value);
                
            case 'repeater':
                if (!is_array($value)) {
                    return array();
                }
                return $this->sanitize_repeater_value($value, $field);
                
            case 'single_repeater':
                if (!is_array($value)) {
                    return array();
                }
                return $this->sanitize_single_repeater_value($value, $field);
                
            default:
                return sanitize_text_field(wp_unslash($value));
        }
    }
    
    /**
     * Sanitize repeater field value
     * 
     * @param array $value Repeater value
     * @param array $field Field configuration
     * @return array Sanitized repeater value
     */
    private function sanitize_repeater_value($value, $field) {
        $sub_fields = isset($field['fields']) ? $field['fields'] : array();
        $sanitized = array();
        
        foreach ($value as $index => $item) {
            if (!is_array($item)) {
                continue;
            }
            
            $sanitized_item = array();
            foreach ($sub_fields as $sub_field) {
                $sub_field_id = $sub_field['id'];
                if (isset($item[$sub_field_id])) {
                    $sanitized_item[$sub_field_id] = $this->sanitize_field_value($item[$sub_field_id], $sub_field);
                }
            }
            
            if (!empty($sanitized_item)) {
                $sanitized[] = $sanitized_item;
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Sanitize single repeater field value
     * 
     * @param array $value Single repeater value (array of simple values)
     * @param array $field Field configuration
     * @return array Sanitized single repeater value
     */
    private function sanitize_single_repeater_value($value, $field) {
        $sub_field = isset($field['field']) ? $field['field'] : array('type' => 'text');
        $sanitized = array();
        
        // Re-index array to remove gaps
        $value = array_values($value);
        
        foreach ($value as $item_value) {
            $sanitized_value = $this->sanitize_field_value($item_value, $sub_field);
            
            // Only add non-empty values
            if ($sanitized_value !== '' && $sanitized_value !== null) {
                $sanitized[] = $sanitized_value;
            }
        }
        
        return $sanitized;
    }
}
