<?php
/**
 * BizzPlugin Options Framework - Main Framework Class
 * 
 * @package BizzPlugin_Options_Framework
 * @version 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main Framework Class
 */
class BizzPlugin_Framework {
    
    /**
     * Framework version
     */
    const VERSION = BIZZPLUGIN_FRAMEWORK_VERSION;//'5.0.0';
    
    /**
     * Single instance of the class
     */
    private static $instance = null;

    /**
     * Field type to CSS file mapping
     * Use this in class-bizzplugin-framework.php
     */
    private $field_css_map = array();
    
    /**
     * Registered options panels
     */
    private $panels = array();
    
    /**
     * Registered metaboxes
     */
    private $metaboxes = array();
    
    public $panel = null;

    /**
     * Framework URL
     */
    private $framework_url = '';
    
    /**
     * Framework path
     */
    private $framework_path = '';
    
    /**
     * Get single instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->framework_path = dirname(__FILE__);
        $this->framework_url = $this->get_framework_url();
        
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * Get framework URL
     */
    public function get_framework_url() {
        $path = wp_normalize_path($this->framework_path);
        $plugins_dir = wp_normalize_path(WP_PLUGIN_DIR);
        $themes_dir = wp_normalize_path(get_theme_root());
        
        if (strpos($path, $plugins_dir) !== false) {
            return plugin_dir_url(__FILE__);
        } elseif (strpos($path, $themes_dir) !== false) {
            $relative = str_replace($themes_dir, '', $path);
            return get_theme_root_uri() . $relative . '/';
        }
        
        return plugin_dir_url(__FILE__);
    }
    
    /**
     * Load dependencies
     */
    private function load_dependencies() {
        require_once $this->framework_path . '/includes/class-setup-wizard.php';
        require_once $this->framework_path . '/includes/class-helper-common.php';
        require_once $this->framework_path . '/includes/class-render-field.php';
        require_once $this->framework_path . '/includes/class-field-sanitizer.php';
        require_once $this->framework_path . '/includes/class-field-validator.php';
        require_once $this->framework_path . '/includes/class-ajax-handler.php';
        require_once $this->framework_path . '/includes/class-api-handler.php';
        require_once $this->framework_path . '/includes/class-webhook-handler.php';
        require_once $this->framework_path . '/class-bizzplugin-panel.php';
        require_once $this->framework_path . '/class-bizzplugin-metabox.php';
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_bizzplugin_save_options', array($this, 'ajax_save_options'));
        add_action('wp_ajax_bizzplugin_reset_section', array($this, 'ajax_reset_section'));
        add_action('wp_ajax_bizzplugin_reset_all', array($this, 'ajax_reset_all'));
        add_action('wp_ajax_bizzplugin_test_webhook', array($this, 'ajax_test_webhook'));
        add_action('wp_ajax_bizzplugin_install_plugin', array($this, 'ajax_install_plugin'));
        add_action('wp_ajax_bizzplugin_activate_plugin', array($this, 'ajax_activate_plugin'));
        add_action('wp_ajax_bizzplugin_generate_api_key', array($this, 'ajax_generate_api_key'));
        add_action('wp_ajax_bizzplugin_delete_api_key', array($this, 'ajax_delete_api_key'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // Allow plugins to extend framework
        do_action('bizzplugin_framework_loaded', $this);
    }
    
    /**
     * Enqueue assets
     */
    public function enqueue_assets($hook) {
        // Check if we're on a registered options page
        $is_options_page = false;
        foreach ($this->panels as $panel) {
            if ($panel->is_current_page($hook)) {
                $is_options_page = true;
                break;
            }
        }
        
        if (!$is_options_page) {
            return;
        }
        if(empty($panel->get_id())) return;
        
        // Enqueue WordPress media uploader
        wp_enqueue_media();
        
        // Enqueue color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Enqueue jQuery UI for datepicker (use WordPress built-in styles)
        wp_enqueue_script('jquery-ui-datepicker');
        
        // Framework CSS (includes datepicker styles)
        // wp_enqueue_style(
        //     'bizzplugin-framework-style',
        //     $this->framework_url . 'assets/css/framework.css',
        //     array(),
        //     self::VERSION
        // );

        /**
         * 
        assets/css/
        ├── framework.css              # মূল ফাইল (অপরিবর্তিত)
        ├── framework-common.css       # কমন CSS
        ├── fields/                    # ফিল্ড-ভিত্তিক CSS
        │   ├── text.css
        │   ├── textarea.css
        │   ├── select.css
        │   ├── checkbox.css
        │   ├── radio.css
        │   ├── switch.css
        │   ├── color.css
        │   ├── date.css
        │   ├── image.css
        │   ├── file.css
        │   ├── image-select.css
        │   ├── option-select.css
        │   ├── post-select.css
        │   ├── slider.css
        │   ├── repeater.css
        │   ├── plugins.css
        │   └── html.css
        └── components/
            └── api-section.css
        * // I have to add all thse files separately because they are required for different fields and components in the framework.
        */
        wp_enqueue_style(
            'bizzplugin-framework-common-style',
            $this->framework_url . 'assets/css/framework-common.css',
            array(),
            self::VERSION
        );
        
        $this->field_css_map = array(
            // Text-like fields
            'text'           => 'text.css',
            'email'          => 'text.css',
            'url'            => 'text.css',
            'number'         => 'text.css',
            'password'       => 'text.css',
            
            // Textarea
            'textarea'       => 'textarea.css',
            
            // Selection fields
            'select'         => 'select.css',
            'multi_select'   => 'select.css',
            'checkbox'       => 'checkbox.css',
            'checkbox_group' => 'checkbox.css',
            'radio'          => 'radio.css',
            
            // Toggle fields
            'on_off'         => 'switch.css',
            'switch'         => 'switch.css',
            
            // Picker fields
            'color'          => 'color.css',
            'alpha_color'    => 'color.css',
            'date'           => 'date.css',
            
            // Media fields
            'image'          => 'image.css',
            'file'           => 'file.css',
            'image_select'   => 'image-select.css',
            
            // Other selection
            'option_select'  => 'option-select.css',
            'post_select'    => 'post-select.css',
            
            // Complex fields
            'slider'         => 'slider.css',
            'range'          => 'slider.css',
            'repeater'       => 'repeater.css',
            'single_repeater' => 'single-repeater.css',
            'plugins'        => 'plugins.css',
            
            // Display-only fields
            'html'           => 'html.css',
            'info'           => 'html.css',
            'notice'         => 'html.css',
            'heading'        => 'html.css',
            'divider'        => 'html.css',
            'link'           => 'html.css',
        );
        $css_map = apply_filters('bizzplugin_framework_field_css_map_' . $panel->get_id(), $this->field_css_map);
        $temp_file = '';
        foreach ($css_map as $field_type => $css_file) {
            if ($temp_file === $css_file) continue; //skip if already enqueued                
            wp_enqueue_style(
                'bizzplugin-framework-field-' . $field_type . '-style',
                $this->framework_url . 'assets/css/fields/' . $css_file,
                array(),
                self::VERSION
            );
            $temp_file = $css_file;
        }
        // Enqueue component-specific styles
        wp_enqueue_style(
            'bizzplugin-framework-component-api-section-style',
            $this->framework_url . 'assets/css/components/api-section.css',
            array(),
            self::VERSION
        );
        
        // Enqueue dark mode styles
        wp_enqueue_style(
            'bizzplugin-framework-dark-mode-style',
            $this->framework_url . 'assets/css/dark-mode.css',
            array(),
            self::VERSION
        );
        
        // Enqueue common JS (shared functionality)
        wp_enqueue_script(
            'bizzplugin-common-script',
            $this->framework_url . 'assets/js/bizzplugin-common.js',
            array('jquery'),
            self::VERSION,
            true
        );
        
        // Framework JS
        wp_enqueue_script(
            'bizzplugin-framework-script',
            $this->framework_url . 'assets/js/framework.js',
            array('jquery', 'wp-color-picker', 'jquery-ui-datepicker', 'bizzplugin-common-script'),
            self::VERSION,
            true
        );

                // Enqueue color picker alpha support
        // wp_enqueue_script(
        //     'wp-color-picker-alpha',
        //     $this->framework_url . 'assets/js/wp-color-picker-alpha.min.js',
        //     array('wp-color-picker', 'jquery-ui-slider'),
        //     self::VERSION,
        //     true
        // );

        // Localize script
        wp_localize_script('bizzplugin-framework-script', 'bizzpluginFramework', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bizzplugin_framework_nonce'),
            'strings' => array(
                'saving' => 'Saving...',
                'saved' => 'Settings Saved!',
                'error' => 'Error saving settings',
                'resetting' => 'Resetting...',
                'reset_success' => 'Section Reset!',
                'reset_all_success' => 'All settings reset successfully!',
                'reset_error' => 'Error resetting section',
                'confirm_reset' => 'Are you sure you want to reset this section to default values?',
                'confirm_reset_all' => 'Are you sure you want to reset ALL settings to default values?',
                'select_image' => 'Select Image',
                'select_file' => 'Select File',
                'use_image' => 'Use this image',
                'use_file' => 'Use this file',
                'webhook_url_required' => 'Please enter a webhook URL first.',
                'testing' => 'Testing...',
                'export_success' => 'Settings exported successfully!',
                'import_success' => 'Settings imported successfully! Please save to apply changes.',
                'import_error' => 'Error parsing import file.',
                'import_invalid' => 'Invalid import file format.',
                'import_panel_mismatch' => 'This export was created from a different panel. Do you want to continue importing?',
                'installing' => 'Installing...',
                'installed_inactive' => 'Installed (Inactive)',
                'activate' => 'Activate',
                'activating' => 'Activating...',
                'active' => 'Active',
                'activated' => 'Activated',
                'install_error' => 'Error installing plugin',
                'activate_error' => 'Error activating plugin',
                'confirm_regenerate_api_key' => 'Are you sure you want to generate a new API key? The old key will be invalidated.',
                'generating' => 'Generating...',
                'regenerate_api_key' => 'Regenerate API Key',
                'api_key_error' => 'Error generating API key',
                'search_result_single' => '1 section found',
                'search_results_plural' => '%d sections found',
                'search_no_results' => 'No results found',
            )
        ));
    }
    
    /**
     * Register a new options panel
     */
    public function create_panel($args) {
        
        $defaults = array(
            'id' => '',
            'title' => '',
            'menu_title' => '',
            'menu_slug' => '',
            'parent_slug' => '',
            'capability' => 'manage_options',
            'icon' => 'dashicons-admin-settings',
            'position' => null,
            'option_name' => '',
            'is_premium' => false,
            'sections' => array(),
            'route_namespace' => 'bizzplugin/v1',
            'enable_search' => true,
            'show_export_import' => true,
            'show_api' => true,
            'theme_mode' => 'light',
            'show_theme_switcher' => false,
        );
        
        $args = wp_parse_args($args, $defaults);
        
        if (empty($args['id']) || empty($args['option_name'])) {
            return false;
        }
        
        $panel = new BizzPlugin_Panel($args, $this);

        $this->panels[$args['id']] = $panel;
        
        // Allow other plugins to add fields to this panel
        do_action('bizzplugin_panel_created', $panel, $args['id']);
        $this->panel = $panel;
        return $this->panel;//;
    }
    
    /**
     * Get a panel by ID
     */
    public function get_panel($panel_id) {
        return isset($this->panels[$panel_id]) ? $this->panels[$panel_id] : null;
    }
    
    /**
     * Get all panels
     */
    public function get_panels() {
        return $this->panels;
    }
    
    /**
     * Register a new metabox for posts, pages, or custom post types
     * 
     * @param array $args Metabox configuration array
     * @return BizzPlugin_Metabox|false Metabox instance or false on failure
     */
    public function create_metabox($args) {
        $defaults = array(
            'id'          => '',
            'title'       => '',
            'post_types'  => array('post'),
            'context'     => 'normal',
            'priority'    => 'default',
            'meta_prefix' => '',
            'capability'  => 'edit_posts',
            'is_premium'  => false,
            'sections'    => array(),
        );
        
        $args = wp_parse_args($args, $defaults);
        
        if (empty($args['id'])) {
            return false;
        }
        
        $metabox = new BizzPlugin_Metabox($args, $this);
        
        $this->metaboxes[$args['id']] = $metabox;
        
        // Allow other plugins to add fields to this metabox
        do_action('bizzplugin_metabox_created', $metabox, $args['id']);
        
        return $metabox;
    }
    
    /**
     * Get a metabox by ID
     * 
     * @param string $metabox_id Metabox ID
     * @return BizzPlugin_Metabox|null Metabox instance or null if not found
     */
    public function get_metabox($metabox_id) {
        return isset($this->metaboxes[$metabox_id]) ? $this->metaboxes[$metabox_id] : null;
    }
    
    /**
     * Get all registered metaboxes
     * 
     * @return array Array of BizzPlugin_Metabox instances
     */
    public function get_metaboxes() {
        return $this->metaboxes;
    }
    
    /**
     * AJAX save options
     */
    public function ajax_save_options() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        $option_name = isset($_POST['option_name']) ? sanitize_text_field(wp_unslash($_POST['option_name'])) : '';
        $panel_id = isset($_POST['panel_id']) ? sanitize_text_field(wp_unslash($_POST['panel_id'])) : '';
        $data = isset($_POST['data']) ? wp_unslash($_POST['data']) : array();
        $hidden_fields = isset($_POST['hidden_fields']) ? wp_unslash($_POST['hidden_fields']) : array();
        
        // Sanitize hidden fields array
        if (!is_array($hidden_fields)) {
            $hidden_fields = array();
        }
        $hidden_fields = array_map('sanitize_text_field', $hidden_fields);
        
        if (empty($option_name) || empty($panel_id)) {
            wp_send_json_error(array('message' => 'Invalid request'));
        }
        
        $panel = $this->get_panel($panel_id);
        if (!$panel) {
            wp_send_json_error(array('message' => 'Panel not found'));
        }
        
        // Sanitize and validate data
        $ajax_handler = new BizzPlugin_Ajax_Handler();
        $sanitized_data = $ajax_handler->sanitize_options($data, $panel);
        $validation_result = $ajax_handler->validate_options($sanitized_data, $panel, $hidden_fields);
        
        if (is_wp_error($validation_result)) {
            wp_send_json_error(array(
                'message' => $validation_result->get_error_message(),
                'errors' => $validation_result->get_error_data()
            ));
        }
        
        // Save options
        $old_options = get_option($option_name, array());
        $saved = update_option($option_name, $sanitized_data);
        
        // Trigger webhook if configured
        do_action('bizzplugin_options_saved', $option_name, $sanitized_data, $old_options, $panel_id);
        
        wp_send_json_success(array(
            'message' => 'Settings saved successfully!',
            'data' => $sanitized_data
        ));
    }
    
    /**
     * AJAX reset section
     */
    public function ajax_reset_section() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        $option_name = isset($_POST['option_name']) ? sanitize_text_field(wp_unslash($_POST['option_name'])) : '';
        $panel_id = isset($_POST['panel_id']) ? sanitize_text_field(wp_unslash($_POST['panel_id'])) : '';
        $section_id = isset($_POST['section_id']) ? sanitize_text_field(wp_unslash($_POST['section_id'])) : '';
        $subsection_id = isset($_POST['subsection_id']) ? sanitize_text_field(wp_unslash($_POST['subsection_id'])) : '';

        if (empty($option_name) || empty($panel_id) || empty($section_id)) {
            wp_send_json_error(array('message' => 'Invalid request'));
        }
        
        $panel = $this->get_panel($panel_id);
        if (!$panel) {
            wp_send_json_error(array('message' => 'Panel not found'));
        }
        
        // Get default values for the section
        $section_defaults = $panel->get_section_defaults($section_id, $subsection_id);
        
        // Get current options and merge with defaults
        $current_options = get_option($option_name, array());
        foreach ($section_defaults as $field_id => $default_value) {
            $current_options[$field_id] = $default_value;
        }
        
        update_option($option_name, $current_options);
        
        wp_send_json_success(array(
            'message' => 'Section reset successfully!',
            'defaults' => $section_defaults
        ));
    }
    
    /**
     * AJAX reset all options
     */
    public function ajax_reset_all() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        $option_name = isset($_POST['option_name']) ? sanitize_text_field(wp_unslash($_POST['option_name'])) : '';
        $panel_id = isset($_POST['panel_id']) ? sanitize_text_field(wp_unslash($_POST['panel_id'])) : '';
        
        if (empty($option_name) || empty($panel_id)) {
            wp_send_json_error(array('message' => 'Invalid request'));
        }
        
        $panel = $this->get_panel($panel_id);
        if (!$panel) {
            wp_send_json_error(array('message' => 'Panel not found'));
        }
        
        // Get all default values
        $all_defaults = $panel->get_all_defaults();
        
        // Update options with defaults
        update_option($option_name, $all_defaults);
        
        wp_send_json_success(array(
            'message' => 'All settings reset successfully!',
            'defaults' => $all_defaults
        ));
    }
    
    /**
     * AJAX test webhook
     */
    public function ajax_test_webhook() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        $option_name = isset($_POST['option_name']) ? sanitize_text_field(wp_unslash($_POST['option_name'])) : '';
        $webhook_url = isset($_POST['webhook_url']) ? esc_url_raw(wp_unslash($_POST['webhook_url'])) : '';
        
        if (empty($webhook_url)) {
            wp_send_json_error(array('message' => 'Webhook URL is required'));
        }
        
        // Validate URL format
        if (!filter_var($webhook_url, FILTER_VALIDATE_URL)) {
            wp_send_json_error(array('message' => 'Invalid webhook URL format'));
        }
        
        // Prepare test payload
        $payload = array(
            'event' => 'webhook_test',
            'option_name' => $option_name,
            'timestamp' => gmdate('c'),
            'site_url' => get_site_url(),
            'message' => 'This is a test webhook from BizzPlugin Options Framework',
        );
        
        // Send test webhook
        $response = wp_remote_post($webhook_url, array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-BizzPlugin-Event' => 'webhook_test',
            ),
            'body' => wp_json_encode($payload),
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array(
                'message' => $response->get_error_message(),
                'payload' => $payload
            ));
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        wp_send_json_success(array(
            'message' => 'Webhook test sent successfully!',
            'response_code' => $response_code,
            'response_body' => $response_body,
            'payload_sent' => $payload
        ));
    }
    
    /**
     * AJAX install plugin
     */
    public function ajax_install_plugin() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('install_plugins')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        $slug = isset($_POST['slug']) ? sanitize_text_field(wp_unslash($_POST['slug'])) : '';
        
        if (empty($slug)) {
            wp_send_json_error(array('message' => 'Plugin slug is required'));
        }
        
        // Validate plugin slug format (lowercase letters, numbers, hyphens only)
        if (!preg_match('/^[a-z0-9-]+$/', $slug)) {
            wp_send_json_error(array('message' => 'Invalid plugin slug format'));
        }
        
        // Include required files for plugin installation
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        
        // Get plugin info from WordPress.org
        $api = plugins_api('plugin_information', array(
            'slug' => $slug,
            'fields' => array(
                'short_description' => false,
                'sections' => false,
                'requires' => false,
                'rating' => false,
                'ratings' => false,
                'downloaded' => false,
                'last_updated' => false,
                'added' => false,
                'tags' => false,
                'compatibility' => false,
                'homepage' => false,
                'donate_link' => false,
            ),
        ));
        
        if (is_wp_error($api)) {
            wp_send_json_error(array('message' => $api->get_error_message()));
        }
        
        // Create a silent skin for the upgrader
        $skin = new WP_Ajax_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);
        
        $result = $upgrader->install($api->download_link);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }
        
        if ($result === false) {
            wp_send_json_error(array('message' => 'Plugin installation failed'));
        }
        
        wp_send_json_success(array(
            'message' => 'Plugin installed successfully!',
            'status' => 'installed'
        ));
    }
    
    /**
     * AJAX activate plugin
     */
    public function ajax_activate_plugin() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('activate_plugins')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        $plugin_file = isset($_POST['file']) ? sanitize_text_field(wp_unslash($_POST['file'])) : '';
        
        if (empty($plugin_file)) {
            wp_send_json_error(array('message' => 'Plugin file is required'));
        }
        
        // Use WordPress validate_file() for comprehensive path validation
        // Returns 0 if valid, 1 if empty, 2 if contains '..' or './', 3 if absolute path
        $validation_result = validate_file($plugin_file);
        if ($validation_result !== 0) {
            wp_send_json_error(array('message' => 'Invalid plugin file path'));
        }
        
        // Additional check: ensure plugin file matches expected format (plugin-folder/plugin-file.php)
        if (!preg_match('/^[a-z0-9_-]+\/[a-z0-9_-]+\.php$/i', $plugin_file)) {
            wp_send_json_error(array('message' => 'Invalid plugin file format'));
        }
        
        // Include plugin functions if not available
        if (!function_exists('activate_plugin')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        
        // Verify the plugin exists before activating
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $installed_plugins = get_plugins();
        if (!array_key_exists($plugin_file, $installed_plugins)) {
            wp_send_json_error(array('message' => 'Plugin not found'));
        }
        
        $result = activate_plugin($plugin_file);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }
        
        wp_send_json_success(array(
            'message' => 'Plugin activated successfully!',
            'status' => 'active'
        ));
    }
    
    /**
     * AJAX handler for generating API key
     */
    public function ajax_generate_api_key() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        // Get panel ID from POST data
        $panel_id = isset($_POST['panel_id']) ? sanitize_text_field($_POST['panel_id']) : null;
        
        if (empty($panel_id)) {
            wp_send_json_error(array('message' => 'Panel ID is required'));
        }
        
        $api_key = BizzPlugin_API_Handler::generate_api_key($panel_id);
        
        wp_send_json_success(array(
            /* translators: %s: panel ID */
            'message' => sprintf('API key generated successfully for panel "%s"!', $panel_id),
            'api_key' => $api_key,
            'panel_id' => $panel_id
        ));
    }

    /**
     * AJAX handler for deleting API key
     */
    public function ajax_delete_api_key() {
        check_ajax_referer('bizzplugin_framework_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
        }
        
        // Get panel ID from POST data
        $panel_id = isset($_POST['panel_id']) ? sanitize_text_field($_POST['panel_id']) : null;
        
        if (empty($panel_id)) {
            wp_send_json_error(array('message' => 'Panel ID is required'));
        }
        
        $deleted = BizzPlugin_API_Handler::delete_api_key($panel_id);
        
        if ($deleted) {
            wp_send_json_success(array(
                'message' => sprintf('API key deleted successfully for panel "%s"!', $panel_id),
                'panel_id' => $panel_id
            ));
        } else {
            wp_send_json_error(array('message' => 'Failed to delete API key'));
        }
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        $api_handler = new BizzPlugin_API_Handler($this);
        $api_handler->register_routes();
    }
    
    /**
     * Get framework URL
     */
    public function get_url() {
        return $this->framework_url;
    }
    
    /**
     * Get framework path
     */
    public function get_path() {
        return $this->framework_path;
    }

    /**
     * Magic method to handle calls to undefined methods
     */
    public function __call($name, $arguments)
    {
        $return = array(
            'method' => $name,
            'arguments' => $arguments,
        );
        return $return;
    }
}

/**
 * Get framework instance
 */
function bizzplugin_framework() {
    return BizzPlugin_Framework::get_instance();
}
