/**
 * BizzPlugin Options Framework - Setup Wizard JavaScript
 * 
 * Handles the setup wizard functionality for initial plugin configuration.
 * 
 * @package BizzPlugin_Options_Framework
 * @version 1.0.0
 */

(function($) {
    'use strict';

    /**
     * BizzPlugin Setup Wizard Module
     */
    const BizzPluginSetupWizardModule = {

        /**
         * Configuration from PHP
         */
        config: {},

        /**
         * DOM Elements
         */
        elements: {
            wrap: null,
            content: null,
            form: null,
            saveButton: null
        },

        /**
         * Wizard state
         */
        state: {
            steps: [],
            currentStep: 0,
            totalSteps: 0,
            hasWizardFields: false
        },

        /**
         * localStorage keys for step persistence
         */
        storageKeys: {
            currentStep: 'bizzplugin_wizard_current_step',
            isCompleted: 'bizzplugin_wizard_completed',
            thankYouReloadCount: 'bizzplugin_wizard_thankyou_reloads',
            panelId:null
        },

        /**
         * Number of reloads on thank you page before clearing localStorage
         */
        thankYouReloadThreshold: 1,

        /**
         * Initialize the setup wizard
         */
        init() {
            
            // Cache DOM elements
            this.cacheElements();            

            // Only initialize if we're on a setup wizard page
            if (!this.elements.wrap.hasClass('bizzplugin-setup-wizard-active')) {
                return;
            }

            this.config = $(this.elements.wrap).data('setup_wizard_localize');

            if(!this.config || !this.config.isActive) {
                return;
            }

            // console.log($(this.elements.wrap).data('setup_wizard_localize'));
            // Build steps from sections with setup_wizard fields
            this.buildSteps();

            // Initialize wizard UI
            this.initWizardUI();

            // Bind events
            this.bindEvents();
        },

        /**
         * Cache DOM elements
         */
        cacheElements() {
            this.elements.wrap = $('.bizzplugin-framework-wrap');
            this.elements.content = this.elements.wrap.find('.bizzplugin-content');
            this.elements.form = $('#bizzplugin-options-form');
            this.elements.saveButton = $('#bizzplugin-save-options');
        },

        /**
         * Build wizard steps from sections with setup_wizard fields
         */
        buildSteps() {
            // return;
            const self = this;
            const steps = [];

            // Find all sections and subsections that have fields with setup_wizard=true
            this.elements.form.find('.bizzplugin-section').each(function() {
                const $section = $(this);
                const sectionId = $section.data('section');
                const subsectionId = $section.data('subsection');
                
                // Find fields with setup_wizard attribute
                const $wizardFields = $section.find('.bizzplugin-field[data-setup_wizard="1"]');
                
                if ($wizardFields.length > 0) {
                    // Get section/subsection info
                    const $header = $section.find('.bizzplugin-section-header');
                    const title = $header.find('h2').text() || '';
                    const description = $header.find('.bizzplugin-section-desc').text() || '';

                    steps.push({
                        sectionId: sectionId,
                        subsectionId: subsectionId || null,
                        elementId: subsectionId ? 'subsection-' + subsectionId : 'section-' + sectionId,
                        title: title,
                        description: description,
                        $element: $section,
                        $fields: $wizardFields
                    });
                }
            });

            this.state.steps = steps;
            this.state.totalSteps = steps.length;
            this.state.hasWizardFields = steps.length > 0;
        },

        /**
         * Initialize wizard UI
         */
        initWizardUI() {
            // Create wizard container
            this.createWizardContainer();

            // If no wizard fields, show thank you page directly
            if (!this.state.hasWizardFields) {
                this.showThankYouPage();
                return;
            }

            // Check if wizard was previously completed (show thank you page)
            if (this.isWizardCompleted()) {
                this.handleThankYouPageReload();
                return;
            }

            // Restore last visited step from localStorage, or start at step 0
            const savedStep = this.getSavedStep();
            this.showStep(savedStep);
        },

        /**
         * Create wizard container with header, progress bar, navigation
         * 
         * The DOM structure will be:
         * .bizzplugin-wizard-wrapper (from PHP)
         *   ├── .bizzplugin-wizard-header (prepended by JS)
         *   ├── .bizzplugin-wizard-progress-wrap (prepended by JS, after header)
         *   ├── .bizzplugin-wizard-step-container (from PHP, contains form fields)
         *   │     └── .bizzplugin-wizard-step-header (prepended by JS in showStep)
         *   ├── .bizzplugin-wizard-navigation (appended by JS)
         *   └── .bizzplugin-wizard-thank-you (appended by JS)
         */
        createWizardContainer() {
            
            const config = this.config.config;
            
            // Get the existing wizard wrapper (rendered by PHP)
            const $wizardWrapper = $('.bizzplugin-wizard-wrapper');
            
            // Get the existing step container (rendered by PHP, contains form content)
            const $stepContainer = $wizardWrapper.find('.bizzplugin-wizard-step-container');
            
            // Create header
            const $header = $('<div class="bizzplugin-wizard-header"></div>');
            const $title = $('<h1 class="bizzplugin-wizard-title"></h1>').text(this.config.setupTitle);
            $header.append($title);
            
            if (config.skipButton) {
                const $skipBtn = $('<button type="button" class="bizzplugin-wizard-skip-btn"></button>').text(config.skipButtonText);
                $header.append($skipBtn);
            }
            
            // Create progress bar
            const $progressWrap = $('<div class="bizzplugin-wizard-progress-wrap"></div>');
            if (!config.showProgress) {
                $progressWrap.hide();
            }
            const $progressBar = $('<div class="bizzplugin-wizard-progress-bar"><div class="bizzplugin-wizard-progress-fill" style="width: 0%;"></div></div>');
            const $progressText = $('<div class="bizzplugin-wizard-progress-text"><span class="bizzplugin-wizard-step-current">1</span> / <span class="bizzplugin-wizard-step-total">' + (this.state.totalSteps || 1) + '</span></div>');
            $progressWrap.append($progressBar).append($progressText);
            
            // Create navigation
            const $navigation = $('<div class="bizzplugin-wizard-navigation"></div>');
            const $prevBtn = $('<button type="button" class="bizzplugin-wizard-prev-btn" style="display:none;"><span class="dashicons dashicons-arrow-left-alt2"></span> </button>');
            $prevBtn.append(document.createTextNode(config.prevText));
            
            const $nextBtn = $('<button type="button" class="bizzplugin-wizard-next-btn"></button>');
            $nextBtn.append(document.createTextNode(config.nextText + ' '));
            $nextBtn.append('<span class="dashicons dashicons-arrow-right-alt2"></span>');
            
            $navigation.append($prevBtn).append($nextBtn);
            
            // Create thank you page container (hidden)
            const $thankYouPage = $('<div class="bizzplugin-wizard-thank-you" style="display:none;"></div>');
            const $thankYouIcon = $('<div class="bizzplugin-wizard-thank-you-icon"><span class="dashicons dashicons-yes-alt"></span></div>');
            const $thankYouTitle = $('<h2 class="bizzplugin-wizard-thank-you-title"></h2>').text(config.thankYouTitle);
            const $thankYouMessage = $('<p class="bizzplugin-wizard-thank-you-message"></p>').text(config.thankYouMessage);
            
            const $thankYouActions = $('<div class="bizzplugin-wizard-thank-you-actions"></div>');
            const $settingsLink = $('<a class="button button-primary"></a>').attr('href', this.config.settingsUrl).html( "<i class='dashicons dashicons-admin-generic'></i>" + config.settingsLinkText);
            const $visitSiteLink = $('<a class="button" target="_blank"></a>').attr('href', this.config.siteUrl).html("<i class='dashicons dashicons-admin-site'></i>" + config.visitSiteText);

            const links = config.links;
            console.log(links,config);

            //check array count and append links
            if (links && Array.isArray(links) && links.length > 0) {
                links.forEach(link => {
                    const icon = link.icon ? '<i class="' + link.icon + '"></i> ' : '';
                    const $link = $('<a class="button ' + (link.class || '') + '" target="' + (link.target || '_blank') + '"></a>').attr('href', link.url).html(icon + link.text);
                    $thankYouActions.append($link);
                });
            }else{  
                $thankYouActions.append($settingsLink).append($visitSiteLink);
            }

            
            
            const $closeBtn = $('<button type="button" class="bizzplugin-wizard-close-btn"><span class="dashicons dashicons-no-alt"></span> </button>');
            $closeBtn.append(document.createTextNode(config.closeText));
            const $editBtn = $('<button type="button" class="bizzplugin-wizard-edit-btn"><span class="dashicons dashicons-edit"></span> </button>');
            $editBtn.append(document.createTextNode(config.editText));
            
            $thankYouPage.append($thankYouIcon).append($thankYouTitle).append($thankYouMessage).append($thankYouActions).append($closeBtn).append($editBtn);
            
            // Assemble wizard wrapper with proper order:
            // Note: prepend() inserts at the beginning, so we prepend in reverse order
            // First prepend $progressWrap (will be second child)
            // Then prepend $header (will be first child, before $progressWrap)
            $wizardWrapper.prepend($progressWrap);
            $wizardWrapper.prepend($header);
            $stepContainer.after($navigation);
            $navigation.after($thankYouPage);
            
            // Store references
            this.elements.wizardWrapper = $wizardWrapper;
            this.elements.stepContainer = $stepContainer;
            this.elements.progressFill = $wizardWrapper.find('.bizzplugin-wizard-progress-fill');
            this.elements.stepCurrent = $wizardWrapper.find('.bizzplugin-wizard-step-current');
            this.elements.prevBtn = $wizardWrapper.find('.bizzplugin-wizard-prev-btn');
            this.elements.nextBtn = $wizardWrapper.find('.bizzplugin-wizard-next-btn');
            this.elements.thankYouPage = $wizardWrapper.find('.bizzplugin-wizard-thank-you');
            this.elements.wizardWrapper.removeClass('bizzplugin-temp-name');
        },

        /**
         * Bind events
         */
        bindEvents() {
            const self = this;

            // Previous button
            $(document).on('click', '.bizzplugin-wizard-prev-btn', function(e) {
                e.preventDefault();
                self.goToPrevStep();
            });

            // Next button
            $(document).on('click', '.bizzplugin-wizard-next-btn', function(e) {
                e.preventDefault();
                self.goToNextStep();
            });

            // Skip button
            $(document).on('click', '.bizzplugin-wizard-skip-btn', function(e) {
                e.preventDefault();
                self.skipWizard();
            });

            // Close button
            $(document).on('click', '.bizzplugin-wizard-close-btn', function(e) {
                e.preventDefault();
                window.location.href = self.config.settingsUrl;
            });
            // Edit button and go to first step
            $(document).on('click', '.bizzplugin-wizard-edit-btn', function(e) {
                e.preventDefault();
                self.clearWizardStorage();
                //and reload the page
                location.reload();
            });
        },

        /**
         * Show a specific step
         */
        showStep(stepIndex) {
            if (stepIndex < 0 || stepIndex >= this.state.totalSteps) {
                return;
            }

            const step = this.state.steps[stepIndex];
            this.state.currentStep = stepIndex;

            // Save current step to localStorage for persistence on reload
            this.saveCurrentStepToStorage(stepIndex);

            // Hide all sections
            this.elements.form.find('.bizzplugin-section').hide();

            // Show current step section
            step.$element.show();

            // Hide all fields in the section first, then show only wizard fields
            step.$element.find('.bizzplugin-field').hide();
            step.$fields.show();

            // Update step container header
            // Remove existing step header (if any) and prepend new one
            // Note: We prepend instead of append so the header appears BEFORE the form fields
            this.elements.stepContainer.find('.bizzplugin-wizard-step-header').remove();
            const $stepHeader = $('<div class="bizzplugin-wizard-step-header"></div>');
            const $stepTitle = $('<h2 class="bizzplugin-wizard-step-title"></h2>').text(step.title);
            $stepHeader.append($stepTitle);
            
            if (step.description) {
                const $stepDesc = $('<p class="bizzplugin-wizard-step-desc"></p>').text(step.description);
                $stepHeader.append($stepDesc);
            }
            this.elements.stepContainer.prepend($stepHeader);

            // Update progress bar
            this.updateProgress();

            // Update navigation buttons
            this.updateNavigation();
        },

        /**
         * Go to previous step
         */
        goToPrevStep() {
            if (this.state.currentStep > 0) {
                this.showStep(this.state.currentStep - 1);
            }
        },

        /**
         * Go to next step
         */
        goToNextStep() {
            const self = this;
            
            // Save settings before moving to next step
            this.saveCurrentStep(function() {
                if (self.state.currentStep < self.state.totalSteps - 1) {
                    self.showStep(self.state.currentStep + 1);
                } else {
                    // Last step - show finish/thank you
                    self.finishWizard();
                }
            });
        },

        /**
         * Save current step settings
         */
        saveCurrentStep(callback) {
            const self = this;
            
            // Check if framework is available
            if (typeof bizzpluginFramework === 'undefined' || !window.BizzPluginFramework) {
                // Show warning that save functionality is not available
                console.warn('BizzPlugin Framework not available. Settings will not be saved.');
                if (window.BizzPluginCommon) {
                    window.BizzPluginCommon.showNotification('Save functionality unavailable. Please reload the page.', 'warning', self.elements.wrap, 3000);
                }
                if (typeof callback === 'function') {
                    callback();
                }
                return;
            }
            
            // Get data attributes with fallbacks
            const panelId = this.elements.wrap.data('panel-id') || '';
            const optionName = this.elements.wrap.data('option-name') || '';
            
            // Validate required data
            if (!panelId || !optionName) {
                console.error('Missing panel-id or option-name data attributes');
                if (typeof callback === 'function') {
                    callback();
                }
                return;
            }
            
            // Disable next button temporarily
            this.elements.nextBtn.prop('disabled', true).addClass('bizzplugin-loading');
            
            // Use the framework's save functionality
            $.ajax({
                url: bizzpluginFramework.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_save_options',
                    nonce: bizzpluginFramework.nonce,
                    panel_id: panelId,
                    option_name: optionName,
                    data: window.BizzPluginFramework.serializeFormData()
                },
                success: function(response) {
                    if (response.success) {
                        if (typeof callback === 'function') {
                            callback();
                        }
                    } else {
                        // Show error notification
                        if (window.BizzPluginCommon) {
                            window.BizzPluginCommon.showNotification(response.data.message || 'Error saving settings', 'error', self.elements.wrap, 3000);
                        }
                    }
                },
                error: function() {
                    if (window.BizzPluginCommon) {
                        window.BizzPluginCommon.showNotification('Error saving settings', 'error', self.elements.wrap, 3000);
                    }
                },
                complete: function() {
                    self.elements.nextBtn.prop('disabled', false).removeClass('bizzplugin-loading');
                }
            });
        },

        /**
         * Update progress bar
         */
        updateProgress() {
            const progress = ((this.state.currentStep + 1) / this.state.totalSteps) * 100;
            
            // Animate progress bar
            this.elements.progressFill.css('width', progress + '%');
            
            // Update step counter
            this.elements.stepCurrent.text(this.state.currentStep + 1);
        },

        /**
         * Update navigation buttons
         */
        updateNavigation() {
            const config = this.config.config;
            
            // Previous button visibility
            if (this.state.currentStep === 0) {
                this.elements.prevBtn.hide();
            } else {
                this.elements.prevBtn.show();
            }

            // Next button text (change to Finish on last step)
            this.elements.nextBtn.empty();
            if (this.state.currentStep === this.state.totalSteps - 1) {
                this.elements.nextBtn.append(document.createTextNode(config.finishText + ' '));
                this.elements.nextBtn.append('<span class="dashicons dashicons-yes"></span>');
            } else {
                this.elements.nextBtn.append(document.createTextNode(config.nextText + ' '));
                this.elements.nextBtn.append('<span class="dashicons dashicons-arrow-right-alt2"></span>');
            }
        },

        /**
         * Finish wizard
         */
        finishWizard() {
            const self = this;
            
            // Save final step
            this.saveCurrentStep(function() {
                self.showThankYouPage();
            });
        },

        /**
         * Skip wizard
         */
        skipWizard() {
            this.showThankYouPage();
        },

        /**
         * Show thank you page
         */
        showThankYouPage() {
            // Mark wizard as completed in localStorage
            this.setWizardCompleted();
            
            // Hide step container and navigation
            this.elements.stepContainer.hide();
            // this.elements.wrap.find('.bizzplugin-wizard-header').hide();
            this.elements.wrap.find('.bizzplugin-wizard-navigation').hide();
            this.elements.wrap.find('.bizzplugin-wizard-progress-wrap').hide();
            this.elements.wrap.find('.bizzplugin-wizard-skip-btn').hide();
            
            // Hide all sections
            this.elements.form.find('.bizzplugin-section').hide();
            
            // Hide the form footer
            this.elements.wrap.find('.bizzplugin-footer').hide();
            
            // Show thank you page with animation
            this.elements.thankYouPage.fadeIn(300);
        },

        /**
         * Get saved step from localStorage
         * @returns {number} The saved step index, or 0 if not found
         */
        getSavedStep() {
            try {
                const savedStep = localStorage.getItem(this.storageKeys.currentStep);
                if (savedStep !== null) {
                    const stepIndex = parseInt(savedStep, 10);
                    // Validate step is within bounds
                    if (!isNaN(stepIndex) && stepIndex >= 0 && stepIndex < this.state.totalSteps) {
                        return stepIndex;
                    }
                }
            } catch (e) {
                console.warn('localStorage not available:', e);
            }
            return 0;
        },

        /**
         * Save current step to localStorage
         * @param {number} stepIndex The step index to save
         */
        saveCurrentStepToStorage(stepIndex) {
            try {
                localStorage.setItem(this.storageKeys.currentStep, stepIndex.toString());
            } catch (e) {
                console.warn('localStorage not available:', e);
            }
        },

        /**
         * Check if wizard was previously completed
         * @returns {boolean} True if wizard is completed
         */
        isWizardCompleted() {
            try {
                return localStorage.getItem(this.storageKeys.isCompleted) === 'true';
            } catch (e) {
                console.warn('localStorage not available:', e);
            }
            return false;
        },

        /**
         * Mark wizard as completed in localStorage
         */
        setWizardCompleted() {
            try {
                localStorage.setItem(this.storageKeys.isCompleted, 'true');
                // Clear the current step when completed
                localStorage.removeItem(this.storageKeys.currentStep);
            } catch (e) {
                console.warn('localStorage not available:', e);
            }
        },

        /**
         * Handle thank you page reload - clear localStorage after threshold reloads
         */
        handleThankYouPageReload() {
            try {
                let reloadCount = parseInt(localStorage.getItem(this.storageKeys.thankYouReloadCount) || '0', 10);
                reloadCount++;
                
                if (reloadCount >= this.thankYouReloadThreshold) {
                    // Clear all wizard localStorage data after threshold reloads and restart wizard
                    this.clearWizardStorage();
                } else {
                    // Increment reload count
                    localStorage.setItem(this.storageKeys.thankYouReloadCount, reloadCount.toString());
                    // Show thank you page
                    this.showThankYouPage();
                }
            } catch (e) {
                console.warn('localStorage not available:', e);
                this.showThankYouPage();
            }
        },

        /**
         * Clear all wizard-related localStorage data
         */
        clearWizardStorage() {
            try {
                localStorage.removeItem(this.storageKeys.currentStep);
                localStorage.removeItem(this.storageKeys.isCompleted);
                localStorage.removeItem(this.storageKeys.thankYouReloadCount);
            } catch (e) {
                console.warn('localStorage not available:', e);
            }
            // After clearing, show first step
            this.showStep(0);
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(function() {
        BizzPluginSetupWizardModule.init();
    });

    /**
     * Expose to global scope for extensibility
     */
    window.BizzPluginSetupWizard = BizzPluginSetupWizardModule;

})(jQuery);