/**
 * BizzPlugin Options Framework - Main JavaScript
 * 
 * Panel-specific functionality for the options framework admin interface.
 * Common functionality is handled by bizzplugin-common.js
 * 
 * @package BizzPlugin_Options_Framework
 * @version 2.0.0
 * @requires bizzplugin-common.js
 */

(function($) {
    'use strict';

    /**
     * BizzPlugin Framework Module
     */
    const BizzPluginFramework = {
        
        /**
         * Configuration
         */
        config: {
            ajaxUrl: '',
            nonce: '',
            strings: {}
        },

        /**
         * DOM Elements
         */
        elements: {
            body: document.body,
            wrap: null,
            form: null,
            saveButton: null,
            saveStatus: null,
            navLinks: null,
            navSublinks: null,
            sections: null
        },

        /**
         * Initialize the framework
         */
        init() {
            // Set configuration from localized data
            if (typeof bizzpluginFramework !== 'undefined') {
                this.config.ajaxUrl = bizzpluginFramework.ajaxUrl;
                this.config.nonce = bizzpluginFramework.nonce;
                this.config.strings = bizzpluginFramework.strings;
                
                // Set common module config
                if (typeof window.BizzPluginCommon !== 'undefined') {
                    window.BizzPluginCommon.setConfig(this.config);
                }
            }

            // Cache DOM elements
            this.cacheElements();

            // Only initialize if we're on a framework page
            if (!this.elements.wrap.length) {
                return;
            }

            // Bind events
            this.bindEvents();

            // Initialize components using common module
            this.initComponents();
        },

        /**
         * Cache DOM elements
         */
        cacheElements() {
            this.elements.wrap = $('.bizzplugin-framework-wrap');
            this.elements.form = $('#bizzplugin-options-form');
            this.elements.saveButton = $('#bizzplugin-save-options');
            this.elements.saveStatus = $('.bizzplugin-save-status');
            this.elements.navLinks = $('.bizzplugin-nav-link');
            this.elements.navSublinks = $('.bizzplugin-nav-sublink');
            this.elements.sections = $('.bizzplugin-section');
        },

        /**
         * Initialize all components
         */
        initComponents() {
            const self = this;
            const Common = window.BizzPluginCommon;
            
            // Use common module for field initialization
            if (Common) {
                Common.initColorPickers(null, () => {
                    self.elements.saveButton.addClass('button-primary-changed');
                });
                Common.initAlphaColorPickers(null, () => {
                    self.elements.saveButton.addClass('button-primary-changed');
                });
                Common.initDatePickers(null, () => {
                    self.elements.saveButton.addClass('button-primary-changed');
                });
                Common.initSliders();
                Common.initImageSelect();
                Common.initOptionSelect();
                Common.initRepeaterSortable(null, ($wrap) => {
                    self.repeaterReindex($wrap);
                    self.elements.saveButton.addClass('button-primary-changed');
                });
                Common.initSingleRepeaterSortable(null, ($wrap) => {
                    self.elements.saveButton.addClass('button-primary-changed');
                });
                
                Common.initConditionalFields();
            }
            
            // Framework-specific initialization
            // this.initConditionalFields();
            this.initApiDocsTabs();
            this.initSearch();
            this.initWebhooks();
            this.initRepeaterColorPickers();
            this.initThemeSwitcher();
        },

        /**
         * Bind events
         */
        bindEvents() {
            const self = this;
            const Common = window.BizzPluginCommon;
            
            // Navigation - main section links
            this.elements.navLinks.on('click', this.handleMainNavigation.bind(this));
            
            // Navigation - arrow icon click for toggle
            $(document).on('click', '.bizzplugin-nav-arrow', this.handleArrowClick.bind(this));
            
            // Navigation - subsection links
            this.elements.navSublinks.on('click', this.handleSubNavigation.bind(this));

            // Save options
            this.elements.saveButton.on('click', this.saveOptions.bind(this));

            // Reset section
            $(document).on('click', '.bizzplugin-reset-section', this.resetSection.bind(this));
            
            // Reset all
            $(document).on('click', '.bizzplugin-reset-all', this.resetAll.bind(this));
            
            // Export options
            $(document).on('click', '#bizzplugin-export-options', this.exportOptions.bind(this));
            
            // Import options - trigger file input
            $(document).on('click', '#bizzplugin-import-trigger', this.triggerImport.bind(this));
            
            // Import options - file change
            $(document).on('change', '#bizzplugin-import-file', this.importOptions.bind(this));
            
            // Theme switcher
            $(document).on('click', '.bizzplugin-theme-switcher', this.toggleTheme.bind(this));

            // Image upload - use common module
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-image-select', function(e) {
                if (Common) {
                    Common.openImageUploader(e, () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    });
                }
            });
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-image-remove', function(e) {
                if (Common) {
                    Common.removeImage(e, () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    });
                }
            });

            // File upload - use common module
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-file-select', function(e) {
                if (Common) {
                    Common.openFileUploader(e, () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    });
                }
            });
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-file-remove', function(e) {
                if (Common) {
                    Common.removeFile(e, () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    });
                }
            });

            // On/Off toggle - trigger conditional fields
            $(document).on('change', '.bizzplugin-toggle-input, .bizzplugin-checkbox', function(e) {
                if (Common) {
                    Common.handleToggleChange(e);
                }
            });

            // Select change - trigger conditional fields
            $(document).on('change', '.bizzplugin-select, .bizzplugin-radio, .bizzplugin-option-select-input', function(e) {
                if (Common) {
                    Common.handleSelectChange(e);
                }
            });

            // Image select - use common module
            $(document).on('change', '.bizzplugin-framework-wrap .bizzplugin-image-select-input', function(e) {
                if (Common) {
                    Common.handleImageSelectChange(e);
                }
            });
            
            // Option select - use common module
            $(document).on('change', '.bizzplugin-framework-wrap .bizzplugin-option-select-input', function(e) {
                if (Common) {
                    Common.handleOptionSelectChange(e);
                }
            });
            
            // Slider input - use common module
            $(document).on('input', '.bizzplugin-framework-wrap .bizzplugin-slider', function(e) {
                if (Common) {
                    Common.handleSliderChange(e, () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    });
                }
            });

            // Form change detection
            this.elements.form.on('change', 'input, select, textarea', () => {
                this.elements.saveButton.addClass('button-primary-changed');
            });

            // Keyboard shortcut for saving (Ctrl+S / Cmd+S)
            $(document).on('keydown', (e) => {
                if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                    e.preventDefault();
                    this.saveOptions();
                }
            });
            
            // Copy button - use common module
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-copy-btn', function(e) {
                if (Common) {
                    Common.handleCopyClick(e);
                }
            });
            
            // Test webhook button
            $(document).on('click', '#bizzplugin-test-webhook', this.testWebhook.bind(this));
            
            // Generate API key button
            $(document).on('click', '#bizzplugin-generate-api-key', this.generateApiKey.bind(this));
            
            // API docs tabs
            $(document).on('click', '.bizzplugin-docs-tab', this.handleDocsTabClick.bind(this));
            
            // Plugin install/activate buttons - use common module
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-install-plugin', function(e) {
                self.installPlugin(e);
            });
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-activate-plugin', function(e) {
                self.activatePlugin(e);
            });
            
            // Search functionality
            $(document).on('input', '#bizzplugin-settings-search', this.handleSearchInput.bind(this));
            $(document).on('click', '.bizzplugin-search-clear', this.clearSearch.bind(this));
            
            // Repeater field events - use common module
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-repeater-add', function(e) {
                self.repeaterAddItem(e);
            });
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-repeater-item-remove', function(e) {
                self.repeaterRemoveItem(e);
            });
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-repeater-item-toggle', function(e) {
                if (Common) {
                    Common.repeaterToggleItem(e);
                }
            });
            // Update repeater item title on input/change for all field types
            $(document).on('input change', '.bizzplugin-framework-wrap .bizzplugin-repeater-item input, .bizzplugin-framework-wrap .bizzplugin-repeater-item select, .bizzplugin-framework-wrap .bizzplugin-repeater-item textarea', function(e) {
                if (Common) {
                    Common.repeaterUpdateItemTitle(e);
                }
            });
            
            // Single repeater field events - use common module
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-single-repeater-add', function(e) {
                self.singleRepeaterAddItem(e);
            });
            $(document).on('click', '.bizzplugin-framework-wrap .bizzplugin-single-repeater-remove', function(e) {
                self.singleRepeaterRemoveItem(e);
            });
        },

        /**
         * Handle arrow icon click for toggling submenu
         */
        handleArrowClick(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const $arrow = $(e.currentTarget);
            const $navItem = $arrow.closest('.bizzplugin-nav-item');
            const $submenu = $navItem.find('.bizzplugin-nav-submenu');
            
            if ($navItem.hasClass('expanded')) {
                // Collapse this submenu
                $navItem.removeClass('expanded');
                $submenu.slideUp(200);
            } else {
                // Expand this submenu
                $navItem.addClass('expanded');
                $submenu.slideDown(200);
            }
        },

        /**
         * Handle main navigation click
         */
        handleMainNavigation(e) {
            e.preventDefault();
            
            // Check if arrow was clicked (handle separately)
            if ($(e.target).hasClass('bizzplugin-nav-arrow')) {
                return;
            }
            
            const $link = $(e.currentTarget);
            const sectionId = $link.data('section');
            const hasFields = $link.data('has-fields') === 1 || $link.data('has-fields') === '1';
            const hasSubsections = $link.data('has-subsections') === 1 || $link.data('has-subsections') === '1';
            
            // Collapse all other expanded menus
            $('.bizzplugin-nav-item').not($link.closest('.bizzplugin-nav-item')).removeClass('expanded');
            $('.bizzplugin-nav-item').not($link.closest('.bizzplugin-nav-item')).find('.bizzplugin-nav-submenu').slideUp(200);
            
            // Update active state on nav items
            $('.bizzplugin-nav-item').removeClass('active');
            $('.bizzplugin-nav-link').removeClass('current');
            $('.bizzplugin-nav-sublink').removeClass('current');
            $('.bizzplugin-nav-subitem').removeClass('active');
            
            $link.closest('.bizzplugin-nav-item').addClass('active');
            
            // Handle subsection expansion/collapse
            if (hasSubsections) {
                $link.closest('.bizzplugin-nav-item').addClass('expanded');
                $link.closest('.bizzplugin-nav-item').find('.bizzplugin-nav-submenu').slideDown(200);
            }
            
            // If section has no fields but has subsections, auto-select first subsection
            if (!hasFields && hasSubsections) {
                const $firstSublink = $link.closest('.bizzplugin-nav-item').find('.bizzplugin-nav-sublink').first();
                if ($firstSublink.length) {
                    $firstSublink.trigger('click');
                    return;
                }
            }
            
            // Mark this link as current
            $link.addClass('current');

            // Show/hide sections - only show section content, not subsections
            this.elements.sections.hide();
            $(`#section-${sectionId}`).fadeIn(200);

            // Update URL without page reload
            const url = new URL(window.location);
            url.searchParams.set('section', sectionId);
            url.searchParams.delete('subsection');
            window.history.replaceState({}, '', url);
        },

        /**
         * Handle subsection navigation click
         */
        handleSubNavigation(e) {
            e.preventDefault();
            
            const $link = $(e.currentTarget);
            const sectionId = $link.data('section');
            const subsectionId = $link.data('subsection');

            // Update active state
            $('.bizzplugin-nav-item').removeClass('active');
            $('.bizzplugin-nav-link').removeClass('current');
            $('.bizzplugin-nav-sublink').removeClass('current');
            $('.bizzplugin-nav-subitem').removeClass('active');
            
            $link.closest('.bizzplugin-nav-item').addClass('active expanded');
            $link.addClass('current');
            $link.closest('.bizzplugin-nav-subitem').addClass('active');
            
            // Keep submenu visible
            $link.closest('.bizzplugin-nav-item').find('.bizzplugin-nav-submenu').show();

            // Show/hide sections - show subsection content
            this.elements.sections.hide();
            $(`#subsection-${subsectionId}`).fadeIn(200);

            // Update URL without page reload
            const url = new URL(window.location);
            url.searchParams.set('section', sectionId);
            url.searchParams.set('subsection', subsectionId);
            window.history.replaceState({}, '', url);
        },

        /**
         * Save options via AJAX
         */
        saveOptions() {
            const $button = this.elements.saveButton;
            
            if ($button.prop('disabled')) {
                return;
            }

            const panelId = this.elements.wrap.data('panel-id');
            const optionName = this.elements.wrap.data('option-name');
            const formData = this.serializeFormData();
            const hiddenFieldIds = this.getHiddenFieldIds();

            // Clear previous errors
            this.clearValidationErrors();

            // Update UI
            this.setStatus('saving');
            $button.prop('disabled', true).addClass('bizzplugin-loading');

            // Send AJAX request
            $.ajax({
                url: this.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_save_options',
                    nonce: this.config.nonce,
                    panel_id: panelId,
                    option_name: optionName,
                    data: formData,
                    hidden_fields: hiddenFieldIds
                },
                success: (response) => {
                    if (response.success) {
                        this.setStatus('saved');
                        $button.removeClass('button-primary-changed');
                        
                        // Trigger custom event
                        $(document).trigger('bizzplugin:options_saved', [formData, response.data]);
                    } else {
                        this.setStatus('error', response.data.message);
                        
                        // Show validation errors
                        if (response.data.errors) {
                            this.showValidationErrors(response.data.errors);
                        }
                    }
                },
                error: (xhr, status, error) => {
                    this.setStatus('error', this.config.strings.error);
                    console.error('BizzPlugin Framework Error:', error);
                },
                complete: () => {
                    $button.prop('disabled', false).removeClass('bizzplugin-loading');
                }
            });
        },

        /**
         * Reset section to defaults
         */
        resetSection(e) {
            e.preventDefault();
            
            if (!confirm(this.config.strings.confirm_reset)) {
                return;
            }

            const $button = $(e.currentTarget);
            const sectionId = $button.data('section');
            const subsectionId = $button.data('subsection');
            const panelId = this.elements.wrap.data('panel-id');
            const optionName = this.elements.wrap.data('option-name');

            $button.prop('disabled', true);
            this.setStatus('saving', this.config.strings.resetting);

            $.ajax({
                url: this.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_reset_section',
                    nonce: this.config.nonce,
                    panel_id: panelId,
                    option_name: optionName,
                    section_id: sectionId,
                    subsection_id: subsectionId || ''
                },
                success: (response) => {
                    if (response.success) {
                        this.setStatus('saved', this.config.strings.reset_success);
                        
                        // Update field values with defaults
                        this.updateFieldValues(response.data.defaults);
                        
                        // Trigger custom event
                        $(document).trigger('bizzplugin:section_reset', [sectionId, response.data.defaults]);
                    } else {
                        this.setStatus('error', response.data.message);
                    }
                },
                error: () => {
                    this.setStatus('error', this.config.strings.reset_error);
                },
                complete: () => {
                    $button.prop('disabled', false);
                }
            });
        },

        /**
         * Reset all settings
         */
        resetAll(e) {
            e.preventDefault();
            
            if (!confirm(this.config.strings.confirm_reset_all || 'Are you sure you want to reset all settings to default values?')) {
                return;
            }
            
            const panelId = this.elements.wrap.data('panel-id');
            const optionName = this.elements.wrap.data('option-name');
            const $button = $(e.currentTarget);
            
            $button.prop('disabled', true);
            this.setStatus('saving', this.config.strings.resetting);
            
            // Send AJAX request to reset all options
            $.ajax({
                url: this.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_reset_all',
                    nonce: this.config.nonce,
                    panel_id: panelId,
                    option_name: optionName
                },
                success: (response) => {
                    if (response.success) {
                        this.setStatus('saved', this.config.strings.reset_all_success || 'All settings reset successfully!');
                        
                        // Update field values with defaults
                        this.updateFieldValues(response.data.defaults);
                        
                        // Trigger custom event
                        $(document).trigger('bizzplugin:all_reset', [response.data.defaults]);
                    } else {
                        this.setStatus('error', response.data.message);
                    }
                },
                error: () => {
                    this.setStatus('error', this.config.strings.reset_error);
                },
                complete: () => {
                    $button.prop('disabled', false);
                }
            });
        },

        /**
         * Serialize form data
         */
        serializeFormData() {
            const formData = {};
            const $form = this.elements.form;
            
            // Selector to exclude hidden template inputs from form serialization
            const templateExclude = '.bizzplugin-repeater-template input, .bizzplugin-repeater-template textarea, .bizzplugin-repeater-template select, .bizzplugin-single-repeater-template input, .bizzplugin-single-repeater-template textarea, .bizzplugin-single-repeater-template select';

            // Text inputs, textareas, selects (exclude hidden template inputs)
            $form.find('input:not([type="checkbox"]):not([type="radio"]), textarea, select').not(templateExclude).each(function() {
                const $field = $(this);
                const name = $field.attr('name');
                
                if (!name) return;

                // Handle repeater fields (e.g., repeater_field[0][subfield])
                const repeaterMatch = name.match(/^([^\[]+)\[(\d+)\]\[([^\]]+)\]$/);
                if (repeaterMatch) {
                    const fieldName = repeaterMatch[1];
                    const index = parseInt(repeaterMatch[2]);
                    const subFieldName = repeaterMatch[3];
                    
                    if (!formData[fieldName]) {
                        formData[fieldName] = [];
                    }
                    if (!formData[fieldName][index]) {
                        formData[fieldName][index] = {};
                    }
                    formData[fieldName][index][subFieldName] = $field.val();
                    return;
                }
                
                // Handle single_repeater fields (e.g., single_repeater_field[0])
                const singleRepeaterMatch = name.match(/^([^\[]+)\[(\d+)\]$/);
                if (singleRepeaterMatch) {
                    const fieldName = singleRepeaterMatch[1];
                    const index = parseInt(singleRepeaterMatch[2]);
                    
                    if (!formData[fieldName]) {
                        formData[fieldName] = [];
                    }
                    formData[fieldName][index] = $field.val();
                    return;
                }

                // Handle array fields (multi-select, checkbox groups)
                if (name.endsWith('[]')) {
                    const baseName = name.slice(0, -2);
                    if (!formData[baseName]) {
                        formData[baseName] = [];
                    }
                    if ($field.is('select[multiple]')) {
                        formData[baseName] = $field.val() || [];
                    } else {
                        formData[baseName].push($field.val());
                    }
                } else {
                    formData[name] = $field.val();
                }
            });

            // Checkboxes (exclude hidden template inputs)
            $form.find('input[type="checkbox"]').not(templateExclude).each(function() {
                const $checkbox = $(this);
                const name = $checkbox.attr('name');
                
                if (!name) return;

                // Handle repeater checkbox fields (e.g., repeater_field[0][subfield])
                const repeaterMatch = name.match(/^([^\[]+)\[(\d+)\]\[([^\]]+)\]$/);
                if (repeaterMatch) {
                    const fieldName = repeaterMatch[1];
                    const index = parseInt(repeaterMatch[2]);
                    const subFieldName = repeaterMatch[3];
                    
                    if (!formData[fieldName]) {
                        formData[fieldName] = [];
                    }
                    if (!formData[fieldName][index]) {
                        formData[fieldName][index] = {};
                    }
                    formData[fieldName][index][subFieldName] = $checkbox.is(':checked') ? '1' : '0';
                    return;
                }

                if (name.endsWith('[]')) {
                    const baseName = name.slice(0, -2);
                    if (!formData[baseName]) {
                        formData[baseName] = [];
                    }
                    if ($checkbox.is(':checked')) {
                        formData[baseName].push($checkbox.val());
                    }
                } else {
                    formData[name] = $checkbox.is(':checked') ? '1' : '0';
                }
            });

            // Radio buttons (exclude hidden template inputs)
            $form.find('input[type="radio"]:checked').not(templateExclude).each(function() {
                const $radio = $(this);
                const name = $radio.attr('name');
                
                if (name) {
                    formData[name] = $radio.val();
                }
            });

            // Compact repeater arrays (remove gaps from sparse arrays)
            for (const key of Object.keys(formData)) {
                if (Array.isArray(formData[key]) && formData[key].length > 0) {
                    // Find the first defined element to check if this is a repeater field (array of objects)
                    const firstDefined = formData[key].find(item => item !== undefined && item !== null);
                    if (firstDefined && typeof firstDefined === 'object') {
                        formData[key] = formData[key].filter(item => item !== undefined && item !== null);
                    } else {
                        // For single_repeater (array of simple values), also filter out empty/undefined values
                        formData[key] = formData[key].filter(item => item !== undefined && item !== null && item !== '');
                    }
                }
            }

            return formData;
        },

        /**
         * Get field IDs that are hidden due to dependency
         * This includes fields with 'hidden' class (field-level dependency)
         * and all fields inside sections/subsections that are hidden by section-level dependency
         */
        getHiddenFieldIds() {
            const hiddenFieldIds = [];
            const $form = this.elements.form;
            const $wrap = this.elements.wrap;
            
            /**
             * Helper function to extract base field ID from input name
             */
            const extractFieldId = (name) => {
                if (!name) return null;
                // Clean up array notation and extract base field ID
                return name.replace(/\[\]$/, '').replace(/\[\d+\].*$/, '');
            };
            
            /**
             * Helper function to collect field IDs from a container
             */
            const collectFieldIds = ($container) => {
                $container.find('.bizzplugin-field').each(function() {
                    const $fieldWrap = $(this);
                    $fieldWrap.find('input, select, textarea').each(function() {
                        const baseFieldId = extractFieldId($(this).attr('name'));
                        if (baseFieldId && !hiddenFieldIds.includes(baseFieldId)) {
                            hiddenFieldIds.push(baseFieldId);
                        }
                    });
                });
            };
            
            // Get fields hidden by field-level dependency (have 'hidden' class on .bizzplugin-field)
            $form.find('.bizzplugin-field.hidden').each(function() {
                const $fieldWrap = $(this);
                $fieldWrap.find('input, select, textarea').each(function() {
                    const baseFieldId = extractFieldId($(this).attr('name'));
                    if (baseFieldId && !hiddenFieldIds.includes(baseFieldId)) {
                        hiddenFieldIds.push(baseFieldId);
                    }
                });
            });
            
            // Get fields hidden by section-level dependency
            // The 'bizzplugin-section-hidden' class is added to nav items (.bizzplugin-nav-item)
            // We need to find corresponding section content and collect fields from there
            $wrap.find('.bizzplugin-nav-item.bizzplugin-section-hidden').each(function() {
                const $navItem = $(this);
                const $navLink = $navItem.find('.bizzplugin-nav-link').first();
                const sectionId = $navLink.data('section');
                
                if (sectionId) {
                    // Find main section content
                    const $sectionContent = $form.find(`#section-${sectionId}`);
                    if ($sectionContent.length) {
                        collectFieldIds($sectionContent);
                    }
                    
                    // Also collect from any subsections of this hidden section
                    $navItem.find('.bizzplugin-nav-sublink').each(function() {
                        const subsectionId = $(this).data('subsection');
                        if (subsectionId) {
                            const $subsectionContent = $form.find(`#subsection-${subsectionId}`);
                            if ($subsectionContent.length) {
                                collectFieldIds($subsectionContent);
                            }
                        }
                    });
                }
            });
            
            // Also check for individually hidden subsections (if subsections have their own dependency)
            $wrap.find('.bizzplugin-nav-subitem.bizzplugin-section-hidden').each(function() {
                const $subItem = $(this);
                const $subLink = $subItem.find('.bizzplugin-nav-sublink').first();
                const subsectionId = $subLink.data('subsection');
                
                if (subsectionId) {
                    const $subsectionContent = $form.find(`#subsection-${subsectionId}`);
                    if ($subsectionContent.length) {
                        collectFieldIds($subsectionContent);
                    }
                }
            });
            
            return hiddenFieldIds;
        },

        /**
         * Update field values (after reset)
         */
        updateFieldValues(defaults) {
            const self = this;
            const Common = window.BizzPluginCommon;
            
            for (const [fieldId, value] of Object.entries(defaults)) {
                // Handle repeater fields (array of objects)
                const $repeaterWrap = $(`.bizzplugin-repeater-wrap[data-field-id="${fieldId}"]`);
                if ($repeaterWrap.length) {
                    self.resetRepeaterField($repeaterWrap, value);
                    continue;
                }
                
                // Handle single_repeater fields (array of simple values)
                const $singleRepeaterWrap = $(`.bizzplugin-single-repeater-wrap[data-field-id="${fieldId}"]`);
                if ($singleRepeaterWrap.length) {
                    self.resetSingleRepeaterField($singleRepeaterWrap, value);
                    continue;
                }
                
                const $field = $(`[name="${fieldId}"], [name="${fieldId}[]"]`);
                
                if (!$field.length) continue;

                const fieldType = $field.attr('type') || $field.prop('tagName').toLowerCase();

                switch (fieldType) {
                    case 'checkbox':
                        if ($field.length > 1) {
                            // Checkbox group
                            $field.prop('checked', false);
                            if (Array.isArray(value)) {
                                value.forEach(v => {
                                    $field.filter(`[value="${v}"]`).prop('checked', true);
                                });
                            }
                        } else {
                            $field.prop('checked', value === '1');
                        }
                        break;

                    case 'radio':
                        $field.filter(`[value="${value}"]`).prop('checked', true);
                        break;

                    case 'select':
                        $field.val(value);
                        break;

                    case 'range':
                        // Slider field - update value and UI
                        $field.val(value);
                        if (Common) {
                            Common.updateSliderUI($field);
                        }
                        break;

                    default:
                        $field.val(value);
                        // Check if this is a slider field by class
                        if ($field.hasClass('bizzplugin-slider') && Common) {
                            Common.updateSliderUI($field);
                        }
                }

                // Trigger change for conditional fields
                $field.trigger('change');
            }

            // Reinitialize color pickers
            if (Common) {
                Common.initColorPickers();
                Common.initAlphaColorPickers();
            }
            
            // Reinitialize sliders to ensure all UI is updated
            if (Common) {
                Common.initSliders();
            }
            $(document).trigger('bizzplugin:update_field', [defaults, self]);
        },
        
        /**
         * Reset repeater field to default values
         * @param {jQuery} $wrap Repeater wrapper element
         * @param {Array} defaultValue Default value (array of objects or empty array)
         */
        resetRepeaterField($wrap, defaultValue) {
            const self = this;
            const Common = window.BizzPluginCommon;
            const $items = $wrap.find('.bizzplugin-repeater-items');
            const $template = $wrap.find('.bizzplugin-repeater-template');
            const fieldId = $wrap.data('field-id');
            
            // Remove all existing items
            $items.find('.bizzplugin-repeater-item').remove();
            
            // If default value is empty or not an array, just show empty state
            if (!defaultValue || !Array.isArray(defaultValue) || defaultValue.length === 0) {
                // Show add button
                const allowAdd = $wrap.data('allow-add') !== 0 && $wrap.data('allow-add') !== '0';
                if (allowAdd) {
                    $wrap.find('.bizzplugin-repeater-add').show();
                }
                return;
            }
            
            // Add items from default values
            defaultValue.forEach((itemData, index) => {
                // Get template HTML and replace placeholders
                let newItemHtml = $template.html();
                newItemHtml = newItemHtml.replace(/\{\{INDEX\}\}/g, index);
                newItemHtml = newItemHtml.replace(/\{\{DISPLAY_INDEX\}\}/g, index + 1);
                
                // Create the new item element
                const $newItem = $($.parseHTML(newItemHtml, document, false));
                
                // Enable disabled inputs in template fields
                $newItem.find('[data-template-field="1"] input, [data-template-field="1"] textarea, [data-template-field="1"] select').prop('disabled', false);
                $newItem.find('[data-template-field]').removeAttr('data-template-field');
                
                // Set values from default data
                if (typeof itemData === 'object') {
                    for (const [subFieldId, subValue] of Object.entries(itemData)) {
                        const $subField = $newItem.find(`[name="${fieldId}[${index}][${subFieldId}]"]`);
                        if ($subField.length) {
                            if ($subField.is(':checkbox')) {
                                $subField.prop('checked', subValue === '1' || subValue === 1 || subValue === true);
                            } else {
                                $subField.val(subValue);
                            }
                        }
                    }
                }
                
                // Add to items container
                $items.append($newItem);
                
                // Initialize fields for new item
                if (Common) {
                    Common.initColorPickers($newItem);
                    Common.initAlphaColorPickers($newItem);
                    Common.initDatePickers($newItem);
                    Common.initSliders($newItem);
                }
            });
            
            // Update remove button visibility
            if (Common) {
                Common.repeaterUpdateRemoveButtons($wrap);
            }

            $(document).trigger('bizzplugin:repeater_reindex_reset', [$template, $wrap]);
            // Update add button visibility
            const maxItems = parseInt($wrap.data('max-items')) || 0;
            const allowAdd = $wrap.data('allow-add') !== 0 && $wrap.data('allow-add') !== '0';
            if (allowAdd && (maxItems === 0 || defaultValue.length < maxItems)) {
                $wrap.find('.bizzplugin-repeater-add').show();
            } else if (maxItems > 0 && defaultValue.length >= maxItems) {
                $wrap.find('.bizzplugin-repeater-add').hide();
            }
        },
        
        /**
         * Reset single repeater field to default values
         * @param {jQuery} $wrap Single repeater wrapper element
         * @param {Array} defaultValue Default value (array of simple values or empty array)
         */
        resetSingleRepeaterField($wrap, defaultValue) {
            const self = this;
            const Common = window.BizzPluginCommon;
            const $items = $wrap.find('.bizzplugin-single-repeater-items');
            const $template = $wrap.find('.bizzplugin-single-repeater-template');
            const fieldId = $wrap.data('field-id');
            
            // Remove all existing items
            $items.find('.bizzplugin-single-repeater-item').remove();
            
            // If default value is empty or not an array, just show empty state
            if (!defaultValue || !Array.isArray(defaultValue) || defaultValue.length === 0) {
                // Show add button
                const allowAdd = $wrap.data('allow-add') !== 0 && $wrap.data('allow-add') !== '0';
                if (allowAdd) {
                    $wrap.find('.bizzplugin-single-repeater-add').show();
                }
                return;
            }
            
            // Add items from default values
            defaultValue.forEach((itemValue, index) => {
                // Get template HTML and replace placeholders
                let newItemHtml = $template.html();
                newItemHtml = newItemHtml.replace(/\{\{INDEX\}\}/g, index);
                
                // Create the new item element
                const $newItem = $($.parseHTML(newItemHtml, document, false));
                
                // Enable disabled inputs in template
                $newItem.find('input, textarea, select').prop('disabled', false);
                $newItem.removeAttr('data-template');
                
                // Set value
                const $input = $newItem.find('input, textarea, select').first();
                if ($input.length) {
                    $input.val(itemValue);
                }
                
                // Add to items container
                $items.append($newItem);
                
                // Initialize fields for new item
                if (Common) {
                    Common.initColorPickers($newItem);
                    Common.initAlphaColorPickers($newItem);
                }
            });
            
            // Update remove button visibility
            if (Common) {
                Common.singleRepeaterUpdateRemoveButtons($wrap);
            }
            $(document).trigger('bizzplugin:repeater_index_reset', [$template, $wrap]);
            // Update add button visibility
            const maxItems = parseInt($wrap.data('max-items')) || 0;
            const allowAdd = $wrap.data('allow-add') !== 0 && $wrap.data('allow-add') !== '0';
            if (allowAdd && (maxItems === 0 || defaultValue.length < maxItems)) {
                $wrap.find('.bizzplugin-single-repeater-add').show();
            } else if (maxItems > 0 && defaultValue.length >= maxItems) {
                $wrap.find('.bizzplugin-single-repeater-add').hide();
            }
        },

        /**
         * Set status message
         */
        setStatus(status, message) {
            const $status = this.elements.saveStatus;

            $(document).trigger('bizzplugin:status_changed', [status, message]);
            $status.removeClass('saving saved error').addClass(status);
            this.showNotification(message || this.config.strings[status] || '', status === 'error' ? 'error' : 'success');
            switch (status) {
                case 'saving':
                    $status.text(message || this.config.strings.saving);
                    break;
                case 'saved':
                    $status.text(message || this.config.strings.saved);
                    setTimeout(() => {
                        $status.removeClass('saved').text('');
                    }, 3000);
                    break;
                case 'error':
                    $status.text(message || this.config.strings.error);
                    break;
            }
        },

        /**
         * Show validation errors
         */
        showValidationErrors(errors) {
            // Track error counts per section and subsection
            const sectionErrors = {};
            
            for (const [fieldId, message] of Object.entries(errors)) {
                // Try to find the field by exact name match first
                let $field = $(`[name="${fieldId}"]`).closest('.bizzplugin-field');
                
                // If not found, try to find repeater fields (names like fieldId[0][subfield])
                if (!$field.length) {
                    // Look for repeater wrapper with matching field-id
                    const $repeaterWrap = $(`.bizzplugin-repeater-wrap[data-field-id="${fieldId}"], .bizzplugin-single-repeater-wrap[data-field-id="${fieldId}"]`);
                    if ($repeaterWrap.length) {
                        $field = $repeaterWrap.closest('.bizzplugin-field');
                    }
                }
                
                // If still not found, try partial name match for array fields
                if (!$field.length) {
                    $field = $(`[name^="${fieldId}["]`).first().closest('.bizzplugin-field');
                }
                
                // Skip if field not found
                if (!$field.length) {
                    console.warn('Field not found for error:', fieldId);
                    continue;
                }
                
                $field.addClass('has-error');
                
                // Remove existing error message
                $field.find('.bizzplugin-field-error').remove();
                
                // Add error message using safe DOM method
                const $errorDiv = $('<div>')
                    .addClass('bizzplugin-field-error')
                    .text(message);
                $field.find('.bizzplugin-field-content').append($errorDiv);
                
                // Find the section/subsection this field belongs to
                const $section = $field.closest('.bizzplugin-section');
                if ($section.length) {
                    const mainSectionId = $section.data('section');
                    const subsectionId = $section.data('subsection');
                    
                    if (mainSectionId) {
                        const key = `section-${mainSectionId}`;
                        if (!sectionErrors[key]) {
                            sectionErrors[key] = { count: 0, subsections: {} };
                        }
                        sectionErrors[key].count++;
                        
                        // Track subsection error if applicable
                        if (subsectionId) {
                            if (!sectionErrors[key].subsections[subsectionId]) {
                                sectionErrors[key].subsections[subsectionId] = 0;
                            }
                            sectionErrors[key].subsections[subsectionId]++;
                        }
                    }
                }
            }
            
            // Update navigation menu with error states
            this.updateNavErrorStates(sectionErrors);
        },
        
        /**
         * Add error count bubble to nav element
         */
        addErrorCountBubble($element, count) {
            $element.find('.bizzplugin-nav-error-count').remove();
            const $bubble = $('<span>')
                .addClass('bizzplugin-nav-error-count')
                .text(count);
            $element.append($bubble);
        },
        
        /**
         * Update navigation menu error states
         */
        updateNavErrorStates(sectionErrors) {
            // Clear existing error states from nav
            $('.bizzplugin-nav-item').removeClass('has-nav-error');
            $('.bizzplugin-nav-subitem').removeClass('has-nav-error');
            $('.bizzplugin-nav-error-count').remove();
            
            // Apply error states
            for (const [sectionKey, errorData] of Object.entries(sectionErrors)) {
                const sectionId = sectionKey.replace('section-', '');
                const $navItem = $(`.bizzplugin-nav-link[data-section="${sectionId}"]`).closest('.bizzplugin-nav-item');
                
                if ($navItem.length && errorData.count > 0) {
                    // Add error class to main section nav item
                    $navItem.addClass('has-nav-error');
                    
                    // Add error count bubble to nav link
                    const $navLink = $navItem.find('> .bizzplugin-nav-link');
                    this.addErrorCountBubble($navLink, errorData.count);
                    
                    // Add error states to subsections
                    for (const [subsectionId, subCount] of Object.entries(errorData.subsections)) {
                        const $subItem = $(`.bizzplugin-nav-sublink[data-subsection="${subsectionId}"]`).closest('.bizzplugin-nav-subitem');
                        if ($subItem.length && subCount > 0) {
                            $subItem.addClass('has-nav-error');
                            const $subLink = $subItem.find('.bizzplugin-nav-sublink');
                            this.addErrorCountBubble($subLink, subCount);
                        }
                    }
                }
            }
        },

        /**
         * Clear validation errors
         */
        clearValidationErrors() {
            $('.bizzplugin-field').removeClass('has-error');
            $('.bizzplugin-field-error').remove();
            
            // Clear navigation error states
            $('.bizzplugin-nav-item').removeClass('has-nav-error');
            $('.bizzplugin-nav-subitem').removeClass('has-nav-error');
            $('.bizzplugin-nav-error-count').remove();
        },

        
        
        /**
         * Export options to JSON file
         */
        exportOptions(e) {
            e.preventDefault();
            
            const panelId = this.elements.wrap.data('panel-id');
            const optionName = this.elements.wrap.data('option-name');
            const formData = this.serializeFormData();
            
            // Create export data
            const exportData = {
                plugin: 'bizzplugin-option-framework',
                panel_id: panelId,
                option_name: optionName,
                exported_at: new Date().toISOString(),
                data: formData
            };
            
            // Create and download file
            const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `${optionName}-export-${new Date().toISOString().substring(0, 10)}.json`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            $(document).trigger('bizzplugin:export_success', [optionName, exportData]);
            this.showNotification(this.config.strings.export_success || 'Settings exported successfully!', 'info');
        },
        
        /**
         * Trigger import file input
         */
        triggerImport(e) {
            e.preventDefault();
            $('#bizzplugin-import-file').trigger('click');
        },
        
        /**
         * Import options from JSON file
         */
        importOptions(e) {
            const file = e.target.files[0];
            if (!file) return;
            
            const reader = new FileReader();
            const self = this;
            
            reader.onload = function(event) {
                try {
                    const importData = JSON.parse(event.target.result);
                    
                    // Validate import data
                    if (!importData.plugin || importData.plugin !== 'bizzplugin-option-framework') {
                        self.setStatus('error', self.config.strings.import_invalid || 'Invalid import file format.');
                        return;
                    }
                    
                    // Check if panel_id matches (optional warning)
                    const currentPanelId = self.elements.wrap.data('panel-id');
                    if (importData.panel_id && importData.panel_id !== currentPanelId) {
                        if (!confirm(self.config.strings.import_panel_mismatch || 'This export was created from a different panel. Do you want to continue importing?')) {
                            return;
                        }
                    }
                    
                    // Update field values
                    if (importData.data) {
                        self.updateFieldValues(importData.data);
                        self.elements.saveButton.addClass('button-primary-changed');
                        self.showNotification(self.config.strings.import_success || 'Settings imported successfully! Please save to apply changes.', 'info');
                        $(document).trigger('bizzplugin:import_success', [importData.panel_id, importData.data]);
                    }
                } catch (error) {
                    self.setStatus('error', self.config.strings.import_error || 'Error parsing import file.');
                    console.error('Import error:', error);
                }
            };
            
            reader.readAsText(file);
            
            // Reset file input
            e.target.value = '';
        },
        
        /**
         * Test webhook
         */
        testWebhook(e) {
            e.preventDefault();
            
            const $button = $(e.currentTarget);
            const webhookUrl = $('#bizzplugin_webhook_url').val();
            const $response = $('#bizzplugin-webhook-response');
            
            if (!webhookUrl) {
                this.setStatus('error', this.config.strings.webhook_url_required || 'Please enter a webhook URL first.');
                return;
            }
            
            $button.prop('disabled', true).text(this.config.strings.testing || 'Testing...');
            
            const panelId = this.elements.wrap.data('panel-id');
            const optionName = this.elements.wrap.data('option-name');
            
            $.ajax({
                url: this.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_test_webhook',
                    nonce: this.config.nonce,
                    panel_id: panelId,
                    option_name: optionName,
                    webhook_url: webhookUrl
                },
                success: (response) => {
                    $response.show().find('.bizzplugin-code-block').text(
                        JSON.stringify(response, null, 2)
                    );
                },
                error: (xhr, status, error) => {
                    $response.show().find('.bizzplugin-code-block').text(
                        'Error: ' + error
                    );
                },
                complete: () => {
                    $button.prop('disabled', false).html('<span class="dashicons dashicons-migrate"></span> Test Webhook');
                }
            });
        },
        
        /**
         * Generate API key
         */
        generateApiKey(e) {
            e.preventDefault();
            
            const $button = $(e.currentTarget);
            const panelId = this.elements.wrap.data('panel-id');
            
            if (!confirm(this.config.strings.confirm_regenerate_api_key || 'Are you sure you want to generate a new API key? The old key will be invalidated.')) {
                return;
            }
            
            $button.prop('disabled', true);
            const originalText = $button.html();
            $button.html('<span class="dashicons dashicons-update bizzplugin-spin"></span> ' + (this.config.strings.generating || 'Generating...'));
            
            $.ajax({
                url: this.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_generate_api_key',
                    nonce: this.config.nonce,
                    panel_id: panelId
                },
                success: (response) => {
                    if (response.success) {
                        const apiKey = response.data.api_key;
                        
                        // Update the API key display
                        const $displayContainer = $('.bizzplugin-api-key-display');
                        
                        // Create new API key row using DOM manipulation for security
                        const $newRow = $('<div>').addClass('bizzplugin-api-key-row');
                        const $codeEl = $('<code>')
                            .addClass('bizzplugin-api-key-code')
                            .attr('id', 'bizzplugin-api-key-value')
                            .text(apiKey);
                        const $copyBtn = $('<button>')
                            .attr('type', 'button')
                            .addClass('button bizzplugin-copy-btn')
                            .attr('data-copy', apiKey)
                            .html('<span class="dashicons dashicons-admin-page"></span>');
                        
                        $newRow.append($codeEl).append($copyBtn);
                        $displayContainer.empty().append($newRow);
                        
                        // Update button text
                        $button.html('<span class="dashicons dashicons-update" style="margin-right: 5px; margin-top: 3px;"></span> ' + (this.config.strings.regenerate_api_key || 'Regenerate API Key'));
                        
                        this.showNotification(response.data.message, 'success');

                        $(document).trigger('bizzplugin:api_key_generated', [apiKey, response]);
                    } else {
                        $button.html(originalText);
                        this.showNotification(response.data.message, 'error');
                    }
                },
                error: (xhr, status, error) => {
                    $button.html(originalText);
                    this.showNotification(this.config.strings.api_key_error || 'Error generating API key', 'error');
                    console.error('API key generation error:', error);
                },
                complete: () => {
                    $button.prop('disabled', false);
                }
            });
        },
        
        /**
         * Initialize API docs tabs
         */
        initApiDocsTabs() {
            // Initial state - show first tab content
            $('.bizzplugin-docs-content[data-tab-content="get"]').addClass('active');
        },
        
        /**
         * Handle docs tab click
         */
        handleDocsTabClick(e) {
            e.preventDefault();
            
            const $tab = $(e.currentTarget);
            const tabId = $tab.data('tab');
            const $container = $tab.closest('.bizzplugin-api-card-body');
            
            // Update tab active state
            $container.find('.bizzplugin-docs-tab').removeClass('active');
            $tab.addClass('active');
            
            // Show/hide content
            $container.find('.bizzplugin-docs-content').removeClass('active');
            $container.find(`.bizzplugin-docs-content[data-tab-content="${tabId}"]`).addClass('active');
        },

        /**
         * Install plugin via AJAX - uses common module
         */
        installPlugin(e) {
            const self = this;
            const Common = window.BizzPluginCommon;
            
            if (Common) {
                Common.installPlugin(e, 
                    (response) => {
                        self.showNotification(response.data.message, 'success');
                    },
                    (response) => {
                        self.showNotification(response.data?.message || self.config.strings.install_error || 'Error installing plugin', 'error');
                    }
                );
            }
        },

        /**
         * Activate plugin via AJAX - uses common module
         */
        activatePlugin(e) {
            const self = this;
            const Common = window.BizzPluginCommon;
            
            if (Common) {
                Common.activatePlugin(e,
                    (response) => {
                        self.showNotification(response.data.message, 'success');
                    },
                    (response) => {
                        self.showNotification(response.data?.message || self.config.strings.activate_error || 'Error activating plugin', 'error');
                    }
                );
            }
        },

        /**
         * Show notification - uses common module or fallback
         */
        showNotification(message, type = 'success') {
            const Common = window.BizzPluginCommon;
            $(document).trigger('bizzplugin:notification', [message, type]);
            if (Common) {
                Common.showNotification(message, type, this.elements.wrap, 2000);
            } else {
                // Fallback if common module not available
                alert(message);
            }
        },
        
        /**
         * Initialize search functionality
         */
        initSearch() {
            this.searchTimeout = null;
            this.isSearchActive = false;
            this.originalNavState = null;
        },
        
        /**
         * Handle search input
         */
        handleSearchInput(e) {
            const searchTerm = $(e.currentTarget).val().trim().toLowerCase();
            const $clearBtn = $('.bizzplugin-search-clear');
            const $resultsInfo = $('.bizzplugin-search-results-info');
            
            // Clear any existing timeout
            if (this.searchTimeout) {
                clearTimeout(this.searchTimeout);
            }
            
            // Show/hide clear button
            if (searchTerm.length > 0) {
                $clearBtn.show();
            } else {
                $clearBtn.hide();
                this.clearSearchHighlights();
                $resultsInfo.hide();
                return;
            }
            
            // Debounce the search
            this.searchTimeout = setTimeout(() => {
                this.performSearch(searchTerm);
            }, 200);
        },
        
        /**
         * Perform the actual search
         */
        performSearch(searchTerm) {
            const $navItems = $('.bizzplugin-nav-item');
            const $navSubitems = $('.bizzplugin-nav-subitem');
            const $fields = $('.bizzplugin-field[data-search-keywords]');
            const $resultsInfo = $('.bizzplugin-search-results-info');
            let matchCount = 0;
            
            // Mark search as active
            this.isSearchActive = true;
            this.elements.wrap.addClass('bizzplugin-search-active');
            
            // Search in navigation items
            $navItems.each((index, item) => {
                const $item = $(item);
                const keywords = $item.data('search-keywords') || '';
                
                if (keywords.indexOf(searchTerm) !== -1) {
                    $item.removeClass('bizzplugin-search-hidden').addClass('bizzplugin-search-match');
                    matchCount++;
                    
                    // If this is a parent with subsections, show matching subsections
                    $item.find('.bizzplugin-nav-subitem').each((subIndex, subItem) => {
                        const $subItem = $(subItem);
                        const subKeywords = $subItem.data('search-keywords') || '';
                        
                        if (subKeywords.indexOf(searchTerm) !== -1) {
                            $subItem.removeClass('bizzplugin-search-hidden').addClass('bizzplugin-search-match');
                        } else {
                            $subItem.addClass('bizzplugin-search-hidden').removeClass('bizzplugin-search-match');
                        }
                    });
                    
                    // Expand the subsections if there are matches
                    $item.addClass('expanded');
                    $item.find('.bizzplugin-nav-submenu').show();
                } else {
                    $item.addClass('bizzplugin-search-hidden').removeClass('bizzplugin-search-match');
                }
            });
            
            // Search in fields (for current visible section)
            $fields.each((index, field) => {
                const $field = $(field);
                const keywords = $field.data('search-keywords') || '';
                
                if (keywords.indexOf(searchTerm) !== -1) {
                    $field.removeClass('bizzplugin-search-field-hidden').addClass('bizzplugin-search-field-match');
                } else {
                    $field.addClass('bizzplugin-search-field-hidden').removeClass('bizzplugin-search-field-match');
                }
            });
            
            // Show results info
            if (matchCount > 0) {
                let resultsText;
                if (matchCount === 1) {
                    resultsText = this.config.strings.search_result_single || '1 section found';
                } else {
                    const pluralTemplate = this.config.strings.search_results_plural || '%d sections found';
                    resultsText = pluralTemplate.replace('%d', matchCount);
                }
                $resultsInfo.text(resultsText).show();
            } else {
                $resultsInfo.text(this.config.strings.search_no_results || 'No results found').show();
            }
        },
        
        /**
         * Clear search
         */
        clearSearch(e) {
            if (e) {
                e.preventDefault();
            }
            
            const $searchInput = $('#bizzplugin-settings-search');
            $searchInput.val('').focus();
            
            this.clearSearchHighlights();
            
            $('.bizzplugin-search-clear').hide();
            $('.bizzplugin-search-results-info').hide();
        },
        
        /**
         * Clear search highlights and restore original state
         */
        clearSearchHighlights() {
            this.isSearchActive = false;
            this.elements.wrap.removeClass('bizzplugin-search-active');
            
            // Remove search classes from nav items
            $('.bizzplugin-nav-item').removeClass('bizzplugin-search-hidden bizzplugin-search-match');
            $('.bizzplugin-nav-subitem').removeClass('bizzplugin-search-hidden bizzplugin-search-match');
            
            // Remove search classes from fields
            $('.bizzplugin-field').removeClass('bizzplugin-search-field-hidden bizzplugin-search-field-match');
            
            // Restore nav submenu state based on active section
            const $activeNavItem = $('.bizzplugin-nav-item.active');
            $('.bizzplugin-nav-item').not($activeNavItem).removeClass('expanded').find('.bizzplugin-nav-submenu').hide();
            if ($activeNavItem.hasClass('has-subsections')) {
                $activeNavItem.addClass('expanded').find('.bizzplugin-nav-submenu').show();
            }
        },
        
        /**
         * Initialize color pickers in repeater items
         */
        initRepeaterColorPickers() {
            const self = this;
            const Common = window.BizzPluginCommon;
            
            $('.bizzplugin-repeater-item .bizzplugin-repeater-color').not('.wp-color-picker').each(function() {
                $(this).wpColorPicker({
                    change: () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    },
                    clear: () => {
                        self.elements.saveButton.addClass('button-primary-changed');
                    }
                });
            });
        },
        
        /**
         * Add new repeater item - uses common module
         */
        repeaterAddItem(e) {
            const self = this;
            const Common = window.BizzPluginCommon;
            // alert('here');
            $(document).trigger('bizzplugin:repeater_item_added', self);
            if (Common) {
                Common.repeaterAddItem(e, 
                    ($newItem, $wrap) => {
                        // Initialize color pickers for new item
                        self.initRepeaterColorPickers();
                        // Mark as changed
                        self.elements.saveButton.addClass('button-primary-changed');
                    },
                    ($newItem) => {
                        // Custom field initialization for framework
                        Common.initColorPickers($newItem, () => {
                            self.elements.saveButton.addClass('button-primary-changed');
                        });
                        Common.initDatePickers($newItem);
                        Common.initSliders($newItem);
                    }
                );
            }
        },
        
        /**
         * Remove repeater item - uses common module
         */
        repeaterRemoveItem(e) {
            const self = this;
            const Common = window.BizzPluginCommon;
            $(document).trigger('bizzplugin:repeater_item_removed', self);
            if (Common) {
                Common.repeaterRemoveItem(e,
                    ($wrap) => {
                        // Mark as changed
                        self.elements.saveButton.addClass('button-primary-changed');
                    },
                    ($wrap) => {
                        // Use common reindex function
                        Common.repeaterReindex($wrap);
                    }
                );
            }
        },
        
        /**
         * Reindex repeater items - uses common module
         */
        repeaterReindex($wrap) {
            const self = this;
            $(document).trigger('bizzplugin:repeater_reindex', self);
            const Common = window.BizzPluginCommon;
            if (Common) {
                Common.repeaterReindex($wrap);
            }
        },
        
        /**
         * Add new single repeater item - uses common module
         */
        singleRepeaterAddItem(e) {
            const self = this;
            const Common = window.BizzPluginCommon;
            $(document).trigger('bizzplugin:repeater_item_added', self);
            if (Common) {
                Common.singleRepeaterAddItem(e, 
                    ($newItem, $wrap) => {
                        // Mark as changed
                        self.elements.saveButton.addClass('button-primary-changed');
                    },
                    ($newItem) => {
                        // Custom field initialization for framework
                        Common.initColorPickers($newItem, () => {
                            self.elements.saveButton.addClass('button-primary-changed');
                        });
                    }
                );
            }
        },
        
        /**
         * Remove single repeater item - uses common module
         */
        singleRepeaterRemoveItem(e) {
            const self = this;
            const Common = window.BizzPluginCommon;
            $(document).trigger('bizzplugin:repeater_item_removed', self);
            if (Common) {
                Common.singleRepeaterRemoveItem(e,
                    ($wrap) => {
                        // Mark as changed
                        self.elements.saveButton.addClass('button-primary-changed');
                    }
                );
            }
        },
        
        /**
         * Initialize webhooks management
         */
        initWebhooks() {
            const self = this;
            const $container = $('.bizzplugin-webhooks-container');
            
            if (!$container.length) {
                return;
            }
            
            // Add webhook
            $container.on('click', '.bizzplugin-add-webhook', function(e) {
                e.preventDefault();
                self.addWebhook($container);
            });
            
            // Remove webhook
            $container.on('click', '.bizzplugin-remove-webhook', function(e) {
                e.preventDefault();
                const $item = $(this).closest('.bizzplugin-webhook-item');
                if (confirm(self.config.strings?.confirmRemove || 'Are you sure you want to remove this webhook?')) {
                    $item.slideUp(300, function() {
                        $(this).remove();
                        self.reindexWebhooks($container);
                    });
                }
            });
            
            // Auth type change
            $container.on('change', '.bizzplugin-webhook-auth-type', function() {
                const $item = $(this).closest('.bizzplugin-webhook-item');
                const authType = $(this).val();
                
                // Hide all auth fields
                $item.find('.bizzplugin-auth-fields').hide();
                
                // Show relevant auth fields
                if (authType !== 'none') {
                    $item.find('.bizzplugin-auth-' + authType).show();
                }
            });
        },
        
        /**
         * Add new webhook item
         */
        addWebhook($container) {
            const $template = $container.find('.bizzplugin-webhook-template');
            const $list = $container.find('.bizzplugin-webhooks-list');
            const newIndex = $list.find('.bizzplugin-webhook-item').length;
            
            // Clone template content (safely parse HTML without scripts)
            let templateHtml = $template.html();
            
            // Replace index placeholders
            templateHtml = templateHtml.replace(/\{\{INDEX\}\}/g, newIndex);
            templateHtml = templateHtml.replace(/\{\{DISPLAY_INDEX\}\}/g, newIndex + 1);
            
            // Use jQuery parseHTML with keepScripts=false to prevent script execution
            const parsed = $.parseHTML(templateHtml, document, false);
            const $newItem = $(parsed);
            
            // Append to list
            $list.append($newItem);
            
            // Trigger auth type change to show correct fields
            $newItem.find('.bizzplugin-webhook-auth-type').trigger('change');
            
            // Scroll to new item
            $newItem[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
        },
        
        /**
         * Reindex webhook items after removal
         */
        reindexWebhooks($container) {
            const $items = $container.find('.bizzplugin-webhooks-list .bizzplugin-webhook-item');
            
            $items.each(function(index) {
                const $item = $(this);
                const oldIndex = $item.data('index');
                
                // Update data attribute
                $item.attr('data-index', index);
                $item.data('index', index);
                
                // Update header text
                $item.find('.bizzplugin-webhook-item-header strong').text('Webhook #' + (index + 1));
                
                // Update all input names and IDs
                $item.find('input, select, textarea').each(function() {
                    const $input = $(this);
                    
                    // Update name attribute
                    const name = $input.attr('name');
                    if (name) {
                        $input.attr('name', name.replace(
                            /bizzplugin_webhooks\[\d+\]/,
                            'bizzplugin_webhooks[' + index + ']'
                        ));
                    }
                    
                    // Update id attribute
                    const id = $input.attr('id');
                    if (id) {
                        $input.attr('id', id.replace(
                            /bizzplugin_webhook_\d+/,
                            'bizzplugin_webhook_' + index
                        ));
                    }
                });
                
                // Update label for attributes
                $item.find('label').each(function() {
                    const $label = $(this);
                    const forAttr = $label.attr('for');
                    if (forAttr) {
                        $label.attr('for', forAttr.replace(
                            /bizzplugin_webhook_\d+/,
                            'bizzplugin_webhook_' + index
                        ));
                    }
                });
            });
        },
        
        /**
         * Initialize theme switcher
         */
        initThemeSwitcher() {
            // Check if user has a saved preference in localStorage
            const panelId = this.elements.wrap.data('panel-id');
            // Sanitize panelId for localStorage key (remove special chars, limit length)
            const sanitizedPanelId = String(panelId || 'default').replace(/[^a-zA-Z0-9_-]/g, '_').substring(0, 50);
            const storageKey = 'bizzplugin_theme_' + sanitizedPanelId;
            const savedTheme = localStorage.getItem(storageKey);
            
            if (savedTheme) {
                // Apply saved theme preference (only allow 'dark' or 'light')
                if (savedTheme === 'dark') {
                    this.elements.wrap.addClass('bizzplugin-dark-mode');
                    this.elements.wrap.attr('data-theme-mode', 'dark');
                } else {
                    this.elements.wrap.removeClass('bizzplugin-dark-mode');
                    this.elements.wrap.attr('data-theme-mode', 'light');
                }

                // let body = document.body;
                if (savedTheme === 'dark') {
                    this.elements.body.classList.add('bizzplugin-theme-dark');
                    this.elements.body.classList.remove('bizzplugin-theme-light');
                } else {
                    this.elements.body.classList.remove('bizzplugin-theme-dark');
                    this.elements.body.classList.add('bizzplugin-theme-light');
                }
            }
        },
        
        /**
         * Toggle theme between light and dark
         */
        toggleTheme(e) {
            e.preventDefault();
            
            const panelId = this.elements.wrap.data('panel-id');
            // Sanitize panelId for localStorage key (remove special chars, limit length)
            const sanitizedPanelId = String(panelId || 'default').replace(/[^a-zA-Z0-9_-]/g, '_').substring(0, 50);
            const storageKey = 'bizzplugin_theme_' + sanitizedPanelId;
            const currentTheme = this.elements.wrap.attr('data-theme-mode') || 'light';
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            // let body = document.body;
            if (newTheme === 'dark') {
                this.elements.body.classList.add('bizzplugin-theme-dark');
                this.elements.body.classList.remove('bizzplugin-theme-light');
            } else {
                this.elements.body.classList.remove('bizzplugin-theme-dark');
                this.elements.body.classList.add('bizzplugin-theme-light');
            }
            // Toggle class
            if (newTheme === 'dark') {
                this.elements.wrap.addClass('bizzplugin-dark-mode');


            } else {
                this.elements.wrap.removeClass('bizzplugin-dark-mode');
            }
            


            // Update data attribute
            this.elements.wrap.attr('data-theme-mode', newTheme);
            $(document).trigger('bizzplugin:theme_changed', [newTheme,storageKey,currentTheme,panelId]);
            // Save preference to localStorage
            localStorage.setItem(storageKey, newTheme);
        },
        
        log(...args) {
            console.log('BizzPlugin Framework is initialized.');
            if (window?.console) console.log('[BizzPluginFramework]', ...args);
        }
    };

    /**
     * Initialize on document ready
     */
    $(document).ready(() => {
        BizzPluginFramework.init();
    });

    /**
     * Expose to global scope for extensibility
     */
    window.BizzPluginFramework = BizzPluginFramework;

})(jQuery);
