/**
 * BizzPlugin Common JavaScript
 * 
 * Shared functionality used by both framework.js and metabox.js
 * This file consolidates all common UI interactions to avoid code duplication.
 * 
 * @package BizzPlugin_Options_Framework
 * @version 2.0.0
 */

(function($) {
    'use strict';

    /**
     * BizzPlugin Common Module
     * Contains shared functions for color pickers, date pickers, 
     * image uploaders, repeaters, dependency handling, and plugin management
     */
    window.BizzPluginCommon = {

        /**
         * Configuration (can be overridden by framework or metabox)
         */
        config: {
            ajaxUrl: '',
            nonce: '',
            strings: {}
        },

        /**
         * Set configuration
         * @param {Object} config Configuration object
         */
        setConfig: function(config) {
            this.config = $.extend({}, this.config, config);
        },

        /**
         * Initialize conditional fields
         */
        initConditionalFields() {
            const $conditionalFields = $('[data-dependency]');
            console.log('Initializing conditional fields:', $conditionalFields);
            $conditionalFields.each((index, element) => {
                const $field = $(element);
                const dependencyField = $field.data('dependency');
                const dependencyValue = String($field.data('dependency-value'));
                
                // Find the dependency field
                const $dependency = $(`[name="${dependencyField}"]`);
                
                // Initial state
                this.updateConditionalField($field, $dependency, dependencyValue);
            });
            
            // Initialize section dependencies
            this.initSectionDependencies();
        },
        
        /**
         * Initialize section-level dependencies
         */
        initSectionDependencies() {
            const $conditionalSections = $('[data-section-dependency]');
            
            $conditionalSections.each((index, element) => {
                const $section = $(element);
                const dependencyField = $section.data('section-dependency');
                const dependencyValue = String($section.data('section-dependency-value'));
                
                // Find the dependency field
                const $dependency = $(`[name="${dependencyField}"]`);
                
                // Initial state
                this.updateSectionDependency($section, $dependency, dependencyValue);
            });
        },
        
        /**
         * Update section visibility based on dependency
         */
        updateSectionDependency($section, $dependency, dependencyValue) {
            let currentValue;
            
            if ($dependency.is(':checkbox')) {
                currentValue = $dependency.is(':checked') ? '1' : '0';
            } else if ($dependency.is(':radio')) {
                currentValue = $dependency.filter(':checked').val();
            } else {
                currentValue = $dependency.val();
            }

            // Handle multiple values (comma-separated)
            const requiredValues = dependencyValue.split(',').map(v => v.trim());
            
            if (requiredValues.includes(String(currentValue))) {
                $section.removeClass('bizzplugin-section-hidden').slideDown(200);
            } else {
                $section.addClass('bizzplugin-section-hidden').slideUp(200);
                
                // If this section is currently active, navigate to first visible section
                if ($section.hasClass('active')) {
                    const $firstVisible = $('.bizzplugin-nav-item').not('.bizzplugin-section-hidden').first();
                    if ($firstVisible.length) {
                        $firstVisible.find('.bizzplugin-nav-link').trigger('click');
                    }
                }
            }
        },

        /**
         * Update conditional field visibility
         */
        updateConditionalField($field, $dependency, dependencyValue) {
            let currentValue;
            
            if ($dependency.is(':checkbox')) {
                currentValue = $dependency.is(':checked') ? '1' : '0';
            } else if ($dependency.is(':radio')) {
                currentValue = $dependency.filter(':checked').val();
            } else {
                currentValue = $dependency.val();
            }

            // Handle multiple values (comma-separated)
            const requiredValues = dependencyValue.split(',').map(v => v.trim());
            
            if (requiredValues.includes(String(currentValue))) {
                $field.removeClass('hidden').slideDown(200);
            } else {
                $field.addClass('hidden').fadeOut(200);
            }
        },

        /**
         * Handle toggle/checkbox change for conditional fields
         */
        handleToggleChange(e) {
            const $toggle = $(e.currentTarget);
            const fieldId = $toggle.attr('id');
            const fieldName = $toggle.attr('name');
            
            // Find fields dependent on this toggle by ID
            $(`[data-dependency="${fieldId}"]`).each((index, element) => {
                const $field = $(element);
                const dependencyValue = String($field.data('dependency-value'));
                this.updateConditionalField($field, $toggle, dependencyValue);
            });
            
            // Also find fields dependent on this toggle by name
            if (fieldName && fieldName !== fieldId) {
                $(`[data-dependency="${fieldName}"]`).each((index, element) => {
                    const $field = $(element);
                    const dependencyValue = String($field.data('dependency-value'));
                    this.updateConditionalField($field, $toggle, dependencyValue);
                });
            }
            
            // Find sections dependent on this toggle by ID
            $(`[data-section-dependency="${fieldId}"]`).each((index, element) => {
                const $section = $(element);
                const dependencyValue = String($section.data('section-dependency-value'));
                this.updateSectionDependency($section, $toggle, dependencyValue);
            });
            
            // Also find sections dependent on this toggle by name
            if (fieldName && fieldName !== fieldId) {
                $(`[data-section-dependency="${fieldName}"]`).each((index, element) => {
                    const $section = $(element);
                    const dependencyValue = String($section.data('section-dependency-value'));
                    this.updateSectionDependency($section, $toggle, dependencyValue);
                });
            }
        },

        /**
         * Handle select/radio change for conditional fields
         */
        handleSelectChange(e) {
            const $select = $(e.currentTarget);
            const fieldId = $select.attr('id');
            const fieldName = $select.attr('name');
            console.log('Handling select change for field:', fieldId, fieldName, $select);
            
            // Find fields dependent on this field by ID
            $(`[data-dependency="${fieldId}"]`).each((index, element) => {
                const $field = $(element);
                const dependencyValue = String($field.data('dependency-value'));
                this.updateConditionalField($field, $select, dependencyValue);
            });
            
            // Also find fields dependent on this field by name (for radio buttons like option_select)
            if (fieldName && fieldName !== fieldId) {
                $(`[data-dependency="${fieldName}"]`).each((index, element) => {
                    const $field = $(element);
                    const dependencyValue = String($field.data('dependency-value'));
                    this.updateConditionalField($field, $select, dependencyValue);
                });
            }
            
            // Find sections dependent on this field by ID
            $(`[data-section-dependency="${fieldId}"]`).each((index, element) => {
                const $section = $(element);
                const dependencyValue = String($section.data('section-dependency-value'));
                this.updateSectionDependency($section, $select, dependencyValue);
            });
            
            // Also find sections dependent on this field by name (for radio buttons like option_select)
            if (fieldName && fieldName !== fieldId) {
                $(`[data-section-dependency="${fieldName}"]`).each((index, element) => {
                    const $section = $(element);
                    const dependencyValue = String($section.data('section-dependency-value'));
                    this.updateSectionDependency($section, $select, dependencyValue);
                });
            }
        },
        /**
         * Initialize color pickers (standard HEX only)
         * @param {jQuery} $container Container to search within (optional)
         * @param {Function} onChange Callback on change (optional)
         */
        initColorPickers: function($container, onChange) {
            var $elements = $container ? 
                $container.find('.bizzplugin-color-picker') : 
                $('.bizzplugin-color-picker');
            
            $elements.not('.wp-color-picker').each(function() {
                $(this).wpColorPicker({
                    change: onChange || function() {},
                    clear: onChange || function() {}
                });
            });
        },
        
        /**
         * Initialize alpha color pickers (with opacity support)
         * @param {jQuery} $container Container to search within (optional)
         * @param {Function} onChange Callback on change (optional)
         */
        initAlphaColorPickers: function($container, onChange) {
            var $elements = $container ? 
                $container.find('.bizzplugin-alpha-color-picker') : 
                $('.bizzplugin-alpha-color-picker');
            
            $elements.not('.wp-color-picker').each(function() {
                var $input = $(this);
                
                var options = {
                    change: onChange || function() {},
                    clear: onChange || function() {},
                    palettes: true,
                    alpha: true,
                    alphaStep: parseFloat($input.data('alpha-step')) || 0.01
                };
                
                $input.wpColorPicker(options);
            });
        },

        /**
         * Initialize date pickers
         * @param {jQuery} $container Container to search within (optional)
         * @param {Function} onSelect Callback on select (optional)
         */
        initDatePickers: function($container, onSelect) {
            var $elements = $container ? 
                $container.find('.bizzplugin-date-picker') : 
                $('.bizzplugin-date-picker');
            
            $elements.each(function() {
                var $el = $(this);
                if (!$el.hasClass('hasDatepicker')) {
                    $el.datepicker({
                        dateFormat: 'yy-mm-dd',
                        changeMonth: true,
                        changeYear: true,
                        onSelect: onSelect || function() {}
                    });
                }
            });
        },

        /**
         * Initialize sliders
         * @param {jQuery} $container Container to search within (optional)
         */
        initSliders: function($container) {
            var self = this;
            var $elements = $container ? 
                $container.find('.bizzplugin-slider') : 
                $('.bizzplugin-slider');
            
            $elements.each(function() {
                self.updateSliderUI($(this));
            });
        },

        /**
         * Update slider UI (value display and track background)
         * @param {jQuery} $slider Slider element
         */
        updateSliderUI: function($slider) {
            var value = $slider.val();
            var min = parseFloat($slider.attr('min')) || 0;
            var max = parseFloat($slider.attr('max')) || 100;
            var progress = ((value - min) / (max - min)) * 100;
            
            // Update the value display - check multiple possible containers
            var $wrap = $slider.closest('.bizzplugin-slider-wrap');
            if ($wrap.length) {
                $wrap.find('.bizzplugin-slider-value-number').text(value);
            }
            
            var $container = $slider.closest('.bizzplugin-slider-container');
            if ($container.length) {
                $container.find('.bizzplugin-slider-value-number').text(value);
            }
            
            // Also check siblings
            $slider.siblings('.bizzplugin-slider-value').find('.bizzplugin-slider-value-number').text(value);
            
            // Update the slider track background
            $slider.css('--slider-progress', progress + '%');
        },

        /**
         * Handle slider change event
         * @param {Event} e Event object
         * @param {Function} onChange Optional callback
         */
        handleSliderChange: function(e, onChange) {
            var $slider = $(e.currentTarget);
            this.updateSliderUI($slider);
            
            if (typeof onChange === 'function') {
                onChange($slider);
            }
        },

        /**
         * Open WordPress media uploader for images
         * @param {Event} e Event object
         * @param {Function} onSuccess Callback on successful selection
         */
        openImageUploader: function(e, onSuccess) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $field = $button.closest('.bizzplugin-image-upload');
            var $input = $field.find('.bizzplugin-image-input');
            var $preview = $field.find('.bizzplugin-image-preview');
            
            var frame = wp.media({
                title: self.config.strings.select_image || 'Select Image',
                button: {
                    text: self.config.strings.use_image || 'Use this image'
                },
                multiple: false,
                library: {
                    type: 'image'
                }
            });
            
            frame.on('select', function() {
                var attachment = frame.state().get('selection').first().toJSON();
                $input.val(attachment.id).trigger('change');
                
                var imageUrl = attachment.sizes && attachment.sizes.thumbnail ? 
                    attachment.sizes.thumbnail.url : attachment.url;
                
                $preview.find('img').attr('src', imageUrl);
                $preview.show();
                
                if (typeof onSuccess === 'function') {
                    onSuccess(attachment);
                }
            });
            
            frame.open();
        },

        /**
         * Remove image
         * @param {Event} e Event object
         * @param {Function} onSuccess Callback on success
         */
        removeImage: function(e, onSuccess) {
            e.preventDefault();
            
            var $button = $(e.currentTarget);
            var $field = $button.closest('.bizzplugin-image-upload');
            var $input = $field.find('.bizzplugin-image-input');
            var $preview = $field.find('.bizzplugin-image-preview');
            
            $input.val('').trigger('change');
            $preview.hide();
            $preview.find('img').attr('src', '');
            
            if (typeof onSuccess === 'function') {
                onSuccess();
            }
        },

        /**
         * Open WordPress media uploader for files
         * @param {Event} e Event object
         * @param {Function} onSuccess Callback on successful selection
         */
        openFileUploader: function(e, onSuccess) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $field = $button.closest('.bizzplugin-file-upload');
            var $input = $field.find('.bizzplugin-file-input');
            var $fileName = $field.find('.bizzplugin-file-name');
            var $removeBtn = $field.find('.bizzplugin-file-remove');
            
            var frame = wp.media({
                title: self.config.strings.select_file || 'Select File',
                button: {
                    text: self.config.strings.use_file || 'Use this file'
                },
                multiple: false
            });
            
            frame.on('select', function() {
                var attachment = frame.state().get('selection').first().toJSON();
                $input.val(attachment.id).trigger('change');
                $fileName.text(attachment.filename);
                $removeBtn.show();
                
                if (typeof onSuccess === 'function') {
                    onSuccess(attachment);
                }
            });
            
            frame.open();
        },

        /**
         * Remove file
         * @param {Event} e Event object
         * @param {Function} onSuccess Callback on success
         */
        removeFile: function(e, onSuccess) {
            e.preventDefault();
            
            var $button = $(e.currentTarget);
            var $field = $button.closest('.bizzplugin-file-upload');
            var $input = $field.find('.bizzplugin-file-input');
            var $fileName = $field.find('.bizzplugin-file-name');
            
            $input.val('').trigger('change');
            $fileName.text('');
            $button.hide();
            
            if (typeof onSuccess === 'function') {
                onSuccess();
            }
        },

        /**
         * Handle image select field change
         * @param {Event} e Event object
         */
        handleImageSelectChange: function(e) {
            var $input = $(e.currentTarget);
            var $wrap = $input.closest('.bizzplugin-image-select-wrap');
            
            $wrap.find('.bizzplugin-image-select-item').removeClass('selected');
            $input.closest('.bizzplugin-image-select-item').addClass('selected');
        },

        /**
         * Handle option select field change
         * @param {Event} e Event object
         */
        handleOptionSelectChange: function(e) {
            var $input = $(e.currentTarget);
            var $wrap = $input.closest('.bizzplugin-option-select-wrap');
            
            $wrap.find('.bizzplugin-option-select-item').removeClass('selected');
            $input.closest('.bizzplugin-option-select-item').addClass('selected');
        },

        /**
         * Initialize image select (mark selected items)
         * @param {jQuery} $container Container to search within (optional)
         */
        initImageSelect: function($container) {
            var $elements = $container ? 
                $container.find('.bizzplugin-image-select-input:checked') : 
                $('.bizzplugin-image-select-input:checked');
            
            $elements.closest('.bizzplugin-image-select-item').addClass('selected');
        },

        /**
         * Initialize option select (mark selected items)
         * @param {jQuery} $container Container to search within (optional)
         */
        initOptionSelect: function($container) {
            var $elements = $container ? 
                $container.find('.bizzplugin-option-select-input:checked') : 
                $('.bizzplugin-option-select-input:checked');
            
            $elements.closest('.bizzplugin-option-select-item').addClass('selected');
        },

        /**
         * Install plugin via AJAX
         * @param {Event} e Event object
         * @param {Function} onSuccess Callback on success
         * @param {Function} onError Callback on error
         */
        installPlugin: function(e, onSuccess, onError) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $card = $button.closest('.bizzplugin-plugin-card');
            var slug = $button.data('slug');
            
            if ($card.hasClass('bizzplugin-plugin-loading')) {
                return;
            }
            
            // Add loading state
            $card.addClass('bizzplugin-plugin-loading');
            var originalText = $button.html();
            $button.empty()
                .append($('<span>').addClass('dashicons dashicons-update'))
                .append(' ' + (self.config.strings.installing || 'Installing...'));
            
            $.ajax({
                url: self.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_install_plugin',
                    nonce: self.config.nonce,
                    slug: slug
                },
                success: function(response) {
                    if (response.success) {
                        // Update UI to show activate button
                        $card.find('.bizzplugin-plugin-status')
                            .removeClass('bizzplugin-plugin-status-not-installed')
                            .addClass('bizzplugin-plugin-status-installed')
                            .text(self.config.strings.installed_inactive || 'Installed (Inactive)');
                        
                        $button
                            .removeClass('bizzplugin-install-plugin')
                            .addClass('bizzplugin-activate-plugin')
                            .empty()
                            .append($('<span>').addClass('dashicons dashicons-yes-alt'))
                            .append(' ' + (self.config.strings.activate || 'Activate'));
                        
                        if (typeof onSuccess === 'function') {
                            onSuccess(response);
                        }
                    } else {
                        $button.html(originalText);
                        if (typeof onError === 'function') {
                            onError(response);
                        }
                    }
                },
                error: function(xhr, status, error) {
                    $button.html(originalText);
                    if (typeof onError === 'function') {
                        onError({success: false, data: {message: error}});
                    }
                    console.error('Plugin installation error:', error);
                },
                complete: function() {
                    $card.removeClass('bizzplugin-plugin-loading');
                }
            });
        },

        /**
         * Activate plugin via AJAX
         * @param {Event} e Event object
         * @param {Function} onSuccess Callback on success
         * @param {Function} onError Callback on error
         */
        activatePlugin: function(e, onSuccess, onError) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $card = $button.closest('.bizzplugin-plugin-card');
            var file = $button.data('file');
            
            if ($card.hasClass('bizzplugin-plugin-loading')) {
                return;
            }
            
            // Add loading state
            $card.addClass('bizzplugin-plugin-loading');
            var originalText = $button.html();
            $button.empty()
                .append($('<span>').addClass('dashicons dashicons-update'))
                .append(' ' + (self.config.strings.activating || 'Activating...'));
            
            $.ajax({
                url: self.config.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'bizzplugin_activate_plugin',
                    nonce: self.config.nonce,
                    file: file
                },
                success: function(response) {
                    if (response.success) {
                        // Update UI to show activated state
                        $card.find('.bizzplugin-plugin-status')
                            .removeClass('bizzplugin-plugin-status-installed bizzplugin-plugin-status-not-installed')
                            .addClass('bizzplugin-plugin-status-active')
                            .text(self.config.strings.active || 'Active');
                        
                        var $activatedSpan = $('<span>').addClass('bizzplugin-plugin-activated')
                            .append($('<span>').addClass('dashicons dashicons-yes'))
                            .append($('<span>').text(self.config.strings.activated || 'Activated'));
                        $card.find('.bizzplugin-plugin-actions').empty().append($activatedSpan);
                        
                        if (typeof onSuccess === 'function') {
                            onSuccess(response);
                        }
                    } else {
                        $button.html(originalText);
                        if (typeof onError === 'function') {
                            onError(response);
                        }
                    }
                },
                error: function(xhr, status, error) {
                    $button.html(originalText);
                    if (typeof onError === 'function') {
                        onError({success: false, data: {message: error}});
                    }
                    console.error('Plugin activation error:', error);
                },
                complete: function() {
                    $card.removeClass('bizzplugin-plugin-loading');
                }
            });
        },

        /**
         * Initialize repeater sortable
         * @param {jQuery} $container Container to search within
         * @param {Function} onUpdate Callback on update
         */
        initRepeaterSortable: function($container, onUpdate) {
            if (!$.fn.sortable) {
                return;
            }
            
            var $repeaters = $container ? 
                $container.find('.bizzplugin-repeater-wrap[data-sortable="1"] .bizzplugin-repeater-items') :
                $('.bizzplugin-repeater-wrap[data-sortable="1"] .bizzplugin-repeater-items');
            
            $repeaters.sortable({
                handle: '.bizzplugin-repeater-item-handle',
                axis: 'y',
                placeholder: 'bizzplugin-repeater-placeholder',
                update: function(event, ui) {
                    var $wrap = $(this).closest('.bizzplugin-repeater-wrap');
                    if (typeof onUpdate === 'function') {
                        onUpdate($wrap);
                    }
                }
            });
        },

        /**
         * Add new repeater item
         * @param {Event} e Event object
         * @param {Function} onAdd Callback after adding item
         * @param {Function} initFieldsFn Function to initialize fields in new item
         */
        repeaterAddItem: function(e, onAdd, initFieldsFn) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $wrap = $button.closest('.bizzplugin-repeater-wrap');
            var $items = $wrap.find('.bizzplugin-repeater-items');
            var $template = $wrap.find('.bizzplugin-repeater-template');
            var maxItems = parseInt($wrap.data('max-items')) || 0;
            var currentCount = $items.find('.bizzplugin-repeater-item').length;
            
            // Check max items limit
            if (maxItems > 0 && currentCount >= maxItems) {
                return;
            }
            
            // Get the new index
            var newIndex = currentCount;
            
            // Get template HTML and replace placeholders
            var newItemHtml = $template.html();
            newItemHtml = newItemHtml.replace(/\{\{INDEX\}\}/g, newIndex);
            newItemHtml = newItemHtml.replace(/\{\{DISPLAY_INDEX\}\}/g, newIndex + 1);
            
            // Create the new item element using jQuery.parseHTML for safer HTML parsing
            var $newItem = $($.parseHTML(newItemHtml, document, false));
            
            // Enable disabled inputs in template fields
            $newItem.find('[data-template-field="1"] input, [data-template-field="1"] textarea, [data-template-field="1"] select').prop('disabled', false);
            $newItem.find('[data-template-field]').removeAttr('data-template-field');
            
            // Add to items container
            $items.append($newItem);
            
            // Initialize fields for new item
            if (typeof initFieldsFn === 'function') {
                initFieldsFn($newItem);
            } else {
                // Default field initialization
                self.initColorPickers($newItem);
                self.initDatePickers($newItem);
                self.initSliders($newItem);
            }
            
            // Update remove button visibility based on min items
            self.repeaterUpdateRemoveButtons($wrap);
            
            // Hide add button if max reached
            if (maxItems > 0 && currentCount + 1 >= maxItems) {
                $button.hide();
            }
            
            // Callback
            if (typeof onAdd === 'function') {
                onAdd($newItem, $wrap);
            }
            
            // Scroll to new item
            $newItem[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
            // $(document).trigger('bizzplugin:repeater_item_added', [$newItem, $wrap]);
        },

        /**
         * Remove repeater item
         * @param {Event} e Event object
         * @param {Function} onRemove Callback after removing item
         * @param {Function} reindexFn Function to reindex items
         */
        repeaterRemoveItem: function(e, onRemove, reindexFn) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $item = $button.closest('.bizzplugin-repeater-item');
            var $wrap = $button.closest('.bizzplugin-repeater-wrap');
            var minItems = parseInt($wrap.data('min-items')) || 0;
            var currentCount = $wrap.find('.bizzplugin-repeater-item').length;
            
            // Check min items limit
            if (currentCount <= minItems) {
                return;
            }
            
            // Confirm removal
            if (!confirm(self.config.strings.confirm_remove_item || 'Are you sure you want to remove this item?')) {
                return;
            }
            
            // Remove the item with animation
            $item.slideUp(200, function() {
                $item.remove();
                
                // Reindex items
                if (typeof reindexFn === 'function') {
                    reindexFn($wrap);
                }
                
                self.repeaterUpdateRemoveButtons($wrap);
                
                // Show add button if it was hidden
                var allowAdd = $wrap.data('allow-add') !== 0 && $wrap.data('allow-add') !== '0';
                var maxItems = parseInt($wrap.data('max-items')) || 0;
                var newCount = $wrap.find('.bizzplugin-repeater-item').length;
                if (allowAdd && (maxItems === 0 || newCount < maxItems)) {
                    $wrap.find('.bizzplugin-repeater-add').show();
                }
                
                // Callback
                if (typeof onRemove === 'function') {
                    onRemove($wrap);
                }
            });
        },

        /**
         * Toggle repeater item content visibility
         * @param {Event} e Event object
         */
        repeaterToggleItem: function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            var $button = $(e.currentTarget);
            var $item = $button.closest('.bizzplugin-repeater-item');
            var $content = $item.find('.bizzplugin-repeater-item-content');
            var $icon = $button.find('.dashicons');
            
            $content.slideToggle(200);
            $item.toggleClass('collapsed open');
            
            // Toggle icon
            if ($item.hasClass('collapsed')) {
                $icon.removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
            } else {
                $icon.removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
            }
        },

        /**
         * Handle repeater header click (toggle)
         * @param {Event} e Event object
         */
        repeaterHandleHeaderClick: function(e) {
            // Don't toggle if clicking on buttons
            if ($(e.target).closest('.bizzplugin-repeater-item-actions').length) {
                return;
            }
            
            var $item = $(e.currentTarget).closest('.bizzplugin-repeater-item');
            $item.toggleClass('open');
            $item.find('.bizzplugin-repeater-item-content').slideToggle(200);
        },

        /**
         * Update repeater item title from subfields
         * Supports header_template and all field types
         * @param {Event} e Event object
         */
        repeaterUpdateItemTitle: function(e) {
            var $input = $(e.currentTarget);
            var $item = $input.closest('.bizzplugin-repeater-item');
            var $wrap = $item.closest('.bizzplugin-repeater-wrap');
            var $title = $item.find('.bizzplugin-repeater-item-title');
            var index = $item.data('index');
            var headerTemplate = $wrap.data('header-template');
            
            var title = '';
            
            if (headerTemplate) {
                // Use header template
                title = headerTemplate;
                
                // Replace all placeholders with actual values from subfields
                $item.find('.bizzplugin-repeater-subfield').each(function() {
                    var $subfield = $(this);
                    var $input = $subfield.find('input, select, textarea').first();
                    
                    if (!$input.length) return;
                    
                    // Extract field ID from input name (e.g., fieldname[0][subfield_id])
                    var inputName = $input.attr('name') || '';
                    var matches = inputName.match(/\[[^\]]+\]\[([^\]]+)\]$/);
                    var fieldId = matches ? matches[1] : '';
                    
                    if (!fieldId) return;
                    
                    var value = '';
                    
                    if ($input.is('select')) {
                        // For select, get the selected option text
                        value = $input.find('option:selected').text() || '';
                    } else if ($input.is(':checkbox')) {
                        value = $input.is(':checked') ? 'Yes' : 'No';
                    } else if ($input.is(':radio')) {
                        var checkedRadio = $item.find('input[name="' + $input.attr('name') + '"]:checked');
                        value = checkedRadio.val() || '';
                    } else {
                        value = $input.val() || '';
                    }
                    
                    // Escape special regex characters in fieldId to prevent RegExp injection
                    var escapedFieldId = fieldId.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                    // Replace placeholder
                    title = title.replace(new RegExp('\\{' + escapedFieldId + '\\}', 'g'), value);
                });
                
                // Clean up any unreplaced placeholders
                title = title.replace(/\{[^}]+\}/g, '').trim();
            } else {
                // No template - get first non-empty field value
                $item.find('.bizzplugin-repeater-subfield').each(function() {
                    if (title) return false; // Already found a value
                    
                    var $subfield = $(this);
                    var $input = $subfield.find('input, select, textarea').first();
                    
                    if (!$input.length) return;
                    
                    var value = '';
                    
                    if ($input.is('select')) {
                        // For select, get the selected option text
                        value = $input.find('option:selected').text() || '';
                    } else if ($input.is(':checkbox')) {
                        // Skip checkboxes for auto-title
                        return;
                    } else if ($input.is(':radio')) {
                        var checkedRadio = $item.find('input[name="' + $input.attr('name') + '"]:checked');
                        value = checkedRadio.val() || '';
                    } else {
                        value = $input.val() || '';
                    }
                    
                    if (value.trim()) {
                        title = value.trim();
                    }
                });
            }
            
            // Set title or default
            if (title) {
                $title.text(title);
            } else {
                $title.text('Item #' + (parseInt(index) + 1));
            }
        },

        /**
         * Update remove button visibility based on min items
         * @param {jQuery} $wrap Repeater wrapper
         */
        repeaterUpdateRemoveButtons: function($wrap) {
            var minItems = parseInt($wrap.data('min-items')) || 0;
            var currentCount = $wrap.find('.bizzplugin-repeater-item').length;
            var $removeButtons = $wrap.find('.bizzplugin-repeater-item-remove');
            
            if (currentCount <= minItems) {
                $removeButtons.hide();
            } else {
                $removeButtons.show();
            }
        },

        /**
         * Reindex repeater items after add/remove/reorder
         * @param {jQuery} $wrap Repeater wrapper
         */
        repeaterReindex: function($wrap) {
            var fieldId = $wrap.data('field-id');
            var $items = $wrap.find('.bizzplugin-repeater-item');
            
            $items.each(function(index) {
                var $item = $(this);
                var oldIndex = $item.data('index');
                
                // Skip if oldIndex is undefined or same as new index
                if (oldIndex === undefined || oldIndex === null) {
                    $item.attr('data-index', index).data('index', index);
                    return;
                }
                
                // Update data attribute
                $item.attr('data-index', index).data('index', index);
                
                // Skip regex replacement if indices are the same
                if (oldIndex === index) {
                    return;
                }
                
                // Escape special regex characters in fieldId
                var escapedFieldId = fieldId.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                
                // Update all input names and IDs
                $item.find('input, textarea, select').each(function() {
                    var $input = $(this);
                    var name = $input.attr('name');
                    var id = $input.attr('id');
                    
                    if (name) {
                        var newName = name.replace(new RegExp(escapedFieldId + '\\[' + oldIndex + '\\]'), fieldId + '[' + index + ']');
                        // Also handle generic index patterns
                        newName = newName.replace(/\[\d+\]/, '[' + index + ']');
                        $input.attr('name', newName);
                    }
                    
                    if (id) {
                        var newId = id.replace(new RegExp(escapedFieldId + '_' + oldIndex + '_'), fieldId + '_' + index + '_');
                        newId = newId.replace(/_\d+_/, '_' + index + '_');
                        $input.attr('id', newId);
                    }
                });
                
                // Update labels
                $item.find('label').each(function() {
                    var $label = $(this);
                    var forAttr = $label.attr('for');
                    if (forAttr) {
                        var newFor = forAttr.replace(new RegExp(escapedFieldId + '_' + oldIndex + '_'), fieldId + '_' + index + '_');
                        newFor = newFor.replace(/_\d+_/, '_' + index + '_');
                        $label.attr('for', newFor);
                    }
                });
                
                // Update title if it was a default title
                var $title = $item.find('.bizzplugin-repeater-item-title');
                var titleText = $title.text();
                if (titleText.match(/^Item #\d+$/)) {
                    $title.text('Item #' + (index + 1));
                }
            });
        },

        /**
         * Escape a string for use in a CSS selector
         * @param {string} str String to escape
         * @return {string} Escaped string
         */
        escapeSelector: function(str) {
            if (!str) return '';
            return str.replace(/\\/g, '\\\\').replace(/([[\]()])/g, '\\$1');
        },

        /**
         * Initialize field dependencies
         * @param {jQuery} $container Container to search within
         */
        initDependencies: function($container) {
            var self = this;
            var $elements = $container ? 
                $container.find('[data-dependency]') : 
                $('[data-dependency]');
            
            $elements.each(function() {
                var $dependent = $(this);
                var dependencyField = $dependent.data('dependency');
                var expectedValue = String($dependent.data('dependency-value'));
                var $parent = $dependent.closest('.bizzplugin-metabox-wrap, .bizzplugin-framework-wrap');
                
                // Find the trigger field
                var escapedName = self.escapeSelector(dependencyField);
                var $trigger = $parent.find('[name="' + escapedName + '"]');
                
                if (!$trigger.length) {
                    return;
                }
                
                // Get current value
                var currentValue = self.getFieldValue($trigger);
                
                // Check if should be visible
                var expectedValues = expectedValue.split(',').map(function(v) { return v.trim(); });
                var shouldShow = expectedValues.indexOf(String(currentValue)) !== -1;
                
                // Show/hide immediately without animation on initial load
                if (shouldShow) {
                    $dependent.show();
                } else {
                    $dependent.hide();
                }
            });
        },

        /**
         * Handle field dependency change
         * @param {Event} e Event object
         */
        handleDependencyChange: function(e) {
            // return;
            var self = this;
            var $trigger = $(e.currentTarget);
            var triggerName = $trigger.attr('name');
            var triggerValue = self.getFieldValue($trigger);
            
            // Find dependent fields
            var $container = $trigger.closest('.bizzplugin-metabox-wrap, .bizzplugin-framework-wrap');
            $container.find('[data-dependency]').each(function() {
                var $dependent = $(this);
                var dependencyField = $dependent.data('dependency');
                
                if (dependencyField !== triggerName) {
                    return;
                }
                
                var expectedValue = String($dependent.data('dependency-value'));
                var expectedValues = expectedValue.split(',').map(function(v) { return v.trim(); });
                var shouldShow = expectedValues.indexOf(String(triggerValue)) !== -1;
                
                if (shouldShow) {
                    $dependent.slideDown(200);
                } else {
                    $dependent.slideUp(200);
                }
            });
        },

        /**
         * Get field value based on field type
         * @param {jQuery} $field Field element
         * @return {string} Field value
         */
        getFieldValue: function($field) {
            if ($field.is(':checkbox')) {
                return $field.is(':checked') ? '1' : '';
            } else if ($field.is(':radio')) {
                return $field.filter(':checked').val() || '';
            } else {
                return $field.val() || '';
            }
        },

        /**
         * Show notification
         * @param {string} message Message to display
         * @param {string} type Type: success, error, info
         * @param {jQuery} $container Container to append to
         * @param {number} duration Duration in milliseconds (default: 3000)
         */
        showNotification: function(message, type, $container, duration) {
            type = type || 'success';
            duration = duration || 3000;
            
            var $notification = $('<div>')
                .addClass('bizzplugin-notification')
                .addClass('bizzplugin-notification-' + type);
            
            var $message = $('<span>')
                .addClass('bizzplugin-notification-message')
                .text(message);
            
            var $closeBtn = $('<button>')
                .addClass('bizzplugin-notification-close')
                .attr('aria-label', 'Close')
                .html('&times;');
            
            $notification.append($message).append($closeBtn);
            
            $closeBtn.on('click', function() {
                $notification.remove();
            });
            
            if ($container && $container.length) {
                $container.append($notification);
            } else {
                $('body').append($notification);
            }
            
            setTimeout(function() {
                $notification.fadeOut(function() {
                    $notification.remove();
                });
            }, duration);
        },

        /**
         * Handle copy button click
         * @param {Event} e Event object
         */
        handleCopyClick: function(e) {
            e.preventDefault();
            
            var $button = $(e.currentTarget);
            var textToCopy = $button.data('copy');
            
            if (navigator.clipboard && textToCopy) {
                navigator.clipboard.writeText(textToCopy).then(function() {
                    var $icon = $button.find('.dashicons');
                    var originalClass = $icon.attr('class');
                    $icon.removeClass('dashicons-admin-page').addClass('dashicons-yes');
                    
                    setTimeout(function() {
                        $icon.attr('class', originalClass);
                    }, 2000);
                });
            }
        },

        /* =============================================
           Single Repeater Functions
           ============================================= */

        /**
         * Initialize single repeater sortable
         * @param {jQuery} $container Container to search within
         * @param {Function} onUpdate Callback on update
         */
        initSingleRepeaterSortable: function($container, onUpdate) {
            if (!$.fn.sortable) {
                return;
            }
            
            var self = this;
            var $repeaters = $container ? 
                $container.find('.bizzplugin-single-repeater-wrap[data-sortable="1"] .bizzplugin-single-repeater-items') :
                $('.bizzplugin-single-repeater-wrap[data-sortable="1"] .bizzplugin-single-repeater-items');
            
            $repeaters.sortable({
                handle: '.bizzplugin-single-repeater-handle',
                axis: 'y',
                placeholder: 'bizzplugin-single-repeater-placeholder',
                update: function(event, ui) {
                    var $wrap = $(this).closest('.bizzplugin-single-repeater-wrap');
                    self.singleRepeaterReindex($wrap);
                    if (typeof onUpdate === 'function') {
                        onUpdate($wrap);
                    }
                }
            });
        },

        /**
         * Add new single repeater item
         * @param {Event} e Event object
         * @param {Function} onAdd Callback after adding item
         * @param {Function} initFieldsFn Function to initialize fields in new item
         */
        singleRepeaterAddItem: function(e, onAdd, initFieldsFn) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $wrap = $button.closest('.bizzplugin-single-repeater-wrap');
            var $items = $wrap.find('.bizzplugin-single-repeater-items');
            var $template = $wrap.find('.bizzplugin-single-repeater-template');
            var maxItems = parseInt($wrap.data('max-items')) || 0;
            var currentCount = $items.find('.bizzplugin-single-repeater-item').length;
            
            // Check max items limit
            if (maxItems > 0 && currentCount >= maxItems) {
                return;
            }
            
            // Get the new index
            var newIndex = currentCount;
            
            // Get template HTML and replace placeholders
            var newItemHtml = $template.html();
            newItemHtml = newItemHtml.replace(/\{\{INDEX\}\}/g, newIndex);
            
            // Create the new item element
            var $newItem = $($.parseHTML(newItemHtml, document, false));
            
            // Enable disabled inputs in template
            $newItem.find('input, textarea, select').prop('disabled', false);
            $newItem.removeAttr('data-template');
            
            // Add to items container
            $items.append($newItem);
            
            // Initialize fields for new item
            if (typeof initFieldsFn === 'function') {
                initFieldsFn($newItem);
            } else {
                // Default field initialization
                self.initColorPickers($newItem);
                self.initDatePickers($newItem);
            }
            
            // Update remove button visibility
            self.singleRepeaterUpdateRemoveButtons($wrap);
            
            // Hide add button if max reached
            if (maxItems > 0 && currentCount + 1 >= maxItems) {
                $button.hide();
            }
            
            // Callback
            if (typeof onAdd === 'function') {
                onAdd($newItem, $wrap);
            }
            
            // Focus on new input
            $newItem.find('input, textarea, select').first().focus();
        },

        /**
         * Remove single repeater item
         * @param {Event} e Event object
         * @param {Function} onRemove Callback after removing item
         */
        singleRepeaterRemoveItem: function(e, onRemove) {
            e.preventDefault();
            
            var self = this;
            var $button = $(e.currentTarget);
            var $item = $button.closest('.bizzplugin-single-repeater-item');
            var $wrap = $button.closest('.bizzplugin-single-repeater-wrap');
            var minItems = parseInt($wrap.data('min-items')) || 0;
            var currentCount = $wrap.find('.bizzplugin-single-repeater-item').length;
            
            // Check min items limit
            if (currentCount <= minItems) {
                return;
            }
            
            // Remove the item with animation
            $item.slideUp(150, function() {
                $item.remove();
                
                // Reindex items
                self.singleRepeaterReindex($wrap);
                self.singleRepeaterUpdateRemoveButtons($wrap);
                
                // Show add button if it was hidden
                var allowAdd = $wrap.data('allow-add') !== 0 && $wrap.data('allow-add') !== '0';
                var maxItems = parseInt($wrap.data('max-items')) || 0;
                var newCount = $wrap.find('.bizzplugin-single-repeater-item').length;
                if (allowAdd && (maxItems === 0 || newCount < maxItems)) {
                    $wrap.find('.bizzplugin-single-repeater-add').show();
                }
                
                // Callback
                if (typeof onRemove === 'function') {
                    onRemove($wrap);
                }
            });
        },

        /**
         * Update remove button visibility for single repeater based on min items
         * @param {jQuery} $wrap Single repeater wrapper
         */
        singleRepeaterUpdateRemoveButtons: function($wrap) {
            var minItems = parseInt($wrap.data('min-items')) || 0;
            var currentCount = $wrap.find('.bizzplugin-single-repeater-item').length;
            var $removeButtons = $wrap.find('.bizzplugin-single-repeater-remove');
            
            if (currentCount <= minItems) {
                $removeButtons.hide();
            } else {
                $removeButtons.show();
            }
        },

        /**
         * Reindex single repeater items after add/remove/reorder
         * @param {jQuery} $wrap Single repeater wrapper
         */
        singleRepeaterReindex: function($wrap) {
            var fieldId = $wrap.data('field-id');
            var $items = $wrap.find('.bizzplugin-single-repeater-item');
            
            $items.each(function(index) {
                var $item = $(this);
                var oldIndex = $item.data('index');
                
                // Update data attribute
                $item.attr('data-index', index).data('index', index);
                
                // Skip if indices are the same
                if (oldIndex === index) {
                    return;
                }
                
                // Escape special regex characters in fieldId
                var escapedFieldId = fieldId.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
                
                // Update all input names and IDs
                $item.find('input, textarea, select').each(function() {
                    var $input = $(this);
                    var name = $input.attr('name');
                    var id = $input.attr('id');
                    
                    if (name) {
                        var newName = name.replace(new RegExp(escapedFieldId + '\\[\\d+\\]'), fieldId + '[' + index + ']');
                        $input.attr('name', newName);
                    }
                    
                    if (id) {
                        var newId = id.replace(new RegExp(escapedFieldId + '_\\d+'), fieldId + '_' + index);
                        newId = newId.replace(/_\d+$/, '_' + index);
                        $input.attr('id', newId);
                    }
                });
            });
        }
    };

})(jQuery);
