<?php
/**
 * BizzOrder Framework Settings Loader
 * 
 * This class handles the plugin settings page using BizzPlugin Options Framework.
 * All settings are stored in a single option key 'bizzorder_settings'.
 *
 * @package Bizzorder
 * @since 1.1.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class BizzOrder_Framework_Settings
 * 
 * Manages the settings panel for BizzOrder Pre-Order plugin.
 * Uses BizzPlugin Options Framework for settings UI.
 */
class BizzOrder_Framework_Settings {

    /**
     * Singleton instance
     *
     * @var BizzOrder_Framework_Settings
     */
    private static $instance = null;

    /**
     * Panel instance
     *
     * @var BizzPlugin_Panel
     */
    private $panel = null;

    /**
     * Option name for settings storage
     *
     * @var string
     */
    private $option_name = 'bizzorder_settings';

    /**
     * Get singleton instance
     *
     * @return BizzOrder_Framework_Settings
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_framework();
        add_action( 'init', function() {
            $this->init_panel();
        } );
    }

    /**
     * Load the options framework
     */
    private function load_framework() {
        require_once BIZZORDER_PLUGIN_DIR . '/admin/options-framework/options-loader.php';
    }

    /**
     * Initialize the settings panel
     */
    private function init_panel() {
        $framework = bizzplugin_framework();
        
        $this->panel = $framework->create_panel( array(
            'id'                  => 'bizzorder-settings-panel',
            'title'               => __( 'Pre-Order Settings', 'bizzorder' ),
            'menu_title'          => __( 'BizzPreOrder', 'bizzorder' ),
            'menu_slug'           => 'bizzorder-settings',
            'capability'          => 'manage_options',
            'icon'                => 'dashicons-cart',
            'position'            => 55,
            'option_name'         => $this->option_name,
            'route_namespace'     => 'bizzorder/v1',
            'is_premium'          => false,
            'theme_mode'          => 'light',
            'show_theme_switcher' => false,
            'enable_search'       => true,
            'show_export_import'  => true,
            'show_api'            => false,
        ) );

        // Set panel configuration
        $this->panel->set_panel_config( array(
            'title'       => __( 'Bizzorder - Pre Order for WooCommerce', 'bizzorder' ),
            'version'     => defined( 'BIZZORDER_VERSION' ) ? BIZZORDER_VERSION : '1.0.0',
            'footer_text' => __( 'Powered by Bizzplugin', 'bizzorder' ),
            'logo'        => 'https://ps.w.org/bizzorder/assets/icon-128x128.png',
        ) );


        $this->add_setup_wizard();
        $this->add_topbar();

        // Add sections
        $this->add_general_section();
        $this->add_display_section();
        $this->add_cart_section();

        $this->add_recommendations_section();

        // Add resources
        $this->add_resources();
    }

    public function add_setup_wizard() {
        $this->panel->add_setup_wizard(array(
            'title'         => __('Bizzorder Setup', 'bizzorder'),
            'redirect_key'   => BIZZORDER_PLUGIN_BASE, 
            'skip_button'   => true,
            'skip_button_text' => __('Skip Setup', 'bizzorder'),
            'progress'      => true,
            'next_text'   => __('Next', 'bizzorder'),
            'prev_text'   => __('Previous', 'bizzorder'),
            'finish_button' => __('Finish', 'bizzorder'),
            'links'             => array(

                array(
                    'text'  => __('Settings', 'bizzorder'),
                    'url'   => admin_url( 'admin.php?page=bizzorder-settings' ),
                    'target'=> '_self',
                    'icon'  => 'dashicons dashicons-admin-generic',
                    'class' => 'button button-primary',
                ),
                array(
                    'text'  => __('Visit Shop Page', 'bizzorder'),
                    'url'   => get_permalink( wc_get_page_id( 'shop' ) ),
                    'target'=> '_blank',
                    'icon'  => 'dashicons dashicons-sos',
                    'class' => 'button button-secondary',
                ),
                // array(
                //     'text'  => __('Get Premium', 'bizzorder'),
                //     'url'   => 'https://codeastrology.com/downloads/bizzorder-premium/',
                //     'target'=> '_blank',
                //     'icon'  => 'dashicons dashicons-awards',
                //     'class' => 'button button-secondary',
                // ),
            ),
        ));
    }
    /**
     * Add topbar links
     */
    public function add_topbar() {
        $this->panel->set_topbar( array(
            'left_links' => array(
                array(
                    'title'  => __( 'Documentation', 'bizzorder' ),
                    'url'    => 'https://codeastrology.com/docs/docs/bizzorder/',
                    'icon'   => 'dashicons dashicons-media-document',
                    'target' => '_blank',
                ),
                array(
                    'title'  => __( 'Support', 'bizzorder' ),
                    'url'    => 'https://codeastrology.com/my-support',
                    'icon'   => 'dashicons dashicons-sos',
                    'target' => '_blank',
                ),
                array(
                    'title'  => __( 'Browse Plugins', 'bizzorder' ),
                    'url'    => admin_url( 'plugin-install.php?s=codersaiful&tab=search&type=author' ),
                    'icon'   => 'dashicons dashicons-store',
                    'target' => '_blank',
                ),
            ),
            'right_links' => array(
                array(
                    'title'  => __( 'Live Demo', 'bizzorder' ),
                    'url'    => 'https://wpprincipal.xyz?demo=wppo',
                    'icon'   => 'dashicons dashicons-visibility',
                    'target' => '_blank',
                    'class'  => 'bizzplugin-topbar-btn-primary',
                ),
                array(
                    'title'  => __( 'Rate Us', 'bizzorder' ),
                    'url'    => 'https://wordpress.org/support/plugin/bizzorder/reviews/#new-post',
                    'icon'   => 'dashicons dashicons-star-filled',
                    'target' => '_blank',
                    'class'  => 'bizzplugin-topbar-btn-success',
                ),
            ),
        ) );
    }

    /**
     * Add General Settings section
     */
    private function add_general_section() {
        $this->panel->add_section( array(
            'id'          => 'general_settings',
            'title'       => __( 'General', 'bizzorder' ),
            'description' => __( 'Configure the basic pre-order settings.', 'bizzorder' ),
            'icon'        => 'dashicons dashicons-admin-generic',
            'fields'      => array(
                array(
                    'id'          => 'button_text',
                    'type'        => 'text',
                    'title'       => __( 'Pre-Order Button Text', 'bizzorder' ),
                    'description' => __( 'Text displayed on the Add to Cart button for pre-order products.', 'bizzorder' ),
                    'default'     => 'Pre Order Now!',
                    'placeholder' => __( 'Pre Order Now!', 'bizzorder' ),
                    'setup_wizard' => true,
                ),
                array(
                    'id'          => 'status_label',
                    'type'        => 'text',
                    'title'       => __( 'Pre-Order Status Label', 'bizzorder' ),
                    'description' => __( 'Label for the pre-ordered order status.', 'bizzorder' ),
                    'default'     => 'Pre-Ordered',
                    'placeholder' => __( 'Pre-Ordered', 'bizzorder' ),
                ),
                array(
                    'id'          => 'cart_notice',
                    'type'        => 'text',
                    'title'       => __( 'Cart Item Notice', 'bizzorder' ),
                    'description' => __( 'Notice shown below pre-order items in cart. Use {days_left} for days remaining.', 'bizzorder' ),
                    'default'     => 'This item will be available for shipping in {days_left} days',
                    'placeholder' => __( 'This item will be available for shipping in {days_left} days', 'bizzorder' ),
                ),
                array(
                    'id'          => 'available_date_text',
                    'type'        => 'text',
                    'title'       => __( 'Available Date Text', 'bizzorder' ),
                    'description' => __( 'Text showing when product will be available. Use {date_format} for the date.', 'bizzorder' ),
                    'default'     => 'Available on {date_format}',
                    'placeholder' => __( 'Available on {date_format}', 'bizzorder' ),
                ),
            ),
        ) );
    }

    /**
     * Add Display Settings section
     */
    private function add_display_section() {
        $this->panel->add_section( array(
            'id'          => 'display_settings',
            'title'       => __( 'Display', 'bizzorder' ),
            'description' => __( 'Configure how pre-order information is displayed on your store.', 'bizzorder' ),
            'icon'        => 'dashicons dashicons-visibility',
            'fields'      => array(
                array(
                    'id'          => 'show_badge',
                    'type'        => 'switch',
                    'title'       => __( 'Show Pre-Order Badge', 'bizzorder' ),
                    'description' => __( 'Display a "Pre-Order" badge on shop and product pages.', 'bizzorder' ),
                    'default'     => '1',
                    'on_label'    => __( 'yes', 'bizzorder' ),
                    'off_label'   => __( 'no', 'bizzorder' ),
                    'setup_wizard' => true,
                ),
                array(
                    'id'          => 'show_date',
                    'type'        => 'switch',
                    'title'       => __( 'Show Available Date', 'bizzorder' ),
                    'description' => __( 'Display the available date on product pages.', 'bizzorder' ),
                    'default'     => '1',
                    'on_label'    => __( 'Yes', 'bizzorder' ),
                    'off_label'   => __( 'No', 'bizzorder' ),
                    'setup_wizard' => true,
                ),
            ),
        ) );
    }

    /**
     * Add Cart Settings section
     */
    private function add_cart_section() {
        $this->panel->add_section( array(
            'id'          => 'cart_settings',
            'title'       => __( 'Cart', 'bizzorder' ),
            'description' => __( 'Configure cart behavior for pre-order products.', 'bizzorder' ),
            'icon'        => 'dashicons dashicons-cart',
            'fields'      => array(
                array(
                    'id'          => 'either_mode',
                    'type'        => 'switch',
                    'title'       => __( 'Either Mode', 'bizzorder' ),
                    'description' => __( 'Prevent customers from mixing pre-order and regular products in the same cart.', 'bizzorder' ),
                    'default'     => '1',
                    'on_label'    => __( 'Yes', 'bizzorder' ),
                    'off_label'   => __( 'No', 'bizzorder' ),
                    'setup_wizard' => true,
                ),
            ),
        ) );
    }

    /**
     * Add Recommendations section
     */
    private function add_recommendations_section() {
        $this->panel->add_section( array(
            'id'          => 'recommended_plugins',
            'title'       => __( 'Recommended Plugins', 'bizzorder' ),
            'description' => __( 'Enhance your WordPress site with these recommended plugins. Install and activate them directly from here.', 'bizzorder' ),
            'icon'        => 'dashicons dashicons-plugins-checked',
            'fields'      => array(
                array(
                    'id'      => 'recommended_plugins_info',
                    'type'    => 'html',
                    'title'   => __( 'About Recommended Plugins', 'bizzorder' ),
                    'content' => '<div class="bizzplugin-notice bizzplugin-notice-info">
                        <p><strong>' . esc_html__( 'Discover Amazing Plugins!', 'bizzorder' ) . '</strong></p>
                        <p>' . esc_html__( 'These plugins are carefully selected to work perfectly with your setup. Install them with a single click!', 'bizzorder' ) . '</p>
                    </div>',
                    'setup_wizard' => true,
                ),
                array(
                    'id'          => 'recommended_plugins_list',
                    'type'        => 'plugins',
                    'title'       => __( 'Recommended Plugins', 'bizzorder' ),
                    'description' => __( 'Click Install to download and install the plugin, then click Activate to enable it.', 'bizzorder' ),
                    'plugins'     => $this->get_recommended_plugins(),
                    'setup_wizard' => true,
                ),
            ),
        ) );
    }

    /**
     * Get recommended plugins
     *
     * @return array
     */
    private function get_recommended_plugins( $shorted = false ) {
        $plugins = array(
            array(
                'slug'        => 'woo-product-table',
                'name'        => 'Product Table for WooCommerce',
                'description' => __( 'Display WooCommerce products in a table layout.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/woo-product-table/assets/icon-256x256.gif',
                'author'      => 'Bizzplugin',
                'file'        => 'woo-product-table/woo-product-table.php',
                'url'         => 'https://wordpress.org/plugins/woo-product-table/',
            ),
            array(
                'slug'        => 'woo-min-max-quantity-step-control-single',
                'name'        => 'Min Max Control - Control Quantity for WooCommerce',
                'description' => __( 'Control minimum and maximum quantities and step values for WooCommerce products.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/woo-min-max-quantity-step-control-single/assets/icon-256x256.png',
                'author'      => 'Bizzplugin',
                'file'        => 'woo-min-max-quantity-step-control-single/wcmmq.php',
                'url'         => 'https://wordpress.org/plugins/woo-min-max-quantity-step-control-single/',
            ),
            array(
                'slug'        => 'product-sync-master-sheet',
                'name'        => 'Sync Master Sheet - Sync with Google Sheet',
                'description' => __( 'Sync your product data with a google sheet.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/product-sync-master-sheet/assets/icon-256x256.gif',
                'author'      => 'Bizzplugin',
                'file'        => 'product-sync-master-sheet/product-sync-master-sheet.php',
                'url'         => 'https://wordpress.org/plugins/product-sync-master-sheet/',
            ),
            array(
                'slug'        => 'bizzorder',
                'name'        => 'Bizzview - Quick View for WooCommerce',
                'description' => __( 'Add quick view functionality to your WooCommerce products.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/ca-quick-view/assets/icon-256x256.png?new',
                'author'      => 'Bizzplugin',
                'file'        => 'ca-quick-view/starter.php',
                'url'         => 'https://wordpress.org/plugins/ca-quick-view/',
            ),
            array(
                'slug'        => 'bizzswatches',
                'name'        => 'Bizzswatches - Color and Image Swatches',
                'description' => __( 'Add color and image swatches to your WooCommerce products.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/bizzswatches/assets/icon-256x256.png?new',
                'author'      => 'Bizzplugin',
                'file'        => 'bizzswatches/bizzswatches.php',
                'url'         => 'https://wordpress.org/plugins/bizzswatches/',
            ),
            array(
                'slug'        => 'wc-quantity-plus-minus-button',
                'name'        => 'Quantity Plus Minus Button for WooCommerce',
                'description' => __( 'Add plus and minus buttons to WooCommerce quantity fields.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/wc-quantity-plus-minus-button/assets/icon-256x256.png',
                'author'      => 'Bizzplugin',
                'file'        => 'wc-quantity-plus-minus-button/init.php',
                'url'         => 'https://wordpress.org/plugins/wc-quantity-plus-minus-button/',
            ),
            array(
                'slug'        => 'bizzmudra',
                'name'        => 'Bizzmudra - Multi Currency Switcher',
                'description' => __( 'A multi currency switcher for WooCommerce.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/bizzmudra/assets/icon-256x256.png',
                'author'      => 'Bizzplugin',
                'file'        => 'bizzmudra/bizzmudra.php',
                'url'         => 'https://wordpress.org/plugins/bizzmudra/',
            ),
            array(
                'slug'        => 'sheet-to-wp-table-for-google-sheet',
                'name'        => 'Sheet to Table Live Sync for Google Sheet',
                'description' => __( 'Display Google Sheet data in WordPress tables with live sync.', 'bizzorder' ),
                'thumbnail'   => 'https://ps.w.org/sheet-to-wp-table-for-google-sheet/assets/icon-256x256.png',
                'author'      => 'Bizzplugin',
                'file'        => 'sheet-to-wp-table-for-google-sheet/sheet-to-wp-table-for-google-sheet.php',
                'url'         => 'https://wordpress.org/plugins/sheet-to-wp-table-for-google-sheet/',
            ),
            
        );
        if( $shorted ) {
            shuffle( $plugins ); // Randomize plugin order
            // Return only first 6 plugins
            $plugins = array_slice( $plugins, 0, 6 );
        }
    
        return $plugins;
    }

    /**
     * Add resources to sidebar
     */
    private function add_resources() {
        $this->panel->add_recommended_plugins( $this->get_recommended_plugins( true ) );
    }

    /**
     * Get panel instance
     *
     * @return BizzPlugin_Panel
     */
    public function get_panel() {
        return $this->panel;
    }
}
