<?php
/**
 * Admin Loader Class
 * 
 * Handles all admin-related functionality for BizzOrder Pre-Order plugin.
 *
 * @package Bizzorder
 * @since 1.1.0
 */

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * BizzOrder Admin Loader Class
 */
class BizzOrder_Admin_Loader {

    /**
     * Singleton instance
     *
     * @var BizzOrder_Admin_Loader
     */
    private static $instance = null;

    /**
     * Settings page slug
     *
     * @var string
     */
    private $menu_slug = 'bizzorder-settings';

    /**
     * Get singleton instance
     *
     * @return BizzOrder_Admin_Loader
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Plugin action links
        add_filter( 'plugin_action_links_' . BIZZORDER_PLUGIN_BASE, [ $this, 'plugin_action_links' ] );
        // Row meta links
        add_filter( 'plugin_row_meta', [ $this, 'plugin_row_meta' ], 10, 3 );

        // Enqueue admin scripts
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );

        add_action( 'admin_menu', [ $this, 'add_admin_menu' ], 56 );
        add_filter( 'plugins_api_result', [ $this, 'filter_plugins_api_result' ], 1, 3 );
    }

    /**
     * Enqueue admin scripts
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_scripts( $hook ) {

        if ( strpos( $hook, 'bizzorder' ) === false ) {
            return;
        }

        // WordPress scripts
        wp_enqueue_style( 'woocommerce_admin_styles' );

        // Admin styles
        wp_enqueue_style(
            'bizzorder-admin',
            BIZZORDER_PLUGIN_URL . 'assets/css/admin.css',
            [],
            BIZZORDER_VERSION
        );

        // Admin script
        wp_enqueue_script(
            'bizzorder-admin',
            BIZZORDER_PLUGIN_URL . 'assets/js/admin.js',
            [ 'jquery' ],
            BIZZORDER_VERSION,
            true
        );
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            $this->menu_slug,
            esc_html__( 'Browse Plugins', 'bizzorder' ),
            esc_html__( 'Browse Plugins', 'bizzorder' ),
            'manage_options',
            $this->menu_slug . '-browse-plugins',
            [ $this, 'render_browse_plugins' ],
        );
    }

    /**
     * Render browse plugins page
     */
    public function render_browse_plugins() {
        ?>
        <div class="wrap bizzorder_wrap wpt-content">
            <h1 class="wp-heading "></h1>
            <div class="fieldwrap">

                <div class="wpt-section-panel no-background wpt-clearfix">
                    <?php 
                    
                    $wp_list_table = _get_list_table( 'WP_Plugin_Install_List_Table' );
                    
                    $wp_list_table->prepare_items();

                    echo '<form id="plugin-filter" method="post">';
                    $wp_list_table->display();
                    echo '</form>';
                    ?>
                </div>
            </div>

        </div>
        <?php 
    }

    /**
     * Plugin action links
     *
     * @param array $links Plugin links
     * @return array
     */
    public function plugin_action_links( $links ) {
        $settings_link = '<a href="' . admin_url( 'admin.php?page=' . $this->menu_slug ) . '">' . 
            esc_html__( 'Settings', 'bizzorder' ) . '</a>';
        array_unshift( $links, $settings_link );
        return $links;
    }

    public function plugin_row_meta( $links, $file, $data ) {
        if ( BIZZORDER_PLUGIN_BASE === $file ) {
            $row_meta = [
                'docs' => '<a href="' . esc_url( 'https://codeastrology.com/docs/docs/bizzorder/' ) . '" target="_blank">' . esc_html__( 'Docs', 'bizzorder' ) . '</a>',
                'support' => '<a href="' . esc_url( 'https://codeastrology.com/my-support' ) . '" target="_blank">' . esc_html__( 'Support', 'bizzorder' ) . '</a>',
            ];
            return array_merge( $links, $row_meta );
        }
        return $links;
    }

    /**
     * Filter plugins API result for browse plugins page
     *
     * @param object $res    API response.
     * @param string $action API action.
     * @param object $args   API arguments.
     * @return object
     */
    public function filter_plugins_api_result( $res, $action, $args ) {
    
        if ( $action !== 'query_plugins' ) {
            return $res;
        }
        
        //phpcs:ignore WordPress.Security.NonceVerification.Recommended - Not needed for this read-only request
        if ( isset( $_GET['page'] ) && sanitize_text_field( wp_unslash( $_GET['page'] ) ) === 'bizzorder-settings-browse-plugins' ) {
            // Will Continue
        } else {
            return $res;
        }

        $browse_plugins = get_transient( 'codersaiful_browse_plugins' );

        if ( $browse_plugins ) {
            return $browse_plugins;
        }
        
        $wp_version = get_bloginfo( 'version', 'display' );
        $action     = 'query_plugins';
        $args       = array(
            'page'       => 1,
            'wp_version' => $wp_version,
        );
        $args['author'] = 'codersaiful';

        $url = 'https://api.wordpress.org/plugins/info/1.2/';
        $url = add_query_arg(
            array(
                'action'  => $action,
                'request' => $args,
            ),
            $url
        );

        $http_url = $url;
        $ssl      = wp_http_supports( array( 'ssl' ) );
        if ( $ssl ) {
            $url = set_url_scheme( $url, 'https' );
        }

        $http_args = array(
            'timeout'    => 15,
            'user-agent' => 'WordPress/' . $wp_version . '; ' . home_url( '/' ),
        );
        $request   = wp_remote_get( $url, $http_args );

        if ( $ssl && is_wp_error( $request ) ) {
            $request = wp_remote_get( $http_url, $http_args );
        }

        $res = json_decode( wp_remote_retrieve_body( $request ), true );
        if ( is_array( $res ) ) {
            $res = (object) $res;
            set_transient( 'codersaiful_browse_plugins', $res, 32000 );
        }
        
        return $res;
    }
}
