<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Class BirthdayBash_Cron
 *
 * Handles cron jobs for sending birthday coupons.
 */
class BirthdayBash_Cron {

    protected static $instance = null;

    public static function get_instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    protected function __construct() {
        add_action( 'wp', array( $this, 'schedule_daily_cron_job' ) );
        add_action( 'birthday_bash_daily_cron', array( $this, 'do_daily_birthday_check' ) );
    }

    /**
     * Schedule the daily cron job.
     */
    public function schedule_daily_cron_job() {
        // We will schedule only once to avoid multiple instances for 'every_minute' for testing.
        // For production, you can keep the clear_scheduled_hook for daily, but for 'every_minute'
        // it's usually managed by a flag or specific conditions to avoid rapid re-scheduling.
        // For debugging, 'every_minute' can be disruptive. Let's ensure only one.
        if ( ! wp_next_scheduled( 'birthday_bash_daily_cron' ) ) {
            wp_schedule_event( time(), 'daily', 'birthday_bash_daily_cron' ); // Change 'daily' for production
        }
    }

    /**
     * Perform daily birthday check and send coupons.
     */
    public function do_daily_birthday_check() {
        // Removed: error_log( 'BIRTHDAY_BASH_CRON_DEBUG: Starting daily birthday check.' );

        // Ensure WooCommerce email classes are loaded.
        // This is crucial for cron jobs that might run in a more minimal WordPress context.
        if ( function_exists( 'WC' ) ) {
            // Ensure the WC Mailer is initialized, which triggers 'woocommerce_email_classes' filter.
            // This ensures WC_Email and other base email classes are defined, and your custom email is registered.
            WC()->mailer();
            // Removed: error_log( 'BIRTHDAY_BASH_CRON_DEBUG: Triggered WC()->mailer() to load email classes.' );
        } else {
            // Removed: error_log( 'BIRTHDAY_BASH_CRON_DEBUG: WC() function not found. WooCommerce might not be fully loaded or active.' );
            return; // Cannot proceed without WooCommerce mailer.
        }

        // Now, safely ensure BirthdayBash_Email class is available.
        // It should have been loaded via the 'woocommerce_email_classes' filter when WC()->mailer() ran.
        if ( ! class_exists( 'BirthdayBash_Email' ) ) {
            // Fallback require_once, although the filter should handle this if hooked correctly.
            require_once BIRTHDAY_BASH_PLUGIN_DIR . 'includes/common/class-birthday-bash-email.php';
            // Removed: error_log( 'BIRTHDAY_BASH_CRON_DEBUG: Fallback require_once for class-birthday-bash-email.php.' );
        }
       
        $days_before_birthday = 7; // Send coupon if the birthday is within 7 days from today.

        // Get current date/time in WordPress's configured timezone as DateTime object.
        $today_wp_datetime = new DateTime( current_time( 'mysql' ), wp_timezone() );
        $current_year      = (int) wp_date( 'Y' ); // Current year in WP timezone for meta key.

        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- This query runs on a daily cron and is the standard way to query user meta.
        $users = get_users( array(
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- This meta_query is part of the standard get_users API and runs on a daily cron.
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key'     => 'birthday_bash_birthday_day',
                    'compare' => 'EXISTS',
                ),
                array(
                    'key'     => 'birthday_bash_birthday_month',
                    'compare' => 'EXISTS',
                ),
            ),
            'fields'     => array( 'ID', 'user_email', 'display_name' ),
        ) );

        foreach ( $users as $user ) {
            $birthday_day   = (int) get_user_meta( $user->ID, 'birthday_bash_birthday_day', true );
            $birthday_month = (int) get_user_meta( $user->ID, 'birthday_bash_birthday_month', true );

            // Skip if day or month data is missing or invalid.
            if ( ! $birthday_day || ! $birthday_month ) {
                // Removed: error_log( 'BirthdayBash_Cron: User ' . $user->ID . ' has incomplete or invalid birthday data (day/month missing).' );
                continue;
            }

            // Construct this year's birthday date string (e.g., '2025-07-15').
            $this_year_birthday_date_str = sprintf( '%d-%02d-%02d', $current_year, $birthday_month, $birthday_day );

            // Create a DateTime object for this year's birthday, using WordPress's timezone.
            $this_year_birthday_dt = DateTime::createFromFormat( 'Y-m-d', $this_year_birthday_date_str, wp_timezone() );

            // Handle invalid dates (e.g., Feb 30th, or if the format doesn't match).
            if ( false === $this_year_birthday_dt ) {
                // Removed: error_log( 'BirthdayBash_Cron: Failed to parse birthday date for user ' . $user->ID . '. Original input: ' . $this_year_birthday_date_str );
                continue;
            }

            // Determine the next upcoming birthday.
            // Start by assuming it's this year's birthday.
            $upcoming_birthday_dt = clone $this_year_birthday_dt;

            // If this year's birthday has already passed today's date, then the next birthday is next year.
            // Compare only dates, ignoring time component for this check.
            $today_date_only = new DateTime( $today_wp_datetime->format('Y-m-d'), wp_timezone() );
            $upcoming_birthday_date_only = new DateTime( $upcoming_birthday_dt->format('Y-m-d'), wp_timezone() );

            if ( $upcoming_birthday_date_only < $today_date_only ) {
                $upcoming_birthday_dt->modify( '+1 year' );
            }

            // Calculate the difference in days from today to the upcoming birthday.
            $interval = $today_wp_datetime->diff( $upcoming_birthday_dt );
            $days_until_birthday = $interval->days;

            // The core condition: if (days until birthday) is less than $days_before_birthday.
            // This means if $days_before_birthday is 7, it will send if 0, 1, 2, 3, 4, 5, or 6 days remain.
            if ( $days_until_birthday < $days_before_birthday ) {

                // Check if coupon already issued for this user for current year's birthday.
                // This is crucial to prevent sending multiple coupons within the X-day window.
                $coupon_issued_meta_key = '_birthday_bash_coupon_issued_' . $current_year; // e.g., _birthday_bash_coupon_issued_2025
                $issued_this_year       = get_user_meta( $user->ID, $coupon_issued_meta_key, true );

                if ( $issued_this_year ) {
                    // Removed: error_log( 'BirthdayBash_Cron: Coupon already issued for user ID ' . $user->ID . ' for ' . $current_year . '.' );
                    continue; // Coupon already issued for this year, skip.
                }

                // Check if user has unsubscribed.
                $unsubscribed = get_user_meta( $user->ID, 'birthday_bash_unsubscribed', true );
                if ( $unsubscribed ) {
                    // Removed: error_log( 'BirthdayBash_Cron: User ID ' . $user->ID . ' has unsubscribed from birthday emails.' );
                    continue; // User has unsubscribed, skip.
                }

                // Generate and send coupon.
                $coupon_code = strtoupper( BirthdayBash_Helper::generate_unique_coupon_code() ); // FIXED: Ensure coupon code is uppercase
                $coupon_id   = BirthdayBash_Helper::create_woocommerce_coupon( $coupon_code, $user->ID );

                if ( $coupon_id ) {
                    // Update user meta to indicate coupon issued for this year.
                    update_user_meta( $user->ID, $coupon_issued_meta_key, time() );

                    // Log coupon issuance in custom table.
                    // Use the upcoming birthday's MM-DD format for logging.
                    $upcoming_birthday_md = $upcoming_birthday_dt->format( 'm-d' );
                    BirthdayBash_DB::insert_coupon_log( $coupon_id, $user->ID, $upcoming_birthday_md, current_time( 'mysql' ), $coupon_code );

                    // Removed: error_log( 'BIRTHDAY_BASH_EMAIL_DEBUG: Attempting to send email for user ID: ' . $user->ID . ' with coupon ID: ' . $coupon_id );
                    
                    // Obtain the email instance from WooCommerce's mailer and send the email correctly.
                    $email_object = WC()->mailer()->get_emails();
                    $birthday_coupon_email_instance = $email_object['BirthdayBash_Birthday_Coupon'];

                    if ( $birthday_coupon_email_instance ) {
                        // Set email properties on the instance managed by WooCommerce's mailer.
                        $coupon = new WC_Coupon($coupon_id); // Re-instantiate coupon object for fresh data if needed.
                        $user_data_obj = get_userdata($user->ID); // Ensure user object is fresh.

                        $birthday_coupon_email_instance->object    = $user_data_obj;
                        $birthday_coupon_email_instance->recipient = $user_data_obj->user_email;

                        $birthday_coupon_email_instance->placeholders['{customer_name}']    = $user_data_obj->display_name;
                        $birthday_coupon_email_instance->placeholders['{coupon_code}']      = $coupon->get_code();
                        $birthday_coupon_email_instance->placeholders['{coupon_amount}']    = $coupon->get_amount();
                        $birthday_coupon_email_instance->placeholders['{coupon_type_text}'] = BirthdayBash_Helper::get_coupon_amount_text( $coupon );
                        $birthday_coupon_email_instance->placeholders['{coupon_expiry_date}'] = $coupon->get_date_expires() ? date_i18n( wc_date_format(), $coupon->get_date_expires()->getOffsetTimestamp() ) : esc_html__( 'Never', 'birthday-bash' );

                        // Trigger the email send via WooCommerce's mailer.
                        $email_sent = WC()->mailer()->send(
                            $birthday_coupon_email_instance->recipient,
                            $birthday_coupon_email_instance->get_subject(),
                            $birthday_coupon_email_instance->get_content_html(),
                            $birthday_coupon_email_instance->get_headers(),
                            $birthday_coupon_email_instance->get_attachments()
                        );

                        if ( ! $email_sent ) {
                            // Removed: error_log( 'BIRTHDAY_BASH_EMAIL_DEBUG: Email send function called but FAILED for user ID: ' . $user->ID . ' with coupon ID: ' . $coupon_id );
                        }
                    } else {
                        // Removed: error_log( 'BIRTHDAY_BASH_EMAIL_DEBUG: Custom email type "BirthdayBash_Birthday_Coupon" not found in WC Mailer for user ID: ' . $user->ID );
                    }
                    
                    // Removed: error_log( 'BirthdayBash_Cron: Coupon ' . $coupon_code . ' sent to user ID ' . $user->ID . ' for birthday ' . $upcoming_birthday_md . '.' );
                } else {
                    // Removed: error_log( 'BirthdayBash_Cron: Failed to create coupon for user ID ' . $user->ID . '.' );
                }
            }
        }
        // Removed: error_log( 'BIRTHDAY_BASH_CRON_DEBUG: Finished daily birthday check.' );
    }
}