<?php
/*
Plugin Name: Bible Reader
Description: Bible reader with a 'Verse of the Day' and a 'Hebrew Date Converter'.
Version: 1.0.1
Requires at least: 5.6
Requires PHP: 7.4
Author: Quincy van Riet
Text Domain: bible-reader
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) exit;

class BibleReader {
    private $api_key;

    private $special_versions = [
        'ara' => ['Arabic' => 'b17e246951402e50-01'],
        'fas' => ['Persian' => '7cd100148df29c08-01'],
        'yid' => ['Yiddish' => 'b6e21a7696dccae7-01'],
        'ell' => [
            '1904 Patriarchal' => '901dcd9744e1bf69-01',
            'Brenton LXX' => 'c114c33098c4fef1-01',
            'Textus Receptus' => '3aefb10641485092-01',
            'Solid Rock NT' => '47f396bad37936f0-01',
            'Text-Critical NT' => '7644de2e4c5188e5-01',
            'Family 35' => '5e29945cf530b0f6-01'
        ],
        'est' => [
            'Biblica® Open Estonian Contemporary Version™' => 'c1c43b459a6be524-01',
            'Piibel Kõigile' => 'e1c5cd49c0757432-01'
        ]
    ];

    // Share options
    private $share_options = [
        'x' => [
            'name' => 'X',
            'icon' => 'fab fa-x-twitter',
            'url' => 'https://x.com/intent/tweet?text=[TEXT]&url=[URL]'
        ],
        'truthsocial' => [
            'name' => 'Truth Social',
            'icon' => 'custom',
            'url' => 'https://truthsocial.com/intent/post?text=[TEXT]%0A%0A[URL]'
        ],
        'gab' => [
            'name' => 'Gab',
            'icon' => 'custom',
            'url' => 'https://gab.com/compose?text=[TEXT]%0A%0A[URL]'
        ],
        'whatsapp' => [
            'name' => 'WhatsApp',
            'icon' => 'fab fa-whatsapp',
            'url' => 'https://wa.me/?text=[TEXT]%20[URL]'
        ],
        'telegram' => [
            'name' => 'Telegram',
            'icon' => 'fab fa-telegram',
            'url' => 'https://t.me/share/url?url=[URL]&text=[TEXT]'
        ],
        'email' => [
            'name' => 'Email',
            'icon' => 'fas fa-envelope',
            'url' => 'mailto:?subject=Bible Verse&body=[TEXT]%0A%0A[URL]'
        ]
    ];

    public function __construct() {
        $this->api_key = get_option('bible_reader_api_key', '');
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_shortcode('bible_reader', [$this, 'render_bible_reader']);
        add_shortcode('verse_of_the_day', [$this, 'render_verse_of_the_day']);
        add_shortcode('hebrew_date_converter', [$this, 'render_hebrew_date_converter']);
        add_action('wp_ajax_bible_api_request', [$this, 'handle_api_request']);
        add_action('wp_ajax_nopriv_bible_api_request', [$this, 'handle_api_request']);
        add_action('wp_ajax_get_random_verse', [$this, 'handle_random_verse_request']);
        add_action('wp_ajax_nopriv_get_random_verse', [$this, 'handle_random_verse_request']);
        add_action('wp_ajax_convert_to_hebrew_date', [$this, 'handle_hebrew_date_conversion']);
        add_action('wp_ajax_nopriv_convert_to_hebrew_date', [$this, 'handle_hebrew_date_conversion']);
        add_action('wp_ajax_search_bible_api', [$this, 'handle_search_request']);
        add_action('wp_ajax_nopriv_search_bible_api', [$this, 'handle_search_request']);
        add_action('wp_ajax_bible_reader_search', [$this, 'handle_search_request']);
        add_action('wp_ajax_nopriv_bible_reader_search', [$this, 'handle_search_request']);
        add_action('init', [$this, 'maybe_clear_old_verses']);
    }

    public function add_admin_menu() {
        add_options_page(
            'Bible Reader Settings',
            'Bible Reader',
            'manage_options',
            'bible-reader-settings',
            [$this, 'settings_page_html']
        );
    }

    public function register_settings() {
        register_setting(
            'bible_reader_settings', 
            'bible_reader_api_key', 
            [
                'sanitize_callback' => 'sanitize_text_field'
            ]
        );
        
        add_settings_section(
            'bible_reader_api_section',
            'API Settings',
            [$this, 'api_section_html'],
            'bible-reader-settings'
        );

        add_settings_field(
            'bible_reader_api_key',
            'API.Bible Key',
            [$this, 'api_key_field_html'],
            'bible-reader-settings',
            'bible_reader_api_section'
        );
    }

    public function api_section_html() {
        echo '<p>Get your free API key from <a href="https://scripture.api.bible/" target="_blank">scripture.api.bible</a></p>';
    }

    public function api_key_field_html() {
        $api_key = get_option('bible_reader_api_key');
        echo '<input type="text" name="bible_reader_api_key" value="' . esc_attr($api_key) . '" class="regular-text">';
    }

    public function settings_page_html() {
        if (!current_user_can('manage_options')) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('bible_reader_settings');
                do_settings_sections('bible-reader-settings');
                submit_button('Save Settings');
                ?>
            </form>
            <h2>Plugin Usage</h2>
            <div class="card">
                <h3>Shortcodes</h3>
                <p><strong>Bible Reader:</strong> <code>[bible_reader]</code></p>
                <p><strong>Verse of the Day:</strong> <code>[verse_of_the_day]</code></p>
                <p><strong>Hebrew Date Converter:</strong> <code>[hebrew_date_converter]</code></p>
                
                <h3>Requirements</h3>
                <p>You must register for a free API key at <a href="https://scripture.api.bible/" target="_blank">scripture.api.bible</a></p>
            </div>
        </div>
        <?php
    }

    public function enqueue_assets() {
        wp_enqueue_style(
            'bible-api-css',
            plugin_dir_url(__FILE__) . 'bible-api.css',
            array(),
            filemtime(plugin_dir_path(__FILE__) . 'bible-api.css')
        );
        
        wp_enqueue_script(
            'bible-api-js',
            plugin_dir_url(__FILE__) . 'bible-api.js',
            ['jquery'],
            '1.0',
            true
        );

        wp_localize_script('bible-api-js', 'bibleReader', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bible_reader_nonce'),
            'pluginUrl' => plugin_dir_url(__FILE__),
            'apiKey' => $this->api_key,
            'specialVersions' => $this->special_versions,
            'shareOptions' => $this->share_options,
            'siteUrl' => get_site_url(),
            'excludedWordsUrls' => [
                'eng' => plugin_dir_url(__FILE__) . 'excluded-words-eng.json',
                'ara' => plugin_dir_url(__FILE__) . 'excluded-words-ara.json',
                'est' => plugin_dir_url(__FILE__) . 'excluded-words-est.json',
                'ell' => plugin_dir_url(__FILE__) . 'excluded-words-ell.json',
                'fas' => plugin_dir_url(__FILE__) . 'excluded-words-fas.json',
                'fra' => plugin_dir_url(__FILE__) . 'excluded-words-fra.json',
                'ger' => plugin_dir_url(__FILE__) . 'excluded-words-ger.json',
                'heb' => plugin_dir_url(__FILE__) . 'excluded-words-heb.json',
                'hin' => plugin_dir_url(__FILE__) . 'excluded-words-hin.json',
                'ind' => plugin_dir_url(__FILE__) . 'excluded-words-ind.json',
                'ita' => plugin_dir_url(__FILE__) . 'excluded-words-ita.json',
                'nld' => plugin_dir_url(__FILE__) . 'excluded-words-nld.json',
                'pan' => plugin_dir_url(__FILE__) . 'excluded-words-pan.json',
                'pol' => plugin_dir_url(__FILE__) . 'excluded-words-pol.json',
                'por' => plugin_dir_url(__FILE__) . 'excluded-words-por.json',
                'spa' => plugin_dir_url(__FILE__) . 'excluded-words-spa.json',
                'tur' => plugin_dir_url(__FILE__) . 'excluded-words-tur.json',
                'urd' => plugin_dir_url(__FILE__) . 'excluded-words-urd.json',
                'yid' => plugin_dir_url(__FILE__) . 'excluded-words-yid.json'
            ]
        ]);
    }

    public function maybe_clear_old_verses() {
        $last_clear = get_option('bible_reader_last_clear', '');
        $today = gmdate('Y-m-d');
        
        if ($last_clear !== $today) {
            // Clear localStorage key (via JS)
            wp_add_inline_script(
                'bible-api-js',
                'localStorage.removeItem("verseOfTheDay");'
            );
            update_option('bible_reader_last_clear', $today);
        }
    }

    public function render_bible_reader() {
        ob_start(); ?>
        <div class="bible-reader-wrapper">
            <div class="bible-reader-container">
                <div class="bible-controls">
                    <div class="control-group">
                        <label for="bible-language">Language</label>
                        <select id="bible-language" class="bible-select">
                            <option value="eng" selected>🇬🇧 English</option>
                            <option value="ara">🇸🇦 Arabic (العربية)</option>
                            <option value="est">🇪🇪 Estonian (Eesti keel)</option>
                            <option value="ell">🇬🇷 Greek (Ελληνικά)</option>
                            <option value="fas">🇮🇷 Persian (فارسی)</option>
                            <option value="fra">🇫🇷 French (Français)</option>
                            <option value="deu">🇩🇪 German (Deutsch)</option>
                            <option value="heb">🇮🇱 Hebrew (עברית)</option>
                            <option value="hin">🇮🇳 Hindi (हिन्दी)</option>
                            <option value="ind">🇮🇩 Indonesian (Bahasa Indonesia)</option>
                            <option value="ita">🇮🇹 Italian (Italiano)</option>
                            <option value="nld">🇳🇱 Dutch (Nederlands)</option>
                            <option value="pan">🇮🇳 Punjabi (ਪੰਜਾਬੀ)</option>
                            <option value="pol">🇵🇱 Polish (Polski)</option>
                            <option value="por">🇵🇹 Portuguese (Português)</option>
                            <option value="spa">🇪🇸 Spanish (Español)</option>
                            <option value="tur">🇹🇷 Turkish (Türkçe)</option>
                            <option value="urd">🇵🇰 Urdu (اُردُو)</option>
                            <option value="yid">🇮🇱 Yiddish (ייִדיש)</option>
                        </select>
                    </div>
                    
                    <div class="control-group">
                        <label for="bible-version">Version</label>
                        <select id="bible-version" class="bible-select" disabled>
                            <option value="">Select language first</option>
                        </select>
                    </div>
                    
                    <div class="control-group">
                        <label for="bible-book">Book</label>
                        <select id="bible-book" class="bible-select" disabled>
                            <option value="">Select version first</option>
                        </select>
                    </div>
                    
                    <div class="control-group">
                        <label for="bible-chapter">Chapter</label>
                        <select id="bible-chapter" class="bible-select" disabled>
                            <option value="">Select book first</option>
                        </select>
                    </div>
                    
                    <button id="bible-load" class="bible-button" disabled>
                        <span class="button-text"><?php echo esc_html__('Load Passage', 'bible-reader'); ?></span>
                        <span class="loading-spinner" style="display:none;"></span>
                    </button>

                    <div class="control-group search-group">                        
                        <div class="search-container">
                            <input type="text" id="bible-search" placeholder="Search across all Bibles...">
                            <button id="search-button" class="search-button">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="bible-content-container">
                    <div id="chapter-navigation" style="display: none;">
                        <div class="chapter-navigation">
                            <a href="#" id="prev-chapter" class="nav-button">&larr; <span id="prev-chapter-num"></span></a>
                            <h1 id="chapter-title"></h1>
                            <a href="#" id="next-chapter" class="nav-button"><span id="next-chapter-num"></span> &rarr;</a>
                        </div>
                    </div>
                    <div id="word-cloud-container" style="display:none;">
                        <h4>Key Terms in this chapter</h4>
                        <div id="word-cloud-display"></div>
                    </div>
                    <div class="bible-content"></div>
                    <div class="verse-toolbar" style="display: none;">
                        <div class="selected-reference"></div>
                        <button class="copy-btn">
                            <i class="fas fa-copy"></i> Copy
                        </button>
                        <button class="share-btn">
                            <i class="fas fa-share"></i> Share
                        </button>
                        <div class="share-dropdown">
                            <a href="#" class="share-option" data-service="x">
                                <i class="fab fa-x-twitter"></i> X
                            </a>
                            <a href="#" class="share-option" data-service="truthsocial">
                                <img src="<?php echo esc_url(plugin_dir_url(__FILE__)) . 'assets/icons/truth-social.svg'; ?>" class="truthsocial-icon" alt="Truth Social">
                                <span>Truth Social</span>
                            </a>
                            <a href="#" class="share-option" data-service="gab">
                                <img src="<?php echo esc_url(plugin_dir_url(__FILE__)) . 'assets/icons/gab.svg'; ?>" class="gab-icon" alt="Gab">
                                <span>Gab</span>
                            </a>
                            <a href="#" class="share-option" data-service="whatsapp">
                                <i class="fab fa-whatsapp"></i> WhatsApp
                            </a>
                            <a href="#" class="share-option" data-service="telegram">
                                <i class="fab fa-telegram"></i> Telegram
                            </a>
                            <a href="#" class="share-option" data-service="email">
                                <i class="fas fa-envelope"></i> Email
                            </a>
                        </div>
                        <span class="copied-notice">Copied!</span>
                    </div>
                    <div class="bible-error" style="display:none;"></div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function handle_search_request() {
        check_ajax_referer('bible_reader_nonce', 'security');
        
        $request_type = isset($_POST['request_type']) ? sanitize_text_field(wp_unslash($_POST['request_type'])) : '';
        $page = isset($_POST['page']) ? max(1, absint($_POST['page'])) : 1;
        $per_page = 10;
        
        try {
            if ($request_type === 'api') {
                $bible_id = isset($_POST['bible_id']) ? sanitize_text_field(wp_unslash($_POST['bible_id'])) : '';
                $search_term = isset($_POST['search_term']) ? sanitize_text_field(wp_unslash($_POST['search_term'])) : '';
    
                if (empty($bible_id) || empty($search_term)) {
                    throw new Exception('Missing parameters');
                }
    
                // First get the book names for this Bible
                $books_url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/books?include-chapters=false';
                $books_response = wp_remote_get($books_url, [
                    'headers' => ['api-key' => $this->api_key],
                    'timeout' => 15
                ]);
    
                if (is_wp_error($books_response)) {
                    throw new Exception('Failed to get book list: ' . $books_response->get_error_message());
                }
    
                $books_data = json_decode(wp_remote_retrieve_body($books_response), true);
                $book_order = [];
                if (!empty($books_data['data'])) {
                    foreach ($books_data['data'] as $book) {
                        $book_order[$book['id']] = [
                            'name' => $book['name'],
                            'order' => count($book_order) + 1
                        ];
                    }
                }
    
                // Perform search with exact phrase matching
                $search_url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/search?query=' . urlencode('"' . $search_term . '"') . 
                  '&offset=' . (($page - 1) * $per_page) . 
                  '&limit=' . $per_page;
                $search_response = wp_remote_get($search_url, [
                    'headers' => ['api-key' => $this->api_key],
                    'timeout' => 30
                ]);
    
                if (is_wp_error($search_response)) {
                    throw new Exception('Search failed: ' . $search_response->get_error_message());
                }
    
                $search_data = json_decode(wp_remote_retrieve_body($search_response), true);
                $results = [];
                $total_results = 0;
    
                if (!empty($search_data['data'])) {
                    $total_results = $search_data['data']['total'] ?? 0;
                    
                    if (!empty($search_data['data']['verses'])) {
                        foreach ($search_data['data']['verses'] as $verse) {
                            if (preg_match('/(.+?)\s(\d+):(\d+)/', $verse['reference'], $matches)) {
                                $book_name = $matches[1];
                                $chapter = (int)$matches[2];
                                $verse_num = (int)$matches[3];
                                
                                // Find book order
                                $book_id = array_search($book_name, array_column($books_data['data'], 'name'));
                                $book_order_num = $book_id !== false ? $book_order[$books_data['data'][$book_id]['id']]['order'] : 999;
                                
                                $results[] = [
                                    'reference' => $verse['reference'],
                                    'text' => wp_strip_all_tags($verse['text']),
                                    'book_order' => $book_order_num,
                                    'chapter' => $chapter,
                                    'verse' => $verse_num,
                                    'book_id' => $book_id !== false ? $books_data['data'][$book_id]['id'] : ''
                                ];
                            }
                        }
                        
                        // Sort by biblical order (since API might not return in order)
                        usort($results, function($a, $b) {
                            if ($a['book_order'] !== $b['book_order']) {
                                return $a['book_order'] - $b['book_order'];
                            }
                            if ($a['chapter'] !== $b['chapter']) {
                                return $a['chapter'] - $b['chapter'];
                            }
                            return $a['verse'] - $b['verse'];
                        });
                    }
                }
            
                wp_send_json_success([
                    'results' => $results,
                    'total' => $total_results,
                    'page' => $page,
                    'per_page' => $per_page,
                    'total_pages' => ceil($total_results / $per_page)
                ]);
            } else {
                throw new Exception('Invalid search request type');
            }
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    private function find_book_id($books, $book_name) {
        foreach ($books as $id => $name) {
            if (strpos($book_name, $name) !== false) {
                return $id;
            }
        }
        return null;
    }
    
    private function handle_api_search() {
        check_ajax_referer('bible_reader_nonce', 'security');
        
        $bible_id = isset($_POST['bible_id']) ? sanitize_text_field(wp_unslash($_POST['bible_id'])) : '';
        $search_term = isset($_POST['search_term']) ? sanitize_text_field(wp_unslash($_POST['search_term'])) : '';
        
        if (empty($bible_id) || empty($search_term)) {
            wp_send_json_error('Missing parameters');
        }
        
        $url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/search?query=' . urlencode($search_term);
        
        $response = wp_remote_get($url, [
            'headers' => ['api-key' => $this->api_key],
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (empty($body['data']['verses'])) {
            wp_send_json_success(['results' => []]);
        }
        
        $results = [];
        foreach ($body['data']['verses'] as $verse) {
            $reference = $verse['reference'];
            $text = wp_strip_all_tags($verse['text']); // Remove HTML tags
            
            $results[] = [
                'reference' => $reference,
                'text' => $text
            ];
        }
        
        wp_send_json_success(['results' => $results]);
    }

    public function render_verse_of_the_day() {
        ob_start(); ?>
        <div class="verse-of-the-day-wrapper">
            <div class="verse-of-the-day-container">
                <h3>Verse of the Day</h3>
                <br>
                <div class="verse-content"></div>
                <div class="verse-reference"></div>
                <div class="verse-error" style="display:none;"></div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function render_hebrew_date_converter() {
        ob_start(); ?>
        <div class="hebrew-date-converter">
            <h4>Hebrew Date Converter</h4>
            <div class="hebrew-converter">
                <input type="text" id="gregorian-date" placeholder="DD-MM-YYYY" 
                    pattern="\d{2}-\d{2}-\d{4}" 
                    title="Use DD-MM-YYYY format (01-01-0001 to present)">
                <button id="convert-to-hebrew">Convert to Hebrew</button>
                
                <div id="hebrew-date-result" class="hebrew-result">
                    <div class="hebrew-date-display"></div>
                    <div class="hebrew-holiday"></div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function handle_hebrew_date_conversion() {
        check_ajax_referer('bible_reader_nonce', 'security');

        $day = absint(wp_unslash($_POST['day'] ?? 0));
        $month = absint(wp_unslash($_POST['month'] ?? 0));
        $year = absint(wp_unslash($_POST['year'] ?? 0));
        
        // Basic validation
        if ($day < 1 || $day > 31 || $month < 1 || $month > 12 || $year < 1 || $year > 9999) {
            wp_send_json_error('Invalid date (use 01-01-0001 to present)');
            return;
        }
        
        try {
            // Format date as YYYY-MM-DD
            $dateStr = sprintf("%04d-%02d-%02d", $year, $month, $day);
            
            // Call HebCal API
            $apiUrl = "https://www.hebcal.com/converter?cfg=json&gy=$year&gm=$month&gd=$day&g2h=1";
            $response = wp_remote_get($apiUrl, ['timeout' => 15]);
            
            if (is_wp_error($response)) {
                throw new Exception('Failed to connect to date conversion service');
            }
    
            $response_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            
            if ($response_code !== 200) {
                throw new Exception('Date conversion service returned an error');
            }
    
            $data = json_decode($body, true);
        
            if (empty($data) || !isset($data['hebrew'])) {
                throw new Exception('Invalid response from date conversion service');
            }
    
            // Extract Hebrew date information
            $hebrewDate = $data['hebrew'];
            $hebrewYear = $data['hy'] ?? '';
            $hebrewMonth = $data['hm'] ?? '';
            $hebrewDay = $data['hd'] ?? '';

            // Convert month to biblical name and number
            $monthData = $this->hebrewMonthToBiblical($hebrewMonth);
            $monthNumber = $monthData['number'];
            $biblicalMonth = $monthData['biblical'];

            // Format the date with biblical month name
            $formattedDate = "$hebrewDay $biblicalMonth $hebrewYear";

            // Events
            $events = [
                // Passover and Unleavened Bread (Leviticus 23:5-8)
                '1-14' => 'Preparation for Pesach (Passover Eve)',
                '1-15' => 'Pesach (Passover) Day 1',
                '1-16' => 'Pesach (Passover) Day 2',
                '1-17' => 'Pesach (Passover) Day 3 (Chol Hamoed)',
                '1-18' => 'Pesach (Passover) Day 4 (Chol Hamoed)',
                '1-19' => 'Pesach (Passover) Day 5 (Chol Hamoed)',
                '1-20' => 'Pesach (Passover) Day 6 (Chol Hamoed)',
                '1-21' => 'Pesach (Passover) Day 7',
                
                // Shavuot (Leviticus 23:15-21)
                '3-6' => 'Shavuot (Pentecost) (Feast of Weeks)',
                
                // Rosh Hashanah (Leviticus 23:23-25)
                '7-1' => 'Rosh Hashanah (Feast of Trumpets) Day 1',
                '7-2' => 'Rosh Hashanah (Feast of Trumpets) Day 2',
                
                // Yom Kippur (Leviticus 23:26-32)
                '7-10' => 'Yom Kippur (Day of Atonement)',
                
                // Sukkot (Leviticus 23:33-36)
                '7-15' => 'Sukkot (Feast of Tabernacles) Day 1',
                '7-16' => 'Sukkot (Feast of Tabernacles) Day 2',
                '7-17' => 'Sukkot (Feast of Tabernacles) Day 3 (Chol Hamoed)',
                '7-18' => 'Sukkot (Feast of Tabernacles) Day 4 (Chol Hamoed)',
                '7-19' => 'Sukkot (Feast of Tabernacles) Day 5 (Chol Hamoed)',
                '7-20' => 'Sukkot (Feast of Tabernacles) Day 6 (Chol Hamoed)',
                '7-21' => 'Sukkot (Feast of Tabernacles) Day 7',
                '7-22' => 'Shemini Atzeret',
                
                // Hanukkah (John 10:22 - Feast of Dedication)
                '9-25' => 'Hanukkah (Feast of Dedication) Day 1',
                '9-26' => 'Hanukkah (Feast of Dedication) Day 2',
                '9-27' => 'Hanukkah (Feast of Dedication) Day 3',
                '9-28' => 'Hanukkah (Feast of Dedication) Day 4',
                '9-29' => 'Hanukkah (Feast of Dedication) Day 5',
                '9-30' => 'Hanukkah (Feast of Dedication) Day 6',
                '10-1' => 'Hanukkah (Feast of Dedication) Day 7',
                '10-2' => 'Hanukkah (Feast of Dedication) Day 8',
                
                // Purim (Esther 9:26-28) - Not strictly biblical but in the Bible
                '12-14' => 'Purim',
                '12-15' => 'Shushan Purim'
            ];
            
            // Check for your custom holidays
            $eventKey = "$monthNumber-$hebrewDay";
            $event = $events[$eventKey] ?? 'Regular day';

            wp_send_json_success([
                'hebrewDate' => $formattedDate,
                'hebrewYear' => $hebrewYear,
                'holiday' => $event
            ]);
        } catch (Exception $e) {
            wp_send_json_error('Unable to convert date. Please try again later.');
        }
    }

    private function hebrewMonthToBiblical($monthName) {
        // Normalize the month name (trim and capitalize first letter)
        $normalizedMonth = ucfirst(strtolower(trim($monthName)));
        
        $monthMap = [
            'Nisan' => ['number' => 1, 'biblical' => 'Abib'],
            'Iyar' => ['number' => 2, 'biblical' => 'Ziv'],
            'Sivan' => ['number' => 3, 'biblical' => 'Sivan'],
            'Tammuz' => ['number' => 4, 'biblical' => 'Tammuz'],
            'Av' => ['number' => 5, 'biblical' => 'Av'],
            'Elul' => ['number' => 6, 'biblical' => 'Elul'],
            'Tishrei' => ['number' => 7, 'biblical' => 'Eitanim'],
            'Cheshvan' => ['number' => 8, 'biblical' => 'Bul'],
            'Kislev' => ['number' => 9, 'biblical' => 'Kislev'],
            'Tevet' => ['number' => 10, 'biblical' => 'Tevet'],
            'Shevat' => ['number' => 11, 'biblical' => 'Shevat'],
            'Adar' => ['number' => 12, 'biblical' => 'Adar'],
            'Adar I' => ['number' => 12, 'biblical' => 'Adar'],
            'Adar II' => ['number' => 13, 'biblical' => 'Adar II'],
            // Add variations if needed
            'Iyvar' => ['number' => 2, 'biblical' => 'Ziv'],
            'Iyyar' => ['number' => 2, 'biblical' => 'Ziv']
        ];
        
        return $monthMap[$normalizedMonth] ?? ['number' => 0, 'biblical' => $monthName];
    }

    public function handle_api_request() {
        // 1. Verify nonce first
        if (!isset($_POST['security']) || false === wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['security'])), 'bible_reader_nonce')) {
            wp_send_json_error('Security verification failed', 403);
            return;
        }
    
        // 2. Verify user capability (if needed)
        if (!current_user_can('read')) {
            wp_send_json_error('Unauthorized', 401);
            return;
        }
    
        // 3. Then process the request
        if (!isset($_POST['request_type'])) {
            wp_send_json_error('Missing request type', 400);
            return;
        }
    
        $request_type = sanitize_text_field(wp_unslash($_POST['request_type']));
        $bible_id = isset($_POST['bible_id']) ? sanitize_text_field(wp_unslash($_POST['bible_id'])) : '';
        
        // Handle special versions
        if ($request_type === 'versions' && isset($_POST['language'])) {
            $language = sanitize_text_field(wp_unslash($_POST['language']));
            if (isset($this->special_versions[$language])) {
                $versions = [];
                foreach ($this->special_versions[$language] as $name => $id) {
                    $versions[] = ['id' => $id, 'name' => $name];
                }
                wp_send_json_success(['data' => $versions]);
                return;
            }
        }
    
        $url = 'https://api.scripture.api.bible/v1/bibles';
        $params = [];
    
        switch ($request_type) {
            case 'versions':
                if (!isset($_POST['language'])) {
                    wp_send_json_error('Missing language parameter', 400);
                    return;
                }
                $params = ['language' => sanitize_text_field(wp_unslash($_POST['language']))];
                break;
                
            case 'books':
                if (empty($bible_id)) {
                    wp_send_json_error('Missing bible_id parameter', 400);
                    return;
                }
                $url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/books';
                break;
                
            case 'chapters':
                if (empty($bible_id) || !isset($_POST['book_id'])) {
                    wp_send_json_error('Missing parameters', 400);
                    return;
                }
                $book_id = sanitize_text_field(wp_unslash($_POST['book_id']));
                $url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/books/' . $book_id . '/chapters';
                break;
                
            case 'passage':
                if (empty($bible_id) || !isset($_POST['book_id']) || !isset($_POST['chapter'])) {
                    wp_send_json_error('Missing parameters', 400);
                    return;
                }
                $book_id = sanitize_text_field(wp_unslash($_POST['book_id']));
                $chapter = sanitize_text_field(wp_unslash($_POST['chapter']));
                $url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/passages/' . $book_id . '.' . $chapter . '?content-type=html&include-notes=false&include-titles=false';
                break;
                
            default:
                wp_send_json_error('Invalid request');
                return;
        }
        
        $response = wp_remote_get(add_query_arg($params, $url), [
            'headers' => ['api-key' => $this->api_key],
            'timeout' => 15
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error($response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        wp_send_json_success($body['data'] ?? $body);
    }

    public function handle_random_verse_request() {
        // Verify API key is configured
        if (empty($this->api_key)) {
            wp_send_json_error('API configuration error');
        }
    
        // Check if we have a cached verse for today
        $today = gmdate('Y-m-d');
        $transient_key = 'bible_verse_' . $today;
        
        // Try to get cached verse first
        $cached_verse = get_transient($transient_key);
        if ($cached_verse !== false) {
            wp_send_json_success($cached_verse);
            return;
        }

        $bible_id = 'de4e12af7f28f599-02'; // KJV
        
        try {
            // Skip introductory content
            if (strpos($verse_id, 'intro') !== false) {
                // Try again with a different verse
                return $this->handle_random_verse_request();
            }

            // 1. Get list of all books first (more reliable than /random endpoint)
            $books_url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/books';
            $books_response = wp_remote_get($books_url, [
                'headers' => ['api-key' => $this->api_key],
                'timeout' => 10
            ]);
    
            if (is_wp_error($books_response)) {
                throw new Exception('Failed to fetch books: ' . $books_response->get_error_message());
            }
    
            $books_data = json_decode(wp_remote_retrieve_body($books_response), true);
            if (empty($books_data['data'])) {
                throw new Exception('No books data received');
            }
    
            // 2. Select a random book
            $random_book = $books_data['data'][array_rand($books_data['data'])];
            $book_id = $random_book['id'];
    
            // 3. Get chapters for this book
            $chapters_url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/books/' . $book_id . '/chapters';
            $chapters_response = wp_remote_get($chapters_url, [
                'headers' => ['api-key' => $this->api_key],
                'timeout' => 10
            ]);
    
            if (is_wp_error($chapters_response)) {
                throw new Exception('Failed to fetch chapters: ' . $chapters_response->get_error_message());
            }
    
            $chapters_data = json_decode(wp_remote_retrieve_body($chapters_response), true);
            if (empty($chapters_data['data'])) {
                throw new Exception('No chapters data received');
            }
    
            // 4. Select a random chapter
            $random_chapter = $chapters_data['data'][array_rand($chapters_data['data'])];
            $chapter_id = $random_chapter['id'];
    
            // 5. Get verses for this chapter
            $verses_url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/chapters/' . $chapter_id . '/verses';
            $verses_response = wp_remote_get($verses_url, [
                'headers' => ['api-key' => $this->api_key],
                'timeout' => 10
            ]);
    
            if (is_wp_error($verses_response)) {
                throw new Exception('Failed to fetch verses: ' . $verses_response->get_error_message());
            }
    
            $verses_data = json_decode(wp_remote_retrieve_body($verses_response), true);
            if (empty($verses_data['data'])) {
                throw new Exception('No verses data received');
            }
    
            // 6. Select a random verse
            $random_verse = $verses_data['data'][array_rand($verses_data['data'])];
            $verse_id = $random_verse['id'];
    
            // 7. Get the verse content
            $content_url = 'https://api.scripture.api.bible/v1/bibles/' . $bible_id . '/verses/' . $verse_id . '?content-type=text';
            $content_response = wp_remote_get($content_url, [
                'headers' => ['api-key' => $this->api_key],
                'timeout' => 10
            ]);
    
            if (is_wp_error($content_response)) {
                throw new Exception('Failed to fetch verse content: ' . $content_response->get_error_message());
            }
    
            $content_data = json_decode(wp_remote_retrieve_body($content_response), true);
            if (empty($content_data['data'])) {
                throw new Exception('No verse content received');
            }
    
            // 8. Clean the verse text
            $verse_text = trim($content_data['data']['content']);
            $verse_text = preg_replace('/^[\d\s\[\]\.:-]+/', '', $verse_text); // Remove verse numbers
            $verse_text = ucfirst($verse_text); // Capitalize first letter
    
            // 9. Parse the reference
            $verse_parts = explode('.', $verse_id);
            $book_name = $random_book['name'];

            // Clean the book name
            $book_name = rtrim($book_name, "'");
            $chapter_num = $verse_parts[1] ?? '1'; // Fallback to chapter 1
            $verse_num = $verse_parts[2] ?? '1';  // Fallback to verse 1

            // Format the reference properly (e.g., "John 3:16" instead of "John3:16")
            $reference = "$book_name $chapter_num:$verse_num";
    
            // When you successfully get a verse:
            $verse_data = [
                'content' => $verse_text,
                'reference' => $reference,
                'source' => 'api',
                'date' => $today
            ];

            // Cache for 24 hours (86400 seconds)
            set_transient($transient_key, $verse_data, 86400);

            // 10. Return the verse
            wp_send_json_success([
                'content' => $verse_text,
                'reference' => $reference,
                'source' => 'api'
            ]);
    
        } catch (Exception $e) {
            // Fallback to John 3:16 only if API is completely down
            wp_send_json_success([
                'content' => 'For God so loved the world, that he gave his only Son, that whoever believes in him should not perish but have eternal life.',
                'reference' => 'John 3:16',
                'source' => 'fallback'
            ]);
        }
    }
}

new BibleReader();
