jQuery(document).ready(function($) {
    const { ajaxurl, specialVersions } = bibleReader;
    let excludedWords = [];

    // Verse Selection Variables
    let selectedVerses = [];
    let lastSelectedVerse = null;

    function initHebrewDateConverter() {
        // Set today's date in DD-MM-YYYY format
        const today = new Date();
        $('#gregorian-date').val(
            `${String(today.getDate()).padStart(2, '0')}-` +
            `${String(today.getMonth() + 1).padStart(2, '0')}-` +
            today.getFullYear()
        );
    
        // Convert Gregorian to Hebrew
        $('#convert-to-hebrew').click(function(e) {
            e.preventDefault();
            
            const gregDate = $('#gregorian-date').val();
            const [day, month, year] = gregDate.split('-').map(Number);
            
            // Validate date (0001-01-01 onward)
            if (isNaN(day) || isNaN(month) || isNaN(year) || 
                day < 1 || day > 31 || 
                month < 1 || month > 12 || 
                year < 1 || year > 9999) {
                $('#hebrew-date-result').html(
                    '<div class="hebrew-error">Please use DD-MM-YYYY (01-01-0001 to present)</div>'
                );
                return;
            }

            // Show loading
            $('#hebrew-date-result').html('<div class="hebrew-loading">Converting...</div>');
            
            // Convert via AJAX to your server
            $.ajax({
                url: ajaxurl,
                method: 'POST',
                dataType: 'json',
                data: {
                    action: 'convert_to_hebrew_date',
                    security: bibleReader.nonce,
                    day: day,
                    month: month,
                    year: year
                },
                success: function(response) {
                    if (response.success) {
                        displayHebrewDate(response.data);
                    } else {
                        $('#hebrew-date-result').html(
                            `<div class="hebrew-error">Error: ${response.data || 'Conversion failed'}</div>`
                        );
                    }
                },
                error: function(xhr) {
                    $('#hebrew-date-result').html(
                        `<div class="hebrew-error">Error: ${xhr.responseText || 'Request failed'}</div>`
                    );
                }
            });
        });
    
        // Initialize on page load
        $('#convert-to-hebrew').trigger('click');

        // Helper functions
        function displayHebrewDate(data) {
            const $result = $('#hebrew-date-result');
            $result.empty(); // Clear previous content
            
            // Always show date and year
            $result.append(`
                <div class="hebrew-date-display">
                    <strong>${data.hebrewDate}</strong>
                    <div>${data.hebrewYear}</div>
                </div>
            `);
            
            // Show holiday if available
            if (data.holiday) {
                $result.append(`
                    <div class="hebrew-holiday">Holiday: ${data.holiday}</div>
                `);
            } else {
                $result.append(`
                    <div class="hebrew-holiday">Regular day</div>
                `);
            }
            
            $result.removeClass('hebrew-error').addClass('hebrew-info');
        }
    
        function showHebrewDateError(message) {
            const $result = $('#hebrew-date-result');
            $result.html(`<div class="hebrew-error">${message}</div>`);
            $result.removeClass('hebrew-info').addClass('hebrew-error');
        }
    }

    // First load excluded words, then initialize everything
    loadExcludedWords().then(() => {
        initBibleReader();
        initVerseSelection(); // Initialize verse selection system
        initHebrewDateConverter();
        
        $('#convert-to-hebrew').trigger('click');

        if ($('.verse-of-the-day-container').length) {
            fetchRandomVerse();
        }
    }).catch(() => {
        // Fallback if JSON fails to load
        excludedWords = [
            'that', 'this', 'with', 'have', 'were', 
            'said', 'unto', 'them', 'they', 'shall'
        ];
        initBibleReader();
        initVerseSelection(); // Initialize verse selection system
        initHebrewDateConverter();
    });

    // ======================
    // VERSE SELECTION SYSTEM
    // ======================
    function initVerseSelection() {
        // Remove any existing event listeners to avoid duplicates
        $(document).off('click', '.bible-content p', handleVerseClick);
        $(document).off('click', handleDocumentClick);
        $(document).off('click', '.copy-btn');
        
        // Add the toolbar if it doesn't exist
        if ($('.verse-toolbar').length === 0) {
            $('.bible-content-container').append(`
                <div class="verse-toolbar" style="display: none;">
                    <div class="selected-reference"></div>
                    <button class="copy-btn">
                        <i class="fas fa-copy"></i> Copy
                    </button>
                    <button class="share-btn">
                        <i class="fas fa-share-alt"></i> Share
                    </button>
                    <div class="share-dropdown">
                        ${Object.entries(bibleReader.shareOptions).map(([key, option]) => `
                            <a href="#" class="share-option" data-service="${key}">
                                <i class="${option.icon}"></i> ${option.name}
                            </a>
                        `).join('')}
                    </div>
                    <span class="copied-notice">Copied!</span>
                </div>
            `);
        }
    
        // Set up event listeners with proper order
        $(document).on('click', '.bible-content p', handleVerseClick);
        $(document).on('click', handleDocumentClick);
        
        // Update the copy button click handler to stop propagation
        $(document).on('click', '.copy-btn', function(e) {
            e.preventDefault();
            e.stopPropagation();
            copySelectedVerses();
        });
        
        // Share button handlers
        $(document).on('click', '.share-btn', function(e) {
            e.stopPropagation();
            $('.share-dropdown').toggleClass('show');
        });
        
        $(document).on('click', '.share-option', function(e) {
            e.preventDefault();
            const service = $(this).data('service');
            shareVerse(service);
            $('.share-dropdown').removeClass('show');
        });
    }

    function handleVerseClick(e) {
        // Stop event propagation immediately
        e.stopPropagation();
        
        // If user is selecting text, don't interfere
        if (window.getSelection().toString().length > 0) return;
        
        const $verse = $(this);
        let verseNum = $verse.data('verse') || $verse.find('.verse-number').text() || '';
        
        // For API Bibles where verse number might be in the text
        if (!verseNum && $verse.text().match(/^\d+/)) {
            verseNum = $verse.text().match(/^\d+/)[0];
        }
        
        // Generate a unique ID if needed
        if (!$verse.data('verse-id')) {
            $verse.attr('data-verse-id', 'verse-' + Date.now() + '-' + verseNum);
        }
        
        // Handle shift+click for range selection
        if (e.shiftKey && lastSelectedVerse) {
            selectVerseRange($verse);
            updateVerseToolbar();
            return;
        }
        
        toggleVerseSelection($verse);
        updateVerseToolbar();
    }

    function toggleVerseSelection($verse) {
        const verseId = $verse.data('verse-id');
        let verseNum = $verse.find('.verse-number').text() || $verse.data('verse') || '';
        let verseText = $verse.text();
        
        // For API Bibles where verse number is at start of text
        if (!verseNum && verseText.match(/^\d+/)) {
            verseNum = verseText.match(/^\d+/)[0];
            verseText = verseText.replace(/^\d+\s*/, '').trim();
        } else if (verseNum) {
            // For local Bibles where verse number is in a span
            verseText = verseText.replace(verseNum, '').trim();
        }
        
        if ($verse.hasClass('selected')) {
            // Deselect
            $verse.removeClass('selected');
            selectedVerses = selectedVerses.filter(v => v.id !== verseId);
        } else {
            // Select
            $verse.addClass('selected');
            selectedVerses.push({
                id: verseId,
                number: verseNum,
                text: verseText
            });
        }
        
        lastSelectedVerse = $verse.hasClass('selected') ? $verse : null;
    }

    function selectVerseRange($endVerse) {
        const $verses = $('.bible-content p');
        const startIdx = $verses.index(lastSelectedVerse);
        const endIdx = $verses.index($endVerse);
        
        if (startIdx === -1 || endIdx === -1) return;
        
        // Clear previous selection
        $verses.removeClass('selected');
        selectedVerses = [];
        
        // Select range
        for (let i = Math.min(startIdx, endIdx); i <= Math.max(startIdx, endIdx); i++) {
            const $verse = $($verses[i]);
            $verse.addClass('selected');
            
            let verseNum = $verse.find('.verse-number').text() || $verse.data('verse') || '';
            let verseText = $verse.text();
            
            // For API Bibles where verse number is at start of text
            if (!verseNum && verseText.match(/^\d+/)) {
                verseNum = verseText.match(/^\d+/)[0];
                verseText = verseText.replace(/^\d+\s*/, '').trim();
            } else if (verseNum) {
                // For local Bibles where verse number is in a span
                verseText = verseText.replace(verseNum, '').trim();
            }
            
            selectedVerses.push({
                id: $verse.data('verse-id'),
                number: verseNum,
                text: verseText
            });
        }
        
        lastSelectedVerse = $endVerse;
    }

    function updateVerseToolbar() {
        const $toolbar = $('.verse-toolbar');
        
        if (selectedVerses.length > 0) {
            // Update reference
            const book = $('#bible-book option:selected').text();
            const chapter = $('#bible-chapter').val();
            let reference = `${book} ${chapter}:`;
            
            if (selectedVerses.length === 1) {
                reference += selectedVerses[0].number;
            } else {
                const firstNum = selectedVerses[0].number;
                const lastNum = selectedVerses[selectedVerses.length-1].number;
                reference += `${firstNum}-${lastNum}`;
            }
            
            $('.selected-reference').text(reference);
            $toolbar.fadeIn();
        } else {
            $toolbar.fadeOut();
        }
    }

    function copySelectedVerses() {
        if (selectedVerses.length === 0) return;
        
        try {
            // Sort verses by their position in DOM
            selectedVerses.sort((a, b) => {
                return $(`[data-verse-id="${a.id}"]`).index() - $(`[data-verse-id="${b.id}"]`).index();
            });
            
            // Format text
            let textToCopy = selectedVerses.map(v => {
                const num = v.number ? `${v.number} ` : '';
                return `${num}${v.text}`;
            }).join('\n');
            
            // Add reference
            const book = $('#bible-book option:selected').text();
            const chapter = $('#bible-chapter').val();
            textToCopy += `\n\n${book} ${chapter}:${selectedVerses[0].number}`;
            if (selectedVerses.length > 1) {
                textToCopy += `-${selectedVerses[selectedVerses.length-1].number}`;
            }
            
            // Create hidden textarea for copying
            const textarea = document.createElement('textarea');
            textarea.value = textToCopy;
            textarea.style.position = 'fixed';  // Prevent scrolling to bottom
            document.body.appendChild(textarea);
            textarea.select();
            
            try {
                // Try modern clipboard API first
                if (navigator.clipboard) {
                    navigator.clipboard.writeText(textToCopy).then(() => {
                        showCopySuccess();
                    }).catch(() => {
                        fallbackCopy(textarea);
                    });
                } else {
                    // Fallback for older browsers
                    fallbackCopy(textarea);
                }
            } catch (err) {
                fallbackCopy(textarea);
            } finally {
                document.body.removeChild(textarea);
            }
        } catch (error) {
            console.error('Copy failed:', error);
            alert('An error occurred while copying. Please try again.');
        }
    }

    function fallbackCopy(textarea) {
        try {
            const successful = document.execCommand('copy');
            if (successful) {
                showCopySuccess();
            } else {
                throw new Error('Copy command failed');
            }
        } catch (err) {
            throw err;
        }
    }
    
    function showCopySuccess() {
        $('.copied-notice').fadeIn().delay(1500).fadeOut();
    }

    function shareVerse(service) {
        if (selectedVerses.length === 0) return;
        
        // Get the verse text
        const versesText = selectedVerses.map(v => {
            const num = v.number ? `${v.number} ` : '';
            return `${num}${v.text}`;
        }).join('\n');
        
        // Get the reference
        const book = $('#bible-book option:selected').text();
        const chapter = $('#bible-chapter').val();
        let reference = `${book} ${chapter}:${selectedVerses[0].number}`;
        if (selectedVerses.length > 1) {
            reference += `-${selectedVerses[selectedVerses.length-1].number}`;
        }
        
        // Create the full text (max 280 chars for Twitter)
        let fullText = `${versesText}\n\n— ${reference}`;
        if (service === 'x' && fullText.length > 280) {
            fullText = `${reference}\n\n${versesText.substring(0, 280 - reference.length - 5)}...`;
        }
        
        // Get the current page URL
        const currentUrl = encodeURIComponent(window.location.href.split('#')[0]);
        const text = encodeURIComponent(fullText);
        
        // Define share services with updated URLs
        const shareServices = {
            'x': {
                url: `https://x.com/intent/tweet?text=${text}&url=${currentUrl}`,
                width: 550,
                height: 420
            },
            'truthsocial': {
                url: `https://truthsocial.com/share?text=${text}%0A%0A${currentUrl}`,
                width: 600,
                height: 600
            },
            'gab': {
                url: `https://gab.com/compose?text=${text}%0A%0A${currentUrl}`,
                width: 600,
                height: 600
            },
            'whatsapp': {
                url: `https://wa.me/?text=${text}%20${currentUrl}`,
                width: 800,
                height: 600
            },
            'telegram': {
                url: `https://t.me/share/url?url=${currentUrl}&text=${text}`,
                width: 600,
                height: 500
            },
            'email': {
                url: `mailto:?subject=Bible Verse (${reference})&body=${text}%0A%0A${currentUrl}`
            }
        };
        
        // Get the share URL
        const shareConfig = shareServices[service];
        if (!shareConfig) return;
        
        // Open the share window
        if (service === 'email') {
            window.location.href = shareConfig.url;
        } else {
            const width = shareConfig.width || 600;
            const height = shareConfig.height || 400;
            const left = (screen.width - width) / 2;
            const top = (screen.height - height) / 2;
            
            window.open(
                shareConfig.url,
                '_blank',
                `width=${width},height=${height},left=${left},top=${top}`
            );
        }
    }

    function handleDocumentClick(e) {
        const $target = $(e.target);
        
        // Don't hide if clicking on toolbar or its children
        if ($target.closest('.verse-toolbar').length > 0) return;
        
        // Don't hide if clicking on a verse
        if ($target.closest('.bible-content p').length > 0) return;
        
        // Hide toolbar and clear selection
        $('.bible-content p').removeClass('selected');
        selectedVerses = [];
        $('.verse-toolbar').fadeOut();
        
        // Hide dropdown if visible
        $('.share-dropdown').removeClass('show');
    }

    function loadExcludedWords(language = 'eng') {
        const url = bibleReader.excludedWordsUrls[language] || bibleReader.excludedWordsUrls['eng'];
        
        return $.ajax({
            url: url,
            dataType: 'json',
            cache: false
        }).then(function(data) {
            if (data && Array.isArray(data.excludedWords)) {
                excludedWords = data.excludedWords.map(word => word.toLowerCase());
                return excludedWords;
            }
            throw new Error('Invalid JSON format');
        }).catch(function(error) {
            console.error('Error loading excluded words:', error);
            // Fallback words
            excludedWords = getFallbackExcludedWords(language);
            return excludedWords;
        });
    }
    
    function getFallbackExcludedWords(language) {
        if (language === 'nld') {
            return [
                'de', 'het', 'een', 'en', 'van',
                'dat', 'die', 'in', 'te', 'met',
                'zijn', 'op', 'aan', 'voor', 'niet',
                'om', 'als', 'er', 'maar', 'dan'
            ].map(w => w.toLowerCase());
        }
        return [
            'that', 'this', 'with', 'have', 'were', 
            'said', 'unto', 'them', 'they', 'shall'
        ].map(w => w.toLowerCase());
    }

    // Add this function to handle word highlighting
    function highlightWord(wordToHighlight) {
        // Remove all existing highlights first
        $('.highlighted-word').each(function() {
            const word = $(this).data('word');
            const originalText = $(this).text();
            $(this).replaceWith(originalText);
        });
        
        // Remove active class from all word tags
        $('.word-tag').removeClass('active');
        
        if (!wordToHighlight) return;
        
        // Escape special regex characters (works for non-Latin too)
        const escapedWord = wordToHighlight.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const regex = new RegExp(`(^|\\s|>|\\(|")(${escapedWord})($|\\s|<|\\)|"|;|,|\\.|!)`, 'gi');
        
        const content = $('.bible-content').html();
        const newContent = content.replace(regex, (match, p1, p2, p3) => {
            return `${p1}<span class="highlighted-word" data-word="${escapeHTML(wordToHighlight)}">${p2}</span>${p3}`;
        });
        
        $('.bible-content').html(newContent);
        $(`.word-tag[data-word="${escapeHTML(wordToHighlight)}"]`).addClass('active');
    }

    function generateWordCloud(text) {
        // Get current language from dropdown
        const language = $('#bible-language').val();
        
        // Clean the text while preserving all scripts
        const cleanText = text
            .replace(/<[^>]*>/g, ' ')       // Remove HTML tags
            .replace(/['"‘’“”]/g, '')       // Remove quotes
            .replace(/[0-9#¶.,;:!?]/g, '')  // Remove numbers and punctuation
            .replace(/\s{2,}/g, ' ')        // Fix multiple spaces
            .trim();
        
        // Get all words (minimum 2 characters) in any script
        const words = cleanText.match(/\p{L}[\p{L}\p{M}']{1,}/gu) || [];
        
        const freq = {};
        
        words.forEach(word => {
            // Skip short words (1-2 characters)
            if (word.length <= 2) return;
            
            // For Dutch, normalize special characters before comparison
            let compareWord = word.toLowerCase();
            if (language === 'nld') {
                compareWord = compareWord
                    .replace(/[éèêë]/g, 'e')
                    .replace(/[íìîï]/g, 'i')
                    .replace(/[óòôö]/g, 'o')
                    .replace(/[úùûü]/g, 'u')
                    .replace(/[áàâä]/g, 'a');
            }
            
            if (excludedWords.includes(compareWord)) {
                return;
            }
            
            // Count the word (preserve original case in display)
            freq[word] = (freq[word] || 0) + 1;
        });
    
        // Get top words, sorted by frequency
        const topWords = Object.entries(freq)
            .sort((a, b) => b[1] - a[1])
            .slice(0, 10) // Show top 10 words
            .map(([word]) => word);
    
        // Display results
        const cloudContainer = $('#word-cloud-display');
        cloudContainer.empty();
        
        if (topWords.length > 0) {
            topWords.forEach(word => {
                cloudContainer.append(
                    `<span class="word-tag" data-word="${escapeHTML(word)}">${word}</span>`
                );
            });
            $('#word-cloud-container').show();
        } else {
            $('#word-cloud-container').hide();
        }
    }

    // Helper to safely escape HTML in words
    function escapeHTML(str) {
        return str.replace(/[&<>'"]/g, 
            tag => ({
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                "'": '&#39;',
                '"': '&quot;'
            }[tag]));
    }

    // Add click handler for word tags
    $(document).on('click', '.word-tag', function() {
        const word = $(this).data('word');
        const isActive = $(this).hasClass('active');
        
        if (isActive) {
            highlightWord(null);
        } else {
            highlightWord(word);
        }
    });

    function initBibleReader() {
        // Set default language to English
        const defaultLanguage = 'eng';
        $('#bible-language').val(defaultLanguage);
        
        // Immediately load versions for default language
        loadVersions(defaultLanguage);
        
        // Set up event handlers
        $('#bible-language').change(function() {
            const language = $(this).val();
            // Load excluded words for this language first
            loadExcludedWords(language).then(() => {
                loadVersions(language);
            }).catch(() => {
                loadVersions(language);
            });
        });
        
        $('#bible-version').change(function() {
            const bibleId = $(this).val();
            resetDependentFields(true);
            if (bibleId) loadBooks(bibleId);
        });
        
        $('#bible-book').change(function() {
            const bibleId = $('#bible-version').val();
            const bookId = $(this).val();
            if (bibleId && bookId) loadChapters(bibleId, bookId);
        });
        
        $('#bible-chapter').change(function() {
            const bibleId = $('#bible-version').val();
            const bookId = $('#bible-book').val();
            const chapter = $(this).val();
            $('#bible-load').prop('disabled', !(bibleId && bookId && chapter));
        });
        
        $('#bible-load').click(loadPassage);
    }
    
    function fetchRandomVerse() {
        const today = new Date().toISOString().split('T')[0];
    
        // Check localStorage first
        const cachedVerse = localStorage.getItem('verseOfTheDay');
        if (cachedVerse) {
            try {
                const parsed = JSON.parse(cachedVerse);
                if (parsed.date === today && parsed.content) {
                    displayVerse(parsed);
                    return; // Don't make API call if we have today's verse
                }
            } catch(e) { /* Empty */ }
        }

        // Show loading state
        $('.verse-content').html('<div class="loading-verse">Loading today\'s verse...</div>');
        $('.verse-error').hide();
    
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            dataType: 'json',
            data: {
                action: 'get_random_verse',
                security: bibleReader.nonce
            },
            success: function(response) {
                if (response && response.success && response.data) {
                    // Check if this is a fallback
                    if (response.data.source === 'fallback') {
                        $('.verse-error').text('Could not fetch daily verse. Showing John 3:16.').show();
                    }
                    
                    // Store in localStorage
                    const today = new Date().toISOString().split('T')[0];
                    const verseData = {
                        content: response.data.content,
                        reference: response.data.reference,
                        date: today,
                        source: response.data.source
                    };
                    
                    localStorage.setItem('verseOfTheDay', JSON.stringify(verseData));
                    displayVerse(verseData);
                } else {
                    showFallbackVerse();
                }
            },
            error: function() {
                showFallbackVerse();
            }
        });
    }
    
    function displayVerse(verseData) {
        if (!verseData.content) {
            showFallbackVerse();
            return;
        }
        
        // Clean the reference (remove any intro:o text)
        let cleanReference = verseData.reference.replace('intro:o', '').trim();
        
        // Display with better formatting
        $('.verse-content').html(`${verseData.content}`);
        $('.verse-reference').html(`<cite>${cleanReference}</cite>`);
    }
    
    function showFallbackVerse() {
        // Try cached verse first
        const cachedVerse = localStorage.getItem('verseOfTheDay');
        if (cachedVerse) {
            try {
                const parsed = JSON.parse(cachedVerse);
                if (parsed.content) {
                    displayVerse(parsed);
                    $('.verse-error').text('Network issue. Showing cached verse.').show();
                    return;
                }
            } catch(e) {
                /* */
            }
        }
        
        // Ultimate fallback
        displayVerse({
            content: 'For God so loved the world, that he gave his only Son, that whoever believes in him should not perish but have eternal life.',
            reference: 'John 3:16',
            source: 'fallback'
        });
        $('.verse-error').text('Unable to load verse. Showing John 3:16.').show();
    }
    
    function resetDependentFields(resetBooks = false) {
        if (resetBooks) {
            $('#bible-book').prop('disabled', true).html('<option value="">Select version first</option>');
        }
        $('#bible-chapter').prop('disabled', true).html('<option value="">Select book first</option>');
        $('#bible-load').prop('disabled', true);
        $('.bible-content').empty();
        $('.bible-error').hide();
    }
    
    function loadVersions(language) {      
        const $select = $('#bible-version');
        $select.empty().append('<option value="">Loading versions...</option>');

        // Check for special languages
        if (specialVersions[language]) {
            const $select = $('#bible-version');
            $select.empty().append('<option value="">Select Version</option>');
            
            for (const [name, id] of Object.entries(specialVersions[language])) {
                $select.append(`<option value="${id}">${name}</option>`);
            }
            
            $select.prop('disabled', false);
            return;
        }

        // Load Bible Versions
        showLoading('#bible-version', 'Loading versions...');
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'bible_api_request',
                request_type: 'versions',
                language: language,
                security: bibleReader.nonce
            },
            success: function(response) {
                if (response.success && response.data.length > 0) {
                    const $select = $('#bible-version');
                    $select.empty().append('<option value="">Select Version</option>');
                    
                    response.data.forEach(version => {
                        $select.append(`<option value="${version.id}">${version.name}</option>`);
                    });
                    
                    $select.prop('disabled', false);
                    hideLoading();
                } else {
                    showError('No versions available for this language');
                }
            },
            error: function() {
                $('#bible-version').html('<option value="">Load failed</option>');
            }
        });
    }
    
    function loadBooks(bibleId) {
        showLoading('#bible-book', 'Loading books...');

        // Special handling for Arabic/Persian/Yiddish
        if (['b17e246951402e50-01', '7cd100148df29c08-01', 'b6e21a7696dccae7-01'].includes(bibleId)) {
            $.ajax({
                url: ajaxurl,
                method: 'POST',
                data: {
                    action: 'bible_api_request',
                    request_type: 'books',
                    bible_id: bibleId,
                    security: bibleReader.nonce
                },
                success: function(response) {
                    if (response?.data?.length) {
                        const $select = $('#bible-book');
                        $select.empty().append('<option value="">Select Book</option>');
                        
                        response.data.forEach(book => {
                            $select.append(`<option value="${book.id}">${book.name}</option>`);
                        });
                        
                        $select.prop('disabled', false);
                        hideLoading();
                    } else {
                        showError('No books available for this version');
                    }
                },
                error: function() {
                    $('#bible-book').html('<option value="">Load failed</option>');
                    showError('Failed to load books');
                }
            });
            return;
        }
        
        // Regular API request for other versions
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'bible_api_request',
                request_type: 'books',
                bible_id: bibleId,
                security: bibleReader.nonce
            },
            success: function(response) {
                if (response?.data?.length) {
                    const $select = $('#bible-book');
                    $select.empty().append('<option value="">Select Book</option>');
                    
                    response.data.forEach(book => {
                        $select.append(`<option value="${book.id}">${book.name}</option>`);
                    });
                    
                    $select.prop('disabled', false);
                    hideLoading();
                } else {
                    showError('No books available for this version');
                }
            },
            error: function() {
                $('#bible-book').html('<option value="">Load failed</option>');
                showError('Failed to load books');
            }
        });
    }
    
    function loadChapters(bibleId, bookId) {
        showLoading('#bible-chapter', 'Loading chapters...');
        
        // Special handling for Arabic/Persian/Yiddish
        if (['b17e246951402e50-01', '7cd100148df29c08-01', 'b6e21a7696dccae7-01'].includes(bibleId)) {
            $.ajax({
                url: ajaxurl,
                method: 'POST',
                data: {
                    action: 'bible_api_request',
                    request_type: 'chapters',
                    bible_id: bibleId,
                    book_id: bookId,
                    security: bibleReader.nonce
                },
                success: function(response) {
                    if (response?.data?.length) {
                        const $select = $('#bible-chapter');
                        $select.empty().append('<option value="">Select Chapter</option>');
                        
                        response.data.forEach(chapter => {
                            $select.append(`<option value="${chapter.number}">Chapter ${chapter.number}</option>`);
                        });
                        
                        $select.prop('disabled', false);
                    } else {
                        $('#bible-chapter').html('<option value="">No chapters available</option>');
                    }
                },
                error: function() {
                    $('#bible-chapter').html('<option value="">Load failed</option>');
                }
            });
            return;
        }
        
        // Regular API request for other versions
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'bible_api_request',
                request_type: 'chapters',
                bible_id: bibleId,
                book_id: bookId,
                security: bibleReader.nonce
            },
            success: function(response) {
                if (response.success && response.data.length > 0) {
                    const $select = $('#bible-chapter');
                    $select.empty().append('<option value="">Select Chapter</option>');
                    
                    response.data.forEach(chapter => {
                        $select.append(`<option value="${chapter.number}">Chapter ${chapter.number}</option>`);
                    });
                    
                    $select.prop('disabled', false);
                    hideLoading();
                } else {
                    showError('No chapters available for this book');
                }
            },
            error: function() {
                showError('Failed to load chapters');
            }
        });
    }
    
    function loadPassage() {
        const bibleId = $('#bible-version').val();
        const bookId = $('#bible-book').val();
        const chapter = $('#bible-chapter').val();
        const isRTL = ['ara','fas','yid'].includes($('#bible-language').val());
        const language = $('#bible-language').val();
        
        if (!bibleId || !bookId || !chapter) return;
        
        // Set loading state
        showButtonLoading(true);
        $('#bible-load').prop('disabled', true);
        $('.loading-spinner').show();
        $('.button-text').hide();

        $('.bible-content').empty();
        $('.bible-content').attr('dir', isRTL ? 'rtl' : 'ltr');
        $('.bible-content')
            .toggleClass('greek-text', language == 'ell')
            .toggleClass('estonian-text', language == 'est');
        
        if (!$('.font-controls-container').length) {
            $('.bible-content').before(`
                <div class="font-controls-container">
                    <div class="font-controls">
                        <button id="font-decrease" title="Decrease font size">A-</button>
                        <button id="font-reset" title="Reset font size">A</button>
                        <button id="font-increase" title="Increase font size">A+</button>
                    </div>
                </div>
            `);
        } else {
            $('.font-controls-container').show();
        }
        
        // Initialize font size
        let fontSize = 1;
        $('#font-increase').click(() => {
            fontSize = Math.min(1.5, fontSize + 0.1);
            $('.bible-content').css('font-size', `${fontSize}em`);
        });
        $('#font-decrease').click(() => {
            fontSize = Math.max(0.8, fontSize - 0.1);
            $('.bible-content').css('font-size', `${fontSize}em`);
        });
        $('#font-reset').click(() => {
            fontSize = 1;
            $('.bible-content').css('font-size', '');
        });

        $('.bible-error').hide();
        
        // Handle API Bibles
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            dataType: 'json',
            data: {
                action: 'bible_api_request',
                request_type: 'passage',
                bible_id: $('#bible-version').val(),
                book_id: $('#bible-book').val(),
                chapter: $('#bible-chapter').val(),
                security: bibleReader.nonce
            },
            success: function(response) {
                // Always remove loading state first
                $('#bible-load').prop('disabled', false);
                $('.loading-spinner').hide();
                $('.button-text').show();
                
                if (response.success && response.data && response.data.content) {
                    // Get the book name from the dropdown
                    const bookName = $('#bible-book option:selected').text();
                    
                    // Update the chapter navigation
                    updateChapterNavigation(bookName, chapter);
                    
                    // Display the Bible text
                    $('.bible-content').html(response.data.content);
                    
                    // Generate word cloud
                    try {
                        generateWordCloud(response.data.content);
                    } catch (e) {
                        console.error("Word cloud generation failed:", e);
                    }
                } else {
                    showError('Invalid response format');
                }
            },
            error: function(xhr, status, error) {
                // Always remove loading state
                $('#bible-load').prop('disabled', false);
                $('.loading-spinner').hide();
                $('.button-text').show();
                
                showButtonLoading(false);
                showError('Failed to load: ' + error);
            }
        });
    }

    function updateChapterNavigation(bookName, chapter) {
        const navContainer = $('#chapter-navigation');
        const currentChapter = $('#chapter-title');
        const prevNum = $('#prev-chapter-num');
        const nextNum = $('#next-chapter-num');
        const chapterSelect = $('#bible-chapter');
        
        // Show navigation
        navContainer.show();
        
        // Update current chapter display
        currentChapter.text(`${bookName} ${chapter}`);
        
        // Get all chapter options (excluding the first default option)
        const chapterOptions = chapterSelect.find('option:not(:first)');
        const maxChapterNum = chapterOptions.length;
        const currentChapterNum = parseInt(chapter);
        
        // Update previous chapter number
        const prevChapter = currentChapterNum - 1;
        prevNum.text(prevChapter);
        
        // Update next chapter number
        const nextChapter = currentChapterNum + 1;
        nextNum.text(nextChapter);
        
        // Handle previous button visibility
        if (currentChapterNum <= 1) {
            $('#prev-chapter').hide();
        } else {
            $('#prev-chapter').show();
        }
        
        // Handle next button visibility
        if (currentChapterNum >= maxChapterNum) {
            $('#next-chapter').hide();
        } else {
            $('#next-chapter').show();
        }
        
        // Set click handlers
        $('#prev-chapter').off('click').on('click', function(e) {
            e.preventDefault();
            if (prevChapter >= 1) {
                chapterSelect.val(prevChapter).trigger('change');
                $('#bible-load').trigger('click');
            }
        });
        
        $('#next-chapter').off('click').on('click', function(e) {
            e.preventDefault();
            if (nextChapter <= maxChapterNum) {
                chapterSelect.val(nextChapter).trigger('change');
                $('#bible-load').trigger('click');
            }
        });
    }

    // Search functionality
    let currentSearchResults = [];
    let currentSearchTerm = '';
    let currentSearchPage = 1;
    const resultsPerPage = 10;

    $('#search-button').click(function() {
        const term = $('#bible-search').val().trim();
        if (term.length < 2) {
            alert('Please enter at least 2 characters');
            return;
        }
        
        performSearch(term);
    });

    $('#bible-search').on('keypress', function(e) {
        if (e.which === 13) { // Enter key
            $('#search-button').trigger('click');
        }
    });

    function performSearch(searchTerm) {
        const language = $('#bible-language').val();
        const version = $('#bible-version').val();
        
        if (!version || version === '') {
            $('.bible-content').html('<div class="search-error">Please select a Bible version first</div>');
            return;
        }
    
        // Show loading state
        $('.bible-content').html(`
            <div class="search-loading">
                <i class="fas fa-spinner fa-spin"></i> Searching for exact phrase "${searchTerm}"...
            </div>
        `);
        
        // Reset pagination
        currentSearchPage = 1;
        currentSearchTerm = searchTerm;
    
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            dataType: 'json',
            data: {
                action: 'bible_reader_search',
                security: bibleReader.nonce,
                request_type: 'api',
                language: language,
                bible_id: version,
                search_term: searchTerm,
                page: currentSearchPage
            },
            success: function(response) {
                if (response.success && response.data && response.data.results) {
                    currentSearchResults = response.data.results;
                    const totalResults = response.data.total || 0;
                    const totalPages = response.data.total_pages || 1;
                    
                    displaySearchResults(response.data.results, currentSearchPage, totalPages, totalResults);
                } else {
                    const errorMsg = response.data || 'No results found';
                    $('.bible-content').html(`<div class="search-error">${errorMsg}</div>`);
                }
            },
            error: function(xhr, status, error) {
                $('.bible-content').html(`<div class="search-error">Search error: ${error}</div>`);
            }
        });
    }
    
    function displaySearchResults(results, currentPage, totalPages, totalResults) {
        $('.bible-content').empty();
    
        if (!results || results.length === 0) {
            $('.bible-content').html('<div class="search-error">No results found</div>');
            return;
        }
    
        // Create results container
        const $container = $('<div class="search-results-container"></div>');
        
        // Add summary
        $container.append(`
            <div class="search-summary">
                Found ${totalResults} verses
                <div class="search-pagination-info">Page ${currentPage} of ${totalPages}</div>
            </div>
        `);
    
        // Add results
        results.forEach(result => {
            $container.append(`
                <div class="search-result" data-book="${result.book_id || ''}" data-chapter="${result.chapter || ''}">
                    <div class="search-result-reference">${result.reference}</div>
                    <div class="search-result-text">${highlightSearchTerm(result.text, currentSearchTerm)}</div>
                </div>
            `);
        });
    
        // Add pagination if needed
        if (totalPages > 1) {
            const $pagination = $('<div class="search-pagination"></div>');
            
            // Previous button
            $pagination.append(`<button class="prev-page" ${currentPage <= 1 ? 'disabled' : ''}>&laquo; Previous</button>`);
            
            // Page numbers - show up to 5 pages around current page
            const maxVisiblePages = 5;
            let startPage = Math.max(1, currentPage - 2);
            let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
            
            // Adjust if we're at the end
            if (endPage - startPage + 1 < maxVisiblePages) {
                startPage = Math.max(1, endPage - maxVisiblePages + 1);
            }
            
            // First page and ellipsis if needed
            if (startPage > 1) {
                $pagination.append('<button class="page-number" data-page="1">1</button>');
                if (startPage > 2) {
                    $pagination.append('<span class="page-ellipsis">...</span>');
                }
            }
            
            // Visible page range
            for (let i = startPage; i <= endPage; i++) {
                $pagination.append(`<button class="page-number ${i === currentPage ? 'active' : ''}" data-page="${i}">${i}</button>`);
            }
            
            // Last page and ellipsis if needed
            if (endPage < totalPages) {
                if (endPage < totalPages - 1) {
                    $pagination.append('<span class="page-ellipsis">...</span>');
                }
                $pagination.append(`<button class="page-number" data-page="${totalPages}">${totalPages}</button>`);
            }
            
            // Next button
            $pagination.append(`<button class="next-page" ${currentPage >= totalPages ? 'disabled' : ''}>Next &raquo;</button>`);
            
            $container.append($pagination);
        }
    
        $('.bible-content').append($container);
    
        // Smooth scroll to results
        $('html, body').animate({
            scrollTop: $('.bible-content').offset().top - 20
        }, 200);
    }

    function highlightSearchTerm(text, term) {
        if (!term) return text;
        
        // Escape special regex characters
        const escapedTerm = term.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const regex = new RegExp(`(${escapedTerm})`, 'gi');
        
        return text.replace(regex, '<span class="search-highlight">$1</span>');
    }

    // Handle pagination clicks
    $(document).on('click', '.search-pagination button', function() {
        const $btn = $(this);
        const version = $('#bible-version').val();
        
        if ($btn.hasClass('prev-page')) {
            currentSearchPage--;
        } else if ($btn.hasClass('next-page')) {
            currentSearchPage++;
        } else if ($btn.hasClass('page-number')) {
            currentSearchPage = parseInt($btn.data('page'));
        }
        
        // Show loading state
        $('.bible-content').html(`
            <div class="search-loading">
                <i class="fas fa-spinner fa-spin"></i> Loading page ${currentSearchPage}...
            </div>
        `);
    
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            dataType: 'json',
            data: {
                action: 'bible_reader_search',
                security: bibleReader.nonce,
                request_type: 'api',
                language: $('#bible-language').val(),
                bible_id: version,
                search_term: currentSearchTerm,
                page: currentSearchPage
            },
            success: function(response) {
                if (response.success && response.data && response.data.results) {
                    const totalResults = response.data.total || 0;
                    const totalPages = response.data.total_pages || 1;
                    
                    displaySearchResults(response.data.results, currentSearchPage, totalPages, totalResults);
                } else {
                    const errorMsg = response.data || 'No results found';
                    $('.bible-content').html(`<div class="search-error">${errorMsg}</div>`);
                }
            },
            error: function(xhr, status, error) {
                $('.bible-content').html(`<div class="search-error">Error loading page: ${error}</div>`);
            }
        });
    });

    // Handle result clicks to load the passage
    $(document).on('click', '.search-result', function() {
        const bookId = $(this).data('book');
        const chapter = $(this).data('chapter');
        
        $('#bible-book').val(bookId).trigger('change');
        
        setTimeout(() => {
            $('#bible-chapter').val(chapter).trigger('change');
            $('#bible-load').trigger('click');
        }, 500);
    });

    function showLoading(selector, message) {
        $(selector).html(`<option value="">${message}</option>`);
    }
    
    function hideLoading() {
        // Loading states are cleared when options are populated
    }
    
    function showButtonLoading(show) {
        if (show) {
            $('#bible-load').prop('disabled', true);
            $('.button-text').hide();
            $('.loading-spinner').show();
        } else {
            const bibleId = $('#bible-version').val();
            const bookId = $('#bible-book').val();
            const chapter = $('#bible-chapter').val();
            $('#bible-load').prop('disabled', !(bibleId && bookId && chapter));
            $('.button-text').show();
            $('.loading-spinner').hide();
        }
    }
    
    function showError(message) {
        $('.bible-error').text(message).show();
    }

    $(document).on('click', '.share-image-btn', function(e) {
        e.preventDefault();
        
        // Check if any verses are selected
        if (selectedVerses.length === 0) {
            alert("Please select verses first by clicking on them!");
            return;
        }
    
        const $btn = $(this);
        $btn.html('<i class="fas fa-spinner fa-spin"></i> Generating...').prop('disabled', true);
    
        // Create container for SELECTED verses only
        const $imageContainer = $('<div class="verse-image-container"></div>').css({
            position: 'fixed',
            left: '-9999px',
            background: '#fff',
            padding: '30px',
            borderRadius: '8px',
            maxWidth: '600px'
        });
    
        // Add reference
        const book = $('#bible-book option:selected').text();
        const chapter = $('#bible-chapter').val();
        const firstVerse = selectedVerses[0].number;
        const lastVerse = selectedVerses[selectedVerses.length - 1].number;
        const reference = `${book} ${chapter}:${firstVerse}${selectedVerses.length > 1 ? `-${lastVerse}` : ''}`;
        
        $imageContainer.append(
            `<h3 style="text-align:center;color:#6e3b3b;margin-bottom:20px;font-family:sans-serif">
                ${reference}
            </h3>`
        );
    
        // Add ONLY selected verses
        selectedVerses.forEach(verse => {
            $imageContainer.append(`
                <p style="margin-bottom: 15px;">
                    <sup style="font-weight:bold;color:#777;">${verse.number}</sup> 
                    ${verse.text}
                </p>
            `);
        });
    
        $('body').append($imageContainer);
    
        // Generate image
        html2canvas($imageContainer[0], {
            backgroundColor: '#ffffff',
            scale: 2,
            logging: false
        }).then(canvas => {
            $imageContainer.remove();
            if (confirm(`Download ${reference} as an image?`)) {
                const link = document.createElement('a');
                link.download = `bible-${reference.replace(/ /g, '-')}.png`;
                link.href = canvas.toDataURL('image/png');
                link.click();
            }
            $btn.html('<i class="fas fa-image"></i> Share as Image').prop('disabled', false);
        }).catch(err => {
            console.error("Error:", err);
            $btn.html('<i class="fas fa-image"></i> Share as Image').prop('disabled', false);
        });
    });
});
