/**
 * Product parsing utility for chat messages
 * Parses markdown-style product links: [![Product Name](image_url)](product_url)
 */

export interface ProductInfo {
  product_name: string;
  image_url: string;
  product_url: string;
}

/**
 * Parse markdown-style product links from text
 * Format: [![Product Name](image_url)](product_url)
 * @param text - The text to parse
 * @returns Array of parsed products
 */
export function parseProducts(text: string): ProductInfo[] {
  // Regex to match [![Product Name](image_url)](product_url)
  // Structure: [![name](image_url)](product_url)
  // Breakdown:
  // \[!\[ - matches the opening [![
  // ([^\]]+) - captures the product name (any character except ])
  // \]\( - matches ](
  // ([^)]+) - captures the image URL (any character except ))
  // \)\]\( - matches )]( where ) closes image URL, ] closes [![, and ( opens product URL
  // ([^)]+) - captures the product URL (any character except ))
  // \) - matches the closing )
  const productRegex = /\[!\[([^\]]+)\]\(([^)]+)\)\]\(([^)]+)\)/g;
  const products: ProductInfo[] = [];
  let match;

  while ((match = productRegex.exec(text)) !== null) {
    const [, name, imageUrl, productUrl] = match;

    // Validate URLs
    if (isValidUrl(imageUrl) && isValidUrl(productUrl)) {
      products.push({
        product_name: name.trim(),
        image_url: cleanUrl(imageUrl.trim()),
        product_url: cleanUrl(productUrl.trim()),
      });
    }
  }

  return products;
}

/**
 * Check if text contains any products
 * @param text - The text to check
 * @returns True if text contains products
 */
export function hasProducts(text: string): boolean {
  return /\[!\[([^\]]+)\]\(([^)]+)\)\]\(([^)]+)\)/.test(text);
}

/**
 * Remove products from text (used when displaying products separately)
 * Also removes extra newlines left after product removal
 * @param text - The text to clean
 * @returns Text with products removed and cleaned up
 */
export function removeProductsFromText(text: string): string {
  const productRegex = /\[!\[([^\]]+)\]\(([^)]+)\)\]\(([^)]+)\)/g;
  let cleaned = text.replace(productRegex, '');

  // Remove extra newlines that might be left after removing products
  // This handles cases like "text\n\n[![...](...)](...)\n\ntext" -> "text\n\ntext"
  // Replace 3+ consecutive newlines with 2 to preserve paragraph breaks
  cleaned = cleaned.replace(/\n{3,}/g, '\n\n');

  // Clean up any remaining multiple spaces
  cleaned = cleaned.replace(/[ \t]+/g, ' ');

  // Final trim
  cleaned = cleaned.trim();

  return cleaned;
}

/**
 * Validate URL
 * @param url - The URL to validate
 * @returns True if URL is valid
 */
function isValidUrl(url: string): boolean {
  try {
    const cleanUrl = url.trim();
    if (!cleanUrl) return false;

    // Check for basic URL pattern
    const urlPattern = /^(https?:\/\/|mailto:|\/)/i;
    return urlPattern.test(cleanUrl) || cleanUrl.startsWith('/');
  } catch {
    return false;
  }
}

/**
 * Clean and normalize URL
 * @param url - The URL to clean
 * @returns Cleaned URL with protocol if missing
 */
function cleanUrl(url: string): string {
  try {
    let cleanUrl = url.trim();

    // Add protocol if missing and it's not a relative path or mailto
    if (!cleanUrl.match(/^https?:\/\//i) && !cleanUrl.match(/^(mailto:|\/)/)) {
      cleanUrl = 'https://' + cleanUrl;
    }

    return cleanUrl;
  } catch {
    return url;
  }
}
