/**
 * Theme Service
 * Handles widget theming and CSS custom properties
 */

import { WidgetConfiguration, ColorModeConfig } from '../types/api';

export class ThemeService {
  private hostElement: HTMLElement;
  private currentTheme: 'light' | 'dark' | 'auto' = 'auto';

  constructor(hostElement: HTMLElement) {
    this.hostElement = hostElement;
  }

  /**
   * Detect the language of the website the widget is embedded on
   * Returns 'pl' for Polish, 'en' for English (default)
   */
  async detectWebsiteLanguage(): Promise<'pl' | 'en'> {
    try {
      // Method 1: Check meta http-equiv="content-language"
      const metaLang = document.querySelector('meta[http-equiv="content-language"]');
      if (metaLang) {
        const content = metaLang.getAttribute('content')?.toLowerCase();
        if (content?.includes('pl')) return 'pl';
        if (content?.includes('en')) return 'en';
      }

      // Method 2: Check meta name="language"
      const metaNameLang = document.querySelector('meta[name="language"]');
      if (metaNameLang) {
        const content = metaNameLang.getAttribute('content')?.toLowerCase();
        if (content?.includes('pl')) return 'pl';
        if (content?.includes('en')) return 'en';
      }

      // Method 3: Check for Polish text patterns in the page (improved)
      const bodyText = document.body.textContent?.toLowerCase() || '';

      // More comprehensive Polish word detection
      const polishWords = [
        // Polish articles and pronouns
        'jest',
        'są',
        'ma',
        'mają',
        'być',
        'może',
        'można',
        'możemy',
        'można',
        'możemy',
        // Polish common verbs
        'jest',
        'są',
        'ma',
        'mają',
        'być',
        'może',
        'można',
        'możemy',
        'można',
        'możemy',
        // Polish common nouns
        'strona',
        'stronie',
        'strony',
        'stron',
        'stroną',
        'stronę',
        'stronie',
        'strony',
        'stron',
        'stroną',
        'stronę',
        'informacje',
        'informacji',
        'informacjami',
        'informacjami',
        'informacjami',
        'informacjami',
        'informacjami',
        'kontakt',
        'kontakcie',
        'kontaktu',
        'kontaktem',
        'kontaktem',
        'kontaktem',
        'kontaktem',
        'kontaktem',
        // Polish common adjectives
        'nowy',
        'nowa',
        'nowe',
        'nowego',
        'nowej',
        'nowego',
        'nowym',
        'nową',
        'nowym',
        'nowym',
        'dobry',
        'dobra',
        'dobre',
        'dobrego',
        'dobrej',
        'dobrego',
        'dobrym',
        'dobrą',
        'dobrym',
        'dobrym',
        // Polish common conjunctions
        'oraz',
        'lub',
        'ale',
        'jednak',
        'więc',
        'dlatego',
        'ponieważ',
        'gdy',
        'gdyż',
        'jeśli',
        'jeżeli',
        // Polish common prepositions
        'w',
        'na',
        'z',
        'do',
        'od',
        'po',
        'przy',
        'dla',
        'przez',
        'bez',
        'pod',
        'nad',
        'między',
        'przed',
        'za',
        // Polish common particles
        'nie',
        'już',
        'jeszcze',
        'tylko',
        'także',
        'również',
        'też',
        'tak',
        'nie',
        'już',
        'jeszcze',
        'tylko',
        'także',
        'również',
        'też',
        'tak',
      ];

      // Count Polish words with better matching
      const polishWordCount = polishWords.filter(word => {
        // Look for word boundaries to avoid false positives
        const regex = new RegExp(`\\b${word}\\b`, 'gi');
        return regex.test(bodyText);
      }).length;

      // Lower threshold for detection
      if (polishWordCount >= 2) {
        return 'pl';
      }

      // Method 4: Check for Polish characters (ą, ć, ę, ł, ń, ó, ś, ź, ż)
      const polishChars = /[ąćęłńóśźż]/gi;
      const polishCharCount = (bodyText.match(polishChars) || []).length;

      if (polishCharCount >= 5) {
        return 'pl';
      }

      // Method 5: Check navigator.language as fallback
      const browserLang = navigator.language.toLowerCase().split('-')[0];
      if (browserLang === 'pl') {
        return 'pl';
      }

      // Method 6: Check for common Polish website patterns
      const polishPatterns = [
        'www.',
        '.pl',
        'strona główna',
        'o nas',
        'kontakt',
        'oferta',
        'usługi',
        'produkty',
        'cennik',
        'galeria',
        'aktualności',
        'news',
        'blog',
        'pomoc',
        'regulamin',
        'polityka prywatności',
      ];

      const polishPatternCount = polishPatterns.filter(pattern => bodyText.includes(pattern.toLowerCase())).length;

      if (polishPatternCount >= 2) {
        return 'pl';
      }

      // Method 7: Try to use Google Translate API for detection (optional)
      try {
        const detectedLang = await this.detectLanguageWithGoogleTranslate(bodyText.substring(0, 1000));
        if (detectedLang) {
          if (detectedLang === 'pl') return 'pl';
          if (detectedLang === 'en') return 'en';
        }
      } catch (error) {
        console.error('Error detecting language with online translation:', error);
      }

      // Method 8: Check HTML lang attribute (highest priority)
      const htmlLang = document.documentElement.lang || document.documentElement.getAttribute('lang');
      if (htmlLang) {
        const langCode = htmlLang.toLowerCase().split('-')[0];
        if (langCode === 'pl') return 'pl';
        if (langCode === 'en') return 'en';
      }

      // Method 9: Check for English words to balance detection
      const englishWords = [
        'the',
        'and',
        'or',
        'but',
        'in',
        'on',
        'at',
        'to',
        'for',
        'of',
        'with',
        'by',
        'is',
        'are',
        'was',
        'were',
        'be',
        'been',
        'being',
        'have',
        'has',
        'had',
        'do',
        'does',
        'did',
        'will',
        'would',
        'could',
        'should',
        'may',
        'might',
        'can',
        'this',
        'that',
        'these',
        'those',
        'a',
        'an',
        'some',
        'any',
        'all',
        'every',
        'each',
        'other',
        'another',
        'such',
        'no',
        'not',
        'only',
        'also',
        'very',
        'much',
        'more',
        'most',
        'less',
        'least',
        'many',
        'few',
        'little',
        'big',
        'small',
        'good',
        'bad',
        'new',
        'old',
        'first',
        'last',
        'next',
        'previous',
        'here',
        'there',
        'where',
        'when',
        'why',
        'how',
        'what',
        'who',
      ];

      const englishWordCount = englishWords.filter(word => {
        const regex = new RegExp(`\\b${word}\\b`, 'gi');
        return regex.test(bodyText);
      }).length;

      // If we have more English words than Polish words, prefer English
      if (englishWordCount > polishWordCount) {
        return 'en';
      }

      // Method 10: Check for common English website patterns
      const englishPatterns = [
        'home',
        'about',
        'contact',
        'services',
        'products',
        'pricing',
        'gallery',
        'news',
        'blog',
        'help',
        'support',
        'terms',
        'privacy',
        'login',
        'register',
        'sign up',
        'sign in',
        'logout',
        'dashboard',
        'profile',
        'settings',
        'account',
        'billing',
        'payment',
        'order',
        'cart',
        'checkout',
        'shipping',
        'delivery',
        'return',
        'refund',
      ];

      const englishPatternCount = englishPatterns.filter(pattern => bodyText.includes(pattern.toLowerCase())).length;

      // If we have more English patterns, prefer English
      if (englishPatternCount > polishPatternCount) {
        return 'en';
      }

      return 'en';
    } catch (error) {
      console.error('Error detecting language', error);
      return 'en';
    }
  }

  /**
   * Detect language using Google Translate API (fallback method)
   */
  private async detectLanguageWithGoogleTranslate(text: string): Promise<string | null> {
    try {
      // Use Google Translate's language detection API
      const response = await fetch(`https://translate.googleapis.com/translate_a/single?client=gtx&sl=auto&tl=en&dt=t&q=${encodeURIComponent(text)}`);

      if (!response.ok) {
        throw new Error('Google Translate API request failed');
      }

      const data = await response.json();
      const detectedLang = data[2]; // The detected language code

      return detectedLang;
    } catch {
      return null;
    }
  }

  /**
   * Apply widget configuration to CSS custom properties
   */
  applyConfig(config: WidgetConfiguration, theme: 'light' | 'dark' | 'auto' = 'auto'): void {
    this.currentTheme = theme;

    const resolvedTheme = this.resolveTheme(theme);
    const colorConfig = resolvedTheme === 'dark' ? config.dark_mode : config.light_mode;

    if (colorConfig) {
      this.applyColorConfig(colorConfig);
    }
  }

  /**
   * Apply color configuration to CSS custom properties
   */
  private applyColorConfig(config: ColorModeConfig): void {
    const properties = {
      '--bcx-primary': config.primary_color,
      '--bcx-secondary': config.secondary_color,
      '--bcx-background': config.background_color,
      '--bcx-text': config.text_color,
    };

    Object.entries(properties).forEach(([property, value]) => {
      if (value) {
        this.hostElement.style.setProperty(property, value);
      }
    });
  }

  /**
   * Resolve theme based on preference and system settings
   */
  private resolveTheme(theme: 'light' | 'dark' | 'auto'): 'light' | 'dark' {
    if (theme === 'auto') {
      return this.detectWebsiteColorScheme();
    }
    return theme;
  }

  /**
   * Universal color scheme detection for any website
   * Checks multiple methods in order of reliability
   */
  private detectWebsiteColorScheme(): 'light' | 'dark' {
    try {
      // Method 1: Check CSS custom properties (most reliable for modern sites)
      const cssTheme = this.detectCSSTheme();
      if (cssTheme) return cssTheme;

      // Method 2: Check data attributes (common in frameworks)
      const dataTheme = this.detectDataTheme();
      if (dataTheme) return dataTheme;

      // Method 3: Check class names on html/body
      const classTheme = this.detectClassTheme();
      if (classTheme) return classTheme;

      // Method 4: Check meta theme-color
      const metaTheme = this.detectMetaTheme();
      if (metaTheme) return metaTheme;

      // Method 5: Check computed styles of body/html
      const computedTheme = this.detectComputedTheme();
      if (computedTheme) return computedTheme;

      // No system preference fallback - default to light mode
      return 'light';
    } catch {
      return 'light';
    }
  }

  /**
   * Method 1: Detect CSS custom properties
   * Checks for common CSS variables used by frameworks
   */
  private detectCSSTheme(): 'light' | 'dark' | null {
    const root = document.documentElement;
    const computedStyle = window.getComputedStyle(root);

    // Check common CSS custom properties
    const cssVars = ['--color-scheme', '--theme', '--mode', '--color-mode', '--dark-mode', '--light-mode', '--app-theme', '--ui-theme'];

    for (const varName of cssVars) {
      const value = computedStyle.getPropertyValue(varName).trim();
      if (value) {
        if (value.includes('dark') || value === 'dark') return 'dark';
        if (value.includes('light') || value === 'light') return 'light';
      }
    }

    // Check for dark mode indicators in CSS variables
    const darkIndicators = ['--bg-color', '--background-color', '--primary-bg', '--surface-color'];

    for (const varName of darkIndicators) {
      const value = computedStyle.getPropertyValue(varName).trim();
      if (value) {
        // Check if it's a dark color (basic heuristic)
        if (this.isDarkColor(value)) return 'dark';
        if (this.isLightColor(value)) return 'light';
      }
    }

    return null;
  }

  /**
   * Method 2: Detect data attributes
   * Common in frameworks like Next.js, Nuxt.js, etc.
   */
  private detectDataTheme(): 'light' | 'dark' | null {
    const elements = [document.documentElement, document.body];
    const dataAttrs = ['data-theme', 'data-mode', 'data-color-scheme', 'data-color-mode'];

    for (const element of elements) {
      if (!element) continue;

      for (const attr of dataAttrs) {
        const value = element.getAttribute(attr);
        if (value) {
          if (value.includes('dark') || value === 'dark') return 'dark';
          if (value.includes('light') || value === 'light') return 'light';
        }
      }
    }

    return null;
  }

  /**
   * Method 3: Detect class names
   * Common in Tailwind, Bootstrap, and other frameworks
   */
  private detectClassTheme(): 'light' | 'dark' | null {
    const elements = [document.documentElement, document.body];
    const darkClasses = ['dark', 'dark-mode', 'theme-dark', 'dark-theme', 'is-dark', 'dark-theme', 'night-mode'];

    for (const element of elements) {
      if (!element) continue;

      const classList = element.classList;
      for (const className of darkClasses) {
        if (classList.contains(className)) return 'dark';
      }
    }

    return null;
  }

  /**
   * Method 4: Detect meta theme-color
   * Some sites use meta theme-color to indicate dark mode
   */
  private detectMetaTheme(): 'light' | 'dark' | null {
    const metaTheme = document.querySelector('meta[name="theme-color"]');
    if (metaTheme) {
      const content = metaTheme.getAttribute('content');
      if (content && this.isDarkColor(content)) return 'dark';
      if (content && this.isLightColor(content)) return 'light';
    }
    return null;
  }

  /**
   * Method 5: Detect computed styles
   * Analyze background and text colors of the page
   */
  private detectComputedTheme(): 'light' | 'dark' | null {
    try {
      const body = document.body;
      if (!body) return null;

      const computedStyle = window.getComputedStyle(body);
      const bgColor = computedStyle.backgroundColor;
      const textColor = computedStyle.color;

      // If we can't get colors, skip this method
      if (!bgColor || bgColor === 'rgba(0, 0, 0, 0)' || !textColor) return null;

      const isDarkBg = this.isDarkColor(bgColor);
      const isDarkText = this.isDarkColor(textColor);

      // If background is dark and text is light, it's dark mode
      if (isDarkBg && !isDarkText) return 'dark';

      // If background is light and text is dark, it's light mode
      if (!isDarkBg && isDarkText) return 'light';

      return null;
    } catch {
      return null;
    }
  }

  /**
   * Helper: Check if a color is dark
   */
  private isDarkColor(color: string): boolean {
    try {
      // Convert to RGB values
      const rgb = this.parseColor(color);
      if (!rgb) return false;

      // Calculate luminance
      const luminance = (0.299 * rgb.r + 0.587 * rgb.g + 0.114 * rgb.b) / 255;
      return luminance < 0.5;
    } catch {
      return false;
    }
  }

  /**
   * Helper: Check if a color is light
   */
  private isLightColor(color: string): boolean {
    try {
      const rgb = this.parseColor(color);
      if (!rgb) return false;

      const luminance = (0.299 * rgb.r + 0.587 * rgb.g + 0.114 * rgb.b) / 255;
      return luminance > 0.7;
    } catch {
      return false;
    }
  }

  /**
   * Helper: Parse color string to RGB
   */
  private parseColor(color: string): { r: number; g: number; b: number } | null {
    // Remove whitespace
    color = color.trim();

    // Handle hex colors
    if (color.startsWith('#')) {
      const hex = color.slice(1);
      if (hex.length === 3) {
        return {
          r: parseInt(hex[0] + hex[0], 16),
          g: parseInt(hex[1] + hex[1], 16),
          b: parseInt(hex[2] + hex[2], 16),
        };
      } else if (hex.length === 6) {
        return {
          r: parseInt(hex.slice(0, 2), 16),
          g: parseInt(hex.slice(2, 4), 16),
          b: parseInt(hex.slice(4, 6), 16),
        };
      }
    }

    // Handle rgb/rgba colors
    const rgbMatch = color.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/);
    if (rgbMatch) {
      return {
        r: parseInt(rgbMatch[1]),
        g: parseInt(rgbMatch[2]),
        b: parseInt(rgbMatch[3]),
      };
    }

    // Handle hsl colors (basic conversion)
    const hslMatch = color.match(/hsla?\((\d+),\s*(\d+)%,\s*(\d+)%/);
    if (hslMatch) {
      const h = parseInt(hslMatch[1]) / 360;
      const s = parseInt(hslMatch[2]) / 100;
      const l = parseInt(hslMatch[3]) / 100;

      const hue2rgb = (p: number, q: number, t: number) => {
        if (t < 0) t += 1;
        if (t > 1) t -= 1;
        if (t < 1 / 6) return p + (q - p) * 6 * t;
        if (t < 1 / 2) return q;
        if (t < 2 / 3) return p + (q - p) * (2 / 3 - t) * 6;
        return p;
      };

      const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
      const p = 2 * l - q;

      return {
        r: Math.round(hue2rgb(p, q, h + 1 / 3) * 255),
        g: Math.round(hue2rgb(p, q, h) * 255),
        b: Math.round(hue2rgb(p, q, h - 1 / 3) * 255),
      };
    }

    return null;
  }

  /**
   * Set default theme colors
   */
  setDefaultTheme(): void {
    const defaultColors = {
      '--bcx-primary': '#007bff',
      '--bcx-secondary': '#6c757d',
      '--bcx-background': '#ffffff',
      '--bcx-text': '#212529',
      '--bcx-border': '#dee2e6',
      '--bcx-shadow': 'rgba(0, 0, 0, 0.1)',
      '--bcx-success': '#28a745',
      '--bcx-warning': '#ffc107',
      '--bcx-error': '#dc3545',
      '--bcx-info': '#17a2b8',
    };

    Object.entries(defaultColors).forEach(([property, value]) => {
      this.hostElement.style.setProperty(property, value);
    });

    // Also apply to document root for inheritance
    Object.entries(defaultColors).forEach(([property, value]) => {
      document.documentElement.style.setProperty(property, value);
    });
  }

  /**
   * Apply custom CSS properties from host page
   */
  applyCustomProperties(customProperties: Record<string, string>): void {
    Object.entries(customProperties).forEach(([property, value]) => {
      this.hostElement.style.setProperty(property, value);
    });
  }

  /**
   * Listen for system theme changes
   */
  watchSystemTheme(callback: (theme: 'light' | 'dark') => void): () => void {
    const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');

    const handler = (e: MediaQueryListEvent) => {
      callback(e.matches ? 'dark' : 'light');
    };

    mediaQuery.addEventListener('change', handler);

    // Return cleanup function
    return () => {
      mediaQuery.removeEventListener('change', handler);
    };
  }

  /**
   * Watch for website theme changes dynamically
   * This is useful for sites that change themes without page reload
   */
  watchWebsiteTheme(callback: (theme: 'light' | 'dark') => void): () => void {
    // Check if MutationObserver is available (not available in test environment)
    if (typeof MutationObserver === 'undefined') {
      return () => {};
    }

    const observers: (() => void)[] = [];

    // Watch for class changes on html and body
    const watchClassChanges = () => {
      const elements = [document.documentElement, document.body];
      const observer = new MutationObserver(mutations => {
        mutations.forEach(mutation => {
          if (mutation.type === 'attributes' && mutation.attributeName === 'class') {
            const newTheme = this.detectWebsiteColorScheme();
            callback(newTheme);
          }
        });
      });

      elements.forEach(element => {
        if (element) {
          observer.observe(element, { attributes: true, attributeFilter: ['class'] });
        }
      });

      observers.push(() => observer.disconnect());
    };

    // Watch for data attribute changes
    const watchDataChanges = () => {
      const elements = [document.documentElement, document.body];
      const observer = new MutationObserver(mutations => {
        mutations.forEach(mutation => {
          if (mutation.type === 'attributes' && ['data-theme', 'data-mode', 'data-color-scheme'].includes(mutation.attributeName)) {
            const newTheme = this.detectWebsiteColorScheme();
            callback(newTheme);
          }
        });
      });

      elements.forEach(element => {
        if (element) {
          observer.observe(element, {
            attributes: true,
            attributeFilter: ['data-theme', 'data-mode', 'data-color-scheme'],
          });
        }
      });

      observers.push(() => observer.disconnect());
    };

    // Watch for CSS custom property changes
    const watchCSSChanges = () => {
      const root = document.documentElement;
      const observer = new MutationObserver(mutations => {
        mutations.forEach(mutation => {
          if (mutation.type === 'attributes' && mutation.attributeName === 'style') {
            const newTheme = this.detectWebsiteColorScheme();
            callback(newTheme);
          }
        });
      });

      if (root) {
        observer.observe(root, { attributes: true, attributeFilter: ['style'] });
      }

      observers.push(() => observer.disconnect());
    };

    // Start all watchers
    watchClassChanges();
    watchDataChanges();
    watchCSSChanges();

    // Return cleanup function
    return () => {
      observers.forEach(cleanup => cleanup());
    };
  }

  /**
   * Get current detected theme
   */
  getCurrentDetectedTheme(): 'light' | 'dark' {
    return this.detectWebsiteColorScheme();
  }

  /**
   * Get current applied theme
   */
  getCurrentTheme(): 'light' | 'dark' {
    return this.currentTheme === 'auto' ? this.detectWebsiteColorScheme() : this.currentTheme;
  }

  /**
   * Set theme explicitly (from prop)
   * If 'auto', will detect from website
   * If 'light' or 'dark', will use that theme directly without detection
   */
  setTheme(theme: 'light' | 'dark' | 'auto'): void {
    this.currentTheme = theme;
  }
}
