/**
 * Authentication Service
 * Handles widget session creation and token management
 */

import { WidgetSessionRequest, WidgetSessionResponse, APIError } from '../types/api';

export class AuthService {
  private baseUrl: string;
  private sessionToken?: string;
  private sessionExpiresAt?: Date;
  private chatId?: string; // Store chat ID for conversation context
  private widgetKey?: string; // Store widget key for token refresh
  private origin?: string; // Store origin for token refresh
  private refreshPromise?: Promise<boolean>; // Promise for ongoing refresh to prevent concurrent refreshes

  constructor(baseUrl: string = 'http://localhost:8000') {
    this.baseUrl = baseUrl;
  }

  /**
   * Create a new widget session
   */
  async createSession(widgetKey: string, origin: string): Promise<Partial<WidgetSessionResponse>> {
    // Store widget key and origin for automatic token refresh
    this.widgetKey = widgetKey;
    this.origin = origin;

    const request: WidgetSessionRequest = {
      widget_key: widgetKey,
      origin: origin,
    };

    const response = await fetch(`${this.baseUrl}/api/widgets/session/create/`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(request),
    });

    if (!response.ok) {
      const error: APIError = await response.json();
      throw new Error(error.error || `HTTP ${response.status}: ${response.statusText}`);
    }

    const sessionData: WidgetSessionResponse = await response.json();

    // Store session data in memory only
    this.sessionToken = sessionData.data.token;
    this.sessionExpiresAt = new Date(sessionData.data.expires_at);

    return sessionData.data as Partial<WidgetSessionResponse>;
  }

  /**
   * Set chat ID from response headers
   */
  setChatId(chatId: string): void {
    this.chatId = chatId;
  }

  /**
   * Get current chat ID
   */
  getChatId(): string | undefined {
    return this.chatId;
  }

  /**
   * Get current session token
   * Automatically refreshes token if expired (seamless token management)
   * This ensures the widget continues to work even after long periods of inactivity
   */
  async getToken(): Promise<string | undefined> {
    // If token is valid, return it immediately
    if (this.isTokenValid()) {
      return this.sessionToken;
    }

    // If token expired and we have credentials, try to refresh
    if (this.widgetKey && this.origin) {
      // If refresh is already in progress, wait for it
      if (this.refreshPromise) {
        const refreshed = await this.refreshPromise;
        if (refreshed && this.isTokenValid()) {
          return this.sessionToken;
        }
        return undefined;
      }

      // Start refresh process
      this.refreshPromise = this.refreshSessionIfNeeded(this.widgetKey, this.origin);

      try {
        const refreshed = await this.refreshPromise;
        this.refreshPromise = undefined; // Clear promise after completion

        if (refreshed && this.isTokenValid()) {
          return this.sessionToken;
        }
      } catch {
        this.refreshPromise = undefined; // Clear promise on error
        // Refresh failed, return undefined
        // Error will be handled by calling code
      }
    }

    // No credentials available or refresh failed
    return undefined;
  }

  /**
   * Check if current token is valid and not expired
   */
  isTokenValid(): boolean {
    if (!this.sessionToken || !this.sessionExpiresAt) {
      return false;
    }
    return new Date() < this.sessionExpiresAt;
  }

  /**
   * Clear current session
   */
  clearSession(): void {
    this.sessionToken = undefined;
    this.sessionExpiresAt = undefined;
    this.chatId = undefined; // Clear chat ID as well
    this.widgetKey = undefined; // Clear widget key
    this.origin = undefined; // Clear origin
    this.refreshPromise = undefined; // Clear any pending refresh
  }

  /**
   * Get authorization header for API requests
   * Automatically refreshes token if expired
   */
  async getAuthHeader(): Promise<{ Authorization: string } | {}> {
    const token = await this.getToken();
    const headers = token ? { Authorization: `Bearer ${token}` } : {};
    return headers;
  }

  /**
   * Get chat ID header for API requests
   */
  getChatIdHeader(): { 'X-BCX-Chat-ID': string } | {} {
    const header = this.chatId ? { 'X-BCX-Chat-ID': this.chatId } : {};
    return header;
  }

  /**
   * Get all headers for API requests (auth + chat ID)
   * Automatically refreshes token if expired
   */
  async getAllHeaders(): Promise<Record<string, string>> {
    const headers = {
      ...(await this.getAuthHeader()),
      ...this.getChatIdHeader(),
    };
    return headers;
  }

  /**
   * Refresh session if needed
   */
  async refreshSessionIfNeeded(widgetKey: string, origin: string): Promise<boolean> {
    if (this.isTokenValid()) {
      return true;
    }

    try {
      await this.createSession(widgetKey, origin);
      return true;
    } catch {
      return false;
    }
  }
}
