/**
 * API Service
 * Handles all API communication with backend services
 */

import { AuthService } from './auth.service';
import { OrganizationWidgetConfiguration, MessageRequest, APIError, PaginatedMessagesResponse } from '../types/api';

export class ApiService {
  private authService: AuthService;
  private dbServiceUrl: string;
  private aiServiceUrl: string;

  constructor(dbServiceUrl: string = 'http://localhost:8000', aiServiceUrl: string = 'http://localhost:8081', authService?: AuthService) {
    this.dbServiceUrl = dbServiceUrl;
    this.aiServiceUrl = aiServiceUrl;
    this.authService = authService || new AuthService(dbServiceUrl);
  }

  /**
   * Fetch widget configuration from backend
   */
  async getWidgetConfig(organizationId: number): Promise<OrganizationWidgetConfiguration> {
    const token = await this.authService.getToken();
    if (!token) {
      throw new Error('No valid session token available');
    }

    const response = await fetch(`${this.dbServiceUrl}/api/widgets/org/${organizationId}/widget-config/`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        ...(await this.authService.getAuthHeader()),
      },
    });

    if (!response.ok) {
      const error: APIError = await response.json();
      throw new Error(error.error || `HTTP ${response.status}: ${response.statusText}`);
    }

    return await response.json();
  }

  /**
   * Send a chat message to the AI service
   */
  async sendMessage(message: string, images?: File[]): Promise<ReadableStream<Uint8Array> | null> {
    const token = await this.authService.getToken();

    if (!token) {
      throw new Error('No valid session token available');
    }

    const headers = {
      ...(await this.authService.getAllHeaders()), // Use all headers (auth + chat ID)
    };

    let body: FormData | string;
    let contentType: string;

    if (images && images.length > 0) {
      // Send as form data with images
      const formData = new FormData();
      formData.append('content', message);

      images.forEach(image => {
        formData.append('images', image);
      });

      body = formData;
      contentType = 'multipart/form-data';
    } else {
      // Send as JSON without images
      const request: MessageRequest = {
        content: message,
      };

      body = JSON.stringify(request);
      contentType = 'application/json';
      headers['Content-Type'] = contentType;
    }

    const response = await fetch(`${this.aiServiceUrl}/widget`, {
      method: 'POST',
      headers: headers,
      body: body,
      credentials: 'omit',
    });

    // Convert headers to object for logging
    const responseHeaders: Record<string, string> = {};
    response.headers.forEach((value, key) => {
      responseHeaders[key] = value;
    });

    if (!response.ok) {
      const error: APIError = await response.json();
      throw new Error(error.error || `HTTP ${response.status}: ${response.statusText}`);
    }

    // Extract chat ID from response headers if present
    const chatId = response.headers.get('x-bcx-chat-id');

    if (chatId && !this.authService.getChatId()) {
      // Only set chat ID if we don't have one yet (first request)
      this.authService.setChatId(chatId);
    }

    // Return the response stream for streaming
    return response.body;
  }

  /**
   * Parse streaming response from AI service
   */
  async *parseStreamResponse(stream: ReadableStream<Uint8Array>): AsyncGenerator<{ type: string; content: string }, void, unknown> {
    const reader = stream.getReader();
    const decoder = new TextDecoder();
    let currentEventType = 'streaming_output'; // Default event type

    try {
      while (true) {
        const { done, value } = await reader.read();
        if (done) {
          break;
        }

        const chunk = decoder.decode(value, { stream: true });
        const lines = chunk.split('\n');

        for (const line of lines) {
          if (line.startsWith('event: ')) {
            currentEventType = line.slice(7).trim();
            continue;
          }

          if (line.startsWith('data: ')) {
            const data = line.slice(6);
            if (data === '[DONE]') {
              return;
            }
            try {
              const parsed = JSON.parse(data);
              if (parsed.content) {
                yield { type: currentEventType, content: parsed.content };
              }
            } catch {
              // Skip invalid JSON
              continue;
            }
          }
        }
      }
    } finally {
      reader.releaseLock();
    }
  }

  /**
   * Get paginated chat messages for a specific chat
   */
  async getChatMessages(chatId: string, page: number = 1, pageSize: number = 20): Promise<PaginatedMessagesResponse> {
    const token = await this.authService.getToken();
    if (!token) {
      throw new Error('No valid session token available');
    }

    const params = new URLSearchParams({
      chat_id: chatId,
      page: page.toString(),
      page_size: pageSize.toString(),
    });

    const response = await fetch(`${this.dbServiceUrl}/api/widgets/chat-messages/?${params.toString()}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        ...(await this.authService.getAuthHeader()),
      },
    });

    if (!response.ok) {
      const error: APIError = await response.json();
      throw new Error(error.error || `HTTP ${response.status}: ${response.statusText}`);
    }

    const responseData = await response.json();

    // Extract data from wrapped response structure {status, message, data: {...}}
    if (responseData.data && typeof responseData.data === 'object') {
      return responseData.data as PaginatedMessagesResponse;
    }

    // Fallback: if response is already in expected format, return it directly
    return responseData as PaginatedMessagesResponse;
  }

  /**
   * Get the auth service instance
   */
  getAuthService(): AuthService {
    return this.authService;
  }
}
