import { ThemeService } from '../theme.service';

describe('ThemeService', () => {
  let service: ThemeService;
  let hostElement: HTMLElement;

  beforeEach(() => {
    hostElement = document.createElement('div');
    service = new ThemeService(hostElement);

    // Reset document state
    document.documentElement.lang = '';
    document.body.textContent = '';
    document.head.innerHTML = '';
    document.body.className = '';
    document.documentElement.className = '';
    document.body.removeAttribute('style');
    document.documentElement.removeAttribute('style');

    // Mock window.matchMedia
    Object.defineProperty(window, 'matchMedia', {
      writable: true,
      value: jest.fn().mockImplementation(query => ({
        matches: false,
        media: query,
        onchange: null,
        addListener: jest.fn(), // Deprecated
        removeListener: jest.fn(), // Deprecated
        addEventListener: jest.fn(),
        removeEventListener: jest.fn(),
        dispatchEvent: jest.fn(),
      })),
    });

    // Mock navigator.language
    Object.defineProperty(navigator, 'language', {
      value: 'en-US',
      configurable: true,
      writable: true,
    });

    // Mock getComputedStyle
    window.getComputedStyle = jest.fn().mockReturnValue({
      getPropertyValue: jest.fn().mockReturnValue(''),
      backgroundColor: '',
      color: '',
    } as any);
  });

  describe('detectWebsiteLanguage', () => {
    it('should return "en" by default when no signals are present', async () => {
      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('en');
    });

    it('should detect "pl" from meta http-equiv="content-language"', async () => {
      const meta = document.createElement('meta');
      meta.setAttribute('http-equiv', 'content-language');
      meta.content = 'pl-PL';
      document.head.appendChild(meta);

      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('pl');
    });

    it('should detect "pl" from meta name="language"', async () => {
      const meta = document.createElement('meta');
      meta.name = 'language';
      meta.content = 'pl';
      document.head.appendChild(meta);

      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('pl');
    });

    it('should detect "pl" based on high frequency of Polish words', async () => {
      // Provide a text with Polish common words
      document.body.textContent = 'To jest strona o nas. Mamy kontakt i informacje.';
      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('pl');
    });

    it('should detect "pl" based on Polish diacritics', async () => {
      document.body.textContent = 'Zażółć gęślą jaźń. To bardzo ważne.';
      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('pl');
    });

    it('should detect "pl" from navigator.language fallback', async () => {
      // Mock navigator
      Object.defineProperty(navigator, 'language', {
        value: 'pl-PL',
        configurable: true,
        writable: true
      });

      // Ensure other methods fail so it falls back
      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('pl');
    });

    it('should prefer "en" if English words outnumber Polish words', async () => {
      document.body.textContent = 'This is a home page with news and contact info. Purely English content.';
      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('en');
    });

    it('should use Google Translate API as fallback if configured', async () => {
      // Mock fetch for Google Translate API
      global.fetch = jest.fn().mockResolvedValue({
        ok: true,
        json: async () => [[], [], 'pl'] // Mock response structure based on implementation
      } as any);

      // Provide ambiguous text that fails other checks but triggers API
      document.body.textContent = 'Unclear text content that fails heuristics';

      const lang = await service.detectWebsiteLanguage();
      expect(lang).toBe('pl');
    });
  });

  describe('detectWebsiteColorScheme', () => {
    it('should default to "light"', () => {
      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('light');
    });

    it('should detect "dark" from CSS variable --color-scheme', () => {
      window.getComputedStyle = jest.fn().mockReturnValue({
        getPropertyValue: (prop: string) => {
          if (prop === '--color-scheme') return 'dark';
          return '';
        },
        backgroundColor: '',
        color: ''
      } as any);

      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('dark');
    });

    it('should detect "dark" from data-theme attribute', () => {
      document.documentElement.setAttribute('data-theme', 'dark');
      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('dark');
    });

    it('should detect "dark" from class name', () => {
      document.body.classList.add('dark-mode');
      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('dark');
    });

    it('should detect "dark" from meta theme-color (if dark color)', () => {
      const meta = document.createElement('meta');
      meta.name = 'theme-color';
      meta.content = '#000000'; // Black is dark
      document.head.appendChild(meta);

      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('dark');
    });

    it('should detect "dark" from computed background color', () => {
      window.getComputedStyle = jest.fn().mockReturnValue({
        getPropertyValue: () => '',
        backgroundColor: 'rgb(0, 0, 0)', // Black bg
        color: 'rgb(255, 255, 255)' // White text
      } as any);

      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('dark');
    });

    it('should detect "light" from computed background color', () => {
      window.getComputedStyle = jest.fn().mockReturnValue({
        getPropertyValue: () => '',
        backgroundColor: 'rgb(255, 255, 255)', // White bg
        color: 'rgb(0, 0, 0)' // Black text
      } as any);

      const theme = service.getCurrentDetectedTheme();
      expect(theme).toBe('light');
    });
  });

  describe('watchWebsiteTheme', () => {
    it('should observe DOM changes', () => {
      const mockObserver = {
        observe: jest.fn(),
        disconnect: jest.fn()
      };

      window.MutationObserver = jest.fn().mockImplementation(() => mockObserver) as any;
      (global as any).MutationObserver = window.MutationObserver;

      const cleanup = service.watchWebsiteTheme(() => {});

      expect(window.MutationObserver).toHaveBeenCalled();
      expect(mockObserver.observe).toHaveBeenCalled();

      cleanup();
      expect(mockObserver.disconnect).toHaveBeenCalled();
    });
  });

  describe('applyConfig', () => {
    it('should apply CSS variables to host element', () => {
      const config = {
        light_mode: {
          primary_color: '#123456',
          secondary_color: '#654321',
          background_color: '#ffffff',
          text_color: '#000000'
        }
      };

      service.applyConfig(config as any, 'light');

      expect(hostElement.style.getPropertyValue('--bcx-primary')).toBe('#123456');
      expect(hostElement.style.getPropertyValue('--bcx-secondary')).toBe('#654321');
    });

    it('should apply dark mode config when resolved theme is dark', () => {
      const config = {
        dark_mode: {
          primary_color: '#000000',
          secondary_color: '#111111',
          background_color: '#222222',
          text_color: '#ffffff'
        }
      };

      service.applyConfig(config as any, 'dark');

      expect(hostElement.style.getPropertyValue('--bcx-primary')).toBe('#000000');
    });
  });
});

