import { ChatStorageService } from '../chat-storage.service';

describe('ChatStorageService', () => {
  const mockLocalStorage = (() => {
    let store: Record<string, string> = {};
    return {
      getItem: jest.fn((key: string) => store[key] || null),
      setItem: jest.fn((key: string, value: string) => {
        store[key] = value.toString();
      }),
      removeItem: jest.fn((key: string) => {
        delete store[key];
      }),
      clear: jest.fn(() => {
        store = {};
      }),
    };
  })();

  beforeAll(() => {
    Object.defineProperty(window, 'localStorage', {
      value: mockLocalStorage,
    });
  });

  beforeEach(() => {
    mockLocalStorage.clear();
    jest.clearAllMocks();
  });

  describe('saveChat', () => {
    it('should save a new chat', () => {
      ChatStorageService.saveChat('chat-1', 'Hello', '2023-01-01T10:00:00Z');

      const stored = JSON.parse(mockLocalStorage.getItem('bcx_chat_list') || '[]');
      expect(stored).toHaveLength(1);
      expect(stored[0]).toEqual({
        chatId: 'chat-1',
        lastMessage: 'Hello',
        lastMessageTimestamp: '2023-01-01T10:00:00Z'
      });
    });

    it('should update an existing chat', () => {
      // Initial save
      ChatStorageService.saveChat('chat-1', 'Hello', '2023-01-01T10:00:00Z');
      // Update
      ChatStorageService.saveChat('chat-1', 'World', '2023-01-01T10:05:00Z');

      const stored = JSON.parse(mockLocalStorage.getItem('bcx_chat_list') || '[]');
      expect(stored).toHaveLength(1);
      expect(stored[0]).toEqual({
        chatId: 'chat-1',
        lastMessage: 'World',
        lastMessageTimestamp: '2023-01-01T10:05:00Z'
      });
    });

    it('should push updated chat to the top of the list', () => {
      ChatStorageService.saveChat('chat-1', 'Msg 1', '2023-01-01T10:00:00Z');
      ChatStorageService.saveChat('chat-2', 'Msg 2', '2023-01-01T10:01:00Z'); // chat-2 is now top

      // Update chat-1
      ChatStorageService.saveChat('chat-1', 'Msg 1 Updated', '2023-01-01T10:02:00Z'); // chat-1 should move to top

      const stored = JSON.parse(mockLocalStorage.getItem('bcx_chat_list') || '[]');
      expect(stored[0].chatId).toBe('chat-1');
      expect(stored[1].chatId).toBe('chat-2');
    });
  });

  describe('getChats', () => {
    it('should return empty array if nothing stored', () => {
      const chats = ChatStorageService.getChats();
      expect(chats).toEqual([]);
    });

    it('should return stored chats', () => {
      const mockChats = [
        { chatId: 'c1', lastMessage: 'm1', lastMessageTimestamp: 't1' }
      ];
      mockLocalStorage.setItem('bcx_chat_list', JSON.stringify(mockChats));

      const chats = ChatStorageService.getChats();
      expect(chats).toEqual(mockChats);
    });
  });

  describe('removeChat', () => {
    it('should remove a specific chat', () => {
      ChatStorageService.saveChat('chat-1', 'm1', 't1');
      ChatStorageService.saveChat('chat-2', 'm2', 't2');

      ChatStorageService.removeChat('chat-1');

      const stored = ChatStorageService.getChats();
      expect(stored).toHaveLength(1);
      expect(stored[0].chatId).toBe('chat-2');
    });
  });

  describe('clearAll', () => {
    it('should remove all chats', () => {
      ChatStorageService.saveChat('chat-1', 'm1', 't1');
      ChatStorageService.clearAll();

      expect(mockLocalStorage.removeItem).toHaveBeenCalledWith('bcx_chat_list');
      expect(ChatStorageService.getChats()).toEqual([]);
    });
  });
});

