import { ApiService } from '../api.service';
import { AuthService } from '../auth.service';

// Mock AuthService
const mockAuthService = {
  getToken: jest.fn(),
  getAuthHeader: jest.fn(),
  getAllHeaders: jest.fn(),
  getChatId: jest.fn(),
  setChatId: jest.fn(),
} as unknown as AuthService;

describe('ApiService', () => {
  let service: ApiService;
  let fetchMock: jest.Mock;

  beforeEach(() => {
    fetchMock = jest.fn();
    global.fetch = fetchMock;
    service = new ApiService('http://db.url', 'http://widget.url', mockAuthService);
    jest.clearAllMocks();
  });

  describe('getWidgetConfig', () => {
    it('should fetch config successfully', async () => {
      (mockAuthService.getToken as jest.Mock).mockResolvedValue('fake-token');
      (mockAuthService.getAuthHeader as jest.Mock).mockResolvedValue({ Authorization: 'Bearer fake-token' });

      const mockResponse = { attrs: { light_mode: {} } };
      fetchMock.mockResolvedValue({
        ok: true,
        json: async () => mockResponse,
      });

      const result = await service.getWidgetConfig(123);

      expect(fetchMock).toHaveBeenCalledWith('http://db.url/api/widgets/org/123/widget-config/', expect.objectContaining({ method: 'GET' }));
      expect(result).toEqual(mockResponse);
    });

    it('should throw error if no token', async () => {
      (mockAuthService.getToken as jest.Mock).mockResolvedValue(undefined);

      await expect(service.getWidgetConfig(123)).rejects.toThrow('No valid session token available');
    });
  });

  describe('sendMessage', () => {
    it('should send message as JSON when no images', async () => {
      (mockAuthService.getToken as jest.Mock).mockResolvedValue('fake-token');
      (mockAuthService.getAllHeaders as jest.Mock).mockResolvedValue({ Authorization: 'Bearer fake-token' });

      const mockStream = new ReadableStream();
      fetchMock.mockResolvedValue({
        ok: true,
        headers: new Map(),
        body: mockStream,
      });

      const result = await service.sendMessage('Hello');

      expect(fetchMock).toHaveBeenCalledWith(
        'http://widget.url/widget',
        expect.objectContaining({
          method: 'POST',
          headers: expect.objectContaining({ 'Content-Type': 'application/json' }),
          body: JSON.stringify({ content: 'Hello' }),
          credentials: 'omit',
        }),
      );
      expect(result).toBe(mockStream);
    });

    it('should send message as FormData when images are present', async () => {
      (mockAuthService.getToken as jest.Mock).mockResolvedValue('fake-token');
      (mockAuthService.getAllHeaders as jest.Mock).mockResolvedValue({ Authorization: 'Bearer fake-token' });

      const mockStream = new ReadableStream();
      fetchMock.mockResolvedValue({
        ok: true,
        headers: new Map(),
        body: mockStream,
      });

      const file = new File([''], 'test.png', { type: 'image/png' });
      await service.sendMessage('Hello', [file]);

      // When sending FormData, we check if the body is FormData instance
      const callArgs = fetchMock.mock.calls[0][1];
      expect(callArgs.body).toBeInstanceOf(FormData);
      // Note: We don't manually set Content-Type for FormData, browser does it with boundary
      expect(callArgs.headers['Content-Type']).toBeUndefined();
    });
  });

  describe('getChatMessages', () => {
    it('should fetch paginated messages', async () => {
      (mockAuthService.getToken as jest.Mock).mockResolvedValue('fake-token');
      (mockAuthService.getAuthHeader as jest.Mock).mockResolvedValue({ Authorization: 'Bearer fake-token' });

      const mockData = {
        count: 10,
        results: [{ id: '1', content: 'test' }],
      };

      // Mock the wrapped response structure { data: ... } if that's what API returns,
      // OR direct response. Implementation handles both.
      fetchMock.mockResolvedValue({
        ok: true,
        json: async () => ({ data: mockData }),
      });

      const result = await service.getChatMessages('chat-123', 1, 20);

      expect(fetchMock).toHaveBeenCalledWith(expect.stringContaining('/api/widgets/chat-messages/?chat_id=chat-123'), expect.objectContaining({ method: 'GET' }));
      expect(result).toEqual(mockData);
    });
  });

  describe('parseStreamResponse', () => {
    it('should parse SSE events correctly', async () => {
      const encoder = new TextEncoder();
      const stream = new ReadableStream({
        start(controller) {
          controller.enqueue(encoder.encode('event: streaming_output\n'));
          controller.enqueue(encoder.encode('data: {"content": "Hello "}\n\n'));
          controller.enqueue(encoder.encode('event: streaming_output\n'));
          controller.enqueue(encoder.encode('data: {"content": "World"}\n\n'));
          controller.close();
        },
      });

      const generator = service.parseStreamResponse(stream);
      const results = [];

      for await (const chunk of generator) {
        results.push(chunk);
      }

      expect(results).toHaveLength(2);
      expect(results[0]).toEqual({ type: 'streaming_output', content: 'Hello ' });
      expect(results[1]).toEqual({ type: 'streaming_output', content: 'World' });
    });

    it('should handle tool events', async () => {
      const encoder = new TextEncoder();
      const stream = new ReadableStream({
        start(controller) {
          controller.enqueue(encoder.encode('event: tool\n'));
          controller.enqueue(encoder.encode('data: {"content": "tool_data"}\n\n'));
          controller.close();
        },
      });

      const generator = service.parseStreamResponse(stream);
      const results = [];

      for await (const chunk of generator) {
        results.push(chunk);
      }

      expect(results[0]).toEqual({ type: 'tool', content: 'tool_data' });
    });
  });
});
