import { Component, Prop, State, Element, Watch, Method, Event, EventEmitter, Host, h } from '@stencil/core';
import { AuthService } from '../../services/auth.service';
import { ApiService } from '../../services/api.service';
import { ThemeService } from '../../services/theme.service';
import { ChatStorageService } from '../../services/chat-storage.service';
import { WidgetState, WidgetEvent, ChatMessage, Product, PaginatedMessagesResponse, BackendMessage } from '../../types/api';
import { parseProducts, removeProductsFromText } from '../../utils/product-parser';
import dayjs from 'dayjs';
import relativeTime from 'dayjs/plugin/relativeTime';
import 'dayjs/locale/pl';
import 'dayjs/locale/en';

@Component({
  tag: 'bettercx-widget',
  styleUrl: 'bettercx-widget.scss',
  shadow: true,
})
export class BetterCXWidget {
  @Element() el: HTMLElement;

  // Simplified public properties - only essential ones
  @Prop() publicKey: string;
  @Prop() theme: 'light' | 'dark' | 'auto' = 'auto';
  @Prop() debug: boolean = false;
  @Prop() baseUrl: string = 'https://api.bettercx.ai';
  @Prop() aiServiceUrl: string = 'https://widget.bettercx.ai/prod';
  @Prop() autoInit: boolean = true;
  @Prop() position: 'left' | 'right' = 'right';
  @Prop() language: 'pl' | 'en' | 'auto' = 'auto';
  @Prop() embedded: boolean = false;
  @Prop() embeddedSize: 'full' | 'medium' | 'small' = 'full';
  @Prop() embeddedPlacement: 'top' | 'center' | 'bottom' = 'center';
  @Prop() isAttachmentsDisabled: boolean = false;

  // Internal state
  @State() state: WidgetState = {
    isOpen: false,
    isAuthenticated: false,
    isLoading: true,
    messages: [],
    isTyping: false,
    showPingMessage: false,
    currentPage: 1,
    hasNextPage: false,
    isLoadingMore: false,
  };

  // Language is now handled by @Prop() above
  private currentLanguage: 'pl' | 'en' = 'en';

  // Services
  private authService: AuthService;
  private apiService: ApiService;
  private themeService: ThemeService;

  // Refs
  private messagesContainerRef: HTMLDivElement;
  private messagesEndRef: HTMLDivElement; // Anchor element at the bottom for scrollIntoView

  // Store session colors for theme changes
  private sessionColors: { dark_mode?: Record<string, unknown>; light_mode?: Record<string, unknown> } | null = null;

  // Ping message handling
  private pingMessageTimeout: ReturnType<typeof setTimeout> | null = null;

  // Time update handling for message timestamps
  @State() private timeUpdateTrigger: number = 0;
  private timeUpdateInterval: ReturnType<typeof setInterval> | null = null;

  // Dropdown menu state
  @State() private isDropdownOpen: boolean = false;
  private dropdownRef: HTMLDivElement;

  // Fullscreen state
  @State() private isFullscreen: boolean = false;

  // Chat list state
  @State() private showChatList: boolean = false;

  // Track active chat load request to prevent race conditions
  private pendingLoadChatId: string | null = null;

  // FAQs animation state - track if FAQs have been shown to prevent re-animation
  private faqsHaveBeenShown: boolean = false;

  // Track programmatic scrolling to prevent handleScroll from triggering during auto-scroll
  private isProgrammaticScroll: boolean = false;

  // Delay message handling
  private delayMessageTimeout: ReturnType<typeof setTimeout> | null = null;
  @State() private showDelayMessage: boolean = false;
  @State() private delayMessageType: 'selecting' | 'thinking' = null; // 'selecting' = no AI messages, 'thinking' = has AI messages

  // Events
  @Event() widgetEvent: EventEmitter<WidgetEvent>;

  @Watch('publicKey')
  async onPublicKeyChange() {
    if (this.publicKey) {
      await this.initialize();
    }
  }

  @Watch('theme')
  async onThemeChange() {
    if (this.themeService) {
      this.themeService.setTheme(this.theme);
      // Reapply custom colors for the new theme
      this.applyCustomColorsFromSession();
      this.applyColorsToMessageComposer();
    }
  }

  @Watch('language')
  async onLanguageChange() {
    // Update language immediately if not auto
    if (this.language !== 'auto') {
      this.currentLanguage = this.language as 'pl' | 'en';
      dayjs.locale(this.currentLanguage === 'pl' ? 'pl' : 'en');
    } else if (this.themeService) {
      // If auto, detect language
      this.currentLanguage = await this.themeService.detectWebsiteLanguage();
      dayjs.locale(this.currentLanguage === 'pl' ? 'pl' : 'en');
    }
  }

  async componentWillLoad() {
    // Initialize dayjs with relativeTime plugin
    dayjs.extend(relativeTime);

    if (this.publicKey && this.autoInit) {
      await this.initialize();
    }
  }

  async componentDidLoad() {
    if (this.themeService) {
      // Only watch for website theme changes if theme prop is 'auto'
      if (this.theme === 'auto') {
        this.themeService.watchWebsiteTheme(_newTheme => {
          this.themeService.setDefaultTheme();
          // Reapply custom colors for the new theme
          this.applyCustomColorsFromSession();
          this.applyColorsToMessageComposer();
        });
      }
    }

    // Set up viewport handling for mobile browsers
    this.setupViewportHandling();

    // Start time update interval for message timestamps
    this.startTimeUpdateInterval();

    // Add click outside listener for dropdown
    document.addEventListener('click', this.handleClickOutside);
  }

  componentDidUpdate() {
    // Scroll restoration is handled naturally by the browser's scroll anchoring
    // or by specific logic in onMessagesChange for auto-scrolling to bottom
  }

  @Watch('state.messages')
  onMessagesChange() {
    // Auto-scroll to bottom when messages change (similar to useLayoutEffect in React)
    // Only scroll if user is near bottom (to avoid interrupting reading)
    // Skip during initial load (isLoading handles that)
    if (!this.state.isLoading && this.isNearBottom() && this.messagesEndRef) {
      requestAnimationFrame(() => {
        if (this.messagesEndRef && this.isNearBottom() && !this.isProgrammaticScroll) {
          this.isProgrammaticScroll = true;
          this.messagesEndRef.scrollIntoView({ behavior: 'auto', block: 'end', inline: 'nearest' });
          setTimeout(() => {
            this.isProgrammaticScroll = false;
          }, 50);
        }
      });
    }
  }

  disconnectedCallback() {
    // Clean up viewport listeners
    this.cleanupViewportHandling();

    // Clean up time update interval
    this.stopTimeUpdateInterval();

    // Remove click outside listener
    document.removeEventListener('click', this.handleClickOutside);

    // Clean up delay message timeout
    if (this.delayMessageTimeout) {
      clearTimeout(this.delayMessageTimeout);
      this.delayMessageTimeout = null;
    }
  }

  async initialize() {
    if (!this.publicKey) {
      this.setState({ isLoading: false, isAuthenticated: false });
      return;
    }

    try {
      this.setState({ isLoading: true, error: undefined });

      this.authService = new AuthService(this.baseUrl);
      const origin = window.location.origin;
      const sessionData = await this.authService.createSession(this.publicKey, origin);

      this.apiService = new ApiService(this.baseUrl, this.aiServiceUrl, this.authService);
      this.themeService = new ThemeService(this.el);

      // Set theme from prop (auto/light/dark)
      this.themeService.setTheme(this.theme);

      // Set language based on prop or auto-detect
      if (this.language === 'auto') {
        this.currentLanguage = await this.themeService.detectWebsiteLanguage();
      } else {
        this.currentLanguage = this.language as 'pl' | 'en';
      }

      // Set dayjs locale based on detected language
      dayjs.locale(this.currentLanguage === 'pl' ? 'pl' : 'en');

      this.themeService.setDefaultTheme();

      if ('attrs' in sessionData && sessionData.attrs) {
        this.sessionColors = sessionData.attrs as { dark_mode?: Record<string, unknown>; light_mode?: Record<string, unknown> };
        this.applyCustomColors(this.sessionColors);
        this.applyColorsToMessageComposer();
      }

      // Prepare basic session data (triggers, config, etc.)
      const welcomeMessage = (
        'welcome_message' in sessionData && sessionData.welcome_message ? sessionData.welcome_message : this.getTranslation('welcome_message_default')
      ) as string;
      const welcomeMessagePlacement = ('welcome_message_placement' in sessionData ? sessionData.welcome_message_placement : 'header') as 'header' | 'message';
      const triggerMessages = ('trigger_messages' in sessionData ? sessionData.trigger_messages : []) as Array<{
        id: number;
        url: string;
        message: string;
        is_default: boolean;
        trigger_after_seconds: number;
        created_at: string;
        updated_at: string;
      }>;

      const agentName = ('agent_name' in sessionData ? sessionData.agent_name : undefined) as string | undefined;
      const selectedTriggerMessage = this.selectTriggerMessage(triggerMessages);

      const commonStateUpdates: Partial<WidgetState> = {
        isAuthenticated: true,
        exampleQuestions: ('example_questions' in sessionData ? sessionData.example_questions : []) as Array<{
          id: number;
          question_text: string;
          answer_text: string;
          created_at: number;
        }>,
        title: ('title' in sessionData ? sessionData.title : undefined) as string | undefined,
        showPoweredByBetterCX: ('show_powered_by_bettercx' in sessionData ? sessionData.show_powered_by_bettercx : undefined) as boolean | undefined,
        logo: ('logo' in sessionData ? sessionData.logo : undefined) as string | undefined,
        welcomeMessage: welcomeMessage,
        welcomeMessagePlacement: welcomeMessagePlacement,
        triggerMessages: triggerMessages,
        selectedTriggerMessage: selectedTriggerMessage,
        agentName: agentName,
      };

      // --- OPTIMIZED SESSION RESTORATION LOGIC ---
      // Restore recent conversation (<24h) from localStorage for seamless UX
      // This is lightweight, synchronous, and requires no additional API calls

      const recentChats = ChatStorageService.getChats();
      const latestChat = recentChats[0];
      const ONE_DAY_MS = 24 * 60 * 60 * 1000;
      let shouldRestoreSession = false;

      // Validate and check if latest chat is recent (<24h)
      if (latestChat?.chatId && latestChat?.lastMessageTimestamp) {
        const lastActivityTime = new Date(latestChat.lastMessageTimestamp).getTime();
        const isRecent = !isNaN(lastActivityTime) && Date.now() - lastActivityTime < ONE_DAY_MS;
        const isValidChatId = typeof latestChat.chatId === 'string' && latestChat.chatId.trim().length > 0;

        if (isRecent && isValidChatId) {
          shouldRestoreSession = true;
        }
      }

      // Apply common state updates (config, theme, etc.) first
      this.setState({
        ...commonStateUpdates,
        messages: [], // Will be populated by loadChat or welcome message
      });

      if (shouldRestoreSession) {
        // SCENARIUSZ 1: Restore recent conversation (<24h)
        // loadChat() handles loading state, error handling, and race conditions
        await this.loadChat(latestChat.chatId);

        // Check if restoration failed (loadChat sets error state on failure)
        // If error occurred, gracefully fall back to welcome message
        if (this.state.error) {
          console.warn('[BetterCX] Chat restoration failed, falling back to welcome message');
          shouldRestoreSession = false; // Trigger fallback to welcome message
        }
        // If no error, loadChat succeeded - user sees restored conversation
        // No further action needed
      }

      // SCENARIUSZ 2: New session or restoration failed - show welcome message
      if (!shouldRestoreSession) {
        // Add welcome message as first message if placement is 'message'
        const initialMessages: ChatMessage[] = [];
        if (welcomeMessage && welcomeMessage.trim() && welcomeMessagePlacement === 'message') {
          initialMessages.push({
            id: 'welcome-' + Date.now(),
            content: welcomeMessage.trim(),
            author: 'assistant',
            timestamp: new Date().toISOString(),
            streamingFinished: true,
          });
        }

        this.setState({
          messages: initialMessages,
          isLoading: false,
          error: undefined, // Clear error to ensure clean welcome message display
        });

        // Start ping message timer only for new sessions
        if (selectedTriggerMessage?.message?.trim()) {
          this.startPingMessageTimer(selectedTriggerMessage);
        }
      }

      this.emitEvent('session-created', { origin });

      // Handle embedded/fullscreen logic
      if (this.embedded) {
        this.applyEmbeddedStyles();
        this.isFullscreen = true;
        this.setState({ isOpen: true });
      } else {
        this.setState({ isOpen: false });
      }
    } catch (error) {
      this.setState({
        isLoading: false,
        isAuthenticated: false,
      });
      this.emitEvent('error', { error: error.message });
    }
  }

  @Method()
  async open() {
    if (this.state.isAuthenticated) {
      this.setState({ isOpen: true, showPingMessage: false });
      this.emitEvent('opened');

      // Clear ping message timer when chat is opened
      this.clearPingMessageTimer();

      this.applyColorsToMessageComposer();

      // Scroll to bottom when opening widget (if there are messages)
      // Skip if only welcome message exists
      requestAnimationFrame(() => {
        if (this.state.messages.length === 0 || (this.state.messages.length === 1 && this.state.messages[0].id?.startsWith('welcome-'))) {
          return;
        }

        this.forceScrollToBottom();
      });
    }
  }

  @Method()
  async close() {
    // Exit fullscreen mode if active when closing widget
    if (this.isFullscreen && !this.embedded) {
      const widgetElement = this.el;
      widgetElement.classList.remove('bcx-widget--fullscreen');
      this.isFullscreen = false;
    }

    this.setState({ isOpen: false });
    this.emitEvent('closed');
  }

  @Method()
  async toggle() {
    if (this.state.isOpen) {
      await this.close();
    } else {
      await this.open();
    }
  }

  @Method()
  async sendMessage(content: string, images?: File[]) {
    if (!this.state.isAuthenticated || (!content.trim() && (!images || images.length === 0))) {
      return;
    }

    // Convert images to data URLs for display
    const imageDataUrls: string[] = [];
    if (images && images.length > 0) {
      for (const image of images) {
        const dataUrl = await this.fileToDataUrl(image);
        imageDataUrls.push(dataUrl);
      }
    }

    const userMessage: ChatMessage = {
      content: content.trim(),
      author: 'user',
      timestamp: new Date().toISOString(),
      id: this.generateId(),
      images: imageDataUrls.length > 0 ? imageDataUrls : undefined,
    };

    this.setState({
      messages: [...this.state.messages, userMessage],
      isTyping: true,
    });

    // Save chat to localStorage when user sends a message
    const chatId = this.authService.getChatId();
    if (chatId) {
      ChatStorageService.saveChat(chatId, content.trim(), new Date().toISOString());
    }

    // Force scroll to bottom after user sends message
    // User action = always scroll (they expect to see their message)
    this.forceScrollToBottom();

    this.emitEvent('message-sent', userMessage as unknown as Record<string, unknown>);

    // Check if there are any AI messages (excluding welcome message)
    const hasAiMessages = this.state.messages.some(msg => msg.author === 'assistant' && !msg.id?.startsWith('welcome-'));

    // Set delay message type based on whether there are AI messages
    const delayMessageType: 'selecting' | 'thinking' = hasAiMessages ? 'thinking' : 'selecting';

    // Clear any existing delay message timeout
    if (this.delayMessageTimeout) {
      clearTimeout(this.delayMessageTimeout);
      this.delayMessageTimeout = null;
    }

    // Show delay message after 5 seconds if streaming hasn't started
    this.delayMessageTimeout = setTimeout(() => {
      if (this.state.isTyping && !this.showDelayMessage) {
        this.showDelayMessage = true;
        this.delayMessageType = delayMessageType;
        this.setState({});
        // Scroll to bottom when delay message appears
        this.forceScrollToBottom();
      }
    }, 5000);

    try {
      const stream = await this.apiService.sendMessage(content, images);
      if (stream) {
        let assistantMessage: ChatMessage | null = null;
        let isStreamingStarted = false;
        const collectedProducts: Product[] = []; // Collect products before message creation

        const streamParser = await this.apiService.parseStreamResponse(stream);

        for await (const chunk of streamParser) {
          if (chunk.type === 'streaming_output') {
            if (!isStreamingStarted) {
              // Clear delay message when streaming starts
              if (this.delayMessageTimeout) {
                clearTimeout(this.delayMessageTimeout);
                this.delayMessageTimeout = null;
              }
              this.showDelayMessage = false;
              this.delayMessageType = null;

              assistantMessage = {
                content: '',
                author: 'assistant',
                timestamp: new Date().toISOString(),
                id: this.generateId(),
                products: [...collectedProducts], // Add any collected products
                streamingFinished: false,
              };

              this.setState({
                messages: [...this.state.messages, assistantMessage],
                isTyping: false,
              });
              isStreamingStarted = true;
            }

            if (assistantMessage) {
              // Just accumulate content during streaming
              assistantMessage.content += chunk.content;

              this.setState({
                messages: [...this.state.messages.slice(0, -1), { ...assistantMessage }],
              });

              // Auto-scroll during streaming if user is near bottom
              // This provides real-time feedback as message streams in
              this.scrollToBottomIfNeeded(true);
            }
          } else if (chunk.type === 'tool') {
            // Handle tool events, specifically display_product
            const toolData = this.parseToolData(chunk.content);
            if (toolData && toolData.name === 'display_product' && toolData.data) {
              if (!isStreamingStarted) {
                // Collect products but don't create message yet - wait for streaming_output
                collectedProducts.push(toolData.data as Product);
                this.setState({ isTyping: true }); // Show typing indicator
              } else if (assistantMessage) {
                // Add product to existing message
                if (!assistantMessage.products) {
                  assistantMessage.products = [];
                }
                assistantMessage.products.push(toolData.data as Product);

                this.setState({
                  messages: [...this.state.messages.slice(0, -1), { ...assistantMessage }],
                });

                // Scroll after product is added (product slider changes message height)
                this.scrollToBottomIfNeeded(true);
              }
            }
          } else {
            if (!isStreamingStarted) {
              this.setState({ isTyping: true });
            }
          }
        }

        // After streaming is complete, parse products from full content
        if (assistantMessage) {
          // Parse products from the complete content
          const parsedProducts = parseProducts(assistantMessage.content);

          if (parsedProducts.length > 0) {
            if (!assistantMessage.products) {
              assistantMessage.products = [];
            }
            // Convert ProductInfo to Product format and add to message
            const products = parsedProducts.map(p => ({
              product_name: p.product_name,
              image_url: p.image_url,
              product_url: p.product_url,
            }));
            assistantMessage.products = [...assistantMessage.products, ...products];

            // Remove product markdown from content
            assistantMessage.content = removeProductsFromText(assistantMessage.content);
          }
        }

        // Mark streaming as finished
        if (assistantMessage) {
          assistantMessage.streamingFinished = true;
          this.setState({
            messages: [...this.state.messages.slice(0, -1), { ...assistantMessage }],
          });

          // Ensure we're scrolled to bottom after streaming completes
          // Use slight delay to allow product slider to render if present
          requestAnimationFrame(() => {
            requestAnimationFrame(() => {
              this.scrollToBottomIfNeeded(true);
            });
          });
        }

        if (assistantMessage) {
          this.emitEvent('message-received', assistantMessage as unknown as Record<string, unknown>);
        }

        // Update chat ID in state after first message
        const chatId = this.authService.getChatId();
        if (chatId) {
          this.setState({ chatId });
          // Save chat to localStorage with last message
          const lastMessage = assistantMessage?.content || userMessage.content;
          ChatStorageService.saveChat(chatId, lastMessage, new Date().toISOString());
        }
      }
    } catch (error) {
      this.setState({ error: 'Failed to send message' });
      this.emitEvent('error', { error: error.message });
    } finally {
      // Clear delay message timeout on error or completion
      if (this.delayMessageTimeout) {
        clearTimeout(this.delayMessageTimeout);
        this.delayMessageTimeout = null;
      }
      this.showDelayMessage = false;
      this.delayMessageType = null;
      this.setState({ isTyping: false });
    }
  }

  private mapBackendMessageToChatMessage(msg: BackendMessage): ChatMessage {
    try {
      const chatMsg: ChatMessage = {
        id: msg.id,
        content: msg.content,
        author: msg.author === 'user' ? 'user' : 'assistant',
        timestamp: msg.created_at,
        products: [],
        streamingFinished: true,
        images: msg.attachments?.map(att => att.file_path),
      };

      // Only parse products for assistant messages to save resources
      if (chatMsg.author === 'assistant' && msg.content) {
        chatMsg.products = parseProducts(msg.content);
        chatMsg.content = removeProductsFromText(msg.content);
      }

      return chatMsg;
    } catch (err) {
      console.error('[BetterCX] Error mapping message:', err, msg);
      // Return a fallback or minimal message in case of mapping error to prevent crash
      return {
        id: msg.id || Math.random().toString(36),
        content: msg.content || 'Error displaying message',
        author: 'assistant',
        timestamp: new Date().toISOString(),
        streamingFinished: true,
      } as ChatMessage;
    }
  }

  /**
   * Load a chat conversation into the main widget
   * Implements safe loading with race condition checks
   */
  private async loadChat(chatId: string) {
    if (!chatId) return;

    // 1. Set pending request ID to track this specific load operation
    this.pendingLoadChatId = chatId;

    // 2. Clear current state immediately to avoid showing "stale" data from previous chat
    //    while the new one is loading. This provides better UX feedback.
    this.setState({
      isLoading: true,
      messages: [], // Clear messages immediately
      isTyping: false, // Reset typing indicator
      error: undefined, // Clear any previous errors
      currentPage: 1,
      hasNextPage: false,
      isLoadingMore: false,
    });

    // Hide chat list immediately to show the loading state in the main chat view
    this.showChatList = false;

    try {
      // Load latest messages (page 1)
      const response: PaginatedMessagesResponse = await this.apiService.getChatMessages(chatId, 1, 20);

      // 3. Race Condition Check:
      //    If the user clicked another chat while this request was pending,
      //    pendingLoadChatId will have changed. In that case, ignore this result.
      if (this.pendingLoadChatId !== chatId) {
        console.warn(`[BetterCX] Ignoring stale response for chat ${chatId} (current: ${this.pendingLoadChatId})`);
        return;
      }

      // Map BackendMessage to ChatMessage safely
      // Backend returns messages from oldest to newest, so we reverse to show newest at bottom
      const messages: ChatMessage[] = response.results.map(msg => this.mapBackendMessageToChatMessage(msg)).reverse();

      // 4. Update State only if we are still on the same chat request
      if (this.pendingLoadChatId === chatId) {
        // Skip auto-scroll in setState - we'll handle it manually after DOM is fully rendered
        this.setState(
          {
            messages: messages,
            chatId: chatId,
            isLoading: false,
            currentPage: 1,
            hasNextPage: response.has_next,
          },
          true, // skipAutoScroll = true
        );

        this.authService.setChatId(chatId);

        // Force scroll to bottom after initial load
        // Use multiple RAF cycles to ensure DOM is fully rendered (Stencil + images + product sliders)
        // Inspired by Frontend ChatMessagesList useLayoutEffect pattern
        requestAnimationFrame(() => {
          requestAnimationFrame(() => {
            requestAnimationFrame(() => {
              if (this.pendingLoadChatId === null) {
                // Only scroll if we're still on the same chat (no race condition)
                this.forceScrollToBottom();

                // Auto-fetch more content if container is not full (inspired by Frontend)
                // This ensures we fill the viewport with messages if first page is too short
                if (this.messagesContainerRef && response.has_next) {
                  const { scrollHeight, clientHeight } = this.messagesContainerRef;
                  const isContentShorterThanViewport = scrollHeight <= clientHeight;

                  if (isContentShorterThanViewport && !this.state.isLoadingMore) {
                    // Load next page to fill viewport
                    this.loadMoreMessages();
                  }
                }
              }
            });
          });
        });
      }
    } catch (error) {
      console.error('Failed to load chat:', error);

      // Only show error if this is still the active request
      if (this.pendingLoadChatId === chatId) {
        this.setState({
          isLoading: false,
          error: this.language === 'pl' ? 'Nie udało się załadować rozmowy.' : 'Failed to load conversation.',
        });
      }
    } finally {
      // Reset pending ID if this request finished (whether success or fail)
      // and it matches the current pending one
      if (this.pendingLoadChatId === chatId) {
        this.pendingLoadChatId = null;
      }
    }
  }

  private handleScroll = () => {
    // Ignore scroll events during programmatic scrolling (auto-scroll, initial load)
    if (this.isProgrammaticScroll) {
      return;
    }

    // Don't trigger pagination during initial load or if already loading more
    if (this.state.isLoading || this.state.isLoadingMore) {
      return;
    }

    // Check if scrolled to top
    // Using 100px threshold to match React implementation 1:1
    if (this.messagesContainerRef && this.messagesContainerRef.scrollTop <= 100 && this.state.hasNextPage) {
      this.loadMoreMessages();
    }
  };

  private async loadMoreMessages() {
    if (this.state.isLoadingMore || !this.state.hasNextPage || !this.state.chatId) {
      return;
    }

    this.setState({ isLoadingMore: true });

    try {
      const nextPage = (this.state.currentPage || 1) + 1;
      const response = await this.apiService.getChatMessages(this.state.chatId, nextPage, 20);

      const newMessages = response.results.map(msg => this.mapBackendMessageToChatMessage(msg)).reverse();

      this.setState(
        {
          messages: [...newMessages, ...this.state.messages],
          currentPage: nextPage,
          hasNextPage: response.has_next,
          isLoadingMore: false,
        },
        true, // skipAutoScroll
      );
    } catch (error) {
      console.error('Failed to load more messages:', error);
      this.setState({ isLoadingMore: false });
    }
  }

  private setState(updates: Partial<WidgetState>, skipAutoScroll: boolean = false) {
    const previousMessages = this.state.messages;
    this.state = { ...this.state, ...updates };

    // Smart auto-scroll: only scroll if user is near bottom
    // Prevents interrupting user if they're reading older messages
    if (!skipAutoScroll && updates.messages && updates.messages !== previousMessages) {
      // Use RAF for smooth, performant scrolling
      requestAnimationFrame(() => {
        this.scrollToBottomIfNeeded(true);
      });
    }
  }

  private emitEvent(type: WidgetEvent['type'], data?: Record<string, unknown>) {
    const event: WidgetEvent = {
      type,
      data,
      timestamp: new Date().toISOString(),
    };
    this.widgetEvent.emit(event);
  }

  private generateId(): string {
    return Math.random().toString(36).substr(2, 9);
  }

  /**
   * Parse tool data from streaming response
   */
  private parseToolData(content: string): { name: string; data: unknown } | null {
    try {
      const parsed = JSON.parse(content);
      return parsed;
    } catch (error) {
      console.warn('Failed to parse tool data:', content, error);
      return null;
    }
  }

  private fileToDataUrl(file: File): Promise<string> {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onload = e => resolve(e.target?.result as string);
      reader.onerror = reject;
      reader.readAsDataURL(file);
    });
  }

  private getTranslation(key: string): string {
    const translations = {
      common_questions: {
        en: 'Common questions',
        pl: 'Często zadawane pytania',
      },
      new_conversation: {
        en: 'New conversation',
        pl: 'Nowa konwersacja',
      },
      frequently_asked_questions: {
        en: 'Frequently Asked Questions',
        pl: 'Często zadawane pytania',
      },
      message_placeholder: {
        en: 'Type your message...',
        pl: 'Wpisz swoją wiadomość...',
      },
      terms_agreement_start: {
        en: 'By using this chat, you agree to ',
        pl: 'Korzystając z tego chatu, akceptujesz ',
      },
      powered_by: {
        en: 'Powered by ',
        pl: 'Napędzane przez ',
      },
      terms_of_service: {
        en: 'Terms of Service',
        pl: 'Warunki korzystania',
      },
      privacy_policy: {
        en: 'Privacy Policy',
        pl: 'Politykę prywatności',
      },
      and: {
        en: ' and ',
        pl: ' i ',
      },
      start_chat: {
        en: 'Start Chat',
        pl: 'Rozpocznij czat',
      },
      author_assistant: {
        en: 'Assistant',
        pl: 'Asystent',
      },
      author_user: {
        en: 'You',
        pl: 'Ty',
      },
      menu_fullscreen: {
        en: 'Full screen',
        pl: 'Pełny ekran',
      },
      menu_minimize: {
        en: 'Minimize',
        pl: 'Minimalizuj',
      },
      menu_privacy_policy: {
        en: 'Privacy Policy',
        pl: 'Polityka prywatności',
      },
      menu_all_conversations: {
        en: 'All conversations',
        pl: 'Wszystkie konwersacje',
      },
      menu_close: {
        en: 'Close',
        pl: 'Zamknij',
      },
      welcome_message_default: {
        en: 'How can I help you today?',
        pl: 'Jak mogę Ci dziś pomóc?',
      },
      instant_response: {
        en: 'Instant response',
        pl: 'Natychmiastowa odpowiedź',
      },
      selecting_assistant: {
        en: 'Connecting to assistant',
        pl: 'Łączenie z asystentem',
      },
      thinking: {
        en: 'Working on the answer',
        pl: 'Pracuję nad odpowiedzią',
      },
    };

    return translations[key]?.[this.currentLanguage] || translations[key]?.['en'] || key;
  }

  private parseLinks(text: string): string {
    // Parse markdown-style links: [text](url)
    const linkRegex = /\[([^\]]+)\]\(([^)]+)\)/g;

    return text.replace(linkRegex, (match, linkText, url) => {
      // Validate URL format
      let validUrl = url.trim();

      // Add protocol if missing
      if (!validUrl.match(/^https?:\/\//i)) {
        validUrl = 'https://' + validUrl;
      }

      // Basic URL validation
      try {
        new URL(validUrl);
        return `<a href="${validUrl}" target="_blank" rel="noopener noreferrer" class="bcx-widget__message-link">${linkText}</a>`;
      } catch {
        // If URL is invalid, return original text
        return match;
      }
    });
  }

  private formatMessageTime(timestamp: string): string {
    // Determine locale based on current language
    const locale = this.currentLanguage === 'pl' ? 'pl' : 'en';

    // Use dayjs with locale directly on the instance
    // dayjs automatically handles locale loading if imported
    return dayjs(timestamp).locale(locale).fromNow();
  }

  /**
   * Check if user is near the bottom of the messages container
   * Used to determine if we should auto-scroll on new messages
   * Threshold: 150px from bottom (matches Frontend implementation)
   */
  private isNearBottom(threshold: number = 150): boolean {
    if (!this.messagesContainerRef) {
      return true; // If container doesn't exist, assume we should scroll
    }

    const { scrollTop, scrollHeight, clientHeight } = this.messagesContainerRef;
    const distanceFromBottom = scrollHeight - scrollTop - clientHeight;
    return distanceFromBottom < threshold;
  }

  /**
   * Scroll to bottom using scrollIntoView on messagesEndRef
   * This is more reliable than scrollTop as it handles all edge cases
   * Inspired by Frontend ChatMessagesList implementation
   */
  private scrollToBottom(smooth: boolean = true) {
    if (this.messagesEndRef) {
      this.isProgrammaticScroll = true;
      this.messagesEndRef.scrollIntoView({
        behavior: smooth ? 'smooth' : 'auto',
        block: 'end',
        inline: 'nearest',
      });
      // Reset flag after scroll completes (smooth scroll takes ~300-500ms)
      setTimeout(
        () => {
          this.isProgrammaticScroll = false;
        },
        smooth ? 600 : 50,
      );
    }
  }

  /**
   * Smart scroll: only scrolls if user is near bottom
   * Prevents interrupting user if they're reading older messages
   * Used during message streaming and updates
   */
  private scrollToBottomIfNeeded(smooth: boolean = true) {
    if (this.isNearBottom()) {
      // Use requestAnimationFrame for smooth, performant scrolling
      requestAnimationFrame(() => {
        this.scrollToBottom(smooth);
      });
    }
  }

  /**
   * Force scroll to bottom (used for initial load, user actions)
   * Uses scrollIntoView which is more reliable than scrollTop
   * Inspired by Frontend ChatMessagesList useLayoutEffect pattern
   */
  private forceScrollToBottom() {
    if (!this.messagesEndRef) return;

    // Use requestAnimationFrame to ensure DOM is updated
    // Similar to useLayoutEffect in React - runs synchronously after DOM mutations
    requestAnimationFrame(() => {
      this.isProgrammaticScroll = true;
      if (this.messagesEndRef) {
        this.messagesEndRef.scrollIntoView({
          behavior: 'auto', // Instant scroll for initial load
          block: 'end',
          inline: 'nearest',
        });
      }
      // Reset flag after instant scroll
      setTimeout(() => {
        this.isProgrammaticScroll = false;
      }, 100);
    });
  }

  private handleToggleClick = () => {
    // Don't allow closing in embedded mode
    if (this.embedded) {
      return;
    }

    if (this.state.isOpen) {
      this.close();
    } else {
      this.open();
    }
  };

  private handleMessageSubmit = (event: CustomEvent<{ content: string; images: File[] }>) => {
    this.sendMessage(event.detail.content, event.detail.images);
  };

  /**
   * Apply custom colors to the widget element only (Shadow DOM isolation)
   * This prevents conflicts with other page elements that might override document root properties
   */
  private applyCustomColors(attrs: { dark_mode?: Record<string, unknown>; light_mode?: Record<string, unknown> }) {
    try {
      const currentTheme = this.themeService.getCurrentTheme();
      const colorMode = currentTheme === 'dark' ? attrs.dark_mode : attrs.light_mode;

      if (colorMode && typeof colorMode === 'object') {
        // Only set CSS custom properties on the widget element (Shadow DOM isolation)
        // This prevents conflicts with other page elements that might override document root properties
        this.el.style.setProperty('--bcx-primary', String(colorMode.primary_color || ''));
        this.el.style.setProperty('--bcx-secondary', String(colorMode.secondary_color || ''));
        this.el.style.setProperty('--bcx-background', String(colorMode.background_color || ''));
        this.el.style.setProperty('--bcx-text', String(colorMode.text_color || ''));
      } else {
        console.warn('[Widget] No valid color mode found for theme:', currentTheme);
      }
    } catch (error) {
      console.error('[Widget] Error applying custom colors:', error);
    }
  }

  private applyCustomColorsFromSession() {
    if (this.sessionColors) {
      this.applyCustomColors(this.sessionColors);
    } else {
      console.warn('[Widget] No session colors available to apply');
    }
  }

  private applyColorsToMessageComposer() {
    setTimeout(() => {
      const messageComposer = this.el.querySelector('bcx-message-composer');
      if (messageComposer) {
        const primaryColor = this.el.style.getPropertyValue('--bcx-primary') || '#007bff';
        const secondaryColor = this.el.style.getPropertyValue('--bcx-secondary') || '#6c757d';
        const backgroundColor = this.el.style.getPropertyValue('--bcx-background') || '#ffffff';
        const textColor = this.el.style.getPropertyValue('--bcx-text') || '#212529';

        messageComposer.style.setProperty('--bcx-primary', primaryColor);
        messageComposer.style.setProperty('--bcx-secondary', secondaryColor);
        messageComposer.style.setProperty('--bcx-background', backgroundColor);
        messageComposer.style.setProperty('--bcx-text', textColor);
      }
    }, 100);
  }

  private handleExampleQuestionClick = (question: { id: number; question_text: string; answer_text: string; created_at: number }) => {
    const userMessage: ChatMessage = {
      content: question.question_text,
      author: 'user',
      timestamp: new Date().toISOString(),
      id: this.generateId(),
    };

    this.setState({
      messages: [...this.state.messages, userMessage],
    });

    this.simulateTypingResponse(question.answer_text);
  };

  private simulateTypingResponse(fullText: string) {
    const assistantMessage: ChatMessage = {
      content: '',
      author: 'assistant',
      timestamp: new Date().toISOString(),
      id: this.generateId(),
    };

    this.setState({
      messages: [...this.state.messages, assistantMessage],
      isTyping: false,
    });

    let currentText = '';
    let index = 0;
    const typingSpeed = 25;

    const typeNextCharacter = () => {
      if (index < fullText.length) {
        currentText += fullText[index];
        index++;

        const updatedMessage = { ...assistantMessage, content: currentText };
        const updatedMessages = [...this.state.messages];
        updatedMessages[updatedMessages.length - 1] = updatedMessage;

        this.setState({
          messages: updatedMessages,
        });

        setTimeout(typeNextCharacter, typingSpeed);
      } else {
        this.emitEvent('message-received', assistantMessage as unknown as Record<string, unknown>);
      }
    };

    setTimeout(typeNextCharacter, 200);
  }

  render() {
    if (!this.state.isAuthenticated) {
      return null;
    }

    return (
      <Host
        class={`bcx-widget ${this.themeService.getCurrentTheme() === 'dark' ? 'dark' : ''} ${this.state.isOpen ? 'bcx-widget--open' : ''} bcx-widget--${this.position}`}
        data-adblock-bypass="true"
        data-role="widget"
        data-widget-type="customer-service"
        aria-label="Customer service chat widget"
        data-tick={this.timeUpdateTrigger}
      >
        {/* Ping Message */}
        {this.state.showPingMessage && this.state.selectedTriggerMessage && this.state.selectedTriggerMessage.message && (
          <div class="bcx-widget__ping-message" data-adblock-bypass="true">
            <div class="bcx-widget__ping-content">
              <div class="bcx-widget__ping-text-wrapper">
                <div class="bcx-widget__ping-text">
                  <div class="bcx-widget__ping-message-text">{this.state.selectedTriggerMessage.message}</div>
                </div>
                <button
                  class="bcx-widget__ping-close"
                  onClick={e => {
                    e.preventDefault();
                    e.stopPropagation();
                    this.handlePingMessageClose();
                  }}
                  onTouchEnd={e => {
                    e.preventDefault();
                    e.stopPropagation();
                    this.handlePingMessageClose();
                  }}
                  aria-label="Close ping message"
                  data-adblock-bypass="true"
                  type="button"
                >
                  <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="18" y1="6" x2="6" y2="18"></line>
                    <line x1="6" y1="6" x2="18" y2="18"></line>
                  </svg>
                </button>
              </div>
              <div class="bcx-widget__ping-header">
                <div class="bcx-widget__ping-status">
                  <span class="bcx-widget__ping-status-dot"></span>
                  <span class="bcx-widget__ping-status-text">Online</span>
                </div>
              </div>
            </div>
            <button class="bcx-widget__ping-action" onClick={this.handlePingMessageClick} aria-label="Open chat" data-adblock-bypass="true">
              <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true">
                <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm0 14H6l-2 2V4h16v12z"></path>
              </svg>
              {this.getTranslation('start_chat')}
            </button>
          </div>
        )}

        {/* Toggle Button */}
        {!this.embedded && (
          <button
            class="bcx-widget__toggle"
            onClick={this.handleToggleClick}
            aria-label={this.state.isOpen ? 'Close chat' : 'Open chat'}
            aria-expanded={this.state.isOpen}
            aria-controls="bcx-widget-chat"
            data-adblock-bypass="true"
          >
            <span class="bcx-widget__toggle-icon" aria-hidden="true">
              {this.state.isOpen ? (
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                  <line x1="18" y1="6" x2="6" y2="18"></line>
                  <line x1="6" y1="6" x2="18" y2="18"></line>
                </svg>
              ) : (
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                  <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                </svg>
              )}
            </span>
          </button>
        )}

        {/* Chat List Overlay */}
        {this.showChatList && this.state.isOpen && (
          <div class="bcx-widget__chat-list-overlay" data-adblock-bypass="true">
            <bcx-chat-list
              apiService={this.apiService}
              language={this.currentLanguage}
              theme={this.themeService.getCurrentTheme()}
              onChatSelected={(e: CustomEvent<string>) => {
                const chatId = e.detail;
                this.loadChat(chatId);
              }}
              onClose={() => {
                this.showChatList = false;
              }}
            />
          </div>
        )}

        {/* Chat Interface */}
        {this.state.isOpen && !this.showChatList && (
          <div id="bcx-widget-chat" class="bcx-widget__chat" role="dialog" aria-labelledby="bcx-widget-title" aria-describedby="bcx-widget-description" data-adblock-bypass="true">
            <div
              class={`bcx-widget__header ${
                this.state.messages.length === 0 || (this.state.messages.length === 1 && this.state.messages[0].id?.startsWith('welcome-'))
                  ? 'bcx-widget__header--initial'
                  : 'bcx-widget__header--compact'
              }`}
            >
              <div class="bcx-widget__header-body">
                <div class="bcx-widget__header-content">
                  <div class="bcx-widget__header-title">
                    {this.state.logo && (
                      <div class="bcx-widget__header-avatar">
                        <img
                          src={this.state.logo}
                          alt="Assistant Avatar"
                          class="bcx-widget__avatar-img"
                          onError={e => {
                            (e.target as HTMLImageElement).style.display = 'none';
                          }}
                        />
                        <div class="bcx-widget__online-indicator" aria-label="Online"></div>
                      </div>
                    )}
                    <h3 id="bcx-widget-title">{this.state.title || 'Chat AI'}</h3>
                  </div>
                  <div class="bcx-widget__header-extra">
                    {this.state.welcomeMessagePlacement === 'header' && <p class="bcx-widget__header-description">{this.state.welcomeMessage}</p>}
                    <p class="bcx-widget__header-subdescription">{this.getTranslation('instant_response')}</p>
                  </div>
                </div>
                <div class="bcx-widget__dropdown" ref={el => (this.dropdownRef = el)} data-adblock-bypass="true">
                  <button
                    class="bcx-widget__dropdown-toggle"
                    onClick={this.toggleDropdown}
                    aria-label="Menu"
                    aria-expanded={this.isDropdownOpen}
                    aria-haspopup="true"
                    data-adblock-bypass="true"
                  >
                    <svg
                      width="18"
                      height="18"
                      viewBox="0 0 24 24"
                      fill="none"
                      stroke="currentColor"
                      stroke-width="2"
                      stroke-linecap="round"
                      stroke-linejoin="round"
                      aria-hidden="true"
                    >
                      <circle cx="12" cy="12" r="1"></circle>
                      <circle cx="19" cy="12" r="1"></circle>
                      <circle cx="5" cy="12" r="1"></circle>
                    </svg>
                  </button>
                  {this.isDropdownOpen && (
                    <div class="bcx-widget__dropdown-menu" role="menu" data-adblock-bypass="true">
                      {this.isDesktop() && !this.embedded && (
                        <button class="bcx-widget__dropdown-item" role="menuitem" onClick={e => this.handleDropdownItemClick('fullscreen', e)} data-adblock-bypass="true">
                          {this.isFullscreen ? (
                            <svg
                              width="16"
                              height="16"
                              viewBox="0 0 24 24"
                              fill="none"
                              stroke="currentColor"
                              stroke-width="2"
                              stroke-linecap="round"
                              stroke-linejoin="round"
                              aria-hidden="true"
                            >
                              <path d="M8 3v3a2 2 0 0 1-2 2H3m18 0h-3a2 2 0 0 1-2-2V3m0 18v-3a2 2 0 0 1 2-2h3M3 16h3a2 2 0 0 1 2 2v3"></path>
                            </svg>
                          ) : (
                            <svg
                              width="16"
                              height="16"
                              viewBox="0 0 24 24"
                              fill="none"
                              stroke="currentColor"
                              stroke-width="2"
                              stroke-linecap="round"
                              stroke-linejoin="round"
                              aria-hidden="true"
                            >
                              <path d="M8 3H5a2 2 0 0 0-2 2v3m18 0V5a2 2 0 0 0-2-2h-3m0 18h3a2 2 0 0 0 2-2v-3M3 16v3a2 2 0 0 0 2 2h3"></path>
                            </svg>
                          )}
                          <span>{this.isFullscreen ? this.getTranslation('menu_minimize') : this.getTranslation('menu_fullscreen')}</span>
                        </button>
                      )}
                      <a
                        href="https://bettercx.ai/privacy"
                        target="_blank"
                        rel="noopener noreferrer nofollow"
                        class="bcx-widget__dropdown-item"
                        role="menuitem"
                        onClick={e => this.handleDropdownItemClick('privacy', e)}
                        data-adblock-bypass="true"
                      >
                        <svg
                          width="16"
                          height="16"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          stroke-width="2"
                          stroke-linecap="round"
                          stroke-linejoin="round"
                          aria-hidden="true"
                        >
                          <path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"></path>
                        </svg>
                        <span>{this.getTranslation('menu_privacy_policy')}</span>
                      </a>
                      <button class="bcx-widget__dropdown-item" role="menuitem" onClick={e => this.handleDropdownItemClick('new_conversation', e)} data-adblock-bypass="true">
                        <svg
                          width="16"
                          height="16"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          stroke-width="2"
                          stroke-linecap="round"
                          stroke-linejoin="round"
                          aria-hidden="true"
                        >
                          <line x1="12" y1="5" x2="12" y2="19"></line>
                          <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                        <span>{this.getTranslation('new_conversation')}</span>
                      </button>
                      <button class="bcx-widget__dropdown-item" role="menuitem" onClick={e => this.handleDropdownItemClick('conversations', e)} data-adblock-bypass="true">
                        <svg
                          width="16"
                          height="16"
                          viewBox="0 0 24 24"
                          fill="none"
                          stroke="currentColor"
                          stroke-width="2"
                          stroke-linecap="round"
                          stroke-linejoin="round"
                          aria-hidden="true"
                        >
                          <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                        </svg>
                        <span>{this.getTranslation('menu_all_conversations')}</span>
                      </button>
                      {!this.embedded && (
                        <button
                          class="bcx-widget__dropdown-item bcx-widget__dropdown-item--danger"
                          role="menuitem"
                          onClick={e => this.handleDropdownItemClick('close', e)}
                          data-adblock-bypass="true"
                        >
                          <svg
                            width="16"
                            height="16"
                            viewBox="0 0 24 24"
                            fill="none"
                            stroke="currentColor"
                            stroke-width="2"
                            stroke-linecap="round"
                            stroke-linejoin="round"
                            aria-hidden="true"
                          >
                            <line x1="18" y1="6" x2="6" y2="18"></line>
                            <line x1="6" y1="6" x2="18" y2="18"></line>
                          </svg>
                          <span>{this.getTranslation('menu_close')}</span>
                        </button>
                      )}
                    </div>
                  )}
                </div>
              </div>
            </div>

            <div
              class="bcx-widget__messages"
              ref={el => (this.messagesContainerRef = el)}
              onScroll={this.handleScroll}
              role="log"
              aria-live="polite"
              aria-label="Chat messages"
              data-adblock-bypass="true"
            >
              {this.state.isLoading && (
                <div class="bcx-widget__loading-container">
                  <div class="bcx-widget__spinner"></div>
                </div>
              )}
              {this.state.isLoadingMore && (
                <div class="bcx-widget__loading-more">
                  <div class="bcx-widget__spinner bcx-widget__spinner--small"></div>
                </div>
              )}
              {this.state.messages.map(message => (
                <div
                  key={message.id || Math.random().toString(36)}
                  class={`bcx-widget__message bcx-widget__message--${message.author}`}
                  role="article"
                  aria-label={`Message from ${message.author}`}
                  data-adblock-bypass="true"
                  data-message-id={message.id || ''}
                >
                  {message.author === 'assistant' && this.state.logo && (
                    <div class="bcx-widget__message-avatar">
                      <img
                        src={this.state.logo}
                        alt="Assistant Avatar"
                        class="bcx-widget__message-avatar-img"
                        onError={e => {
                          (e.target as HTMLImageElement).style.display = 'none';
                        }}
                      />
                    </div>
                  )}
                  <div class="bcx-widget__message-container">
                    <div class="bcx-widget__message-content">
                      {message.content && <div class="bcx-widget__message-text" innerHTML={this.parseLinks(message.content)}></div>}
                      {message.images && message.images.length > 0 && (
                        <div class="bcx-widget__message-images">
                          {message.images.map((image, index) => (
                            <div key={index} class="bcx-widget__message-image">
                              <img src={image} alt={`Image ${index + 1} in message`} class="bcx-widget__message-image-img" data-adblock-bypass="true" />
                            </div>
                          ))}
                        </div>
                      )}
                      {(() => {
                        const shouldRender = message.products && message.products.length > 0 && message.streamingFinished;

                        if (shouldRender) {
                          return <bcx-product-slider products={message.products} language={this.currentLanguage} showAfterStreaming={message.streamingFinished || false} />;
                        }
                        return null;
                      })()}
                    </div>
                    <div class="bcx-widget__message-time">{this.formatMessageTime(message.timestamp)}</div>
                  </div>
                </div>
              ))}
              {/* Example Questions - Modern Design */}
              {(this.state.messages.length === 0 || (this.state.messages.length === 1 && this.state.messages[0].id?.startsWith('welcome-'))) &&
                this.state.exampleQuestions &&
                this.state.exampleQuestions.length > 0 && (
                  <div
                    class={`bcx-widget__faqs ${!this.faqsHaveBeenShown ? 'bcx-widget__faqs--animate' : ''}`}
                    ref={el => {
                      if (el && !this.faqsHaveBeenShown) {
                        // Set flag after animation completes to prevent re-animation on rerenders
                        // Animation duration: 0.4s (container) + 0.5s (items) = ~600ms total
                        setTimeout(() => {
                          this.faqsHaveBeenShown = true;
                        }, 600);
                      }
                    }}
                  >
                    <div class="bcx-widget__faqs-header">
                      <svg
                        width="16"
                        height="16"
                        viewBox="0 0 24 24"
                        fill="none"
                        stroke="currentColor"
                        stroke-width="2"
                        stroke-linecap="round"
                        stroke-linejoin="round"
                        aria-hidden="true"
                      >
                        <circle cx="12" cy="12" r="10"></circle>
                        <path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"></path>
                        <line x1="12" y1="17" x2="12.01" y2="17"></line>
                      </svg>
                      <span class="bcx-widget__faqs-title">{this.getTranslation('common_questions')}</span>
                    </div>
                    <div class="bcx-widget__faqs-list">
                      {this.state.exampleQuestions.slice(0, 3).map((question, index) => (
                        <button
                          key={question.id || Math.random().toString(36)}
                          class="bcx-widget__faq-item"
                          onClick={() => this.handleExampleQuestionClick(question)}
                          role="button"
                          tabIndex={0}
                          aria-label={`Ask: ${question.question_text}`}
                          onKeyDown={e => {
                            if (e.key === 'Enter' || e.key === ' ') {
                              e.preventDefault();
                              this.handleExampleQuestionClick(question);
                            }
                          }}
                          style={{ animationDelay: `${index * 0.1}s` }}
                        >
                          <div class="bcx-widget__faq-item-content">
                            <div class="bcx-widget__faq-item-text">{question.question_text}</div>
                            <svg
                              class="bcx-widget__faq-item-icon"
                              width="16"
                              height="16"
                              viewBox="0 0 24 24"
                              fill="none"
                              stroke="currentColor"
                              stroke-width="2"
                              stroke-linecap="round"
                              stroke-linejoin="round"
                              aria-hidden="true"
                            >
                              <path d="M5 12h14"></path>
                              <path d="M12 5l7 7-7 7"></path>
                            </svg>
                          </div>
                        </button>
                      ))}
                    </div>
                  </div>
                )}

              {this.state.isTyping && (
                <div class="bcx-widget__message bcx-widget__message--assistant bcx-widget__message--typing">
                  {this.state.logo && (
                    <div class="bcx-widget__message-avatar">
                      <img
                        src={this.state.logo}
                        alt="Assistant Avatar"
                        class="bcx-widget__message-avatar-img"
                        onError={e => {
                          (e.target as HTMLImageElement).style.display = 'none';
                        }}
                      />
                    </div>
                  )}
                  <div class="bcx-widget__message-container">
                    <div class="bcx-widget__message-content">
                      <div class="bcx-widget__typing-indicator">
                        <span></span>
                        <span></span>
                        <span></span>
                      </div>
                    </div>
                  </div>
                </div>
              )}

              {/* Anchor element for scrollIntoView - inspired by Frontend ChatMessagesList */}
              <div ref={el => (this.messagesEndRef = el)} />
            </div>

            {/* Delay status message - shown above composer after 5 seconds */}
            {this.showDelayMessage && this.delayMessageType && this.state.isTyping && (
              <div class="bcx-widget__delay-status">
                <span class="bcx-widget__delay-status-text">
                  {this.delayMessageType === 'selecting' ? this.getTranslation('selecting_assistant') : this.getTranslation('thinking')}
                </span>
                <span class="bcx-widget__delay-status-dots">
                  <span class="bcx-widget__delay-dot"></span>
                  <span class="bcx-widget__delay-dot"></span>
                  <span class="bcx-widget__delay-dot"></span>
                </span>
              </div>
            )}

            <div class="bcx-widget__composer" data-adblock-bypass="true" role="form" aria-label="Message composer">
              <bcx-message-composer
                onMessageSubmit={this.handleMessageSubmit}
                disabled={false}
                loading={this.state.isTyping}
                placeholder={this.getTranslation('message_placeholder')}
                data-adblock-bypass="true"
                theme={this.themeService.getCurrentTheme()}
                isAttachmentsDisabled={this.isAttachmentsDisabled}
              />
            </div>

            {/* Powered by BetterCX - always show when flag is enabled */}
            {this.state.showPoweredByBetterCX && (
              <div class="bcx-widget__powered-by" data-adblock-bypass="true" aria-label="Powered by BetterCX">
                <span>{this.getTranslation('powered_by')}</span>
                <a
                  href="https://bettercx.ai/"
                  target="_blank"
                  rel="noopener noreferrer nofollow"
                  class="bcx-widget__powered-by-link"
                  data-adblock-bypass="true"
                  aria-label="Visit BetterCX website"
                >
                  BetterCX
                </a>
              </div>
            )}
          </div>
        )}
      </Host>
    );
  }

  private setupViewportHandling() {
    // Update viewport dimensions on load
    this.updateViewportDimensions();

    // Listen for viewport changes
    window.addEventListener('resize', this.handleViewportChange);
    window.addEventListener('orientationchange', this.handleViewportChange);

    // Listen for visual viewport changes (mobile browsers)
    if (window.visualViewport) {
      window.visualViewport.addEventListener('resize', this.handleViewportChange);
      window.visualViewport.addEventListener('scroll', this.handleViewportChange);
    }
  }

  private cleanupViewportHandling() {
    window.removeEventListener('resize', this.handleViewportChange);
    window.removeEventListener('orientationchange', this.handleViewportChange);

    if (window.visualViewport) {
      window.visualViewport.removeEventListener('resize', this.handleViewportChange);
      window.visualViewport.removeEventListener('scroll', this.handleViewportChange);
    }
  }

  private handleViewportChange = () => {
    // Use requestAnimationFrame for smooth updates without lag
    requestAnimationFrame(() => {
      this.updateViewportDimensions();
      // Reapply embedded styles on viewport change (mobile/desktop switch)
      if (this.embedded) {
        this.applyEmbeddedStyles();
      }
    });
  };

  private updateViewportDimensions() {
    if (window.visualViewport) {
      // Use visual viewport for mobile keyboards
      // We need to account for the visual viewport offset (scrolling)
      // and the height shrinking when keyboard opens
      this.el.style.setProperty('--bcx-viewport-height', `${window.visualViewport.height}px`);
      this.el.style.setProperty('--bcx-viewport-width', `${window.visualViewport.width}px`);
      this.el.style.setProperty('--bcx-viewport-top', `${window.visualViewport.offsetTop}px`);
      this.el.style.setProperty('--bcx-viewport-left', `${window.visualViewport.offsetLeft}px`);
    } else {
      // Fallback for desktop/older browsers
      this.el.style.setProperty('--bcx-viewport-height', `${window.innerHeight}px`);
      this.el.style.setProperty('--bcx-viewport-width', `${window.innerWidth}px`);
      this.el.style.setProperty('--bcx-viewport-top', '0px');
      this.el.style.setProperty('--bcx-viewport-left', '0px');
    }
  }

  /**
   * Select the appropriate trigger message based on current page URL
   * Matches URL patterns and falls back to default message only if origin matches
   */
  private selectTriggerMessage(
    triggerMessages: Array<{
      id: number;
      url: string;
      message: string;
      is_default: boolean;
      trigger_after_seconds: number;
      created_at: string;
      updated_at: string;
    }>,
  ):
    | {
        id: number;
        url: string;
        message: string;
        is_default: boolean;
        trigger_after_seconds: number;
        created_at: string;
        updated_at: string;
      }
    | undefined {
    if (!triggerMessages || triggerMessages.length === 0) {
      return undefined;
    }

    const currentPath = window.location.pathname;
    const currentOrigin = window.location.origin;

    // Check if current origin matches any trigger message origin
    let originMatches = false;
    const defaultMessage = triggerMessages.find(msg => msg.is_default);

    // First, try to find a match for the current full URL (exact match)
    for (const triggerMessage of triggerMessages) {
      if (triggerMessage.is_default) {
        continue; // Skip defaults in first pass
      }

      try {
        const triggerUrl = new URL(triggerMessage.url);
        const triggerOrigin = triggerUrl.origin;
        const triggerPath = triggerUrl.pathname;

        // Track if origin matches (for default fallback check)
        if (triggerOrigin === currentOrigin) {
          originMatches = true;

          // If trigger path is just "/" or empty, it matches homepage
          if (triggerPath === '/' || triggerPath === '') {
            if (currentPath === '/' || currentPath === '') {
              return triggerMessage;
            }
          } else {
            // Check if current path starts with trigger path (for subpage matching like "/cart" matching "/cart/checkout")
            if (currentPath === triggerPath || currentPath.startsWith(triggerPath + '/')) {
              return triggerMessage;
            }
          }
        }
      } catch {
        // Invalid URL format, skip this trigger message
        continue;
      }
    }

    // Check if default message origin matches current origin
    if (defaultMessage) {
      try {
        const defaultUrl = new URL(defaultMessage.url);
        const defaultOrigin = defaultUrl.origin;
        if (defaultOrigin === currentOrigin) {
          originMatches = true;
        }
      } catch {
        // Invalid URL format, skip
      }
    }

    // If no exact match found but origin matches, return the default message
    // If origin doesn't match (different website), return undefined
    if (originMatches && defaultMessage) {
      return defaultMessage;
    }

    return undefined;
  }

  private startPingMessageTimer(triggerMessage: {
    id: number;
    url: string;
    message: string;
    is_default: boolean;
    trigger_after_seconds: number;
    created_at: string;
    updated_at: string;
  }) {
    // Clear any existing timer
    this.clearPingMessageTimer();

    // Use trigger_after_seconds from the message (convert to milliseconds)
    const delayMs = triggerMessage.trigger_after_seconds * 1000;

    this.pingMessageTimeout = setTimeout(() => {
      // Only show ping message if chat is not open and not already shown
      if (!this.state.isOpen && !this.state.showPingMessage && this.state.selectedTriggerMessage) {
        this.setState({ showPingMessage: true });
      }
    }, delayMs);
  }

  private clearPingMessageTimer() {
    if (this.pingMessageTimeout) {
      clearTimeout(this.pingMessageTimeout);
      this.pingMessageTimeout = null;
    }
  }

  private handlePingMessageClose = () => {
    this.setState({ showPingMessage: false });
  };

  private handlePingMessageClick = () => {
    this.setState({ showPingMessage: false });
    this.open();
  };

  /**
   * Start interval to update message timestamps periodically
   * Updates every 30 seconds to keep timestamps fresh
   */
  private startTimeUpdateInterval() {
    // Clear any existing interval
    this.stopTimeUpdateInterval();

    // Update timestamps every 30 seconds
    this.timeUpdateInterval = setInterval(() => {
      // Trigger rerender by updating state
      this.timeUpdateTrigger = Date.now();
    }, 30000); // 30 seconds
  }

  /**
   * Stop the time update interval
   */
  private stopTimeUpdateInterval() {
    if (this.timeUpdateInterval) {
      clearInterval(this.timeUpdateInterval);
      this.timeUpdateInterval = null;
    }
  }

  /**
   * Check if device is desktop (not mobile/tablet)
   */
  private isDesktop(): boolean {
    return window.innerWidth >= 768 && !('ontouchstart' in window || navigator.maxTouchPoints > 0);
  }

  /**
   * Toggle dropdown menu
   */
  private toggleDropdown = (e: MouseEvent) => {
    e.stopPropagation();
    this.isDropdownOpen = !this.isDropdownOpen;
  };

  /**
   * Handle click outside dropdown to close it
   */
  private handleClickOutside = (e: MouseEvent) => {
    if (this.dropdownRef && !this.dropdownRef.contains(e.target as Node)) {
      this.isDropdownOpen = false;
    }
  };

  /**
   * Start a new conversation
   * Clears current chat state and session ID
   */
  private startNewConversation = () => {
    // Clear current chat ID from session
    this.authService.setChatId(null);

    // Add welcome message as first message if placement is 'message'
    const initialMessages: ChatMessage[] = [];
    if (this.state.welcomeMessage && this.state.welcomeMessagePlacement === 'message') {
      initialMessages.push({
        id: 'welcome-' + Date.now(),
        content: this.state.welcomeMessage.trim(),
        author: 'assistant',
        timestamp: new Date().toISOString(),
        streamingFinished: true,
      });
    }

    console.log({
      messages: initialMessages,
      chatId: undefined,
      isTyping: false,
      error: undefined,
    });

    // Update state
    this.setState({
      messages: initialMessages,
      chatId: undefined,
      isTyping: false,
      error: undefined,
    });

    // Close chat list explicitly
    this.showChatList = false;

    // Reset pending load tracker
    this.pendingLoadChatId = null;
  };

  /**
   * Handle dropdown menu item click
   */
  private handleDropdownItemClick = (action: string, e: MouseEvent) => {
    e.stopPropagation();
    this.isDropdownOpen = false;

    switch (action) {
      case 'new_conversation':
        this.startNewConversation();
        break;
      case 'fullscreen':
        this.handleFullscreen();
        break;
      case 'privacy':
        window.open('https://bettercx.ai/privacy', '_blank', 'noopener,noreferrer');
        break;
      case 'conversations':
        this.showChatList = true;
        break;
      case 'close':
        this.toggle();
        break;
    }
  };

  /**
   * Handle fullscreen toggle - makes widget fill the viewport
   */
  private handleFullscreen() {
    // Don't allow exiting fullscreen in embedded mode
    if (this.embedded) {
      return;
    }

    const widgetElement = this.el;

    if (!widgetElement.classList.contains('bcx-widget--fullscreen')) {
      // Enter fullscreen mode - expand widget to fill viewport
      widgetElement.classList.add('bcx-widget--fullscreen');
      this.isFullscreen = true;
    } else {
      // Exit fullscreen mode
      widgetElement.classList.remove('bcx-widget--fullscreen');
      this.isFullscreen = false;
    }
  }

  /**
   * Validate and normalize embedded size prop
   * Returns valid size or 'full' as fallback
   */
  private getValidEmbeddedSize(): 'full' | 'medium' | 'small' {
    const validSizes: Array<'full' | 'medium' | 'small'> = ['full', 'medium', 'small'];
    return validSizes.includes(this.embeddedSize) ? this.embeddedSize : 'full';
  }

  /**
   * Validate and normalize embedded placement prop
   * Returns valid placement or 'center' as fallback
   */
  private getValidEmbeddedPlacement(): 'top' | 'center' | 'bottom' {
    const validPlacements: Array<'top' | 'center' | 'bottom'> = ['top', 'center', 'bottom'];
    return validPlacements.includes(this.embeddedPlacement) ? this.embeddedPlacement : 'center';
  }

  /**
   * Apply embedded styles based on size and placement
   * Mobile always uses full screen regardless of size/placement
   */
  private applyEmbeddedStyles() {
    const widgetElement = this.el;

    // Validate props
    const validSize = this.getValidEmbeddedSize();
    const validPlacement = this.getValidEmbeddedPlacement();

    // Always add fullscreen base class for embedded mode
    widgetElement.classList.add('bcx-widget--fullscreen');

    // Remove any existing size/placement classes
    widgetElement.classList.remove(
      'bcx-widget--embedded-medium',
      'bcx-widget--embedded-small',
      'bcx-widget--embedded-top',
      'bcx-widget--embedded-center',
      'bcx-widget--embedded-bottom',
    );

    // On mobile, always use full screen (no size/placement classes)
    if (!this.isDesktop()) {
      return;
    }

    // Apply size classes only on desktop
    if (validSize !== 'full') {
      widgetElement.classList.add(`bcx-widget--embedded-${validSize}`);
    }

    // Apply placement classes only on desktop and when size is not full
    if (validSize !== 'full') {
      widgetElement.classList.add(`bcx-widget--embedded-${validPlacement}`);
    }
  }

  @Watch('embedded')
  async onEmbeddedChange() {
    if (this.embedded) {
      this.applyEmbeddedStyles();
      this.isFullscreen = true;
      this.setState({ isOpen: true });
    } else {
      // Remove embedded classes when disabled
      const widgetElement = this.el;
      widgetElement.classList.remove('bcx-widget--fullscreen');
      widgetElement.classList.remove(
        'bcx-widget--embedded-medium',
        'bcx-widget--embedded-small',
        'bcx-widget--embedded-top',
        'bcx-widget--embedded-center',
        'bcx-widget--embedded-bottom',
      );
      this.isFullscreen = false;
    }
  }

  @Watch('embeddedSize')
  async onEmbeddedSizeChange() {
    if (this.embedded) {
      this.applyEmbeddedStyles();
    }
  }

  @Watch('embeddedPlacement')
  async onEmbeddedPlacementChange() {
    if (this.embedded) {
      this.applyEmbeddedStyles();
    }
  }
}
