// Mock dependencies before imports
jest.mock('../../../services/theme.service');
jest.mock('../../../services/api.service');
jest.mock('../../../services/auth.service');

import { newSpecPage } from '@stencil/core/testing';
import { BetterCXWidget } from '../bettercx-widget';
import { ThemeService } from '../../../services/theme.service';
import { ApiService } from '../../../services/api.service';
import { AuthService } from '../../../services/auth.service';

describe('bettercx-widget', () => {
  let mockApiService: any;
  let mockThemeService: any;
  let mockAuthService: any;

  beforeEach(() => {
    jest.clearAllMocks();

    // Setup mocks
    mockAuthService = {
      createSession: jest.fn().mockResolvedValue({
        token: 'fake-token',
        attrs: { light_mode: {}, dark_mode: {} }
      }),
      getToken: jest.fn().mockReturnValue('fake-token'),
      setChatId: jest.fn(),
      getChatId: jest.fn(),
      getAuthHeader: jest.fn().mockReturnValue({})
    };

    mockApiService = {
      getChatMessages: jest.fn().mockResolvedValue({
        results: [],
        has_next: false
      }),
      sendMessage: jest.fn(),
      parseStreamResponse: jest.fn(),
    };

    mockThemeService = {
      detectWebsiteLanguage: jest.fn().mockResolvedValue('en'),
      setDefaultTheme: jest.fn(),
      getCurrentTheme: jest.fn().mockReturnValue('light'),
      setTheme: jest.fn(),
      watchWebsiteTheme: jest.fn().mockReturnValue(() => {}),
    };

    // Mock constructors
    (AuthService as jest.Mock).mockImplementation(() => mockAuthService);
    (ApiService as jest.Mock).mockImplementation(() => mockApiService);
    (ThemeService as jest.Mock).mockImplementation(() => mockThemeService);
  });

  afterEach(() => {
    jest.restoreAllMocks();
  });

  it('renders closed by default', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });

    expect(page.root).toHaveClass('bcx-widget');
    expect(page.root).not.toHaveClass('bcx-widget--open');

    const toggleBtn = page.root.shadowRoot.querySelector('.bcx-widget__toggle');
    expect(toggleBtn).not.toBeNull();
  });

  it('initializes correctly', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });

    await page.waitForChanges(); // Wait for async componentWillLoad/initialize

    expect(mockAuthService.createSession).toHaveBeenCalledWith('test-key', expect.any(String));
    expect(mockThemeService.detectWebsiteLanguage).toHaveBeenCalled();
  });

  it('toggles open/close on click', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });
    await page.waitForChanges();

    const widget = page.rootInstance as BetterCXWidget;
    // Force authenticated state
    widget['state'] = { ...widget['state'], isAuthenticated: true, isLoading: false };
    await page.waitForChanges();

    const toggleBtn = page.root.shadowRoot.querySelector('.bcx-widget__toggle') as HTMLElement;

    // Open
    toggleBtn.click();
    await page.waitForChanges();
    expect(page.root).toHaveClass('bcx-widget--open');

    // Close
    toggleBtn.click();
    await page.waitForChanges();
    expect(page.root).not.toHaveClass('bcx-widget--open');
  });

  it('loads chat safely (race condition test)', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });
    await page.waitForChanges();
    const widget = page.rootInstance as BetterCXWidget;

    // Mock API to return messages
    mockApiService.getChatMessages.mockResolvedValue({
      results: [
        { id: '1', content: 'Msg 1', author: 'user', created_at: '2023-01-01' }
      ]
    });

    // Call loadChat via internal method access (since it's private, we cast to any or check how it's exposed)
    // In Stencil tests, we can access private methods of rootInstance
    await (widget as any).loadChat('chat-1');

    expect(mockApiService.getChatMessages).toHaveBeenCalledWith('chat-1', 1, 20);
    expect(widget['state'].messages).toHaveLength(1);
    expect(widget['state'].messages[0].content).toBe('Msg 1');
    expect(widget['state'].chatId).toBe('chat-1');
  });

  it('handles stale responses in loadChat', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });
    await page.waitForChanges();
    const widget = page.rootInstance as BetterCXWidget;

    // Simulate slow response for chat-1
    let resolveChat1: any;
    const chat1Promise = new Promise(resolve => resolveChat1 = resolve);

    mockApiService.getChatMessages.mockImplementation((chatId) => {
      if (chatId === 'chat-1') return chat1Promise;
      return Promise.resolve({ results: [{ id: '2', content: 'Msg 2' }] });
    });

    // Start loading chat 1
    const load1 = (widget as any).loadChat('chat-1');

    // Immediately switch to chat 2
    await (widget as any).loadChat('chat-2');

    // Now resolve chat 1
    resolveChat1({ results: [{ id: '1', content: 'Msg 1' }] });
    await load1;

    // State should REFLECT CHAT 2 (Msg 2), NOT Msg 1
    // Because chat-1 response arrived AFTER we switched to chat-2
    expect(widget['state'].chatId).toBe('chat-2');
    expect(widget['state'].messages[0].content).toBe('Msg 2');
  });

  it('clears state immediately when switching chats', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });
    await page.waitForChanges();
    const widget = page.rootInstance as BetterCXWidget;

    // Set initial state
    widget['state'] = { ...widget['state'], messages: [{ id: 'old', content: 'old', author: 'user' } as any] };
    await page.waitForChanges();

    // Mock slow API
    mockApiService.getChatMessages.mockReturnValue(new Promise(() => {}));

    // Trigger load
    (widget as any).loadChat('new-chat');

    // Check if messages are cleared IMMEDIATELY
    expect(widget['state'].messages).toHaveLength(0);
    expect(widget['state'].isLoading).toBe(true);
  });

  it('starts new conversation correctly', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });
    await page.waitForChanges();
    const widget = page.rootInstance as BetterCXWidget;

    // Force some state
    widget['state'] = {
      ...widget['state'],
      chatId: 'old-chat',
      messages: [{ id: '1', content: 'old', author: 'user', timestamp: '2023' }],
      isAuthenticated: true
    };
    // Manually set private property
    Object.defineProperty(widget, 'showChatList', { value: true, writable: true });

    // Call startNewConversation (private)
    await (widget as any).startNewConversation();
    await page.waitForChanges();

    expect(widget['state'].messages).toHaveLength(0);
    expect(widget['state'].chatId).toBeUndefined();
    expect(widget['showChatList']).toBe(false);
    expect(mockAuthService.setChatId).toHaveBeenCalledWith(null);
  });

  it('hides chat list immediately when loading chat', async () => {
    const page = await newSpecPage({
      components: [BetterCXWidget],
      html: `<bettercx-widget public-key="test-key"></bettercx-widget>`,
    });
    await page.waitForChanges();
    const widget = page.rootInstance as BetterCXWidget;

    Object.defineProperty(widget, 'showChatList', { value: true, writable: true });

    // Mock API
    mockApiService.getChatMessages.mockResolvedValue({ results: [], has_next: false });

    // Load chat
    await (widget as any).loadChat('chat-1');

    // It should be hidden
    expect(widget['showChatList']).toBe(false);
  });
});
